/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>
#include "rwbits.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        7777
#define BUFFSZ      1024    // the max supported is 576

#define GS2_QUERY   "\xfe\xfd\x00" "\x00\x00\x00\x00"                    "\xff\x00\x00" "\x00"
#define GS3_QUERY   "\xfe\xfd\x09" "\x00\x00\x00\x00"
#define GS3_QUERYX  "\xfe\xfd\x00" "\x00\x00\x00\x00" "\x00\x00\x00\x00" "\xff\x00\x00" "\x00"



int fgetz(u8 *data, int datasz, FILE *fd);
int build_info(u8 *buff, int port, int type);
int unreal_build_pck(u8 *buff, int pck, ...);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



    /* unrealfp */



int     aafix           = 0,    // America's Army uses 0x800 instead of 0x3ff
        u3fix           = 0,    // Unreal 3 no longer uses index numbers
        rvfix           = 0,    // RavenShiel uses 0x50f instead of 0x3ff (this is useless since it's enough compatible with AA)
        pariahfix       = 0,    // Pariah
        movfix          = 0;    // Men of Valor



int read_unreal_index(u8 *index_num, int *ret) {
    int     len,
            result;
    u8      b0 = index_num[0],
            b1 = index_num[1],
            b2 = index_num[2],
            b3 = index_num[3],
            b4 = index_num[4];

    if(u3fix) {
        *ret = b0 | (b1 << 8) | (b2 << 16) | (b3 << 24);
        return(4);
    }

    result = 0;
    len    = 1;
    if(b0 & 0x40) {
        len++;
        if(b1 & 0x80) {
            len++;
            if(b2 & 0x80) {
                len++;
                if(b3 & 0x80) {
                    len++;
                    result = b4;
                }
                result = (result << 7) | (b3 & 0x7f);
            }
            result = (result << 7) | (b2 & 0x7f);
        }
        result = (result << 7) | (b1 & 0x7f);
    }
    result = (result << 6) | (b0 & 0x3f);
    if(b0 & 0x80) result = -result;
    *ret = result;
    return(len);
}



int write_unreal_index(int number, u8 *index_num) {
    int     len,
            sign = 1;

    if(u3fix) {
        index_num[0] = number & 0xff;
        index_num[1] = (number >> 8) & 0xff;
        index_num[2] = (number >> 16) & 0xff;
        index_num[3] = (number >> 24) & 0xff;
        return(4);
    }

    if(number < 0) {
        number = -number;
        sign = -1;
    }

    len = 1;
    index_num[0] = (number & 0x3f);
    if(number >>= 6) {
        index_num[0] += 0x40;
        index_num[1] = (number & 0x7f);
        len++;
        if(number >>= 7) {
            index_num[1] += 0x80;
            index_num[2] = (number & 0x7f);
            len++;
            if(number >>= 7) {
                index_num[2] += 0x80;
                index_num[3] = (number & 0x7f);
                len++;
                if(number >>= 7) {
                    index_num[3] += 0x80;
                    index_num[4] = number;
                    len++;
                }
            }
        }
    }
    if(sign < 0) index_num[0] += 0x80;
    return(len);
}



u32 read_bitx(u32 bits, u8 *in, u32 *in_bits) {
    u32     ret;

    ret = read_bits(bits, in, *in_bits);
    *in_bits += bits;
    return(ret);
}



int read_bitmem(u8 *in, int inlen, u8 *out, int bits) {
    for(; inlen--; out++) {
        *out = read_bitx(8, in, &bits);
    }
    return(bits);
}



int write_bitmem(u8 *in, int inlen, u8 *out, int bits) {
    for(; inlen--; in++) {
        bits = write_bits(*in, 8, out, bits);
    }
    return(bits);
}



int write_unrnum(int num, u8 *buff, int bits) {
    int     len;
    u8      mini[5];

    len = write_unreal_index(num, mini);
    return(write_bitmem(mini, len, buff, bits));
}



int read_unrnum(int *num, u8 *buff, int bits) {
    u8      mini[5];

    *num = 0;
    read_bitmem(buff, 5, mini, bits);
    return(bits + (read_unreal_index(mini, num) << 3));
}



int write_unrser(int num, u8 *buff, int bits, int max) {    // forcompability with core.dll
    int     b;

    for(b = 1; b && (b < max); b <<= 1) {
        bits = write_bits((num & b) ? 1 : 0, 1, buff, bits);
    }
    return(bits);
}



int read_unrser(int *num, u8 *buff, int bits, int max) {    // forcompability with core.dll
    int     b;

    *num = 0;
    for(b = 1; b && (b < max); b <<= 1) {
        if(read_bitx(1, buff, &bits)) *num += b;
    }
    return(bits);
}



int unreal_parse_pck(u8 *buff, int size) {
    int     b,
            btmp,
            blen,
            bsize,
            len,
            pck,
            val3ff,
            val8,
            val1000,
            n,
            n1,
            n2,
            n3,
            n4;
    u8      str[BUFFSZ];

    val8 = 0x8;
    if(pariahfix) val8 = 0x4;
    val3ff = 0x3ff;
    if(aafix) val3ff = 0x800;
    if(rvfix) val3ff = 0x50f;   // takes the same number of bits of AA... it's useless
    val1000 = 0x1000;
    if(movfix) val1000 = 0x1e00;

    if(size <= 0) return(-1);
    n = buff[size - 1];
    if(!n) return(-1);
    for(bsize = (size << 3) - 1; !(n & 0x80); n <<= 1, bsize--);

    b = read_unrser(&pck, buff, 0, 0x4000);
    while(b < bsize) {
        if(read_bitx(1, buff, &b)) {
            b = read_unrser(&n, buff, b, 0x4000);
            continue;
        }
        if(read_bitx(1, buff, &b)) {
            n1 = read_bitx(1, buff, &b);
            n2 = read_bitx(1, buff, &b);
        } else {
            n1 = n2 = 0;
        }
        n3 = read_bitx(1, buff, &b);
        b = read_unrser(&n, buff, b, val3ff);
        if(n3) b = read_unrser(&n, buff, b, 0x400);
        if(n1 || n3) {
            b = read_unrser(&n4, buff, b, val8);
        } else {
            n4 = 0;
        }

        b = read_unrser(&blen, buff, b, val1000);
        if((b + blen) > bsize) break;

        if(n4 != 1) {
            b += blen;
            continue;
        }

        for(btmp = b + blen; b < btmp;) {
            b = read_unrnum(&len, buff, b);
            if((len < 0) || (len > (sizeof(str) - 1))) break; 
            b = read_bitmem(buff, len, str, b);
            str[len] = 0;
            printf("  %s\n", str);
        }
        b = btmp;
    }
    return(size);
}



int unreal_send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int *chall, u8 **errmsg) {
    int     len;

    len = send_recv(sd, in, insz, out, outsz, peer, 0);
    if(len < 0) return(-1);
    if(!out) return(0);
    return(unreal_parse_pck(out, len));
}



void activate_fix(int *fix) {
    u8      *str    = NULL;

    aafix       = 0;
    u3fix       = 0;
    rvfix       = 0;
    pariahfix   = 0;
    movfix      = 0;
    if(!fix) {
        str = "full connect without";
    } else {
        *fix = 1;
        if(fix == &aafix)          str = "America's Army";
        else if(fix == &u3fix)     str = "Unreal 3 engine";
        else if(fix == &rvfix)     str = "Raven Shield";
        else if(fix == &pariahfix) str = "Pariah/Warpath";
        else if(fix == &movfix)    str = "Men of Valor";
    }
    printf("- activate the %s compatibility fixes\n", str);
}



    /* unrealfp */



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    FILE    *fd;
    int     i,
            sd,
            len,
            pck,
            psz,
            fix     = 0;
    u16     port    = PORT;
    u8      buff[BUFFSZ],
            line[BUFFSZ],
            *errmsg,
            *fname;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Unreal engine test server "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <port> <file_commands>\n"
            "\n"
            "Options:\n"
            "-x N    activation of a specific compatibility fix, where N is for:\n"
            "        1 = America's Army     2 = Unreal 3 engine    3 = Raven Shield\n"
            "        4 = Pariah/WarPath     5 = Men of Valor\n"
            "\n"
            "Note: only one connection at time is supported\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'x': fix   = atoi(argv[++i]);  break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    port  = atoi(argv[argc]);
    fname = argv[argc + 1];

    fd = fopen(fname, "rb");    // check if file exists
    if(!fd) std_err();
    fclose(fd);

    switch(fix) {
        case 1: activate_fix(&aafix);       break;
        case 2: activate_fix(&u3fix);       break;
        case 3: activate_fix(&rvfix);       break;
        case 4: activate_fix(&pariahfix);   break;
        case 5: activate_fix(&movfix);      break;
        default: break;
    }

    peer.sin_addr.s_addr  = INADDR_ANY;
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    if(bind(sd, (struct sockaddr *)&peer, sizeof(struct sockaddr_in))
      < 0) std_err();

    printf("- ready to receive packets\n");
    for(;;) {
        psz = sizeof(struct sockaddr_in);
        len = recvfrom(sd, buff, BUFFSZ, 0, (struct sockaddr *)&peer, &psz);
        if(len <= 0) continue;

        if(!memcmp(buff, "\\status\\", 8) || !memcmp(buff, "\\info\\", 6)) {
            len = build_info(buff, port, 1);
            send_recv(sd, buff, len, NULL, 0, &peer, 0);
            continue;
        } else if(!memcmp(buff, "\xfe\xfd\x09", 3)) {
            len = build_info(buff, port, 3);
            send_recv(sd, buff, len, NULL, 0, &peer, 0);
            continue;
        } else if(!memcmp(buff, "\xfe\xfd", 2)) {
            len = build_info(buff, port, 2);
            send_recv(sd, buff, len, NULL, 0, &peer, 0);
            continue;
        }

        printf("- %s:%hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));
        fd = fopen(fname, "rb");    // so the user can modify it everytime he wants
        if(!fd) std_err();

        for(pck = 0; fgetz(line, sizeof(line), fd) >= 0;) {  // quick lame way, this is only a test server
            if(!line[0]) continue;
            if((line[0] == '/') && (line[1] == '/')) continue;  // comments
            printf("> %s\n", line);
            len = unreal_build_pck(buff, pck++, line, NULL);
            len = unreal_send_recv(sd, buff, len, buff, BUFFSZ, &peer, NULL, &errmsg);
            if(len < 0) break;
        }
        for(;;) {
            len = unreal_send_recv(sd, NULL, 0, buff, BUFFSZ, &peer, NULL, &errmsg);
            if(len < 0) break;
        }

        fclose(fd);
        printf("- closed\n");
    }

    close(sd);
    return(0);
}



int fgetz(u8 *data, int datasz, FILE *fd) {
    u8      *p;

    if(!fgets(data, datasz, fd)) return(-1);
    for(p = data; *p && (*p != '\n') && (*p != '\r'); p++);
    *p = 0;
    return(p - data);
}



int build_info(u8 *buff, int port, int type) {  // useless, requires gamename and gamever
    int     i;
    u8      *p;

    #define UNREAL_MAP  "DM-HeatRay"
    static const u8 *parval[][2] = {
            { "hostname", "myhost" },
            //{ "gamename","" },
            //{ "gamever","" },
            { "hostport", NULL },
            { "gamemode", "openplaying" },
            { "mapname", UNREAL_MAP },
            { "gametype", "xDeathMatch" },
            //{ "OwningPlayerId", "1234" },
            //{ "NumPublicConnections", "32" },
            //{ "bUsesStats", "True" },
            //{ "bIsDedicated", "True" },
            //{ "OwningPlayerName", "myplayer" },
            //{ "AverageSkillRating", "1000.000000" },
            //{ "EngineVersion", "3601" },
            //{ "MinNetVersion", "3467" },
            { "numplayers", "0" },
            { "numteams", "1" },
            { "maxplayers", "16" },
            { "teamplay", "0" },
            { "fraglimit", "30" },
            { "timelimit", "20" },
            { "password", "0" },
            //{ "s32779", "0" },
            //{ "s0", "2" },
            //{ "s1", "0" },
            //{ "s6", "0" },
            //{ "s7", "0" },
            //{ "s8", "0" },
            //{ "s9", "0" },
            //{ "s10", "0" },
            //{ "s11", "0" },
            //{ "s12", "0" },
            //{ "s13", "1" },
            //{ "s14", "1" },
            //{ "p1073741825", UNREAL_MAP },
            //{ "p1073741826", "UTGame.UTDeathmatch" },
            //{ "p268435704", "30" },
            //{ "p268435705", "15" },
            //{ "p268435703", "1" },
            //{ "p1073741827", "" },
            //{ "p268435717", "0" },
            { NULL, NULL } };

    printf("- query type %d\n", type);

    p = buff;
    if(type == 3) {
        *p++ = 9;
        *p++ = 0; *p++ = 0; *p++ = 0; *p++ = 0;
        p += sprintf(p, "%d", 0) + 1;
        return(p - buff);
    } else if(type == 2) {
        *p++ = 0;
        *p++ = 0; *p++ = 0; *p++ = 0; *p++ = 0;
    } else if(type == 1) {
    }
    for(i = 0; parval[i][0]; i++) {
        if(type == 1) *p++ = '\\';
        strcpy(p, parval[i][0]);
        p += strlen(p);
        *p++ = (type == 1) ? '\\' : 0;
        if(parval[i][1]) {
            strcpy(p, parval[i][1]);
        } else {
            sprintf(p, "%hu", port);
        }
        p += strlen(p);
        if(type == 2) *p++ = 0;
    }
    if(type == 1) {
        p += sprintf(p, "\\final\\\\");
    } else {
        *p++ = 0;
    }
    return(p - buff);
}



int unreal_build_pck(u8 *buff, int pck, ...) {
    va_list ap;
    int     b,
            sl,
            len,
            bpos,
            bsize,
            val3ff,
            val8,
            val1000;
    u8      *s;

        //devastation is not supported, it uses 0x4000 1 1 0x4000 1 1 1 1 0x3ff 0x1000 

    val8 = 0x8;
    if(pariahfix) val8 = 0x4;
    val3ff = 0x3ff;
    if(aafix) val3ff = 0x800;
    if(rvfix) val3ff = 0x50f;   // takes the same number of bits of AA... it's useless
    val1000 = 0x1000;
    if(movfix) val1000 = 0x1e00;

    b = 0;
    b = write_unrser(pck, buff, b, 0x4000);
    b = write_bits(0, 1, buff, b);
    if(!pck) {
        b = write_bits(1, 1, buff, b);  // 1
        b = write_bits(1, 1, buff, b);
        b = write_bits(0, 1, buff, b);
    } else {
        b = write_bits(0, 1, buff, b);  // 0
    }
    if(!pck) {
        b = write_bits(1, 1, buff, b);  // 1
        b = write_unrser(0, buff, b, val3ff);
        b = write_unrser(pck + 1, buff, b, 0x400);
        b = write_unrser(1, buff, b, val8);
    } else {
        b = write_bits(0, 1, buff, b);  // 0
        b = write_unrser(0, buff, b, val3ff);
    }

    bpos = b;
    b = write_unrser(0, buff, b, val1000);
    bsize = b;
    va_start(ap, pck);
    while((s = va_arg(ap, u8 *))) {
        sl = strlen(s) + 1;
        if(sl == 1) continue;               // skip empty
        b = write_unrnum(sl, buff, b);
        b = write_bitmem(s, sl, buff, b);   // in reality they are index numbers
    }
    va_end(ap);
    write_unrser(b - bsize, buff, bpos, val1000);

    write_bits(0, 8, buff, b);  // zero pad
    len = b >> 3;
    if(!(buff[len] & 0x80)) buff[len] |= (1 << (b & 7));
    len++;
    return(len);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err) {
    int     retry = 2,
            len;

    if(in) {
        while(retry--) {
            //fputc('.', stdout);
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) goto quit;
            if(!out) return(0);
            if(!timeout(sd, 2)) break;
        }
    } else {
        if(timeout(sd, 3) < 0) retry = -1;
    }

    if(retry < 0) {
        if(!err) return(-2);
        printf("\nError: socket timeout, no reply received\n\n");
        exit(1);
    }

    //fputc('.', stdout);
    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) goto quit;
    return(len);
quit:
    if(err) std_err();
    return(-1);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


