/* Sample usage:

	struct ftp_conn* ftp;
	unsigned char* buf;
	int len;

	host = resolve_host(hostname);
	ftp = ftp_connect_host(host, port, verbose);
	ftp_login(ftp, username, password);
	ftp_cwd(directory);
	ftp_put(ftp, "filename", FTP_BINARY, buf, len);
	ftp_get(ftp, "filename", FTP_ASCII, &buf);
	ftp_close(ftp);

*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "system.h"
#include "ftp.h"

extern int errno;

/* Resolve a hostname */
in_addr_t resolve_host(char* host)
{
	struct hostent* hp;
	in_addr_t addr;
	int i;
	
	if ((addr = inet_addr(host)) == INADDR_NONE) {

		if (!(hp = gethostbyname(host))) {
			printf("Unable to resolve address %s\n", host);
			exit(1);
		}

		/* we can't handle more than one ip address */
		if (hp->h_addr_list[1]) {
			printf("%s resolves to multiple IP addresses, please select one of them\n", host);
			for (i=0; hp->h_addr_list[i]; i++)
				printf("%s\n", inet_ntoa(*(struct in_addr*)&hp->h_addr_list[i]));
			exit(1);
		}
	
		addr = *(in_addr_t*)hp->h_addr_list[0];
	}

	return addr;
}

/* Connect to a host */
int connect_host(in_addr_t host, int port)
{
	struct sockaddr_in s_in;
	int sock;

	s_in.sin_family = AF_INET;
	s_in.sin_addr.s_addr = host;
	s_in.sin_port = htons(port);

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) <= 0) {
		printf("Could not create a socket\n");
		exit(1);
	}

	if (connect(sock, (struct sockaddr *)&s_in, sizeof(s_in)) < 0) {
		printf("Connection failed: %s\n", strerror(errno));
		exit(1);
	}

	return sock;
}

/* Create a new ftp_conn structure and connect to a host */
struct ftp_conn* ftp_connect_host(in_addr_t host, int port, int verbose)
{
	struct ftp_conn* ftp;

	ftp = (struct ftp_conn*) xmalloc(sizeof(struct ftp_conn));

	/* Initialize the structure */
	ftp->verbose = verbose;
	ftp->reply_code = 0;
	ftp->reply = NULL;

	ftp->host = host;
	ftp->data = 0;

	ftp->sock = connect_host(host, port);

	if (ftp_read_reply(ftp) != 220) {
		printf("Error in server reply: %s\n", ftp->reply);
		exit(1);
	}

	return ftp;
}

/* Closes the socket and destroys the ftp structure */
void ftp_close(struct ftp_conn* ftp)
{
	ftp_send_cmd(ftp, "QUIT\r\n");
	close(ftp->sock);
	
	if (ftp->reply) free(ftp->reply);
	free(ftp);
}

/* Read len bytes from a socket. Returns 0 if the connection is closed */
int read_data(int sock, unsigned char* buf, int len)
{
	int l;
	int to_read = len;

	do {
		if ((l = read(sock, buf, to_read)) < 0) {
			printf("Error in read: %s\n", strerror(errno));
			exit(1);
		}
		else if (!l) {
			return 0;
		}
		to_read -= l;
	} while (to_read > 0);

	return len;
}

/* Read data until EOF */
int read_data_eof(int sock, unsigned char** buf)
{
	int l;
	int len = 0;
	int bufsize = 0;

	*buf = NULL;

	do {
		if (bufsize - len < 1024) {
			*buf = xrealloc(*buf, bufsize + 8192);
			bufsize += 8192;
		}
		
		if ((l = read(sock, *buf + len, bufsize-len)) < 0) {
			printf("Error in read: %s\n", strerror(errno));
			exit(1);
		}

		len += l;
		
	} while (l > 0);

	return len;
}

/* Read a line terminated by '\n' from a socket. Returns a pointer to a
   buffer allocated with malloc(), or NULL if the connection is closed */
char* read_line(int sock)
{
	char buf[1024];
	int l;

	for (l=0; l < 1023; l++) {
		if (!read_data(sock, &buf[l], 1))
			return NULL;

		if (buf[l] == '\n') {
			buf[l+1] = '\0';
			return xstrdup(buf);
		}
	}

	buf[l+1] = '\0';

	printf("Server reply is longer than %d characters.\n", l+1);
	printf("%s\n", buf);

	exit(1);

	return NULL;
}


/* Read a server reply. Stores the reply in ftp->reply_code and ftp->reply.
   Returns the reply code */

int ftp_read_reply(struct ftp_conn* ftp)
{
	char* reply;
	char code[4];

	if (!(reply = read_line(ftp->sock))) {
		printf("Connection closed while reading server reply\n");
		exit(1);
	}

	if ( (strlen(reply) < 4) ||
	     !(ISDIGIT(reply[0]) && ISDIGIT(reply[1]) && ISDIGIT(reply[2])) ||
	     !(reply[3] == ' ' || reply[3] == '-')) {
		printf("Server reply is not valid:\n");
		printf("%s", reply);
		exit(1);
	}

	code[0] = reply[0];
	code[1] = reply[1];
	code[2] = reply[2];

	code[3] = ' ';
	
	while (strncmp(reply, code, 4)) {
		/* free the previous line */
		free(reply);

		/* read the next line of a multiline reply */
		if (!(reply = read_line(ftp->sock))) {
			printf("Connection closed while reading server reply\n");
			exit(1);
		}
	}

	if (ftp->verbose) printf("    %s", reply);

	ftp->reply = reply;
	ftp->reply_code = atoi(code);

	return ftp->reply_code;
}

/* Write to a socket */
void write_data(int sock, unsigned char* buf, int len)
{
	if (send(sock, buf, len, 0) == -1) {
		printf("Error in send: %s\n", strerror(errno));
		exit(1);
	}
}

/* Write a string to a socket */
void ftp_send_cmd(struct ftp_conn* ftp, char* format, ...)
{
	char str[1024];
	va_list args;

	va_start(args, format);
	if (vsnprintf(str, 1023, format, args) >= 1023) {
		printf("Command longer than 1024 characters:\n%s\n", str);
		exit(1);
	}
	str[1023] = '\0';
	va_end(args);

	if (ftp->verbose) printf("    %s", str);

	write_data(ftp->sock, str, strlen(str));
}


/* Log in using a username and password */
void ftp_login(struct ftp_conn* ftp, char* username, char* password)
{
	ftp_send_cmd(ftp, "USER %s\r\n", username);
	if (ftp_read_reply(ftp) != 331) {
		printf("Error in server reply: %s\n", ftp->reply);
		exit(1);
	}
	
	ftp_send_cmd(ftp, "PASS %s\r\n", password);
	if (ftp_read_reply(ftp) != 230) {
		printf("Error in server reply: %s\n", ftp->reply);
		exit(1);
	}
}

void ftp_cwd(struct ftp_conn* ftp, char* directory)
{
	ftp_send_cmd(ftp, "CWD %s\r\n", directory);
	if (ftp_read_reply(ftp) != 250) {
		printf("Error in server reply: %s\n", ftp->reply);
		exit(1);
	}
}

void ftp_del(struct ftp_conn* ftp, char* filename)
{
	ftp_send_cmd(ftp, "DELE %s\r\n", filename);
	ftp_read_reply(ftp);
}

void ftp_passive(struct ftp_conn* ftp)
{
	int port, p1, p2;

	ftp_send_cmd(ftp, "PASV\r\n");
	if (ftp_read_reply(ftp) != 227) {
		printf("Error in server reply: %s", ftp->reply);
		exit(1);
	}		
	
	if (sscanf(ftp->reply, "227 Entering Passive Mode (%*d,%*d,%*d,%*d,%d,%d)", &p1, &p2) != 2) {
		printf("Error decoding PASV response, the server is not running ProFTPd.\n%s\n", ftp->reply);
		exit(1);
	}
	port = (p1 & 0xff) * 256 + (p2 & 0xff);

	ftp->data = connect_host(ftp->host, port);
}

void ftp_put(struct ftp_conn* ftp, char* filename, int mode, unsigned char* buf, int len)
{
	if (mode == FTP_ASCII)
		ftp_send_cmd(ftp, "TYPE A\r\n");
	else
		ftp_send_cmd(ftp, "TYPE I\r\n");

	if (ftp_read_reply(ftp) != 200) {
		printf("Error in server reply: %s\n", ftp->reply);
		exit(1);
	}

	ftp_passive(ftp);
	
	ftp_send_cmd(ftp, "STOR %s\r\n", filename);
	if (ftp_read_reply(ftp) != 150) {
		printf("Error in server reply: %s", ftp->reply);
		exit(1);
	}

	write_data(ftp->data, buf, len);
	close(ftp->data);

	ftp->data = 0;

	if (ftp_read_reply(ftp) != 226) {
		printf("Error in server reply: %s", ftp->reply);
		exit(1);
	}
}

int ftp_get(struct ftp_conn* ftp, char* filename, int mode, unsigned char** buf)
{
	int len;
	
	if (mode == FTP_ASCII)
		ftp_send_cmd(ftp, "TYPE A\r\n");
	else
		ftp_send_cmd(ftp, "TYPE I\r\n");

	if (ftp_read_reply(ftp) != 200) {
		printf("Error in server reply: %s\n", ftp->reply);
		exit(1);
	}

	ftp_passive(ftp);
	
	ftp_send_cmd(ftp, "RETR %s\r\n", filename);
	if (ftp_read_reply(ftp) != 150) {
		printf("Error in server reply: %s", ftp->reply);
		exit(1);
	}

	len = read_data_eof(ftp->data, buf);
	if (ftp->verbose) printf("    %d bytes read\n", len);

	close(ftp->data);
	ftp->data = 0;

/* server is either exploited or crashed, so don't read the reply */

/*	if (ftp_read_reply(ftp) != 226) {
		printf("Error in server reply: %s", ftp->reply);
		exit(1);
	}
*/
	return len;
}

/* EOF */
