//=============================================================================

// File: "GambitREPL.js"

// Copyright (c) 2014-2015 by Marc Feeley, All Rights Reserved.

//=============================================================================

// Filesystem implementation for GambitREPL app.

var sent_requests = {};
var sent_requests_counter = 0;

function sendRequest(request, cont) {
  sent_requests[request.requestId = ++sent_requests_counter] = cont;
  window.location = encodeURI("event:" + encodeURIComponent(JSON.stringify(request)));
}

function receiveResponse(requestId, response) {
  var cont = sent_requests[requestId];
  if (cont !== undefined) {
    delete sent_requests[requestId];
    cont(response);
  }
}

function fs_fileType(path, cont) {
  sendRequest({method:"fileType", path:path},
              cont);
}

Ymacs.prototype.fs_fileType = function (path, cont) {
  fs_fileType(path, cont);
};

function fs_getDirectory(path, cont) {
  sendRequest({method:"getDirectory", path:path},
              cont);
}

Ymacs.prototype.fs_getDirectory = function (path, cont) {
  fs_getDirectory(path, cont);
};

function fs_getFileContents(path, nothrow, cont) {
  sendRequest({method:"getFileContents", path:path, nothrow:nothrow},
              function (response) {
                cont(response[0], response[1]);
              });
}

Ymacs.prototype.fs_getFileContents = function (path, nothrow, cont) {
  fs_getFileContents(path, nothrow, cont);
};

function fs_setFileContents(path, content, stamp, cont) {
  sendRequest({method:"setFileContents", path:path, content:content, stamp:stamp},
              cont);
}

Ymacs.prototype.fs_setFileContents = function (path, content, stamp, cont) {
  fs_setFileContents(path, content, stamp, cont);
};

function fs_deleteFile(path, cont) {
  sendRequest({method:"deleteFile", path:path},
              function (response) {
                cont();
              });
}

Ymacs.prototype.fs_deleteFile = function (path, cont) {
  fs_deleteFile(path, cont);
};

function fs_remapDir(path, cont) {
  sendRequest({method:"remapDir", path:path},
              cont);
}

Ymacs.prototype.fs_remapDir = function (path, cont) {
  fs_remapDir(path, cont);
};

function killTerminal(cont) {
  sendRequest({method:"killTerminal"},
              function (response) {
                cont();
              });
}

Ymacs.prototype.killTerminal = function (cont) {
  killTerminal(cont);
};

function resetKeyboard(cont) {
  sendRequest({method:"resetKeyboard"},
              function (response) {
                cont();
              });
}

Ymacs.prototype.resetKeyboard = function (cont) {
  resetKeyboard(cont);
};

var process_table = {};

function makeProcess(process, cont) {
  sendRequest({method:"makeProcess", process:process},
              cont);
}

function startProcess(processId, cont) {
  sendRequest({method:"startProcess", processId:processId},
              cont);
}

function killProcess(processId, cont) {
  sendRequest({method:"killProcess", processId:processId},
              cont);
}

function sendProcessInterrupt(processId, cont) {
  sendRequest({method:"sendProcessInterrupt", processId:processId},
              function (response) {
                cont();
              });
}

function sendProcessInput(processId, input, cont) {
  sendRequest({method:"sendProcessInput", processId:processId, input:input},
              function (response) {
                cont();
              });
}

function sendMail(target, subject, body, html_body, cont) {
  window.location = "mailto:" + encodeURIComponent(target) +
                    "?subject=" + encodeURIComponent(subject) +
                    "&body=" + encodeURIComponent(html_body);
  cont();
}

function finishSetupEmacs(splash, cont) {
  sendRequest({method:"finishSetupEmacs", splash:splash},
              function (response) {
                cont();
              });
}

//-----------------------------------------------------------------------------

var iCloudStatus = undefined;

Ymacs_Buffer.setq('modeline_custom_handler', function () {
  return iCloudStatus;
});

function iCloudStatus_set(status) {
  iCloudStatus = status;
  ymacs.frames.forEach(function (f) {
    f.redrawModeline();
  });
}

//-----------------------------------------------------------------------------

// Handling of keypress generated by GambitREPL app.

function dispatch_event(ev) {
  DlEvent._genericEventHandler(ev);
}

function add_text_input(input) {
  for (var i=0; i<input.length; i++)
    add_key_input(input[i]);
}

var prefix_key = null;

function add_key_input(input) {

  if (prefix_key === 'CTRL') {
    prefix_key = null;
    if (input === 'CTRL') {
      input = '\u001b'; // CTRL twice = escape
    } else if (input.length === 1) {
        if (input >= 'a' && input <= 'z') {
          input = String.fromCharCode(input.charCodeAt(0)-96);
        } else if (input >= '@' && input <= '_') {
          input = String.fromCharCode(input.charCodeAt(0)-64);
        } else if (input === ' ') {
          input = '^ '; // ctrl-space
        } else if (input === '\u0009') {
          input = '^\u0009'; // ctrl-tab
        }
    } else if (input === 'AR') { // right arrow
        input = '^\u0009'; // ctrl-tab
    } else if (input === 'AL') { // left arrow
        input = 'S^\u0009'; // shift-ctrl-tab
    }
  } else if (input === 'CTRL') {
    prefix_key = input;
    return;
  }

  var ev = { type: 'keypress',
             keyCode: 0,
             charCode: 0,
             ctrlKey: false,
             altKey: false,
             shiftKey: false
           };

  function send_keycode(keyCode) {
    ev.type = 'keydown';
    ev.keyCode = keyCode;
    ev.which = keyCode;
    dispatch_event(ev);
  }

  function send_charcode(charCode) {
    ev.keyCode = ev.charCode = charCode;
    dispatch_event(ev);
  }

  if (input.length > 1) {
    if (input[0] === 'F') {
      send_keycode(111+(+input.slice(1))); // function keys
      return;
    }
    if (input[0] === 'S') {
      ev.shiftKey = true;
      input = input.slice(1);
    }
    if (input.length > 1) {
      if (input[0] === 'M') {
        ev.altKey = true;
        input = input.slice(1);
      }
      if (input.length > 1) {
        if (input[0] === '^') {
          ev.ctrlKey = true;
          input = input.slice(1);
        }
      }
    }
  }

  //BACKSPACE:8,TAB:9,ENTER:13,ESCAPE:27,SPACE:32,DASH:45,PAGE_UP:33,PAGE_DOWN:34,END:35,HOME:36,ARROW_LEFT:37,ARROW_UP:38,ARROW_RIGHT:39,ARROW_DOWN:40,INSERT:45,DELETE:46,F1:112,F2:113,F3:114,F4:115,F5:116,F6:117,F7:118,F8:119,F9:120,F10:121,F11:122,F12:123

  if (input.length === 2) {
    var alt = ev.altKey;
    ev.altKey = false;
    if (input === 'AL') {
      if (!alt)
        input = '\u0002';
      else
        input = '\u0001';
    } else if (input === 'AU') {
      if (!alt)
        input = '\u0010';
      else {
        ev.ctrlKey = true;
        send_keycode(36); // HOME
      }
    } else if (input === 'AR') {
      if (!alt)
        input = '\u0006';
      else
        input = '\u0005';
    } else if (input === 'AD') {
      if (!alt)
        input = '\u000e';
      else {
        ev.ctrlKey = true;
        send_keycode(35); // END
      }
    }
  }

  if (input.length === 1) {

    var code = input.charCodeAt(0);

    if (code >= 32) {
      if (code === 127)
        send_keycode(8); // delete -> backspace
      else
        send_charcode(code);
    } else if (code === 10) {
      send_keycode(13); // return
    } else if (code === 9 || code === 13 || code === 27) {
      send_keycode(code); // tab, return, escape
    } else {
      ev.ctrlKey = true;
      send_charcode(code+64); // some other ctrl character
    }
  }
}

//-----------------------------------------------------------------------------

// Handling of touch events to simulate mouse events.

function handle_touch_event(e) {

  e.preventDefault();

  if (e.touches.length > 1 ||
      (e.type == 'touchend' && e.touches.length > 0))
    return;

  var type = null;
  var t = null;

  switch (e.type) {
    case 'touchstart': type = 'mousedown'; t = e.changedTouches[0]; break;
    case 'touchmove':  type = 'mousemove'; t = e.changedTouches[0]; break;
    case 'touchend':   type = 'mouseup';   t = e.changedTouches[0]; break;
  }

  var ev = { type: type,
             which: e.which,
             keyIdentifier: e.keyIdentifier,
             shiftKey: e.shiftKey,
             altKey: e.altKey,
             ctrlKey: e.ctrlKey,
             button: 0,
             screenX: t.screenX,
             screenY: t.screenY,
             clientX: t.clientX,
             clientY: t.clientY,
             target: e.target
           };

  dispatch_event(ev);
}

window.addEventListener('touchstart', handle_touch_event, false);
window.addEventListener('touchmove', handle_touch_event, false);
window.addEventListener('touchend', handle_touch_event, false);

//=============================================================================
