/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
 *  \file pdl_main.c
 *
 *  \brief All functions that do not fit elsewhere can be found here.
 *
 *  In here one can find the more general functions. Most of them are
 *  accessible to outside sources. For a complete list of usable function
 *  to out side sources, \see pdl.h.
 *
 *
 *  \author  G.M. Venekamp  (venekamp@nikhef.nl)
 *  \version $Revision: 18217 $
 *  \date    $Date: 2015-02-02 12:00:49 +0100 (Mon, 02 Feb 2015) $
 *
 */

#define _XOPEN_SOURCE	500

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "lcmaps_log.h"

#include "evaluationmanager/pdl.h"
#include "evaluationmanager/pdl_variable.h"
#include "evaluationmanager/pdl_policy.h"
#include "evaluationmanager/pdl_rule.h"


#define WARNBUFLEN    ((size_t)2048L)


static char* script_name        = NULL;           /* If non NULL, the name of the configuration script. */
static char* path               = NULL;           /* Path where plugins can be found. */
static const char* undefined	= "(empty string)"; /*String printed instead of (null) */
static int path_lineno          = 0;              /* Path line number. */
static plugin_t* top_plugin     = NULL;           /* First node of the list. */
static BOOL parse_error         = FALSE;          /* Tell if there have been any error during parsing. */
static const char* level_str[PDL_SAME];                 /* When a message is printed, how do we spell warning in a given language. */

static       policy_t* current_policy = NULL;
static const rule_t*   current_rule   = NULL;

int lineno = 1;   /* The first line of a configuration sctipt is labeled 1. */

static void _lcmaps_set_path(const record_t* _path);
static record_t* _lcmaps_concat_strings(const record_t* s1, const record_t* s2, const char* extra);
/* void lcmaps_reduce_policies(void); */
static int lcmaps_init_name_args(plugin_t** plugin, const rule_t* rule, rule_type_t type);
static BOOL lcmaps_plugin_exists(const char* string);
static size_t lcmaps_find_first_space(const char* string);
static int lcmaps_free_plugins(plugin_t ** list);

/*!
 *  Return the current policy.
 *
 */
policy_t * lcmaps_get_current_policy(void)
{
  return current_policy;
}

/*!
 *  Init the pdl engine. The function takes one argument, the name
 *  of a configuration file to use.
 *
 *  \param name Name of the configuration file to use.
 *  \return 0 in case the initialization is successful; -1 in case of
 *            not being successful.
 *
 */
int lcmaps_pdl_init(const char* name)
{
  /* Make sure at least these variables are ALWAYS defined, so start with that.
   * Otherwise we get a problem when printing a warning.
   * This code SHOULD be rewritten for clarity reasons. */
  /*  Init the levels with the appropriate print text.  */
  level_str[PDL_INFO]    = "info";
  level_str[PDL_WARNING] = "warning";
  level_str[PDL_ERROR]   = "error";
  level_str[PDL_UNKNOWN] = "<unknown>";

  /* Initialize lineno to 1, a second run will not have it initialized */
  lineno=1;

  if (name) {
    FILE* file;

    script_name = strdup(name);
    if (script_name==NULL)  {
	lcmaps_pdl_warning(PDL_ERROR, "Out of memory when trying to open '%s'.", name);
	return -1;
    }

    file = fopen(name, "r");
    if (file)
      /* set_active_yyin (file); */
      yyin = file;
    else {
      lcmaps_pdl_warning(PDL_ERROR, "Could not open file '%s'.", name);
      return -1;
    }
  }

  /*  Set the default path  */
  path = NULL;

  /*
   *  Check if there are plugins, if so clear top_plugin and free
   *  all allocated memory.
   */
  if (top_plugin) lcmaps_free_plugins(&top_plugin);
  /* while (top_plugin) {
    if (top_plugin->name) { free(top_plugin->name); }
    if (top_plugin->args) { free(top_plugin->args); }
    top_plugin = top_plugin->next;
  } */

  parse_error = FALSE;

  return 0;
}


/*!
 *  Tell if there were errors/warning during parsing.
 *
 *  \return 0, if the are no errors/warnings, -1 otherwise.
 *
 */
int yyparse_errors(void)
{
  return parse_error ? -1 : 0;
}


/*!
 *  Get a list of plugins as known by the configuration file.
 *
 *  \return Plugin list (linked list).
 *
 */
const plugin_t* lcmaps_get_plugins(void)
{
  plugin_t* plugin;
  policy_t* policy;

  /*
   *  Check if the policies have been reduced. If not, then getting
   *  the plugins and their arguments is useless; the variables have
   *  not been substituted yet.
   */
  if (!lcmaps_policies_have_been_reduced()) {
    lcmaps_log(LOG_ERR, "The policies have not been reduced. Probably the startElevaluationManager has failed or has not been called yet.\n");
    return 0;
  }

  /*
   *  Check if evaluation has been done before, if so use the old
   *  results.
   */
  if (top_plugin)
    return top_plugin;

  policy = lcmaps_get_policies();

  while (policy) {
    rule_t* rule = policy->rule;

    lcmaps_log_debug(2, "processing policy: %s\n", policy->name);
    while (rule) {
      lcmaps_log_debug(4, "  processing rule: %s -> %s | %s\n",
	      rule->state ? rule->state : undefined,
	      rule->true_branch ? rule->true_branch : undefined,
	      rule->false_branch ? rule->false_branch : undefined);

      lcmaps_log_debug(4, "  get_plugins:  initializing...\n");
      if (lcmaps_init_name_args(&plugin, rule, STATE)<0 ||
          lcmaps_init_name_args(&plugin, rule, TRUE_BRANCH)<0 ||
          lcmaps_init_name_args(&plugin, rule, FALSE_BRANCH)<0)	{
	      lcmaps_log(LOG_ERR,"Error initializing plugins.\n");
	      return NULL;
      }
      lcmaps_log_debug(4, "  get_plugins:  initializing done.\n");

      rule = rule->next;
    }
    policy = policy->next;
  }

  return top_plugin;
}

/*!
 *  Free the list of plugins to which "list" points.
 *  Also the pointer to the first plugin is set to NULL.
 *
 *  \param list Pointer to the plugin list.
 *
 *  \retval 0 success
 *  \retval 1 failure
 *
 */
static int lcmaps_free_plugins(
        plugin_t ** list
)
{
    plugin_t * p_list     = NULL;
    plugin_t * tmp_p_list = NULL;

    p_list = *list;
    while (p_list) {
        tmp_p_list = p_list->next;

        lcmaps_log_debug(5, "freeing plugin %s at address %p\n", p_list->name, (void*)p_list);
        if (p_list->name){
            free(p_list->name);
            p_list->name = NULL;
        }
        if (p_list->args){
            free(p_list->args);
            p_list->args = NULL;
        }
        p_list->next = NULL;
        free((plugin_t*)p_list);

        p_list = tmp_p_list;
    }
    *list=p_list=NULL;
    return 0;
}

/*!
 *  Check if a plugin as specified by the string argument exists.
 *
 *  \param string Name of the plugin.
 *  \return TRUE if the plugin exists, FALSE otherwise.
 *
 */
static BOOL lcmaps_plugin_exists(const char* string)
{
  size_t space,string_len;
  plugin_t *plugin;

  string_len= strlen(string);
  space     = lcmaps_find_first_space(string);
  plugin    = top_plugin;

  while (plugin) {
    if (plugin->name && strncmp(plugin->name, string, space)==0) {
      if (plugin->args)
        if (string_len<=space+1 ||
	    strncmp(plugin->args, string+space+1, string_len-space-1)!=0) {
          plugin = plugin->next;
          continue;
        }
      return TRUE;
    }
    plugin = plugin->next;
  }

  return FALSE;
}


/*!
 *
 */
static int lcmaps_init_name_args(plugin_t** plugin, const rule_t* rule, rule_type_t type)
{
  size_t space, remainder, string_len;
  const char* string;
  const char* branch;

  switch (type) {
  case STATE:
    string = rule->state;
    branch = "STATE";
    break;
  case TRUE_BRANCH:
    string = rule->true_branch;
    branch = "TRUE_BRANCH";
    break;
  case FALSE_BRANCH:
    string = rule->false_branch;
    branch = "FALSE_BRANCH";
    break;
  default:
    lcmaps_pdl_warning(PDL_ERROR, "init_name_args: unknown type!");
    return -1;
  }

  if (!string)	{
      lcmaps_log(LOG_DEBUG, "  init_name_args: no plugin on %s\n",branch);
      return 0;
  }

  lcmaps_log_debug(5, "  init_name_args: processing %s: %s\n", branch,string);

  if (lcmaps_plugin_exists(string)) {
    lcmaps_log_debug(5, "  init_name_args: The plugin already exists.\n");
    /* Not an error */
    return 0;
  }

  lcmaps_log_debug(3, "  init_name_args: plugin does not yet exist.\n");

  if (!top_plugin) {
    if ( (top_plugin = (plugin_t*)malloc(sizeof(plugin_t)))==NULL)  {
	lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	return -1;
    }
    *plugin = top_plugin;
  }
  else {
    if ( ((*plugin)->next = (plugin_t*)malloc(sizeof(plugin_t)))==NULL)	{
	lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	return -1;
    }
    *plugin = (*plugin)->next;
  }

  (*plugin)->name = (*plugin)->args = NULL;
  (*plugin)->next = NULL;

  string_len=strlen(string);
  space = lcmaps_find_first_space(string);

  lcmaps_log_debug(5, "  init_name_args: space found a pos: %lu  strlen = %lu.\n",
	  (long unsigned)space, (long unsigned)string_len);

  if ( ((*plugin)->name = (char *)malloc(space+1))==NULL)   {
	lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	return -1;
  }
  strncpy((*plugin)->name, string, space);
  *((*plugin)->name+space) = '\0';

  if (string_len>space+1)   {
    remainder = string_len-space-1;
    if ( ((*plugin)->args   = (char *)malloc(remainder+1))==NULL)   {
	lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	return -1;
    }
    strncpy((*plugin)->args, string+space+1, remainder);
    *((*plugin)->args+remainder) = '\0';
  } else
    (*plugin)->args = NULL;

  (*plugin)->lineno = rule->lineno;
  (*plugin)->next   = NULL;

  lcmaps_log_debug(4, "  init_name_args: plugin->name = %s\n", 
	  (*plugin)->name ? (*plugin)->name : undefined );
  lcmaps_log_debug(4, "  init_name_args: plugin->args = %s\n",
	  (*plugin)->args ? (*plugin)->args : undefined );

  return 0;
}


/*!
 *  Find the first occurrence of a space in a string.
 *
 *  \param string String where the first space needs to be found.
 *  \return Position of the first occurrence of the space. If no space
 *          could be found, the position is set to the length of the
 *          string.
 *
 */
static size_t lcmaps_find_first_space(const char* string)
{
  size_t space, max_length;
  space = 0;
  max_length = strlen(string);

  for (; *string++ != ' ' && space < max_length; ++space);

  return space;
}


/*!
 *  Get the path.
 *
 *  \return Path.
 */
char *lcmaps_pdl_path(void)
{
  return path;
}


/*!
 *  When yacc encounters an error during the parsing process of
 *  the configuration file, it calls yyerror(). The actual message
 *  formatting is done in waring();
 *
 *  \param s error string.
 */
int yyerror(const char* s)
{
  /* lcmaps_log_debug(3, "%s\n", token_name()); */

  lcmaps_pdl_warning(PDL_ERROR, "%s", s);

  return 0;
}


/*!
 *  Function is called when the parser has found the value of the
 *  reserved path word. This function acts as a wrapper for the
 *  _lcmaps_set_path() function.
 *
 *  \param mypath path to be set
 */
void lcmaps_set_path(record_t* mypath)
{
  _lcmaps_set_path(mypath);

  if (mypath)	{
    free(mypath->string);
    free(mypath);
  }
}


/*!
 *  Set the path with the new value. If this function is called more than once,
 *  a warning message is displayed for each occurent.
 *
 *  \param _path The new path.
 *
 */
static void _lcmaps_set_path(const record_t* _path)
{
  /* lcmaps_log_debug(3, "lcmaps_set_path: '%s'.\n", _path); */

  if (path!=NULL) {
    lcmaps_pdl_warning(PDL_WARNING, "path already defined as %s in line: %d; ignoring this instance.", path, path_lineno);
    return;
  }
  if (_path!=NULL)  {  /* Only act when value is provided */
      path_lineno  = _path->lineno;
      if ((_path->string)[0]!='/') {	/* Relative path: prefix LCMAPS_LIB_HOME */
	  path=calloc(strlen(LCMAPS_LIB_HOME)+strlen(_path->string)+2,sizeof(char));
	  if (path==NULL)   {
	      lcmaps_pdl_warning(PDL_ERROR, "Out of memory when setting path.");
	      return;
	  }
	  sprintf(path,"%s/%s",LCMAPS_LIB_HOME,_path->string);
      } else {
	  path = strdup(_path->string);
	  if (path==NULL)   {
	     lcmaps_pdl_warning(PDL_ERROR, "Out of memory when setting path.");
	     return;
	  }
      }
      lcmaps_log_debug(LOG_DEBUG, "Modules search path is set to %s (line %d).\n", path, path_lineno);
   }
}



/*!
 *  Free the string allocated to hold the path
 */
void lcmaps_free_path(void)
{
    free(path);
    path = NULL;
}



/*!
 *  Concatenate two strings. The original two strings are freed. When
 *  the concatenation fails, the original strings are still freed. The
 *  actual concatenation is done by _lcmaps_concat_strings().
 *
 *  \param s1    first half of the string.
 *  \param s2    second half of the string.
 *  \return new string which is the concatenation of s1 and s2.
 *
 */
record_t* lcmaps_concat_strings(record_t* s1, record_t* s2)
{
  record_t* r = _lcmaps_concat_strings(s1, s2, 0);

  free(s1->string);
  free(s2->string);
  free(s1);
  free(s2);

  return r;
}


/*!
 *  Concatenate two string.
 *
 *  \param s1 first half of the string.
 *  \param s2 second half of the string.
 *  \param extra extra string to be inserted between s1 and s2
 *  \return new string which is the concatenation of s1 and s2.
 *
 */
static record_t* _lcmaps_concat_strings(const record_t* s1, const record_t* s2,
                          const char* extra)
{
  size_t len       = strlen(s1->string);
  size_t len_extra = extra ? strlen(extra) : 0;

  record_t* record = (record_t *)malloc(sizeof(record_t));
  if ( record==NULL ||
       (record->string =
	    (char *)malloc(len + len_extra + strlen(s2->string)+1))==NULL ) {
    lcmaps_pdl_warning(PDL_ERROR, "out of memory");
    return NULL;
  }

  strcpy(record->string, s1->string);
  if (extra)
    strcpy(record->string+len, extra);
  strcpy(record->string+len+len_extra, s2->string);

  /* lcmaps_log_debug(3, "lcmaps_concat_strings: '%s' + '%s' = '%s'.\n", s1, s2, s3); */

  return record;
}

/*!
 *  Concatenate two strings. The original two strings are freed. When
 *  the concatenation fails, the original strings are still freed. The
 *  actual concatenation is done by _lcmaps_concat_strings().
 *
 *  \param  s1  First string.
 *  \param  s2  Second string
 *  \return Concatenated strings of s1 + s2.
 *
 */
record_t* lcmaps_concat_strings_with_space(record_t* s1, record_t* s2)
{
  record_t* r;

  /*  Check if there is a string to add.  */
  if (strlen(s2->string)!=0) {
    /*
     *  When the first and second string end with ", do not an extra
     *  space. This is the result of an sequence of escaped ".
     */
    if ((s1->string[strlen(s1->string)-1]=='"') &&
        (s2->string[strlen(s2->string)-1]=='"'))
      r = _lcmaps_concat_strings(s1, s2, 0);
    else
      r = _lcmaps_concat_strings(s1, s2, " ");

    free(s1->string);
    free(s2->string);
    free(s1);
    free(s2);
  } else {
    /*
     *  Since there is nothing to concatenate, we simply copy the
     *  first string (s1) to the result.
     */
    if ( (r = (record_t*)malloc(sizeof(record_t)))==NULL)   {
	lcmaps_pdl_warning(PDL_ERROR, "out of memory");
	return NULL;
    }
    memcpy(r, s1, sizeof(record_t));
  }

  return r;
}


/*!
 *  Find the next plugin to evaluate based on the return status of the
 *  previous plugin evaluation. There are three statuses, two of which
 *  are rather obvious: either the previous evaluation has succeeded
 *  (EVALUATION_SUCCESS), or it has failed (EVALUATION_FAILURE). Based
 *  on these results, the next plugin should be the true_branch or
 *  false_branch respectively. There is one situation where there is
 *  no previous evaluation and that is at the very beginning. The
 *  very first call to this function should have (EVALUATION_START) as
 *  arguments. In this case the current state of the rule is returned
 *  as the next plugin to evaluate.
 *
 *  \param  status Status of previous evaluation.
 *  \return plugin name to be evaluation according to the
 *          configuration file.
 *
 */
char* lcmaps_pdl_next_plugin(plugin_status_t status)
{
  /* int rc       = 0; */
  char* string = NULL;
  char* state  = NULL;
  char* tmp    = NULL;

  switch (status) {
  case EVALUATION_START:
    /*  Make sure that there is a policy list.  */
    if (!(current_policy = lcmaps_get_policies()))
      return NULL;
    if (!(current_rule = current_policy->rule))
      return NULL;

    state = current_rule->state;
    break;

  case EVALUATION_SUCCESS:
    if (current_rule)
      state = current_rule->true_branch;

    if (current_policy && state)
      current_rule = lcmaps_find_state(current_policy->rule, state);
    else
      current_rule = NULL;
    break;

  case EVALUATION_FAILURE:
    /*
     *  Find the next rule for evaluation. The plugins specified by
     *  that rule will not be returned by this iterating of the
     *  function. Instead the static/global variables are setup, such
     *  that on the next iteration the next plugin is returned. Only
     *  after the evaluation of the plugin found during this iteration
     *  can a decision be made which plugin to return.
     *
     *  When the last rule has been evaluated, there is the
     *  possibility that there is another policy in the configration.
     *  Instead of stopping at the end of a rule chain, setup the
     *  variables for the next policy rule.
     *
     */
    if (current_rule)
    {
      state = current_rule->false_branch;
    }

    /* 
     *  Check if either there is a current_rule or a state. If so, we
     *  can continue. If there is no current rule, there might be a
     *  next policy. This needs to be evaluated next. If there is no
     *  state, this is because the false branch does not exist and
     *  hence we need to evaluate possible next policies.
     *
     */
    if (!(current_rule && state)) 
    {
        /* Check if there is a next plug-in policy to run */
        if (current_policy && (current_policy = current_policy->next)) 
        {
            if ((current_rule = current_policy->rule)) 
            {
                state = current_rule->state;
            }
        }
        else
        {
            /* No more policies to run */
            lcmaps_log_debug (5, "evaluationmanager: No more policies to run\n");
        }
    } 
    else 
    {
        /*  We have a valid next state. Let's find the next rule.  */
        if (current_policy)
        {
            current_rule = lcmaps_find_state(current_policy->rule, state);
        }
    }

    break;
  }

  /*
   *  Create the plugin name from the plugin name+arguments as contained in
   *  state. The absolute pluginname is contained in the absname and is
   *  initialized in PluginInit via lcmaps_findplugin(). The part duplicates the
   *  string and strips everything following the first space (i.e. arguments).
   */
  if (state) {
    string=strdup(state);
    if (string==NULL)   {
	lcmaps_pdl_warning(PDL_ERROR, "Out of memory.");
	return NULL;
    }
    tmp = strchr(string,' ');
    if (tmp) *tmp='\0';

    lcmaps_log_debug(3, "evaluationmanager: found plugin: %s\n", string);
  }

  return string;
}


/*!
 *  Free the resources.
 *
 */
void lcmaps_free_resources(void)
{
#if 0
  FILE * my_yyin = NULL;
#endif

  if (script_name) {
    free(script_name);
    script_name = NULL;
  }

  lcmaps_free_path();

  lcmaps_free_variables();
  lcmaps_free_policies();

  lcmaps_free_plugins(&top_plugin);

  /*
   *  YACC takes its input from the stdin by default. When the input
   *  is taken from a file, the yyin variable will have a different
   *  value from stdin and stderr. If so, it is necessary to close it.
   */

  /* By pass function */
#if 0
  my_yyin = get_active_yyin();

  if ((my_yyin!=stdin) && (my_yyin!=stderr)) {
    if (my_yyin)
        fclose(my_yyin);
    /* my_yyin=stdin; */
    set_active_yyin (stdin);
#else
  if (yyin!=stdin && yyin!=stderr) {
    if (yyin)
        fclose(yyin);
    yyin=stdin;
#endif
  }
}


/*!
 *  Display a warning message.
 *
 *  \param error Severity of the error.
 *  \param s     The text string followed by additional values; much like
 *  printf(char *, ...);
 *
 */
void lcmaps_pdl_warning(pdl_error_t error, const char* s, ...)
{
  static const char* level = NULL;
  char buf[WARNBUFLEN];
  int res;
  size_t len;

  va_list args;

  if (error == PDL_ERROR)
    parse_error = TRUE;

  if (!level)
    level = level_str[PDL_UNKNOWN];

  if (error != PDL_SAME)
    level = level_str[error];

  /*
   *  Prepend a standard text before the message. This makes reading
   *  the errors/warnings a little easier.
   */
  res = snprintf(buf, WARNBUFLEN, "%s:%d: [%s] ", script_name, lineno, level);
  if (res<0)	{
      lcmaps_log(LOG_ERR, "Cannot log message: %s\n",strerror(errno));
      len=0;
  } else    {
      len=(size_t)res;
      if (len>WARNBUFLEN-2)	{
	  lcmaps_log(LOG_ERR, "Log message is too long\n");
	  return;
      }
  }

  /*  Print (format) the actual message.  */
  va_start(args, s);
  res = vsnprintf(buf+len, WARNBUFLEN-2-len, s, args);
  va_end(args);
  if (res<0)	{
      lcmaps_log(LOG_ERR, "Cannot log message: %s\n",strerror(errno));
      return;
  }
  len+=(size_t)res;
  if (len>WARNBUFLEN-2) {
      lcmaps_log(LOG_ERR, "Log message is too long\n");
      return;
  }

  /*
   *  The string needs to end in a new line and of course strings are
   *  termintated by the end of string character. Make sure it happens
   *  and respect the buffer size!
   */
  buf[len<WARNBUFLEN-1 ? len++ : WARNBUFLEN-2] = '\n';
  buf[len<WARNBUFLEN   ? len   : WARNBUFLEN-1] = '\0';

  lcmaps_log(LOG_ERR, "%s", buf);
}

