from typing import Dict, List

mitre_attack_version: str = "15.1"
mitre_attack_tactics: Dict[str, str] = {
    "TA0001": "initial-access",
    "TA0002": "execution",
    "TA0003": "persistence",
    "TA0004": "privilege-escalation",
    "TA0005": "defense-evasion",
    "TA0006": "credential-access",
    "TA0007": "discovery",
    "TA0008": "lateral-movement",
    "TA0009": "collection",
    "TA0010": "exfiltration",
    "TA0011": "command-and-control",
    "TA0040": "impact",
    "TA0042": "resource-development",
    "TA0043": "reconnaissance",
}
mitre_attack_techniques: Dict[str, str] = {
    "T1001": "Data Obfuscation",
    "T1001.001": "Junk Data",
    "T1001.002": "Steganography",
    "T1001.003": "Protocol Impersonation",
    "T1003": "OS Credential Dumping",
    "T1003.001": "LSASS Memory",
    "T1003.002": "Security Account Manager",
    "T1003.003": "NTDS",
    "T1003.004": "LSA Secrets",
    "T1003.005": "Cached Domain Credentials",
    "T1003.006": "DCSync",
    "T1003.007": "Proc Filesystem",
    "T1003.008": "/etc/passwd and /etc/shadow",
    "T1005": "Data from Local System",
    "T1006": "Direct Volume Access",
    "T1007": "System Service Discovery",
    "T1008": "Fallback Channels",
    "T1010": "Application Window Discovery",
    "T1011": "Exfiltration Over Other Network Medium",
    "T1011.001": "Exfiltration Over Bluetooth",
    "T1012": "Query Registry",
    "T1014": "Rootkit",
    "T1016": "System Network Configuration Discovery",
    "T1016.001": "Internet Connection Discovery",
    "T1016.002": "Wi-Fi Discovery",
    "T1018": "Remote System Discovery",
    "T1020": "Automated Exfiltration",
    "T1020.001": "Traffic Duplication",
    "T1021": "Remote Services",
    "T1021.001": "Remote Desktop Protocol",
    "T1021.002": "SMB/Windows Admin Shares",
    "T1021.003": "Distributed Component Object Model",
    "T1021.004": "SSH",
    "T1021.005": "VNC",
    "T1021.006": "Windows Remote Management",
    "T1021.007": "Cloud Services",
    "T1021.008": "Direct Cloud VM Connections",
    "T1025": "Data from Removable Media",
    "T1027": "Obfuscated Files or Information",
    "T1027.001": "Binary Padding",
    "T1027.002": "Software Packing",
    "T1027.003": "Steganography",
    "T1027.004": "Compile After Delivery",
    "T1027.005": "Indicator Removal from Tools",
    "T1027.006": "HTML Smuggling",
    "T1027.007": "Dynamic API Resolution",
    "T1027.008": "Stripped Payloads",
    "T1027.009": "Embedded Payloads",
    "T1027.010": "Command Obfuscation",
    "T1027.011": "Fileless Storage",
    "T1027.012": "LNK Icon Smuggling",
    "T1027.013": "Encrypted/Encoded File",
    "T1029": "Scheduled Transfer",
    "T1030": "Data Transfer Size Limits",
    "T1033": "System Owner/User Discovery",
    "T1036": "Masquerading",
    "T1036.001": "Invalid Code Signature",
    "T1036.002": "Right-to-Left Override",
    "T1036.003": "Rename System Utilities",
    "T1036.004": "Masquerade Task or Service",
    "T1036.005": "Match Legitimate Name or Location",
    "T1036.006": "Space after Filename",
    "T1036.007": "Double File Extension",
    "T1036.008": "Masquerade File Type",
    "T1036.009": "Break Process Trees",
    "T1037": "Boot or Logon Initialization Scripts",
    "T1037.001": "Logon Script (Windows)",
    "T1037.002": "Login Hook",
    "T1037.003": "Network Logon Script",
    "T1037.004": "RC Scripts",
    "T1037.005": "Startup Items",
    "T1039": "Data from Network Shared Drive",
    "T1040": "Network Sniffing",
    "T1041": "Exfiltration Over C2 Channel",
    "T1046": "Network Service Discovery",
    "T1047": "Windows Management Instrumentation",
    "T1048": "Exfiltration Over Alternative Protocol",
    "T1048.001": "Exfiltration Over Symmetric Encrypted Non-C2 Protocol",
    "T1048.002": "Exfiltration Over Asymmetric Encrypted Non-C2 Protocol",
    "T1048.003": "Exfiltration Over Unencrypted Non-C2 Protocol",
    "T1049": "System Network Connections Discovery",
    "T1052": "Exfiltration Over Physical Medium",
    "T1052.001": "Exfiltration over USB",
    "T1053": "Scheduled Task/Job",
    "T1053.002": "At",
    "T1053.003": "Cron",
    "T1053.005": "Scheduled Task",
    "T1053.006": "Systemd Timers",
    "T1053.007": "Container Orchestration Job",
    "T1055": "Process Injection",
    "T1055.001": "Dynamic-link Library Injection",
    "T1055.002": "Portable Executable Injection",
    "T1055.003": "Thread Execution Hijacking",
    "T1055.004": "Asynchronous Procedure Call",
    "T1055.005": "Thread Local Storage",
    "T1055.008": "Ptrace System Calls",
    "T1055.009": "Proc Memory",
    "T1055.011": "Extra Window Memory Injection",
    "T1055.012": "Process Hollowing",
    "T1055.013": "Process Doppelgänging",
    "T1055.014": "VDSO Hijacking",
    "T1055.015": "ListPlanting",
    "T1056": "Input Capture",
    "T1056.001": "Keylogging",
    "T1056.002": "GUI Input Capture",
    "T1056.003": "Web Portal Capture",
    "T1056.004": "Credential API Hooking",
    "T1057": "Process Discovery",
    "T1059": "Command and Scripting Interpreter",
    "T1059.001": "PowerShell",
    "T1059.002": "AppleScript",
    "T1059.003": "Windows Command Shell",
    "T1059.004": "Unix Shell",
    "T1059.005": "Visual Basic",
    "T1059.006": "Python",
    "T1059.007": "JavaScript",
    "T1059.008": "Network Device CLI",
    "T1059.009": "Cloud API",
    "T1059.010": "AutoHotKey & AutoIT",
    "T1068": "Exploitation for Privilege Escalation",
    "T1069": "Permission Groups Discovery",
    "T1069.001": "Local Groups",
    "T1069.002": "Domain Groups",
    "T1069.003": "Cloud Groups",
    "T1070": "Indicator Removal",
    "T1070.001": "Clear Windows Event Logs",
    "T1070.002": "Clear Linux or Mac System Logs",
    "T1070.003": "Clear Command History",
    "T1070.004": "File Deletion",
    "T1070.005": "Network Share Connection Removal",
    "T1070.006": "Timestomp",
    "T1070.007": "Clear Network Connection History and Configurations",
    "T1070.008": "Clear Mailbox Data",
    "T1070.009": "Clear Persistence",
    "T1071": "Application Layer Protocol",
    "T1071.001": "Web Protocols",
    "T1071.002": "File Transfer Protocols",
    "T1071.003": "Mail Protocols",
    "T1071.004": "DNS",
    "T1072": "Software Deployment Tools",
    "T1074": "Data Staged",
    "T1074.001": "Local Data Staging",
    "T1074.002": "Remote Data Staging",
    "T1078": "Valid Accounts",
    "T1078.001": "Default Accounts",
    "T1078.002": "Domain Accounts",
    "T1078.003": "Local Accounts",
    "T1078.004": "Cloud Accounts",
    "T1080": "Taint Shared Content",
    "T1082": "System Information Discovery",
    "T1083": "File and Directory Discovery",
    "T1087": "Account Discovery",
    "T1087.001": "Local Account",
    "T1087.002": "Domain Account",
    "T1087.003": "Email Account",
    "T1087.004": "Cloud Account",
    "T1090": "Proxy",
    "T1090.001": "Internal Proxy",
    "T1090.002": "External Proxy",
    "T1090.003": "Multi-hop Proxy",
    "T1090.004": "Domain Fronting",
    "T1091": "Replication Through Removable Media",
    "T1092": "Communication Through Removable Media",
    "T1095": "Non-Application Layer Protocol",
    "T1098": "Account Manipulation",
    "T1098.001": "Additional Cloud Credentials",
    "T1098.002": "Additional Email Delegate Permissions",
    "T1098.003": "Additional Cloud Roles",
    "T1098.004": "SSH Authorized Keys",
    "T1098.005": "Device Registration",
    "T1098.006": "Additional Container Cluster Roles",
    "T1102": "Web Service",
    "T1102.001": "Dead Drop Resolver",
    "T1102.002": "Bidirectional Communication",
    "T1102.003": "One-Way Communication",
    "T1104": "Multi-Stage Channels",
    "T1105": "Ingress Tool Transfer",
    "T1106": "Native API",
    "T1110": "Brute Force",
    "T1110.001": "Password Guessing",
    "T1110.002": "Password Cracking",
    "T1110.003": "Password Spraying",
    "T1110.004": "Credential Stuffing",
    "T1111": "Multi-Factor Authentication Interception",
    "T1112": "Modify Registry",
    "T1113": "Screen Capture",
    "T1114": "Email Collection",
    "T1114.001": "Local Email Collection",
    "T1114.002": "Remote Email Collection",
    "T1114.003": "Email Forwarding Rule",
    "T1115": "Clipboard Data",
    "T1119": "Automated Collection",
    "T1120": "Peripheral Device Discovery",
    "T1123": "Audio Capture",
    "T1124": "System Time Discovery",
    "T1125": "Video Capture",
    "T1127": "Trusted Developer Utilities Proxy Execution",
    "T1127.001": "MSBuild",
    "T1129": "Shared Modules",
    "T1132": "Data Encoding",
    "T1132.001": "Standard Encoding",
    "T1132.002": "Non-Standard Encoding",
    "T1133": "External Remote Services",
    "T1134": "Access Token Manipulation",
    "T1134.001": "Token Impersonation/Theft",
    "T1134.002": "Create Process with Token",
    "T1134.003": "Make and Impersonate Token",
    "T1134.004": "Parent PID Spoofing",
    "T1134.005": "SID-History Injection",
    "T1135": "Network Share Discovery",
    "T1136": "Create Account",
    "T1136.001": "Local Account",
    "T1136.002": "Domain Account",
    "T1136.003": "Cloud Account",
    "T1137": "Office Application Startup",
    "T1137.001": "Office Template Macros",
    "T1137.002": "Office Test",
    "T1137.003": "Outlook Forms",
    "T1137.004": "Outlook Home Page",
    "T1137.005": "Outlook Rules",
    "T1137.006": "Add-ins",
    "T1140": "Deobfuscate/Decode Files or Information",
    "T1176": "Browser Extensions",
    "T1185": "Browser Session Hijacking",
    "T1187": "Forced Authentication",
    "T1189": "Drive-by Compromise",
    "T1190": "Exploit Public-Facing Application",
    "T1195": "Supply Chain Compromise",
    "T1195.001": "Compromise Software Dependencies and Development Tools",
    "T1195.002": "Compromise Software Supply Chain",
    "T1195.003": "Compromise Hardware Supply Chain",
    "T1197": "BITS Jobs",
    "T1199": "Trusted Relationship",
    "T1200": "Hardware Additions",
    "T1201": "Password Policy Discovery",
    "T1202": "Indirect Command Execution",
    "T1203": "Exploitation for Client Execution",
    "T1204": "User Execution",
    "T1204.001": "Malicious Link",
    "T1204.002": "Malicious File",
    "T1204.003": "Malicious Image",
    "T1205": "Traffic Signaling",
    "T1205.001": "Port Knocking",
    "T1205.002": "Socket Filters",
    "T1207": "Rogue Domain Controller",
    "T1210": "Exploitation of Remote Services",
    "T1211": "Exploitation for Defense Evasion",
    "T1212": "Exploitation for Credential Access",
    "T1213": "Data from Information Repositories",
    "T1213.001": "Confluence",
    "T1213.002": "Sharepoint",
    "T1213.003": "Code Repositories",
    "T1216": "System Script Proxy Execution",
    "T1216.001": "PubPrn",
    "T1216.002": "SyncAppvPublishingServer",
    "T1217": "Browser Information Discovery",
    "T1218": "System Binary Proxy Execution",
    "T1218.001": "Compiled HTML File",
    "T1218.002": "Control Panel",
    "T1218.003": "CMSTP",
    "T1218.004": "InstallUtil",
    "T1218.005": "Mshta",
    "T1218.007": "Msiexec",
    "T1218.008": "Odbcconf",
    "T1218.009": "Regsvcs/Regasm",
    "T1218.010": "Regsvr32",
    "T1218.011": "Rundll32",
    "T1218.012": "Verclsid",
    "T1218.013": "Mavinject",
    "T1218.014": "MMC",
    "T1218.015": "Electron Applications",
    "T1219": "Remote Access Software",
    "T1220": "XSL Script Processing",
    "T1221": "Template Injection",
    "T1222": "File and Directory Permissions Modification",
    "T1222.001": "Windows File and Directory Permissions Modification",
    "T1222.002": "Linux and Mac File and Directory Permissions Modification",
    "T1480": "Execution Guardrails",
    "T1480.001": "Environmental Keying",
    "T1482": "Domain Trust Discovery",
    "T1484": "Domain or Tenant Policy Modification",
    "T1484.001": "Group Policy Modification",
    "T1484.002": "Trust Modification",
    "T1485": "Data Destruction",
    "T1486": "Data Encrypted for Impact",
    "T1489": "Service Stop",
    "T1490": "Inhibit System Recovery",
    "T1491": "Defacement",
    "T1491.001": "Internal Defacement",
    "T1491.002": "External Defacement",
    "T1495": "Firmware Corruption",
    "T1496": "Resource Hijacking",
    "T1497": "Virtualization/Sandbox Evasion",
    "T1497.001": "System Checks",
    "T1497.002": "User Activity Based Checks",
    "T1497.003": "Time Based Evasion",
    "T1498": "Network Denial of Service",
    "T1498.001": "Direct Network Flood",
    "T1498.002": "Reflection Amplification",
    "T1499": "Endpoint Denial of Service",
    "T1499.001": "OS Exhaustion Flood",
    "T1499.002": "Service Exhaustion Flood",
    "T1499.003": "Application Exhaustion Flood",
    "T1499.004": "Application or System Exploitation",
    "T1505": "Server Software Component",
    "T1505.001": "SQL Stored Procedures",
    "T1505.002": "Transport Agent",
    "T1505.003": "Web Shell",
    "T1505.004": "IIS Components",
    "T1505.005": "Terminal Services DLL",
    "T1518": "Software Discovery",
    "T1518.001": "Security Software Discovery",
    "T1525": "Implant Internal Image",
    "T1526": "Cloud Service Discovery",
    "T1528": "Steal Application Access Token",
    "T1529": "System Shutdown/Reboot",
    "T1530": "Data from Cloud Storage",
    "T1531": "Account Access Removal",
    "T1534": "Internal Spearphishing",
    "T1535": "Unused/Unsupported Cloud Regions",
    "T1537": "Transfer Data to Cloud Account",
    "T1538": "Cloud Service Dashboard",
    "T1539": "Steal Web Session Cookie",
    "T1542": "Pre-OS Boot",
    "T1542.001": "System Firmware",
    "T1542.002": "Component Firmware",
    "T1542.003": "Bootkit",
    "T1542.004": "ROMMONkit",
    "T1542.005": "TFTP Boot",
    "T1543": "Create or Modify System Process",
    "T1543.001": "Launch Agent",
    "T1543.002": "Systemd Service",
    "T1543.003": "Windows Service",
    "T1543.004": "Launch Daemon",
    "T1543.005": "Container Service",
    "T1546": "Event Triggered Execution",
    "T1546.001": "Change Default File Association",
    "T1546.002": "Screensaver",
    "T1546.003": "Windows Management Instrumentation Event Subscription",
    "T1546.004": "Unix Shell Configuration Modification",
    "T1546.005": "Trap",
    "T1546.006": "LC_LOAD_DYLIB Addition",
    "T1546.007": "Netsh Helper DLL",
    "T1546.008": "Accessibility Features",
    "T1546.009": "AppCert DLLs",
    "T1546.010": "AppInit DLLs",
    "T1546.011": "Application Shimming",
    "T1546.012": "Image File Execution Options Injection",
    "T1546.013": "PowerShell Profile",
    "T1546.014": "Emond",
    "T1546.015": "Component Object Model Hijacking",
    "T1546.016": "Installer Packages",
    "T1547": "Boot or Logon Autostart Execution",
    "T1547.001": "Registry Run Keys / Startup Folder",
    "T1547.002": "Authentication Package",
    "T1547.003": "Time Providers",
    "T1547.004": "Winlogon Helper DLL",
    "T1547.005": "Security Support Provider",
    "T1547.006": "Kernel Modules and Extensions",
    "T1547.007": "Re-opened Applications",
    "T1547.008": "LSASS Driver",
    "T1547.009": "Shortcut Modification",
    "T1547.010": "Port Monitors",
    "T1547.012": "Print Processors",
    "T1547.013": "XDG Autostart Entries",
    "T1547.014": "Active Setup",
    "T1547.015": "Login Items",
    "T1548": "Abuse Elevation Control Mechanism",
    "T1548.001": "Setuid and Setgid",
    "T1548.002": "Bypass User Account Control",
    "T1548.003": "Sudo and Sudo Caching",
    "T1548.004": "Elevated Execution with Prompt",
    "T1548.005": "Temporary Elevated Cloud Access",
    "T1548.006": "TCC Manipulation",
    "T1550": "Use Alternate Authentication Material",
    "T1550.001": "Application Access Token",
    "T1550.002": "Pass the Hash",
    "T1550.003": "Pass the Ticket",
    "T1550.004": "Web Session Cookie",
    "T1552": "Unsecured Credentials",
    "T1552.001": "Credentials In Files",
    "T1552.002": "Credentials in Registry",
    "T1552.003": "Bash History",
    "T1552.004": "Private Keys",
    "T1552.005": "Cloud Instance Metadata API",
    "T1552.006": "Group Policy Preferences",
    "T1552.007": "Container API",
    "T1552.008": "Chat Messages",
    "T1553": "Subvert Trust Controls",
    "T1553.001": "Gatekeeper Bypass",
    "T1553.002": "Code Signing",
    "T1553.003": "SIP and Trust Provider Hijacking",
    "T1553.004": "Install Root Certificate",
    "T1553.005": "Mark-of-the-Web Bypass",
    "T1553.006": "Code Signing Policy Modification",
    "T1554": "Compromise Host Software Binary",
    "T1555": "Credentials from Password Stores",
    "T1555.001": "Keychain",
    "T1555.002": "Securityd Memory",
    "T1555.003": "Credentials from Web Browsers",
    "T1555.004": "Windows Credential Manager",
    "T1555.005": "Password Managers",
    "T1555.006": "Cloud Secrets Management Stores",
    "T1556": "Modify Authentication Process",
    "T1556.001": "Domain Controller Authentication",
    "T1556.002": "Password Filter DLL",
    "T1556.003": "Pluggable Authentication Modules",
    "T1556.004": "Network Device Authentication",
    "T1556.005": "Reversible Encryption",
    "T1556.006": "Multi-Factor Authentication",
    "T1556.007": "Hybrid Identity",
    "T1556.008": "Network Provider DLL",
    "T1556.009": "Conditional Access Policies",
    "T1557": "Adversary-in-the-Middle",
    "T1557.001": "LLMNR/NBT-NS Poisoning and SMB Relay",
    "T1557.002": "ARP Cache Poisoning",
    "T1557.003": "DHCP Spoofing",
    "T1558": "Steal or Forge Kerberos Tickets",
    "T1558.001": "Golden Ticket",
    "T1558.002": "Silver Ticket",
    "T1558.003": "Kerberoasting",
    "T1558.004": "AS-REP Roasting",
    "T1559": "Inter-Process Communication",
    "T1559.001": "Component Object Model",
    "T1559.002": "Dynamic Data Exchange",
    "T1559.003": "XPC Services",
    "T1560": "Archive Collected Data",
    "T1560.001": "Archive via Utility",
    "T1560.002": "Archive via Library",
    "T1560.003": "Archive via Custom Method",
    "T1561": "Disk Wipe",
    "T1561.001": "Disk Content Wipe",
    "T1561.002": "Disk Structure Wipe",
    "T1562": "Impair Defenses",
    "T1562.001": "Disable or Modify Tools",
    "T1562.002": "Disable Windows Event Logging",
    "T1562.003": "Impair Command History Logging",
    "T1562.004": "Disable or Modify System Firewall",
    "T1562.006": "Indicator Blocking",
    "T1562.007": "Disable or Modify Cloud Firewall",
    "T1562.008": "Disable or Modify Cloud Logs",
    "T1562.009": "Safe Mode Boot",
    "T1562.010": "Downgrade Attack",
    "T1562.011": "Spoof Security Alerting",
    "T1562.012": "Disable or Modify Linux Audit System",
    "T1563": "Remote Service Session Hijacking",
    "T1563.001": "SSH Hijacking",
    "T1563.002": "RDP Hijacking",
    "T1564": "Hide Artifacts",
    "T1564.001": "Hidden Files and Directories",
    "T1564.002": "Hidden Users",
    "T1564.003": "Hidden Window",
    "T1564.004": "NTFS File Attributes",
    "T1564.005": "Hidden File System",
    "T1564.006": "Run Virtual Instance",
    "T1564.007": "VBA Stomping",
    "T1564.008": "Email Hiding Rules",
    "T1564.009": "Resource Forking",
    "T1564.010": "Process Argument Spoofing",
    "T1564.011": "Ignore Process Interrupts",
    "T1564.012": "File/Path Exclusions",
    "T1565": "Data Manipulation",
    "T1565.001": "Stored Data Manipulation",
    "T1565.002": "Transmitted Data Manipulation",
    "T1565.003": "Runtime Data Manipulation",
    "T1566": "Phishing",
    "T1566.001": "Spearphishing Attachment",
    "T1566.002": "Spearphishing Link",
    "T1566.003": "Spearphishing via Service",
    "T1566.004": "Spearphishing Voice",
    "T1567": "Exfiltration Over Web Service",
    "T1567.001": "Exfiltration to Code Repository",
    "T1567.002": "Exfiltration to Cloud Storage",
    "T1567.003": "Exfiltration to Text Storage Sites",
    "T1567.004": "Exfiltration Over Webhook",
    "T1568": "Dynamic Resolution",
    "T1568.001": "Fast Flux DNS",
    "T1568.002": "Domain Generation Algorithms",
    "T1568.003": "DNS Calculation",
    "T1569": "System Services",
    "T1569.001": "Launchctl",
    "T1569.002": "Service Execution",
    "T1570": "Lateral Tool Transfer",
    "T1571": "Non-Standard Port",
    "T1572": "Protocol Tunneling",
    "T1573": "Encrypted Channel",
    "T1573.001": "Symmetric Cryptography",
    "T1573.002": "Asymmetric Cryptography",
    "T1574": "Hijack Execution Flow",
    "T1574.001": "DLL Search Order Hijacking",
    "T1574.002": "DLL Side-Loading",
    "T1574.004": "Dylib Hijacking",
    "T1574.005": "Executable Installer File Permissions Weakness",
    "T1574.006": "Dynamic Linker Hijacking",
    "T1574.007": "Path Interception by PATH Environment Variable",
    "T1574.008": "Path Interception by Search Order Hijacking",
    "T1574.009": "Path Interception by Unquoted Path",
    "T1574.010": "Services File Permissions Weakness",
    "T1574.011": "Services Registry Permissions Weakness",
    "T1574.012": "COR_PROFILER",
    "T1574.013": "KernelCallbackTable",
    "T1574.014": "AppDomainManager",
    "T1578": "Modify Cloud Compute Infrastructure",
    "T1578.001": "Create Snapshot",
    "T1578.002": "Create Cloud Instance",
    "T1578.003": "Delete Cloud Instance",
    "T1578.004": "Revert Cloud Instance",
    "T1578.005": "Modify Cloud Compute Configurations",
    "T1580": "Cloud Infrastructure Discovery",
    "T1583": "Acquire Infrastructure",
    "T1583.001": "Domains",
    "T1583.002": "DNS Server",
    "T1583.003": "Virtual Private Server",
    "T1583.004": "Server",
    "T1583.005": "Botnet",
    "T1583.006": "Web Services",
    "T1583.007": "Serverless",
    "T1583.008": "Malvertising",
    "T1584": "Compromise Infrastructure",
    "T1584.001": "Domains",
    "T1584.002": "DNS Server",
    "T1584.003": "Virtual Private Server",
    "T1584.004": "Server",
    "T1584.005": "Botnet",
    "T1584.006": "Web Services",
    "T1584.007": "Serverless",
    "T1584.008": "Network Devices",
    "T1585": "Establish Accounts",
    "T1585.001": "Social Media Accounts",
    "T1585.002": "Email Accounts",
    "T1585.003": "Cloud Accounts",
    "T1586": "Compromise Accounts",
    "T1586.001": "Social Media Accounts",
    "T1586.002": "Email Accounts",
    "T1586.003": "Cloud Accounts",
    "T1587": "Develop Capabilities",
    "T1587.001": "Malware",
    "T1587.002": "Code Signing Certificates",
    "T1587.003": "Digital Certificates",
    "T1587.004": "Exploits",
    "T1588": "Obtain Capabilities",
    "T1588.001": "Malware",
    "T1588.002": "Tool",
    "T1588.003": "Code Signing Certificates",
    "T1588.004": "Digital Certificates",
    "T1588.005": "Exploits",
    "T1588.006": "Vulnerabilities",
    "T1588.007": "Artificial Intelligence",
    "T1589": "Gather Victim Identity Information",
    "T1589.001": "Credentials",
    "T1589.002": "Email Addresses",
    "T1589.003": "Employee Names",
    "T1590": "Gather Victim Network Information",
    "T1590.001": "Domain Properties",
    "T1590.002": "DNS",
    "T1590.003": "Network Trust Dependencies",
    "T1590.004": "Network Topology",
    "T1590.005": "IP Addresses",
    "T1590.006": "Network Security Appliances",
    "T1591": "Gather Victim Org Information",
    "T1591.001": "Determine Physical Locations",
    "T1591.002": "Business Relationships",
    "T1591.003": "Identify Business Tempo",
    "T1591.004": "Identify Roles",
    "T1592": "Gather Victim Host Information",
    "T1592.001": "Hardware",
    "T1592.002": "Software",
    "T1592.003": "Firmware",
    "T1592.004": "Client Configurations",
    "T1593": "Search Open Websites/Domains",
    "T1593.001": "Social Media",
    "T1593.002": "Search Engines",
    "T1593.003": "Code Repositories",
    "T1594": "Search Victim-Owned Websites",
    "T1595": "Active Scanning",
    "T1595.001": "Scanning IP Blocks",
    "T1595.002": "Vulnerability Scanning",
    "T1595.003": "Wordlist Scanning",
    "T1596": "Search Open Technical Databases",
    "T1596.001": "DNS/Passive DNS",
    "T1596.002": "WHOIS",
    "T1596.003": "Digital Certificates",
    "T1596.004": "CDNs",
    "T1596.005": "Scan Databases",
    "T1597": "Search Closed Sources",
    "T1597.001": "Threat Intel Vendors",
    "T1597.002": "Purchase Technical Data",
    "T1598": "Phishing for Information",
    "T1598.001": "Spearphishing Service",
    "T1598.002": "Spearphishing Attachment",
    "T1598.003": "Spearphishing Link",
    "T1598.004": "Spearphishing Voice",
    "T1599": "Network Boundary Bridging",
    "T1599.001": "Network Address Translation Traversal",
    "T1600": "Weaken Encryption",
    "T1600.001": "Reduce Key Space",
    "T1600.002": "Disable Crypto Hardware",
    "T1601": "Modify System Image",
    "T1601.001": "Patch System Image",
    "T1601.002": "Downgrade System Image",
    "T1602": "Data from Configuration Repository",
    "T1602.001": "SNMP (MIB Dump)",
    "T1602.002": "Network Device Configuration Dump",
    "T1606": "Forge Web Credentials",
    "T1606.001": "Web Cookies",
    "T1606.002": "SAML Tokens",
    "T1608": "Stage Capabilities",
    "T1608.001": "Upload Malware",
    "T1608.002": "Upload Tool",
    "T1608.003": "Install Digital Certificate",
    "T1608.004": "Drive-by Target",
    "T1608.005": "Link Target",
    "T1608.006": "SEO Poisoning",
    "T1609": "Container Administration Command",
    "T1610": "Deploy Container",
    "T1611": "Escape to Host",
    "T1612": "Build Image on Host",
    "T1613": "Container and Resource Discovery",
    "T1614": "System Location Discovery",
    "T1614.001": "System Language Discovery",
    "T1615": "Group Policy Discovery",
    "T1619": "Cloud Storage Object Discovery",
    "T1620": "Reflective Code Loading",
    "T1621": "Multi-Factor Authentication Request Generation",
    "T1622": "Debugger Evasion",
    "T1647": "Plist File Modification",
    "T1648": "Serverless Execution",
    "T1649": "Steal or Forge Authentication Certificates",
    "T1650": "Acquire Access",
    "T1651": "Cloud Administration Command",
    "T1652": "Device Driver Discovery",
    "T1653": "Power Settings",
    "T1654": "Log Enumeration",
    "T1656": "Impersonation",
    "T1657": "Financial Theft",
    "T1659": "Content Injection",
    "T1665": "Hide Infrastructure",
}
mitre_attack_techniques_tactics_mapping: Dict[str, List[str]] = {
    "T1001": ["command-and-control"],
    "T1001.001": ["command-and-control"],
    "T1001.002": ["command-and-control"],
    "T1001.003": ["command-and-control"],
    "T1003": ["credential-access"],
    "T1003.001": ["credential-access"],
    "T1003.002": ["credential-access"],
    "T1003.003": ["credential-access"],
    "T1003.004": ["credential-access"],
    "T1003.005": ["credential-access"],
    "T1003.006": ["credential-access"],
    "T1003.007": ["credential-access"],
    "T1003.008": ["credential-access"],
    "T1005": ["collection"],
    "T1006": ["defense-evasion"],
    "T1007": ["discovery"],
    "T1008": ["command-and-control"],
    "T1010": ["discovery"],
    "T1011": ["exfiltration"],
    "T1011.001": ["exfiltration"],
    "T1012": ["discovery"],
    "T1014": ["defense-evasion"],
    "T1016": ["discovery"],
    "T1016.001": ["discovery"],
    "T1016.002": ["discovery"],
    "T1018": ["discovery"],
    "T1020": ["exfiltration"],
    "T1020.001": ["exfiltration"],
    "T1021": ["lateral-movement"],
    "T1021.001": ["lateral-movement"],
    "T1021.002": ["lateral-movement"],
    "T1021.003": ["lateral-movement"],
    "T1021.004": ["lateral-movement"],
    "T1021.005": ["lateral-movement"],
    "T1021.006": ["lateral-movement"],
    "T1021.007": ["lateral-movement"],
    "T1021.008": ["lateral-movement"],
    "T1025": ["collection"],
    "T1027": ["defense-evasion"],
    "T1027.001": ["defense-evasion"],
    "T1027.002": ["defense-evasion"],
    "T1027.003": ["defense-evasion"],
    "T1027.004": ["defense-evasion"],
    "T1027.005": ["defense-evasion"],
    "T1027.006": ["defense-evasion"],
    "T1027.007": ["defense-evasion"],
    "T1027.008": ["defense-evasion"],
    "T1027.009": ["defense-evasion"],
    "T1027.010": ["defense-evasion"],
    "T1027.011": ["defense-evasion"],
    "T1027.012": ["defense-evasion"],
    "T1027.013": ["defense-evasion"],
    "T1029": ["exfiltration"],
    "T1030": ["exfiltration"],
    "T1033": ["discovery"],
    "T1036": ["defense-evasion"],
    "T1036.001": ["defense-evasion"],
    "T1036.002": ["defense-evasion"],
    "T1036.003": ["defense-evasion"],
    "T1036.004": ["defense-evasion"],
    "T1036.005": ["defense-evasion"],
    "T1036.006": ["defense-evasion"],
    "T1036.007": ["defense-evasion"],
    "T1036.008": ["defense-evasion"],
    "T1036.009": ["defense-evasion"],
    "T1037": ["persistence", "privilege-escalation"],
    "T1037.001": ["persistence", "privilege-escalation"],
    "T1037.002": ["persistence", "privilege-escalation"],
    "T1037.003": ["persistence", "privilege-escalation"],
    "T1037.004": ["persistence", "privilege-escalation"],
    "T1037.005": ["persistence", "privilege-escalation"],
    "T1039": ["collection"],
    "T1040": ["credential-access", "discovery"],
    "T1041": ["exfiltration"],
    "T1046": ["discovery"],
    "T1047": ["execution"],
    "T1048": ["exfiltration"],
    "T1048.001": ["exfiltration"],
    "T1048.002": ["exfiltration"],
    "T1048.003": ["exfiltration"],
    "T1049": ["discovery"],
    "T1052": ["exfiltration"],
    "T1052.001": ["exfiltration"],
    "T1053": ["execution", "persistence", "privilege-escalation"],
    "T1053.002": ["execution", "persistence", "privilege-escalation"],
    "T1053.003": ["execution", "persistence", "privilege-escalation"],
    "T1053.005": ["execution", "persistence", "privilege-escalation"],
    "T1053.006": ["execution", "persistence", "privilege-escalation"],
    "T1053.007": ["execution", "persistence", "privilege-escalation"],
    "T1055": ["defense-evasion", "privilege-escalation"],
    "T1055.001": ["defense-evasion", "privilege-escalation"],
    "T1055.002": ["defense-evasion", "privilege-escalation"],
    "T1055.003": ["defense-evasion", "privilege-escalation"],
    "T1055.004": ["defense-evasion", "privilege-escalation"],
    "T1055.005": ["defense-evasion", "privilege-escalation"],
    "T1055.008": ["defense-evasion", "privilege-escalation"],
    "T1055.009": ["defense-evasion", "privilege-escalation"],
    "T1055.011": ["defense-evasion", "privilege-escalation"],
    "T1055.012": ["defense-evasion", "privilege-escalation"],
    "T1055.013": ["defense-evasion", "privilege-escalation"],
    "T1055.014": ["defense-evasion", "privilege-escalation"],
    "T1055.015": ["defense-evasion", "privilege-escalation"],
    "T1056": ["collection", "credential-access"],
    "T1056.001": ["collection", "credential-access"],
    "T1056.002": ["collection", "credential-access"],
    "T1056.003": ["collection", "credential-access"],
    "T1056.004": ["collection", "credential-access"],
    "T1057": ["discovery"],
    "T1059": ["execution"],
    "T1059.001": ["execution"],
    "T1059.002": ["execution"],
    "T1059.003": ["execution"],
    "T1059.004": ["execution"],
    "T1059.005": ["execution"],
    "T1059.006": ["execution"],
    "T1059.007": ["execution"],
    "T1059.008": ["execution"],
    "T1059.009": ["execution"],
    "T1059.010": ["execution"],
    "T1068": ["privilege-escalation"],
    "T1069": ["discovery"],
    "T1069.001": ["discovery"],
    "T1069.002": ["discovery"],
    "T1069.003": ["discovery"],
    "T1070": ["defense-evasion"],
    "T1070.001": ["defense-evasion"],
    "T1070.002": ["defense-evasion"],
    "T1070.003": ["defense-evasion"],
    "T1070.004": ["defense-evasion"],
    "T1070.005": ["defense-evasion"],
    "T1070.006": ["defense-evasion"],
    "T1070.007": ["defense-evasion"],
    "T1070.008": ["defense-evasion"],
    "T1070.009": ["defense-evasion"],
    "T1071": ["command-and-control"],
    "T1071.001": ["command-and-control"],
    "T1071.002": ["command-and-control"],
    "T1071.003": ["command-and-control"],
    "T1071.004": ["command-and-control"],
    "T1072": ["execution", "lateral-movement"],
    "T1074": ["collection"],
    "T1074.001": ["collection"],
    "T1074.002": ["collection"],
    "T1078": ["defense-evasion", "persistence", "privilege-escalation", "initial-access"],
    "T1078.001": ["defense-evasion", "persistence", "privilege-escalation", "initial-access"],
    "T1078.002": ["defense-evasion", "persistence", "privilege-escalation", "initial-access"],
    "T1078.003": ["defense-evasion", "persistence", "privilege-escalation", "initial-access"],
    "T1078.004": ["defense-evasion", "persistence", "privilege-escalation", "initial-access"],
    "T1080": ["lateral-movement"],
    "T1082": ["discovery"],
    "T1083": ["discovery"],
    "T1087": ["discovery"],
    "T1087.001": ["discovery"],
    "T1087.002": ["discovery"],
    "T1087.003": ["discovery"],
    "T1087.004": ["discovery"],
    "T1090": ["command-and-control"],
    "T1090.001": ["command-and-control"],
    "T1090.002": ["command-and-control"],
    "T1090.003": ["command-and-control"],
    "T1090.004": ["command-and-control"],
    "T1091": ["lateral-movement", "initial-access"],
    "T1092": ["command-and-control"],
    "T1095": ["command-and-control"],
    "T1098": ["persistence", "privilege-escalation"],
    "T1098.001": ["persistence", "privilege-escalation"],
    "T1098.002": ["persistence", "privilege-escalation"],
    "T1098.003": ["persistence", "privilege-escalation"],
    "T1098.004": ["persistence", "privilege-escalation"],
    "T1098.005": ["persistence", "privilege-escalation"],
    "T1098.006": ["persistence", "privilege-escalation"],
    "T1102": ["command-and-control"],
    "T1102.001": ["command-and-control"],
    "T1102.002": ["command-and-control"],
    "T1102.003": ["command-and-control"],
    "T1104": ["command-and-control"],
    "T1105": ["command-and-control"],
    "T1106": ["execution"],
    "T1110": ["credential-access"],
    "T1110.001": ["credential-access"],
    "T1110.002": ["credential-access"],
    "T1110.003": ["credential-access"],
    "T1110.004": ["credential-access"],
    "T1111": ["credential-access"],
    "T1112": ["defense-evasion"],
    "T1113": ["collection"],
    "T1114": ["collection"],
    "T1114.001": ["collection"],
    "T1114.002": ["collection"],
    "T1114.003": ["collection"],
    "T1115": ["collection"],
    "T1119": ["collection"],
    "T1120": ["discovery"],
    "T1123": ["collection"],
    "T1124": ["discovery"],
    "T1125": ["collection"],
    "T1127": ["defense-evasion"],
    "T1127.001": ["defense-evasion"],
    "T1129": ["execution"],
    "T1132": ["command-and-control"],
    "T1132.001": ["command-and-control"],
    "T1132.002": ["command-and-control"],
    "T1133": ["persistence", "initial-access"],
    "T1134": ["defense-evasion", "privilege-escalation"],
    "T1134.001": ["defense-evasion", "privilege-escalation"],
    "T1134.002": ["defense-evasion", "privilege-escalation"],
    "T1134.003": ["defense-evasion", "privilege-escalation"],
    "T1134.004": ["defense-evasion", "privilege-escalation"],
    "T1134.005": ["defense-evasion", "privilege-escalation"],
    "T1135": ["discovery"],
    "T1136": ["persistence"],
    "T1136.001": ["persistence"],
    "T1136.002": ["persistence"],
    "T1136.003": ["persistence"],
    "T1137": ["persistence"],
    "T1137.001": ["persistence"],
    "T1137.002": ["persistence"],
    "T1137.003": ["persistence"],
    "T1137.004": ["persistence"],
    "T1137.005": ["persistence"],
    "T1137.006": ["persistence"],
    "T1140": ["defense-evasion"],
    "T1176": ["persistence"],
    "T1185": ["collection"],
    "T1187": ["credential-access"],
    "T1189": ["initial-access"],
    "T1190": ["initial-access"],
    "T1195": ["initial-access"],
    "T1195.001": ["initial-access"],
    "T1195.002": ["initial-access"],
    "T1195.003": ["initial-access"],
    "T1197": ["defense-evasion", "persistence"],
    "T1199": ["initial-access"],
    "T1200": ["initial-access"],
    "T1201": ["discovery"],
    "T1202": ["defense-evasion"],
    "T1203": ["execution"],
    "T1204": ["execution"],
    "T1204.001": ["execution"],
    "T1204.002": ["execution"],
    "T1204.003": ["execution"],
    "T1205": ["defense-evasion", "persistence", "command-and-control"],
    "T1205.001": ["defense-evasion", "persistence", "command-and-control"],
    "T1205.002": ["defense-evasion", "persistence", "command-and-control"],
    "T1207": ["defense-evasion"],
    "T1210": ["lateral-movement"],
    "T1211": ["defense-evasion"],
    "T1212": ["credential-access"],
    "T1213": ["collection"],
    "T1213.001": ["collection"],
    "T1213.002": ["collection"],
    "T1213.003": ["collection"],
    "T1216": ["defense-evasion"],
    "T1216.001": ["defense-evasion"],
    "T1216.002": ["defense-evasion"],
    "T1217": ["discovery"],
    "T1218": ["defense-evasion"],
    "T1218.001": ["defense-evasion"],
    "T1218.002": ["defense-evasion"],
    "T1218.003": ["defense-evasion"],
    "T1218.004": ["defense-evasion"],
    "T1218.005": ["defense-evasion"],
    "T1218.007": ["defense-evasion"],
    "T1218.008": ["defense-evasion"],
    "T1218.009": ["defense-evasion"],
    "T1218.010": ["defense-evasion"],
    "T1218.011": ["defense-evasion"],
    "T1218.012": ["defense-evasion"],
    "T1218.013": ["defense-evasion"],
    "T1218.014": ["defense-evasion"],
    "T1218.015": ["defense-evasion"],
    "T1219": ["command-and-control"],
    "T1220": ["defense-evasion"],
    "T1221": ["defense-evasion"],
    "T1222": ["defense-evasion"],
    "T1222.001": ["defense-evasion"],
    "T1222.002": ["defense-evasion"],
    "T1480": ["defense-evasion"],
    "T1480.001": ["defense-evasion"],
    "T1482": ["discovery"],
    "T1484": ["defense-evasion", "privilege-escalation"],
    "T1484.001": ["defense-evasion", "privilege-escalation"],
    "T1484.002": ["defense-evasion", "privilege-escalation"],
    "T1485": ["impact"],
    "T1486": ["impact"],
    "T1489": ["impact"],
    "T1490": ["impact"],
    "T1491": ["impact"],
    "T1491.001": ["impact"],
    "T1491.002": ["impact"],
    "T1495": ["impact"],
    "T1496": ["impact"],
    "T1497": ["defense-evasion", "discovery"],
    "T1497.001": ["defense-evasion", "discovery"],
    "T1497.002": ["defense-evasion", "discovery"],
    "T1497.003": ["defense-evasion", "discovery"],
    "T1498": ["impact"],
    "T1498.001": ["impact"],
    "T1498.002": ["impact"],
    "T1499": ["impact"],
    "T1499.001": ["impact"],
    "T1499.002": ["impact"],
    "T1499.003": ["impact"],
    "T1499.004": ["impact"],
    "T1505": ["persistence"],
    "T1505.001": ["persistence"],
    "T1505.002": ["persistence"],
    "T1505.003": ["persistence"],
    "T1505.004": ["persistence"],
    "T1505.005": ["persistence"],
    "T1518": ["discovery"],
    "T1518.001": ["discovery"],
    "T1525": ["persistence"],
    "T1526": ["discovery"],
    "T1528": ["credential-access"],
    "T1529": ["impact"],
    "T1530": ["collection"],
    "T1531": ["impact"],
    "T1534": ["lateral-movement"],
    "T1535": ["defense-evasion"],
    "T1537": ["exfiltration"],
    "T1538": ["discovery"],
    "T1539": ["credential-access"],
    "T1542": ["defense-evasion", "persistence"],
    "T1542.001": ["persistence", "defense-evasion"],
    "T1542.002": ["persistence", "defense-evasion"],
    "T1542.003": ["persistence", "defense-evasion"],
    "T1542.004": ["defense-evasion", "persistence"],
    "T1542.005": ["defense-evasion", "persistence"],
    "T1543": ["persistence", "privilege-escalation"],
    "T1543.001": ["persistence", "privilege-escalation"],
    "T1543.002": ["persistence", "privilege-escalation"],
    "T1543.003": ["persistence", "privilege-escalation"],
    "T1543.004": ["persistence", "privilege-escalation"],
    "T1543.005": ["persistence", "privilege-escalation"],
    "T1546": ["privilege-escalation", "persistence"],
    "T1546.001": ["privilege-escalation", "persistence"],
    "T1546.002": ["privilege-escalation", "persistence"],
    "T1546.003": ["privilege-escalation", "persistence"],
    "T1546.004": ["privilege-escalation", "persistence"],
    "T1546.005": ["privilege-escalation", "persistence"],
    "T1546.006": ["privilege-escalation", "persistence"],
    "T1546.007": ["privilege-escalation", "persistence"],
    "T1546.008": ["privilege-escalation", "persistence"],
    "T1546.009": ["privilege-escalation", "persistence"],
    "T1546.010": ["privilege-escalation", "persistence"],
    "T1546.011": ["privilege-escalation", "persistence"],
    "T1546.012": ["privilege-escalation", "persistence"],
    "T1546.013": ["privilege-escalation", "persistence"],
    "T1546.014": ["privilege-escalation", "persistence"],
    "T1546.015": ["privilege-escalation", "persistence"],
    "T1546.016": ["privilege-escalation", "persistence"],
    "T1547": ["persistence", "privilege-escalation"],
    "T1547.001": ["persistence", "privilege-escalation"],
    "T1547.002": ["persistence", "privilege-escalation"],
    "T1547.003": ["persistence", "privilege-escalation"],
    "T1547.004": ["persistence", "privilege-escalation"],
    "T1547.005": ["persistence", "privilege-escalation"],
    "T1547.006": ["persistence", "privilege-escalation"],
    "T1547.007": ["persistence", "privilege-escalation"],
    "T1547.008": ["persistence", "privilege-escalation"],
    "T1547.009": ["persistence", "privilege-escalation"],
    "T1547.010": ["persistence", "privilege-escalation"],
    "T1547.012": ["persistence", "privilege-escalation"],
    "T1547.013": ["persistence", "privilege-escalation"],
    "T1547.014": ["persistence", "privilege-escalation"],
    "T1547.015": ["persistence", "privilege-escalation"],
    "T1548": ["privilege-escalation", "defense-evasion"],
    "T1548.001": ["privilege-escalation", "defense-evasion"],
    "T1548.002": ["privilege-escalation", "defense-evasion"],
    "T1548.003": ["privilege-escalation", "defense-evasion"],
    "T1548.004": ["privilege-escalation", "defense-evasion"],
    "T1548.005": ["privilege-escalation", "defense-evasion"],
    "T1548.006": ["defense-evasion", "privilege-escalation"],
    "T1550": ["defense-evasion", "lateral-movement"],
    "T1550.001": ["defense-evasion", "lateral-movement"],
    "T1550.002": ["defense-evasion", "lateral-movement"],
    "T1550.003": ["defense-evasion", "lateral-movement"],
    "T1550.004": ["defense-evasion", "lateral-movement"],
    "T1552": ["credential-access"],
    "T1552.001": ["credential-access"],
    "T1552.002": ["credential-access"],
    "T1552.003": ["credential-access"],
    "T1552.004": ["credential-access"],
    "T1552.005": ["credential-access"],
    "T1552.006": ["credential-access"],
    "T1552.007": ["credential-access"],
    "T1552.008": ["credential-access"],
    "T1553": ["defense-evasion"],
    "T1553.001": ["defense-evasion"],
    "T1553.002": ["defense-evasion"],
    "T1553.003": ["defense-evasion"],
    "T1553.004": ["defense-evasion"],
    "T1553.005": ["defense-evasion"],
    "T1553.006": ["defense-evasion"],
    "T1554": ["persistence"],
    "T1555": ["credential-access"],
    "T1555.001": ["credential-access"],
    "T1555.002": ["credential-access"],
    "T1555.003": ["credential-access"],
    "T1555.004": ["credential-access"],
    "T1555.005": ["credential-access"],
    "T1555.006": ["credential-access"],
    "T1556": ["credential-access", "defense-evasion", "persistence"],
    "T1556.001": ["credential-access", "defense-evasion", "persistence"],
    "T1556.002": ["credential-access", "defense-evasion", "persistence"],
    "T1556.003": ["credential-access", "defense-evasion", "persistence"],
    "T1556.004": ["credential-access", "defense-evasion", "persistence"],
    "T1556.005": ["credential-access", "defense-evasion", "persistence"],
    "T1556.006": ["credential-access", "defense-evasion", "persistence"],
    "T1556.007": ["credential-access", "defense-evasion", "persistence"],
    "T1556.008": ["credential-access", "defense-evasion", "persistence"],
    "T1556.009": ["credential-access", "defense-evasion", "persistence"],
    "T1557": ["credential-access", "collection"],
    "T1557.001": ["credential-access", "collection"],
    "T1557.002": ["credential-access", "collection"],
    "T1557.003": ["credential-access", "collection"],
    "T1558": ["credential-access"],
    "T1558.001": ["credential-access"],
    "T1558.002": ["credential-access"],
    "T1558.003": ["credential-access"],
    "T1558.004": ["credential-access"],
    "T1559": ["execution"],
    "T1559.001": ["execution"],
    "T1559.002": ["execution"],
    "T1559.003": ["execution"],
    "T1560": ["collection"],
    "T1560.001": ["collection"],
    "T1560.002": ["collection"],
    "T1560.003": ["collection"],
    "T1561": ["impact"],
    "T1561.001": ["impact"],
    "T1561.002": ["impact"],
    "T1562": ["defense-evasion"],
    "T1562.001": ["defense-evasion"],
    "T1562.002": ["defense-evasion"],
    "T1562.003": ["defense-evasion"],
    "T1562.004": ["defense-evasion"],
    "T1562.006": ["defense-evasion"],
    "T1562.007": ["defense-evasion"],
    "T1562.008": ["defense-evasion"],
    "T1562.009": ["defense-evasion"],
    "T1562.010": ["defense-evasion"],
    "T1562.011": ["defense-evasion"],
    "T1562.012": ["defense-evasion"],
    "T1563": ["lateral-movement"],
    "T1563.001": ["lateral-movement"],
    "T1563.002": ["lateral-movement"],
    "T1564": ["defense-evasion"],
    "T1564.001": ["defense-evasion"],
    "T1564.002": ["defense-evasion"],
    "T1564.003": ["defense-evasion"],
    "T1564.004": ["defense-evasion"],
    "T1564.005": ["defense-evasion"],
    "T1564.006": ["defense-evasion"],
    "T1564.007": ["defense-evasion"],
    "T1564.008": ["defense-evasion"],
    "T1564.009": ["defense-evasion"],
    "T1564.010": ["defense-evasion"],
    "T1564.011": ["defense-evasion"],
    "T1564.012": ["defense-evasion"],
    "T1565": ["impact"],
    "T1565.001": ["impact"],
    "T1565.002": ["impact"],
    "T1565.003": ["impact"],
    "T1566": ["initial-access"],
    "T1566.001": ["initial-access"],
    "T1566.002": ["initial-access"],
    "T1566.003": ["initial-access"],
    "T1566.004": ["initial-access"],
    "T1567": ["exfiltration"],
    "T1567.001": ["exfiltration"],
    "T1567.002": ["exfiltration"],
    "T1567.003": ["exfiltration"],
    "T1567.004": ["exfiltration"],
    "T1568": ["command-and-control"],
    "T1568.001": ["command-and-control"],
    "T1568.002": ["command-and-control"],
    "T1568.003": ["command-and-control"],
    "T1569": ["execution"],
    "T1569.001": ["execution"],
    "T1569.002": ["execution"],
    "T1570": ["lateral-movement"],
    "T1571": ["command-and-control"],
    "T1572": ["command-and-control"],
    "T1573": ["command-and-control"],
    "T1573.001": ["command-and-control"],
    "T1573.002": ["command-and-control"],
    "T1574": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.001": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.002": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.004": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.005": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.006": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.007": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.008": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.009": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.010": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.011": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.012": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.013": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1574.014": ["persistence", "privilege-escalation", "defense-evasion"],
    "T1578": ["defense-evasion"],
    "T1578.001": ["defense-evasion"],
    "T1578.002": ["defense-evasion"],
    "T1578.003": ["defense-evasion"],
    "T1578.004": ["defense-evasion"],
    "T1578.005": ["defense-evasion"],
    "T1580": ["discovery"],
    "T1583": ["resource-development"],
    "T1583.001": ["resource-development"],
    "T1583.002": ["resource-development"],
    "T1583.003": ["resource-development"],
    "T1583.004": ["resource-development"],
    "T1583.005": ["resource-development"],
    "T1583.006": ["resource-development"],
    "T1583.007": ["resource-development"],
    "T1583.008": ["resource-development"],
    "T1584": ["resource-development"],
    "T1584.001": ["resource-development"],
    "T1584.002": ["resource-development"],
    "T1584.003": ["resource-development"],
    "T1584.004": ["resource-development"],
    "T1584.005": ["resource-development"],
    "T1584.006": ["resource-development"],
    "T1584.007": ["resource-development"],
    "T1584.008": ["resource-development"],
    "T1585": ["resource-development"],
    "T1585.001": ["resource-development"],
    "T1585.002": ["resource-development"],
    "T1585.003": ["resource-development"],
    "T1586": ["resource-development"],
    "T1586.001": ["resource-development"],
    "T1586.002": ["resource-development"],
    "T1586.003": ["resource-development"],
    "T1587": ["resource-development"],
    "T1587.001": ["resource-development"],
    "T1587.002": ["resource-development"],
    "T1587.003": ["resource-development"],
    "T1587.004": ["resource-development"],
    "T1588": ["resource-development"],
    "T1588.001": ["resource-development"],
    "T1588.002": ["resource-development"],
    "T1588.003": ["resource-development"],
    "T1588.004": ["resource-development"],
    "T1588.005": ["resource-development"],
    "T1588.006": ["resource-development"],
    "T1588.007": ["resource-development"],
    "T1589": ["reconnaissance"],
    "T1589.001": ["reconnaissance"],
    "T1589.002": ["reconnaissance"],
    "T1589.003": ["reconnaissance"],
    "T1590": ["reconnaissance"],
    "T1590.001": ["reconnaissance"],
    "T1590.002": ["reconnaissance"],
    "T1590.003": ["reconnaissance"],
    "T1590.004": ["reconnaissance"],
    "T1590.005": ["reconnaissance"],
    "T1590.006": ["reconnaissance"],
    "T1591": ["reconnaissance"],
    "T1591.001": ["reconnaissance"],
    "T1591.002": ["reconnaissance"],
    "T1591.003": ["reconnaissance"],
    "T1591.004": ["reconnaissance"],
    "T1592": ["reconnaissance"],
    "T1592.001": ["reconnaissance"],
    "T1592.002": ["reconnaissance"],
    "T1592.003": ["reconnaissance"],
    "T1592.004": ["reconnaissance"],
    "T1593": ["reconnaissance"],
    "T1593.001": ["reconnaissance"],
    "T1593.002": ["reconnaissance"],
    "T1593.003": ["reconnaissance"],
    "T1594": ["reconnaissance"],
    "T1595": ["reconnaissance"],
    "T1595.001": ["reconnaissance"],
    "T1595.002": ["reconnaissance"],
    "T1595.003": ["reconnaissance"],
    "T1596": ["reconnaissance"],
    "T1596.001": ["reconnaissance"],
    "T1596.002": ["reconnaissance"],
    "T1596.003": ["reconnaissance"],
    "T1596.004": ["reconnaissance"],
    "T1596.005": ["reconnaissance"],
    "T1597": ["reconnaissance"],
    "T1597.001": ["reconnaissance"],
    "T1597.002": ["reconnaissance"],
    "T1598": ["reconnaissance"],
    "T1598.001": ["reconnaissance"],
    "T1598.002": ["reconnaissance"],
    "T1598.003": ["reconnaissance"],
    "T1598.004": ["reconnaissance"],
    "T1599": ["defense-evasion"],
    "T1599.001": ["defense-evasion"],
    "T1600": ["defense-evasion"],
    "T1600.001": ["defense-evasion"],
    "T1600.002": ["defense-evasion"],
    "T1601": ["defense-evasion"],
    "T1601.001": ["defense-evasion"],
    "T1601.002": ["defense-evasion"],
    "T1602": ["collection"],
    "T1602.001": ["collection"],
    "T1602.002": ["collection"],
    "T1606": ["credential-access"],
    "T1606.001": ["credential-access"],
    "T1606.002": ["credential-access"],
    "T1608": ["resource-development"],
    "T1608.001": ["resource-development"],
    "T1608.002": ["resource-development"],
    "T1608.003": ["resource-development"],
    "T1608.004": ["resource-development"],
    "T1608.005": ["resource-development"],
    "T1608.006": ["resource-development"],
    "T1609": ["execution"],
    "T1610": ["defense-evasion", "execution"],
    "T1611": ["privilege-escalation"],
    "T1612": ["defense-evasion"],
    "T1613": ["discovery"],
    "T1614": ["discovery"],
    "T1614.001": ["discovery"],
    "T1615": ["discovery"],
    "T1619": ["discovery"],
    "T1620": ["defense-evasion"],
    "T1621": ["credential-access"],
    "T1622": ["defense-evasion", "discovery"],
    "T1647": ["defense-evasion"],
    "T1648": ["execution"],
    "T1649": ["credential-access"],
    "T1650": ["resource-development"],
    "T1651": ["execution"],
    "T1652": ["discovery"],
    "T1653": ["persistence"],
    "T1654": ["discovery"],
    "T1656": ["defense-evasion"],
    "T1657": ["impact"],
    "T1659": ["initial-access", "command-and-control"],
    "T1665": ["command-and-control"],
}
mitre_attack_intrusion_sets: Dict[str, str] = {
    "G0001": "Axiom",
    "G0002": "Moafee",
    "G0003": "Cleaver",
    "G0004": "Ke3chang",
    "G0005": "APT12",
    "G0006": "APT1",
    "G0007": "APT28",
    "G0008": "Carbanak",
    "G0009": "Deep Panda",
    "G0010": "Turla",
    "G0011": "PittyTiger",
    "G0012": "Darkhotel",
    "G0013": "APT30",
    "G0016": "APT29",
    "G0017": "DragonOK",
    "G0018": "admin@338",
    "G0019": "Naikon",
    "G0020": "Equation",
    "G0021": "Molerats",
    "G0022": "APT3",
    "G0023": "APT16",
    "G0024": "Putter Panda",
    "G0025": "APT17",
    "G0026": "APT18",
    "G0027": "Threat Group-3390",
    "G0028": "Threat Group-1314",
    "G0029": "Scarlet Mimic",
    "G0030": "Lotus Blossom",
    "G0032": "Lazarus Group",
    "G0033": "Poseidon Group",
    "G0034": "Sandworm Team",
    "G0035": "Dragonfly",
    "G0036": "GCMAN",
    "G0037": "FIN6",
    "G0038": "Stealth Falcon",
    "G0039": "Suckfly",
    "G0040": "Patchwork",
    "G0041": "Strider",
    "G0043": "Group5",
    "G0044": "Winnti Group",
    "G0045": "menuPass",
    "G0046": "FIN7",
    "G0047": "Gamaredon Group",
    "G0048": "RTM",
    "G0049": "OilRig",
    "G0050": "APT32",
    "G0051": "FIN10",
    "G0052": "CopyKittens",
    "G0053": "FIN5",
    "G0054": "Sowbug",
    "G0055": "NEODYMIUM",
    "G0056": "PROMETHIUM",
    "G0059": "Magic Hound",
    "G0060": "BRONZE BUTLER",
    "G0061": "FIN8",
    "G0062": "TA459",
    "G0063": "BlackOasis",
    "G0064": "APT33",
    "G0065": "Leviathan",
    "G0066": "Elderwood",
    "G0067": "APT37",
    "G0068": "PLATINUM",
    "G0069": "MuddyWater",
    "G0070": "Dark Caracal",
    "G0071": "Orangeworm",
    "G0073": "APT19",
    "G0075": "Rancor",
    "G0076": "Thrip",
    "G0077": "Leafminer",
    "G0078": "Gorgon Group",
    "G0079": "DarkHydrus",
    "G0080": "Cobalt Group",
    "G0081": "Tropic Trooper",
    "G0082": "APT38",
    "G0083": "SilverTerrier",
    "G0084": "Gallmaker",
    "G0085": "FIN4",
    "G0087": "APT39",
    "G0088": "TEMP.Veles",
    "G0089": "The White Company",
    "G0090": "WIRTE",
    "G0091": "Silence",
    "G0092": "TA505",
    "G0093": "GALLIUM",
    "G0094": "Kimsuky",
    "G0095": "Machete",
    "G0096": "APT41",
    "G0097": "Bouncing Golf",
    "G0098": "BlackTech",
    "G0099": "APT-C-36",
    "G0100": "Inception",
    "G0102": "Wizard Spider",
    "G0103": "Mofang",
    "G0105": "DarkVishnya",
    "G0106": "Rocke",
    "G0107": "Whitefly",
    "G0108": "Blue Mockingbird",
    "G0112": "Windshift",
    "G0114": "Chimera",
    "G0115": "GOLD SOUTHFIELD",
    "G0117": "Fox Kitten",
    "G0119": "Indrik Spider",
    "G0120": "Evilnum",
    "G0121": "Sidewinder",
    "G0122": "Silent Librarian",
    "G0123": "Volatile Cedar",
    "G0124": "Windigo",
    "G0125": "HAFNIUM",
    "G0126": "Higaisa",
    "G0127": "TA551",
    "G0128": "ZIRCONIUM",
    "G0129": "Mustang Panda",
    "G0130": "Ajax Security Team",
    "G0131": "Tonto Team",
    "G0133": "Nomadic Octopus",
    "G0134": "Transparent Tribe",
    "G0135": "BackdoorDiplomacy",
    "G0136": "IndigoZebra",
    "G0137": "Ferocious Kitten",
    "G0138": "Andariel",
    "G0139": "TeamTNT",
    "G0140": "LazyScripter",
    "G0142": "Confucius",
    "G0143": "Aquatic Panda",
    "G1001": "HEXANE",
    "G1002": "BITTER",
    "G1003": "Ember Bear",
    "G1004": "LAPSUS$",
    "G1005": "POLONIUM",
    "G1006": "Earth Lusca",
    "G1007": "Aoqin Dragon",
    "G1008": "SideCopy",
    "G1009": "Moses Staff",
    "G1011": "EXOTIC LILY",
    "G1012": "CURIUM",
    "G1013": "Metador",
    "G1014": "LuminousMoth",
    "G1015": "Scattered Spider",
    "G1016": "FIN13",
    "G1017": "Volt Typhoon",
    "G1018": "TA2541",
    "G1019": "MoustachedBouncer",
    "G1020": "Mustard Tempest",
    "G1021": "Cinnamon Tempest",
    "G1022": "ToddyCat",
    "G1023": "APT5",
    "G1024": "Akira",
    "G1026": "Malteiro",
    "G1028": "APT-C-23",
}
mitre_attack_software: Dict[str, str] = {
    "S0001": "Trojan.Mebromi",
    "S0002": "Mimikatz",
    "S0003": "RIPTIDE",
    "S0004": "TinyZBot",
    "S0005": "Windows Credential Editor",
    "S0006": "pwdump",
    "S0007": "Skeleton Key",
    "S0008": "gsecdump",
    "S0009": "Hikit",
    "S0010": "Lurid",
    "S0011": "Taidoor",
    "S0012": "PoisonIvy",
    "S0013": "PlugX",
    "S0014": "BS2005",
    "S0015": "Ixeshe",
    "S0016": "P2P ZeuS",
    "S0017": "BISCUIT",
    "S0018": "Sykipot",
    "S0019": "Regin",
    "S0020": "China Chopper",
    "S0021": "Derusbi",
    "S0022": "Uroburos",
    "S0023": "CHOPSTICK",
    "S0024": "Dyre",
    "S0025": "CALENDAR",
    "S0026": "GLOOXMAIL",
    "S0027": "Zeroaccess",
    "S0028": "SHIPSHAPE",
    "S0029": "PsExec",
    "S0030": "Carbanak",
    "S0031": "BACKSPACE",
    "S0032": "gh0st RAT",
    "S0033": "NetTraveler",
    "S0034": "NETEAGLE",
    "S0035": "SPACESHIP",
    "S0036": "FLASHFLOOD",
    "S0037": "HAMMERTOSS",
    "S0038": "Duqu",
    "S0039": "Net",
    "S0040": "HTRAN",
    "S0041": "Wiper",
    "S0042": "LOWBALL",
    "S0043": "BUBBLEWRAP",
    "S0044": "JHUHUGIT",
    "S0045": "ADVSTORESHELL",
    "S0046": "CozyCar",
    "S0047": "Hacking Team UEFI Rootkit",
    "S0048": "PinchDuke",
    "S0049": "GeminiDuke",
    "S0050": "CosmicDuke",
    "S0051": "MiniDuke",
    "S0052": "OnionDuke",
    "S0053": "SeaDuke",
    "S0054": "CloudDuke",
    "S0055": "RARSTONE",
    "S0056": "Net Crawler",
    "S0057": "Tasklist",
    "S0058": "SslMM",
    "S0059": "WinMM",
    "S0060": "Sys10",
    "S0061": "HDoor",
    "S0062": "DustySky",
    "S0063": "SHOTPUT",
    "S0064": "ELMER",
    "S0065": "4H RAT",
    "S0066": "3PARA RAT",
    "S0067": "pngdowner",
    "S0068": "httpclient",
    "S0069": "BLACKCOFFEE",
    "S0070": "HTTPBrowser",
    "S0071": "hcdLoader",
    "S0072": "OwaAuth",
    "S0073": "ASPXSpy",
    "S0074": "Sakula",
    "S0075": "Reg",
    "S0076": "FakeM",
    "S0077": "CallMe",
    "S0078": "Psylo",
    "S0079": "MobileOrder",
    "S0080": "Mivast",
    "S0081": "Elise",
    "S0082": "Emissary",
    "S0083": "Misdat",
    "S0084": "Mis-Type",
    "S0085": "S-Type",
    "S0086": "ZLib",
    "S0087": "Hi-Zor",
    "S0088": "Kasidet",
    "S0089": "BlackEnergy",
    "S0090": "Rover",
    "S0091": "Epic",
    "S0092": "Agent.btz",
    "S0093": "Backdoor.Oldrea",
    "S0094": "Trojan.Karagany",
    "S0095": "ftp",
    "S0096": "Systeminfo",
    "S0097": "Ping",
    "S0098": "T9000",
    "S0099": "Arp",
    "S0100": "ipconfig",
    "S0101": "ifconfig",
    "S0102": "nbtstat",
    "S0103": "route",
    "S0104": "netstat",
    "S0105": "dsquery",
    "S0106": "cmd",
    "S0107": "Cherry Picker",
    "S0108": "netsh",
    "S0109": "WEBC2",
    "S0110": "at",
    "S0111": "schtasks",
    "S0112": "ROCKBOOT",
    "S0113": "Prikormka",
    "S0114": "BOOTRASH",
    "S0115": "Crimson",
    "S0116": "UACMe",
    "S0117": "XTunnel",
    "S0118": "Nidiran",
    "S0119": "Cachedump",
    "S0120": "Fgdump",
    "S0121": "Lslsass",
    "S0122": "Pass-The-Hash Toolkit",
    "S0123": "xCmd",
    "S0124": "Pisloader",
    "S0125": "Remsec",
    "S0126": "ComRAT",
    "S0127": "BBSRAT",
    "S0128": "BADNEWS",
    "S0129": "AutoIt backdoor",
    "S0130": "Unknown Logger",
    "S0131": "TINYTYPHON",
    "S0132": "H1N1",
    "S0133": "Miner-C",
    "S0134": "Downdelph",
    "S0135": "HIDEDRV",
    "S0136": "USBStealer",
    "S0137": "CORESHELL",
    "S0138": "OLDBAIT",
    "S0139": "PowerDuke",
    "S0140": "Shamoon",
    "S0141": "Winnti for Windows",
    "S0142": "StreamEx",
    "S0143": "Flame",
    "S0144": "ChChes",
    "S0145": "POWERSOURCE",
    "S0146": "TEXTMATE",
    "S0147": "Pteranodon",
    "S0148": "RTM",
    "S0149": "MoonWind",
    "S0150": "POSHSPY",
    "S0151": "HALFBAKED",
    "S0152": "EvilGrab",
    "S0153": "RedLeaves",
    "S0154": "Cobalt Strike",
    "S0155": "WINDSHIELD",
    "S0156": "KOMPROGO",
    "S0157": "SOUNDBITE",
    "S0158": "PHOREAL",
    "S0159": "SNUGRIDE",
    "S0160": "certutil",
    "S0161": "XAgentOSX",
    "S0162": "Komplex",
    "S0163": "Janicab",
    "S0164": "TDTESS",
    "S0165": "OSInfo",
    "S0166": "RemoteCMD",
    "S0167": "Matryoshka",
    "S0168": "Gazer",
    "S0169": "RawPOS",
    "S0170": "Helminth",
    "S0171": "Felismus",
    "S0172": "Reaver",
    "S0173": "FLIPSIDE",
    "S0174": "Responder",
    "S0175": "meek",
    "S0176": "Wingbird",
    "S0177": "Power Loader",
    "S0178": "Truvasys",
    "S0179": "MimiPenguin",
    "S0180": "Volgmer",
    "S0181": "FALLCHILL",
    "S0182": "FinFisher",
    "S0183": "Tor",
    "S0184": "POWRUNER",
    "S0185": "SEASHARPEE",
    "S0186": "DownPaper",
    "S0187": "Daserf",
    "S0188": "Starloader",
    "S0189": "ISMInjector",
    "S0190": "BITSAdmin",
    "S0191": "Winexe",
    "S0192": "Pupy",
    "S0193": "Forfiles",
    "S0194": "PowerSploit",
    "S0195": "SDelete",
    "S0196": "PUNCHBUGGY",
    "S0197": "PUNCHTRACK",
    "S0198": "NETWIRE",
    "S0199": "TURNEDUP",
    "S0200": "Dipsind",
    "S0201": "JPIN",
    "S0202": "adbupd",
    "S0203": "Hydraq",
    "S0204": "Briba",
    "S0205": "Naid",
    "S0206": "Wiarp",
    "S0207": "Vasport",
    "S0208": "Pasam",
    "S0210": "Nerex",
    "S0211": "Linfo",
    "S0212": "CORALDECK",
    "S0213": "DOGCALL",
    "S0214": "HAPPYWORK",
    "S0215": "KARAE",
    "S0216": "POORAIM",
    "S0217": "SHUTTERSPEED",
    "S0218": "SLOWDRIFT",
    "S0219": "WINERACK",
    "S0220": "Chaos",
    "S0221": "Umbreon",
    "S0222": "CCBkdr",
    "S0223": "POWERSTATS",
    "S0224": "Havij",
    "S0225": "sqlmap",
    "S0226": "Smoke Loader",
    "S0227": "spwebmember",
    "S0228": "NanHaiShu",
    "S0229": "Orz",
    "S0230": "ZeroT",
    "S0231": "Invoke-PSImage",
    "S0232": "HOMEFRY",
    "S0233": "MURKYTOP",
    "S0234": "Bandook",
    "S0235": "CrossRAT",
    "S0236": "Kwampirs",
    "S0237": "GravityRAT",
    "S0238": "Proxysvc",
    "S0239": "Bankshot",
    "S0240": "ROKRAT",
    "S0241": "RATANKBA",
    "S0242": "SynAck",
    "S0243": "DealersChoice",
    "S0244": "Comnie",
    "S0245": "BADCALL",
    "S0246": "HARDRAIN",
    "S0247": "NavRAT",
    "S0248": "yty",
    "S0249": "Gold Dragon",
    "S0250": "Koadic",
    "S0251": "Zebrocy",
    "S0252": "Brave Prince",
    "S0253": "RunningRAT",
    "S0254": "PLAINTEE",
    "S0255": "DDKONG",
    "S0256": "Mosquito",
    "S0257": "VERMIN",
    "S0258": "RGDoor",
    "S0259": "InnaputRAT",
    "S0260": "InvisiMole",
    "S0261": "Catchamas",
    "S0262": "QuasarRAT",
    "S0263": "TYPEFRAME",
    "S0264": "OopsIE",
    "S0265": "Kazuar",
    "S0266": "TrickBot",
    "S0267": "FELIXROOT",
    "S0268": "Bisonal",
    "S0269": "QUADAGENT",
    "S0270": "RogueRobin",
    "S0271": "KEYMARBLE",
    "S0272": "NDiskMonitor",
    "S0273": "Socksbot",
    "S0274": "Calisto",
    "S0275": "UPPERCUT",
    "S0276": "Keydnap",
    "S0277": "FruitFly",
    "S0278": "iKitten",
    "S0279": "Proton",
    "S0280": "MirageFox",
    "S0281": "Dok",
    "S0282": "MacSpy",
    "S0283": "jRAT",
    "S0284": "More_eggs",
    "S0302": "Twitoor",
    "S0330": "Zeus Panda",
    "S0331": "Agent Tesla",
    "S0332": "Remcos",
    "S0333": "UBoatRAT",
    "S0334": "DarkComet",
    "S0335": "Carbon",
    "S0336": "NanoCore",
    "S0337": "BadPatch",
    "S0338": "Cobian RAT",
    "S0339": "Micropsia",
    "S0340": "Octopus",
    "S0341": "Xbash",
    "S0342": "GreyEnergy",
    "S0343": "Exaramel for Windows",
    "S0344": "Azorult",
    "S0345": "Seasalt",
    "S0346": "OceanSalt",
    "S0347": "AuditCred",
    "S0348": "Cardinal RAT",
    "S0349": "LaZagne",
    "S0350": "zwShell",
    "S0351": "Cannon",
    "S0352": "OSX_OCEANLOTUS.D",
    "S0353": "NOKKI",
    "S0354": "Denis",
    "S0355": "Final1stspy",
    "S0356": "KONNI",
    "S0357": "Impacket",
    "S0358": "Ruler",
    "S0359": "Nltest",
    "S0360": "BONDUPDATER",
    "S0361": "Expand",
    "S0362": "Linux Rabbit",
    "S0363": "Empire",
    "S0364": "RawDisk",
    "S0365": "Olympic Destroyer",
    "S0366": "WannaCry",
    "S0367": "Emotet",
    "S0368": "NotPetya",
    "S0369": "CoinTicker",
    "S0370": "SamSam",
    "S0371": "POWERTON",
    "S0372": "LockerGoga",
    "S0373": "Astaroth",
    "S0374": "SpeakUp",
    "S0375": "Remexi",
    "S0376": "HOPLIGHT",
    "S0377": "Ebury",
    "S0378": "PoshC2",
    "S0379": "Revenge RAT",
    "S0380": "StoneDrill",
    "S0381": "FlawedAmmyy",
    "S0382": "ServHelper",
    "S0383": "FlawedGrace",
    "S0384": "Dridex",
    "S0385": "njRAT",
    "S0386": "Ursnif",
    "S0387": "KeyBoy",
    "S0388": "YAHOYAH",
    "S0389": "JCry",
    "S0390": "SQLRat",
    "S0391": "HAWKBALL",
    "S0393": "PowerStallion",
    "S0394": "HiddenWasp",
    "S0395": "LightNeuron",
    "S0396": "EvilBunny",
    "S0397": "LoJax",
    "S0398": "HyperBro",
    "S0400": "RobbinHood",
    "S0401": "Exaramel for Linux",
    "S0402": "OSX/Shlayer",
    "S0404": "esentutl",
    "S0409": "Machete",
    "S0410": "Fysbis",
    "S0412": "ZxShell",
    "S0413": "MailSniper",
    "S0414": "BabyShark",
    "S0415": "BOOSTWRITE",
    "S0416": "RDFSNIFFER",
    "S0417": "GRIFFON",
    "S0428": "PoetRAT",
    "S0430": "Winnti for Linux",
    "S0431": "HotCroissant",
    "S0433": "Rifdoor",
    "S0434": "Imminent Monitor",
    "S0435": "PLEAD",
    "S0436": "TSCookie",
    "S0437": "Kivars",
    "S0438": "Attor",
    "S0439": "Okrum",
    "S0441": "PowerShower",
    "S0442": "VBShower",
    "S0443": "MESSAGETAP",
    "S0444": "ShimRat",
    "S0445": "ShimRatReporter",
    "S0446": "Ryuk",
    "S0447": "Lokibot",
    "S0448": "Rising Sun",
    "S0449": "Maze",
    "S0450": "SHARPSTATS",
    "S0451": "LoudMiner",
    "S0452": "USBferry",
    "S0453": "Pony",
    "S0454": "Cadelspy",
    "S0455": "Metamorfo",
    "S0456": "Aria-body",
    "S0457": "Netwalker",
    "S0458": "Ramsay",
    "S0459": "MechaFlounder",
    "S0460": "Get2",
    "S0461": "SDBbot",
    "S0462": "CARROTBAT",
    "S0464": "SYSCON",
    "S0465": "CARROTBALL",
    "S0466": "WindTail",
    "S0467": "TajMahal",
    "S0468": "Skidmap",
    "S0469": "ABK",
    "S0470": "BBK",
    "S0471": "build_downer",
    "S0472": "down_new",
    "S0473": "Avenger",
    "S0475": "BackConfig",
    "S0476": "Valak",
    "S0477": "Goopy",
    "S0481": "Ragnar Locker",
    "S0482": "Bundlore",
    "S0483": "IcedID",
    "S0484": "Carberp",
    "S0486": "Bonadan",
    "S0487": "Kessel",
    "S0488": "CrackMapExec",
    "S0491": "StrongPity",
    "S0492": "CookieMiner",
    "S0493": "GoldenSpy",
    "S0495": "RDAT",
    "S0496": "REvil",
    "S0497": "Dacls",
    "S0498": "Cryptoistic",
    "S0499": "Hancitor",
    "S0500": "MCMD",
    "S0501": "PipeMon",
    "S0502": "Drovorub",
    "S0503": "FrameworkPOS",
    "S0504": "Anchor",
    "S0508": "ngrok",
    "S0511": "RegDuke",
    "S0512": "FatDuke",
    "S0513": "LiteDuke",
    "S0514": "WellMess",
    "S0515": "WellMail",
    "S0516": "SoreFang",
    "S0517": "Pillowmint",
    "S0518": "PolyglotDuke",
    "S0519": "SYNful Knock",
    "S0520": "BLINDINGCAN",
    "S0521": "BloodHound",
    "S0526": "KGH_SPY",
    "S0527": "CSPY Downloader",
    "S0528": "Javali",
    "S0530": "Melcoz",
    "S0531": "Grandoreiro",
    "S0532": "Lucifer",
    "S0533": "SLOTHFULMEDIA",
    "S0534": "Bazar",
    "S0537": "HyperStack",
    "S0538": "Crutch",
    "S0543": "Spark",
    "S0546": "SharpStage",
    "S0547": "DropBook",
    "S0552": "AdFind",
    "S0553": "MoleNet",
    "S0554": "Egregor",
    "S0556": "Pay2Key",
    "S0559": "SUNBURST",
    "S0560": "TEARDROP",
    "S0561": "GuLoader",
    "S0562": "SUNSPOT",
    "S0564": "BlackMould",
    "S0565": "Raindrop",
    "S0567": "Dtrack",
    "S0568": "EVILNUM",
    "S0569": "Explosive",
    "S0570": "BitPaymer",
    "S0572": "Caterpillar WebShell",
    "S0574": "BendyBear",
    "S0575": "Conti",
    "S0576": "MegaCortex",
    "S0578": "SUPERNOVA",
    "S0579": "Waterbear",
    "S0581": "IronNetInjector",
    "S0582": "LookBack",
    "S0583": "Pysa",
    "S0584": "AppleJeus",
    "S0585": "Kerrdown",
    "S0586": "TAINTEDSCRIBE",
    "S0587": "Penquin",
    "S0588": "GoldMax",
    "S0589": "Sibot",
    "S0590": "NBTscan",
    "S0591": "ConnectWise",
    "S0592": "RemoteUtilities",
    "S0593": "ECCENTRICBANDWAGON",
    "S0594": "Out1",
    "S0595": "ThiefQuest",
    "S0596": "ShadowPad",
    "S0597": "GoldFinder",
    "S0598": "P.A.S. Webshell",
    "S0599": "Kinsing",
    "S0600": "Doki",
    "S0601": "Hildegard",
    "S0603": "Stuxnet",
    "S0604": "Industroyer",
    "S0605": "EKANS",
    "S0606": "Bad Rabbit",
    "S0607": "KillDisk",
    "S0608": "Conficker",
    "S0610": "SideTwist",
    "S0611": "Clop",
    "S0612": "WastedLocker",
    "S0613": "PS1",
    "S0614": "CostaBricks",
    "S0615": "SombRAT",
    "S0616": "DEATHRANSOM",
    "S0617": "HELLOKITTY",
    "S0618": "FIVEHANDS",
    "S0622": "AppleSeed",
    "S0623": "Siloscape",
    "S0624": "Ecipekac",
    "S0625": "Cuba",
    "S0626": "P8RAT",
    "S0627": "SodaMaster",
    "S0628": "FYAnti",
    "S0629": "RainyDay",
    "S0630": "Nebulae",
    "S0631": "Chaes",
    "S0632": "GrimAgent",
    "S0633": "Sliver",
    "S0634": "EnvyScout",
    "S0635": "BoomBox",
    "S0636": "VaporRage",
    "S0637": "NativeZone",
    "S0638": "Babuk",
    "S0639": "Seth-Locker",
    "S0640": "Avaddon",
    "S0641": "Kobalos",
    "S0642": "BADFLICK",
    "S0643": "Peppy",
    "S0644": "ObliqueRAT",
    "S0645": "Wevtutil",
    "S0646": "SpicyOmelette",
    "S0647": "Turian",
    "S0648": "JSS Loader",
    "S0649": "SMOKEDHAM",
    "S0650": "QakBot",
    "S0651": "BoxCaon",
    "S0652": "MarkiRAT",
    "S0653": "xCaon",
    "S0654": "ProLock",
    "S0657": "BLUELIGHT",
    "S0658": "XCSSET",
    "S0659": "Diavol",
    "S0660": "Clambling",
    "S0661": "FoggyWeb",
    "S0662": "RCSession",
    "S0663": "SysUpdate",
    "S0664": "Pandora",
    "S0665": "ThreatNeedle",
    "S0666": "Gelsemium",
    "S0667": "Chrommme",
    "S0668": "TinyTurla",
    "S0669": "KOCTOPUS",
    "S0670": "WarzoneRAT",
    "S0671": "Tomiris",
    "S0672": "Zox",
    "S0673": "DarkWatchman",
    "S0674": "CharmPower",
    "S0677": "AADInternals",
    "S0678": "Torisma",
    "S0679": "Ferocious",
    "S0680": "LitePower",
    "S0681": "Lizar",
    "S0682": "TrailBlazer",
    "S0683": "Peirates",
    "S0684": "ROADTools",
    "S0685": "PowerPunch",
    "S0686": "QuietSieve",
    "S0687": "Cyclops Blink",
    "S0688": "Meteor",
    "S0689": "WhisperGate",
    "S0690": "Green Lambert",
    "S0691": "Neoichor",
    "S0692": "SILENTTRINITY",
    "S0693": "CaddyWiper",
    "S0694": "DRATzarus",
    "S0695": "Donut",
    "S0696": "Flagpro",
    "S0697": "HermeticWiper",
    "S0698": "HermeticWizard",
    "S0699": "Mythic",
    "S1011": "Tarrask",
    "S1012": "PowerLess",
    "S1013": "ZxxZ",
    "S1014": "DanBot",
    "S1015": "Milan",
    "S1016": "MacMa",
    "S1017": "OutSteel",
    "S1018": "Saint Bot",
    "S1019": "Shark",
    "S1020": "Kevin",
    "S1021": "DnsSystem",
    "S1022": "IceApple",
    "S1023": "CreepyDrive",
    "S1024": "CreepySnail",
    "S1025": "Amadey",
    "S1026": "Mongall",
    "S1027": "Heyoka Backdoor",
    "S1028": "Action RAT",
    "S1029": "AuTo Stealer",
    "S1030": "Squirrelwaffle",
    "S1031": "PingPull",
    "S1032": "PyDCrypt",
    "S1033": "DCSrv",
    "S1034": "StrifeWater",
    "S1035": "Small Sieve",
    "S1037": "STARWHALE",
    "S1039": "Bumblebee",
    "S1040": "Rclone",
    "S1041": "Chinoxy",
    "S1042": "SUGARDUMP",
    "S1043": "ccf32",
    "S1044": "FunnyDream",
    "S1046": "PowGoop",
    "S1047": "Mori",
    "S1048": "macOS.OSAMiner",
    "S1049": "SUGARUSH",
    "S1050": "PcShare",
    "S1051": "KEYPLUG",
    "S1052": "DEADEYE",
    "S1053": "AvosLocker",
    "S1058": "Prestige",
    "S1059": "metaMain",
    "S1060": "Mafalda",
    "S1063": "Brute Ratel C4",
    "S1064": "SVCReady",
    "S1065": "Woody RAT",
    "S1066": "DarkTortilla",
    "S1068": "BlackCat",
    "S1070": "Black Basta",
    "S1071": "Rubeus",
    "S1072": "Industroyer2",
    "S1073": "Royal",
    "S1074": "ANDROMEDA",
    "S1075": "KOPILUWAK",
    "S1076": "QUIETCANARY",
    "S1078": "RotaJakiro",
    "S1081": "BADHATCH",
    "S1084": "QUIETEXIT",
    "S1085": "Sardonic",
    "S1086": "Snip3",
    "S1087": "AsyncRAT",
    "S1088": "Disco",
    "S1089": "SharpDisco",
    "S1090": "NightClub",
    "S1091": "Pacu",
    "S1096": "Cheerscrypt",
    "S1097": "HUI Loader",
    "S1099": "Samurai",
    "S1100": "Ninja",
    "S1101": "LoFiSe",
    "S1102": "Pcexter",
    "S1104": "SLOWPULSE",
    "S1105": "COATHANGER",
    "S1106": "NGLite",
    "S1107": "NKAbuse",
    "S1108": "PULSECHECK",
    "S1109": "PACEMAKER",
    "S1110": "SLIGHTPULSE",
    "S1111": "DarkGate",
    "S1112": "STEADYPULSE",
    "S1113": "RAPIDPULSE",
    "S1114": "ZIPLINE",
    "S1115": "WIREFIRE",
    "S1116": "WARPWIRE",
    "S1117": "GLASSTOKEN",
    "S1118": "BUSHWALK",
    "S1119": "LIGHTWIRE",
    "S1120": "FRAMESTING",
    "S1121": "LITTLELAMB.WOOLTEA",
    "S1122": "Mispadu",
    "S1123": "PITSTOP",
    "S1124": "SocGholish",
    "S1125": "AcidRain",
    "S1129": "Akira",
}
