if (typeof Test === 'undefined') require('../tester');

Test('integerValue', function () {
    var MAX = 1e9;

    function t(expected, value, roundingMode) {
        Test.areEqual(String(expected), new BigNumber(String(value)).integerValue(roundingMode).toString());
    }

    BigNumber.config({
        DECIMAL_PLACES: 20,
        ROUNDING_MODE: 7,
        RANGE: 1E9,
        EXPONENTIAL_AT: 1E9
    });

    t('0', '0.000084888060736883027314038572334303632');
    t('-79998', '-79998');
    t('30845717889906383053', '30845717889906383052.56472015469740823');
    t('-446807', '-446806.9227761147047517');
    t('76', '76.0719');
    t('0', '-0.00686876124450101884528');
    t('71710176', '71710176');
    t('79', '79');
    t('-20187079148819697784', '-20187079148819697784');
    t('6309793381', '6309793381');
    t('76162410487880977', '76162410487880976.81307140573688335764284');
    t('7491318926', '7491318926.312122759153805942088431');
    t('74909422733607112719', '74909422733607112719.13179009875');
    t('559', '559.2110851431205');
    t('-20306363016185', '-20306363016184.760368254019194925');
    t('-6717914940586242526', '-6717914940586242525.574537480672174624149');
    t('-5', '-5');
    t('8', '8.0606265529');
    t('24701408591129838', '24701408591129838.22163');
    t('3806471925924246437202', '3806471925924246437201.71339576511057957');
    t('-85883753924431578007564277', '-85883753924431578007564277.399776');
    t('826068555927524695', '826068555927524695');
    t('0', '0.000058441452091833136219167587256');
    t('47353089', '47353089.2019161006899898');
    t('-372', '-372.03039745');
    t('3103', '3103');
    t('5620', '5620.48101861977');
    t('9105547112', '9105547112.1319443692');
    t('7472595749413646784408602', '7472595749413646784408602.1095168949');
    t('335816732795', '335816732794.51601276961965281205689254');
    t('-7911332887831436027', '-7911332887831436027.40569259650318');
    t('-52182567851715', '-52182567851715');
    t('439059239410', '439059239409.55703238112278');
    t('43465109466157505', '43465109466157505');
    t('5390894521', '5390894520.8738091063016');
    t('-583532700278695860335147', '-583532700278695860335146.6674153779');
    t('485669', '485669');
    t('3', '3.284095344472285718254591781467536534546');
    t('0', '-0.4210910538061556801483189');
    t('3134', '3133.654494231705614');
    t('8136642604255255554627376', '8136642604255255554627375.87998');
    t('-54301', '-54301.2807534165371279131810401278');
    t('27796900', '27796900.0685101');
    t('-36', '-36');
    t('-7930085746684', '-7930085746684.2599205');
    t('736988809325', '736988809325');
    t('-181606874378737533', '-181606874378737532.738571718716743');
    t('577064478018279', '577064478018279');
    t('-862419646909', '-862419646909');
    t('3269018763090', '3269018763089.5711045989917554');

    t(0, 0);
    t(0, -0);
    t(Infinity, Infinity);
    t(-Infinity, -Infinity);
    t(NaN, NaN);

    t(1, 0.5);
    t(1, 0.7);
    t(1, 1);
    t(1, 1.1);
    t(1, 1.49999);
    t(0, -0.5);
    t(-1, -0.5000000000000001);
    t(-1, -0.7);
    t(-1, -1);
    t(-1, -1.1);
    t(-1, -1.49999);
    t(-1, -1.5);

    t(9007199254740990, 9007199254740990);
    t(9007199254740991, 9007199254740991);
    t(-9007199254740990, -9007199254740990);
    t(-9007199254740991, -9007199254740991);

    BigNumber.config({EXPONENTIAL_AT: 100});

    t(Number.MAX_VALUE, Number.MAX_VALUE);
    t(-Number.MAX_VALUE, -Number.MAX_VALUE);

    t(536870911, 536870910.5);
    t(536870911, 536870911);
    t(536870911, 536870911.4);
    t(536870912, 536870911.5);
    t(536870912, 536870912);
    t(536870912, 536870912.4);
    t(536870913, 536870912.5);
    t(536870913, 536870913);
    t(536870913, 536870913.4);
    t(1073741823, 1073741822.5);
    t(1073741823, 1073741823);
    t(1073741823, 1073741823.4);
    t(1073741824, 1073741823.5);
    t(1073741824, 1073741824);
    t(1073741824, 1073741824.4);
    t(1073741825, 1073741824.5);
    t(2147483647, 2147483646.5);
    t(2147483647, 2147483647);
    t(2147483647, 2147483647.4);
    t(2147483648, 2147483647.5);
    t(2147483648, 2147483648);
    t(2147483648, 2147483648.4);
    t(2147483649, 2147483648.5);

    t(0, 0.4);
    t(-0, -0.4);
    t(-0, -0.5);
    t(1, 0.6);
    t(-1, -0.6);
    t(2, 1.5);
    t(2, 1.6);
    t(-2, -1.6);
    t(8640000000000000, 8640000000000000);
    t(8640000000000001, 8640000000000001);
    t(8640000000000002, 8640000000000002);
    t(9007199254740990, 9007199254740990);
    t(9007199254740991, 9007199254740991);
    t(1.7976931348623157e+308, 1.7976931348623157e+308);
    t(-8640000000000000, -8640000000000000);
    t(-8640000000000001, -8640000000000001);
    t(-8640000000000002, -8640000000000002);
    t(-9007199254740990, -9007199254740990);
    t(-9007199254740991, -9007199254740991);
    t(-1.7976931348623157e+308, -1.7976931348623157e+308);
    t(Infinity, Infinity);
    t(-Infinity, -Infinity);

    var ulp = Math.pow(2, -1022 - 52);
    var max_denormal = (Math.pow(2, 52) - 1) * ulp;
    var min_normal = Math.pow(2, -1022);
    var max_fraction = Math.pow(2, 52) - 0.5;
    var min_nonfraction = Math.pow(2, 52);
    var max_non_infinite = Number.MAX_VALUE;

    var max_smi31 = Math.pow(2,30) - 1;
    var min_smi31 = -Math.pow(2,30);
    var max_smi32 = Math.pow(2,31) - 1;
    var min_smi32 = -Math.pow(2,31);

    t(0, ulp);
    t(0, max_denormal);
    t(0, min_normal);
    t(0, 0.49999999999999994);
    t(1, 0.5);
    t(Math.pow(2,52), max_fraction);
    t(min_nonfraction, min_nonfraction);
    t(max_non_infinite, max_non_infinite);

    t(max_smi31, max_smi31 - 0.5);
    t(max_smi31 + 1, max_smi31 + 0.5);
    t(max_smi32, max_smi32 - 0.5);
    t(max_smi32 + 1, max_smi32 + 0.5);

    t(-0, -ulp);
    t(-0, -max_denormal);
    t(-0, -min_normal);
    t(-0, -0.49999999999999994);
    t(-Math.pow(2,52)+1, -max_fraction);
    t(-min_nonfraction, -min_nonfraction);
    t(-max_non_infinite, -max_non_infinite);

    t(min_smi31, min_smi31 - 0.5);
    t(min_smi31 + 1, min_smi31 + 0.5);
    t(min_smi32, min_smi32 - 0.5);
    t(min_smi32 + 1, min_smi32 + 0.5);

    BigNumber.config({EXPONENTIAL_AT: 1E9});

    t('0', '0');
    t('0', '-0');
    t('0', '-0', 0);
    t('0', '-0', 3);
    t('0', '-0', 6);
    t('-7', '-7', 4);
    t('-22243', '-22243', 1);
    t('3808', '3808', 4);
    t('2616', '2616', 5);
    t('-312283993', '-312283992.7', 3);
    t('0', '-0.003', 4);
    t('6738', '6738.1977', 3);
    t('0', '-0.002', 5);
    t('675', '675.76294', 3);
    t('420', '419.83', 6);
    t('-8', '-8', 4);
    t('0', '0.005', 6);
    t('0', '-0.008097491', 2);
    t('-706', '-706', 1);
    t('-226', '-226.80088', 2);
    t('0', '-0.369577856', 5);
    t('-22', '-22', 5);
    t('1', '0.9', 0);

    t('0', '0.0000000000000000006', 4);
    t('0', '0.000004077002116450599035175933786605407147118579202271612670066923117668943811493201471493894', 3);
    t('-29071348593675752783607764808066026', '-29071348593675752783607764808066026', 3);
    t('0', '0.0000083135366316053183734543904737952651532784316140061929170739473518406297062533554026617147464', 1);
    t('-597198', '-597197.628834953506966767991553710700934413500204012426446876175175114500037146677042239668', 0);
    t('19504345494555478351182477703565457647566876858685627027379505944698694624726169090861984', '19504345494555478351182477703565457647566876858685627027379505944698694624726169090861984', 4);
    t('0', '0.000000042979420754347314988434544000447696776811769466455886485920655', 4);
    t('-1', '-0.0000800368117241283577308465283698286330423189888237592540844483', 3);
    t('-148196', '-148195.22330505940168207920977604763714302872718112527157', 3);
    t('-17034881662549539727165518222983064951722670573249', '-17034881662549539727165518222983064951722670573248.5125902187644045205097633', 6);
    t('1', '0.00000000000000000000000000000000000000007958953405109879', 0);
    t('-1', '-0.0719443', 3);
    t('0', '0.0002', 1);
    t('-448376506822431348847333799886', '-448376506822431348847333799886', 1);
    t('0', '-0.000000000000000000000000000001114691322399392322421', 5);
    t('0', '-0.000000000000000000000670532931448014402320805028695397125747680104246730672291510758362070539', 4);
    t('0', '-0.00000000000000000000000008442227169', 5);

    // rounding mode 7

    t('661581721939770480643', '661581721939770480643', 7);
    t('82201282103354391377948704606454107732823853693730778633', '82201282103354391377948704606454107732823853693730778632.679770307020989736730343202980605582', 7);
    t('-6903218163', '-6903218163', 7);
    t('-815303438910393610000000000000000000000000000', '-815303438910393610000000000000000000000000000.5', 7);
    t('48079804879135055337410501154945563557501499846191760613365317572636043210233866737200001', '48079804879135055337410501154945563557501499846191760613365317572636043210233866737200000.5', 7);
    t('775141321458031580151353003447938403271713528287267233801607476638641728594600000000001', '775141321458031580151353003447938403271713528287267233801607476638641728594600000000000.5', 7);
    t('-3246099003639468003580507000000000000000000000000000000000000000000000000000000000000000000000000000', '-3246099003639468003580507000000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('616052744557652711', '616052744557652710.5', 7);
    t('-47298707260000000000000000000000000000000', '-47298707260000000000000000000000000000000.5', 7);
    t('431370739475777039213491701837719630211035950221951900000000000000000001', '431370739475777039213491701837719630211035950221951900000000000000000000.5', 7);
    t('54683100000000000000000000000000000000000000000000000000000000000000000000000000000000000000001', '54683100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('672783487870821678831521822264427999625400000000000000000000000000000000000000001', '672783487870821678831521822264427999625400000000000000000000000000000000000000000.5', 7);
    t('85750000000000000000000000000000000000000001', '85750000000000000000000000000000000000000000.5', 7);
    t('-357280726879877087440504485199000000000000000000000', '-357280726879877087440504485199000000000000000000000.5', 7);
    t('-4341299477000000000000000000000', '-4341299477000000000000000000000.5', 7);
    t('31902000001', '31902000000.5', 7);
    t('333458122140379389691720067000000000000000000000000000000000000000000001', '333458122140379389691720067000000000000000000000000000000000000000000000.5', 7);
    t('-459839025400000000000000000000', '-459839025400000000000000000000.5', 7);
    t('-82390000000000', '-82390000000000.5', 7);
    t('76701', '76700.5', 7);
    t('190430490057932227000000000000000000000000000000000000001', '190430490057932227000000000000000000000000000000000000000.5', 7);
    t('60159411740605592996051862591038032930184607684088677320000000000000000000000001', '60159411740605592996051862591038032930184607684088677320000000000000000000000000.5', 7);
    t('-2234900000', '-2234900000.5', 7);
    t('56189665877093832601689082910000000000000000000000000000000000000001', '56189665877093832601689082910000000000000000000000000000000000000000.5', 7);
    t('2320000000000000000000000000001', '2320000000000000000000000000000.5', 7);
    t('-6305981611547961162835048322675738149574948489012480000000000000', '-6305981611547961162835048322675738149574948489012480000000000000.5', 7);
    t('7135722984978446900000000000000000000000000000000000001', '7135722984978446900000000000000000000000000000000000000.5', 7);
    t('7739620000000000000000000000000000000000000000000000001', '7739620000000000000000000000000000000000000000000000000.5', 7);
    t('-7310071091565097458378697755962864041334940666867342517847595229165394387202247229680000000', '-7310071091565097458378697755962864041334940666867342517847595229165394387202247229680000000.5', 7);
    t('5138337172269533148524367880141880518489245800000000000001', '5138337172269533148524367880141880518489245800000000000000.5', 7);
    t('-6000', '-6000.5', 7);
    t('463879090754017750000000000000001', '463879090754017750000000000000000.5', 7);
    t('-629850000000000000000', '-629850000000000000000.5', 7);
    t('-205429044500000000000000000000', '-205429044500000000000000000000.5', 7);
    t('-5416257015618874559905147873000000000000000000000000000000000000000000000000', '-5416257015618874559905147873000000000000000000000000000000000000000000000000.5', 7);
    t('623364421289248201317000000000000001', '623364421289248201317000000000000000.5', 7);
    t('911141136528601', '911141136528600.5', 7);
    t('-276055672501612195718314543393745315761189100000', '-276055672501612195718314543393745315761189100000.5', 7);
    t('-51740866419136700582070000', '-51740866419136700582070000.5', 7);
    t('56999862161113789759865220000000000000000000000000000000000000000001', '56999862161113789759865220000000000000000000000000000000000000000000.5', 7);
    t('-4111633071145139947224365900000', '-4111633071145139947224365900000.5', 7);
    t('697451872400000000000000000000000000000000000000000000000000000000000000000000000000000001', '697451872400000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('5000000000000000000000001', '5000000000000000000000000.5', 7);
    t('5350000000000000000000000000000000000000000000000000000001', '5350000000000000000000000000000000000000000000000000000000.5', 7);
    t('10529587347000000000000000000000000000000000000000001', '10529587347000000000000000000000000000000000000000000.5', 7);
    t('-2219716000000000000000000000000000', '-2219716000000000000000000000000000.5', 7);
    t('2211360517426407426481005085784959830631111751668394294258799150000000000001', '2211360517426407426481005085784959830631111751668394294258799150000000000000.5', 7);
    t('91550904625902200434312440338781865630142094511605342285084477183074410000000000000000000000001', '91550904625902200434312440338781865630142094511605342285084477183074410000000000000000000000000.5', 7);
    t('658054801757836432123543171404589184297504018642564438772090020000000000000000000001', '658054801757836432123543171404589184297504018642564438772090020000000000000000000000.5', 7);
    t('79944791864921246971621269400000000000000000000001', '79944791864921246971621269400000000000000000000000.5', 7);
    t('360829991900000000000000000000000000000000000000000000000000000000000000000000000000000001', '360829991900000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-1480000000000000000000000000', '-1480000000000000000000000000.5', 7);
    t('-64280200000000', '-64280200000000.5', 7);
    t('-75061307733616254618964203889755925762439000000', '-75061307733616254618964203889755925762439000000.5', 7);
    t('44130000000000000000000001', '44130000000000000000000000.5', 7);
    t('44702568841946912396571635908221096745886412397401280570000000000000000000000000001', '44702568841946912396571635908221096745886412397401280570000000000000000000000000000.5', 7);
    t('604039501485544598558831356363400000000000000001', '604039501485544598558831356363400000000000000000.5', 7);
    t('-19917984485520', '-19917984485520.5', 7);
    t('-311996565804449030000000000000000000000000000000000000000000000000000000000000000', '-311996565804449030000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-173984786383490515959556081806759606423524092895768000000', '-173984786383490515959556081806759606423524092895768000000.5', 7);
    t('-53469189427189740866342899750820440774650016000000000000000000000000000000000000000000000000', '-53469189427189740866342899750820440774650016000000000000000000000000000000000000000000000000.5', 7);
    t('-75', '-75.5', 7);
    t('27983400000000000000000001', '27983400000000000000000000.5', 7);
    t('-55866209', '-55866209.5', 7);
    t('-4236500000000', '-4236500000000.5', 7);
    t('2698000000000000000000001', '2698000000000000000000000.5', 7);
    t('176755986000000000000000000000000000000000000000000000001', '176755986000000000000000000000000000000000000000000000000.5', 7);
    t('10061938000001', '10061938000000.5', 7);
    t('-6910178749421869055443921431000000000', '-6910178749421869055443921431000000000.5', 7);
    t('3577828651898578960680400000000000000000000000000000000000000000000000001', '3577828651898578960680400000000000000000000000000000000000000000000000000.5', 7);
    t('68020066499785661939171192017417847730323210000000000000000000000000000000000001', '68020066499785661939171192017417847730323210000000000000000000000000000000000000.5', 7);
    t('-670000000000000000000000000000', '-670000000000000000000000000000.5', 7);
    t('-40412630000000000000', '-40412630000000000000.5', 7);
    t('-61065894754899019888087081382802643548863976547845002280076808300000000000000', '-61065894754899019888087081382802643548863976547845002280076808300000000000000.5', 7);
    t('100000001', '100000000.5', 7);
    t('-14822039000000000000000000000000', '-14822039000000000000000000000000.5', 7);
    t('-63700', '-63700.5', 7);
    t('-15144885754320000000000000000', '-15144885754320000000000000000.5', 7);
    t('-5272200000000000000000', '-5272200000000000000000.5', 7);
    t('-3247040000000000000', '-3247040000000000000.5', 7);
    t('-2681260809546501000', '-2681260809546501000.5', 7);
    t('87599201647856572613610170000000000000000000000000000000000000001', '87599201647856572613610170000000000000000000000000000000000000000.5', 7);
    t('250000000000000000000000000000000000000000001', '250000000000000000000000000000000000000000000.5', 7);
    t('14130422050230001', '14130422050230000.5', 7);
    t('-215000000000000000000000000000000000000000000000000000000000000000000000000', '-215000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-43726827420000000', '-43726827420000000.5', 7);
    t('-4800000000000000000000000000000000000000000000000000000000000000000', '-4800000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-7300000000000000000000000000', '-7300000000000000000000000000.5', 7);
    t('491352128092860358920632352122032679985453000001', '491352128092860358920632352122032679985453000000.5', 7);
    t('-882188546069284263665240261471556461467006545965824525162469092995249760000', '-882188546069284263665240261471556461467006545965824525162469092995249760000.5', 7);
    t('-85741667839249074', '-85741667839249074.5', 7);
    t('-527599500272614000000000000000000000000000000000000000000000000000000000000000000000000000000000', '-527599500272614000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-832820000000000', '-832820000000000.5', 7);
    t('-198601973381063265872709695924943530652918000000000000000000', '-198601973381063265872709695924943530652918000000000000000000.5', 7);
    t('-10000', '-10000.5', 7);
    t('-673944669918113000000000000000000000000000000', '-673944669918113000000000000000000000000000000.5', 7);
    t('-84188225', '-84188225.5', 7);
    t('771154817579711197000000000000000000000000000000000000000000000000001', '771154817579711197000000000000000000000000000000000000000000000000000.5', 7);
    t('-87389583669272781200000000000000000000000000000000000', '-87389583669272781200000000000000000000000000000000000.5', 7);
    t('2626469664216538014114045286014123593449500000001', '2626469664216538014114045286014123593449500000000.5', 7);
    t('-696131077975000000000000', '-696131077975000000000000.5', 7);
    t('-78690000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '-78690000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-442122610891418980859993032200000', '-442122610891418980859993032200000.5', 7);
    t('5661090000000000000000000000000000000000000000000000000000000000000000000000000000001', '5661090000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-407000000000000000000', '-407000000000000000000.5', 7);
    t('729350422974116892063715545463978850831800000000001', '729350422974116892063715545463978850831800000000000.5', 7);
    t('637954990213103000000001', '637954990213103000000000.5', 7);
    t('-1138365762514721521483834034584523726374412779304481429041180049963600000', '-1138365762514721521483834034584523726374412779304481429041180049963600000.5', 7);
    t('2475696728487804739525182918139219026008027679054000000000000000000000000000000000001', '2475696728487804739525182918139219026008027679054000000000000000000000000000000000000.5', 7);
    t('66193237237200326100000000000000000001', '66193237237200326100000000000000000000.5', 7);
    t('663179971604113808385191042094969369727970974320403970845752700000000001', '663179971604113808385191042094969369727970974320403970845752700000000000.5', 7);
    t('596157270311242', '596157270311241.5', 7);
    t('-430859454062137256578640000000000000000000000000000000000000000', '-430859454062137256578640000000000000000000000000000000000000000.5', 7);
    t('7032813000000000001', '7032813000000000000.5', 7);
    t('767314152225968313318624830508438163176729385050170000000000000000001', '767314152225968313318624830508438163176729385050170000000000000000000.5', 7);
    t('-321648005175685895332610984400000000000', '-321648005175685895332610984400000000000.5', 7);
    t('-5914848242008475427391535555998338828544179518249385611394025548931079438027758333803040000000000', '-5914848242008475427391535555998338828544179518249385611394025548931079438027758333803040000000000.5', 7);
    t('73480016864560536978879028863212252705530701', '73480016864560536978879028863212252705530700.5', 7);
    t('8976064818931304755883643401472118930622270000000000000001', '8976064818931304755883643401472118930622270000000000000000.5', 7);
    t('-5079754732761006567963078886040778070', '-5079754732761006567963078886040778070.5', 7);
    t('-51002521076071416214838521445923939000000000', '-51002521076071416214838521445923939000000000.5', 7);
    t('8684049955981816069703979854267267479779343491340480759376275884260000000000000000000000000001', '8684049955981816069703979854267267479779343491340480759376275884260000000000000000000000000000.5', 7);
    t('37507194800178617805104432695868625191287264624719442564304569000000000000000000000000000000001', '37507194800178617805104432695868625191287264624719442564304569000000000000000000000000000000000.5', 7);
    t('-4281839065682230128346600112666094892582742848537366052855118901071834009400000000000000', '-4281839065682230128346600112666094892582742848537366052855118901071834009400000000000000.5', 7);
    t('-333000', '-333000.5', 7);
    t('-502583575785454685986915469822211140422238465305000000000000000000000000000000000000', '-502583575785454685986915469822211140422238465305000000000000000000000000000000000000.5', 7);
    t('-32030575969596778652213137431653825274000000000000000000', '-32030575969596778652213137431653825274000000000000000000.5', 7);
    t('-4506117101680962000000000000000000000000', '-4506117101680962000000000000000000000000.5', 7);
    t('-64730463575631254944664602047280130000000000000000000000000000000000000', '-64730463575631254944664602047280130000000000000000000000000000000000000.5', 7);
    t('27570663944600021244014689614081019825668139000000000000000000000000001', '27570663944600021244014689614081019825668139000000000000000000000000000.5', 7);
    t('-350302102100000000000000000', '-350302102100000000000000000.5', 7);
    t('-3918501626000000000', '-3918501626000000000.5', 7);
    t('51812774862668048505100000000000000000000000000000000000000000000000000000000001', '51812774862668048505100000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-61696491490996741116958188157141279933341663428216950000000000000000000000', '-61696491490996741116958188157141279933341663428216950000000000000000000000.5', 7);
    t('4656808510000000000000000000000000000000000000000000000000000000000000000001', '4656808510000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('23272197523886798346038846952829185419450839280000000000000000000000000000000000000001', '23272197523886798346038846952829185419450839280000000000000000000000000000000000000000.5', 7);
    t('4065544662345173976170000000000000000000000000000000001', '4065544662345173976170000000000000000000000000000000000.5', 7);
    t('-422635258000000000000000000000000000000000000000000000000000', '-422635258000000000000000000000000000000000000000000000000000.5', 7);
    t('-3582000', '-3582000.5', 7);
    t('7740826542065848739126877465966868569935787226324870000000000000000001', '7740826542065848739126877465966868569935787226324870000000000000000000.5', 7);
    t('1614965644012243386199869835574326000000001', '1614965644012243386199869835574326000000000.5', 7);
    t('-20000', '-20000.5', 7);
    t('-2291933618897112688229402464046600000000000000000000000000000000000000000000000000000000000000', '-2291933618897112688229402464046600000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-663605069281623297326832485568849000000000000000000000000000000000000000000000000000000000000000000', '-663605069281623297326832485568849000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('1939060273775317149434400000000000000000000000000000000000000000000000000000000000000000000001', '1939060273775317149434400000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-726114440793216017690000000000000000000000', '-726114440793216017690000000000000000000000.5', 7);
    t('-8529164154875627047383959874881458179234088946000000000000000000000000000000000000000000000000000', '-8529164154875627047383959874881458179234088946000000000000000000000000000000000000000000000000000.5', 7);
    t('359551952970277110582000000000000000000000000000000000001', '359551952970277110582000000000000000000000000000000000000.5', 7);
    t('-6822800000000000000000000000000000000000000000000000', '-6822800000000000000000000000000000000000000000000000.5', 7);
    t('28024094231179167233916852000000000000000000000000000000001', '28024094231179167233916852000000000000000000000000000000000.5', 7);
    t('-5768619642107599734543714041179604138353000000000000', '-5768619642107599734543714041179604138353000000000000.5', 7);
    t('-4336937232847962038320625210133958133684710802758818489207914168472208196240530780800000', '-4336937232847962038320625210133958133684710802758818489207914168472208196240530780800000.5', 7);
    t('294276367161162465235912532501878152936548550758709465137169762240500722165978566', '294276367161162465235912532501878152936548550758709465137169762240500722165978565.5', 7);
    t('-5853377924485346629722284422247261298818024606786760212628942481993981005105147485025005541304000', '-5853377924485346629722284422247261298818024606786760212628942481993981005105147485025005541304000.5', 7);
    t('-57376528507859392114198303459327987043203713812970277355663102705761801186615364048872085338570', '-57376528507859392114198303459327987043203713812970277355663102705761801186615364048872085338570.5', 7);
    t('8981899774674151341273723045246779558611526093370000000000000000000000000000001', '8981899774674151341273723045246779558611526093370000000000000000000000000000000.5', 7);
    t('6880184505510000000000000000000000000000000000000000000000000000000000000000000000001', '6880184505510000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-9144765542574759986504560099796367000000000000000000000000000000000000000000000000000000000000000000', '-9144765542574759986504560099796367000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('-33550940230832658386583641510', '-33550940230832658386583641510.5', 7);
    t('570500000001', '570500000000.5', 7);
    t('21171552652016197991647323966415490225947817225636864710000000000000000000000000001', '21171552652016197991647323966415490225947817225636864710000000000000000000000000000.5', 7);
    t('-219451479377926100000000000000000000000', '-219451479377926100000000000000000000000.5', 7);
    t('-46023648500000000000000000000000000000000000000000000000', '-46023648500000000000000000000000000000000000000000000000.5', 7);
    t('-5788386698147635042359128255204287799666980072971472360314579333990755726292753882057820000', '-5788386698147635042359128255204287799666980072971472360314579333990755726292753882057820000.5', 7);
    t('48425486699992611674833028561543095971459492320411160921', '48425486699992611674833028561543095971459492320411160920.5', 7);
    t('-59344633366981652200000000000', '-59344633366981652200000000000.5', 7);
    t('-65482014253687833342600000000000000000000000', '-65482014253687833342600000000000000000000000.5', 7);
    t('-30128014553929563000000000000000000000000000000000000000000000000000000000000', '-30128014553929563000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('794768001', '794768000.5', 7);
    t('-150865959000000000000000000000000000000000000000000000000000000000000000000000000000000', '-150865959000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 7);
    t('873381282566874884976602503384155621488078946785261238', '873381282566874884976602503384155621488078946785261237.5', 7);
    t('-76364822618102400000000000000000000000000000000000000000000000000', '-76364822618102400000000000000000000000000000000000000000000000000.5', 7);
    t('1590000000000000000000000000000000000000000000000000000000001', '1590000000000000000000000000000000000000000000000000000000000.5', 7);
    t('15883352501', '15883352500.5', 7);
    t('-1678796930161119374892', '-1678796930161119374892.5', 7);
    t('7519225332612480000000000000001', '7519225332612480000000000000000.5', 7);
    t('26782458868551013018628206420511402980374696236993201', '26782458868551013018628206420511402980374696236993200.5', 7);
    t('84182962247257426278629190001', '84182962247257426278629190000.5', 7);
    t('-674633568509895706371390517304490463246553534599391649648000000', '-674633568509895706371390517304490463246553534599391649648000000.5', 7);
    t('2036599383754410655598351317310763047828000000000000000000000000000001', '2036599383754410655598351317310763047828000000000000000000000000000000.5', 7);
    t('-1747000000', '-1747000000.5', 7);
    t('3960685307896400000000000001', '3960685307896400000000000000.5', 7);
    t('-1', '-1.5', 7);
    t('5', '4.5', 7);
    t('-8542007911699875753350816000000000', '-8542007911699875753350816000000000.5', 7);
    t('229448019839399741', '229448019839399740.5', 7);
    t('4340972004000000000000000000000000000000000001', '4340972004000000000000000000000000000000000000.5', 7);
    t('-773300000000', '-773300000000.5', 7);
    t('7155533863912051817808706333952649054169243324075500000000000000000000000000000000001', '7155533863912051817808706333952649054169243324075500000000000000000000000000000000000.5', 7);
    t('994879015661912752196001000000000000000001', '994879015661912752196001000000000000000000.5', 7);
    t('9735231726716596452594218428908047051010710000000000000000000000000000000001', '9735231726716596452594218428908047051010710000000000000000000000000000000000.5', 7);
    t('830000001', '830000000.5', 7);
    t('-621154', '-621154.5', 7);
    t('-7026505943196190557292140834274961200000000', '-7026505943196190557292140834274961200000000.5', 7);
    t('-5698210301557000000000000000000000000000', '-5698210301557000000000000000000000000000.5', 7);
    t('633325185790000001', '633325185790000000.5', 7);
    t('-64165596081004870', '-64165596081004870.5', 7);
    t('-41646941216693066756777307741427764177371639363417043000000000000000000000000000000000000000', '-41646941216693066756777307741427764177371639363417043000000000000000000000000000000000000000.5', 7);
    t('-30276599950453', '-30276599950453.5', 7);
    t('844934410001', '844934410000.5', 7);
    t('193536068479533070654088534028865405146725922969081990000000000000000000000000001', '193536068479533070654088534028865405146725922969081990000000000000000000000000000.5', 7);
    t('3571390090807278635466910226000000000001', '3571390090807278635466910226000000000000.5', 7);
    t('33758731615573853269038256552240314238547949393669248932000000000000000000000000000000000000000001', '33758731615573853269038256552240314238547949393669248932000000000000000000000000000000000000000000.5', 7);

    // rounding mode 8

    t('9113500111924420134168224777108947212846008765793966704145', '9113500111924420134168224777108947212846008765793966704145.431817424332', 8);
    t('-269', '-268.5690822533536300367', 8);
    t('8225834668202811700000', '8225834668202811700000.5', 8);
    t('-67268029621398311238261431331851556661886545942854113916050734581000000000001', '-67268029621398311238261431331851556661886545942854113916050734581000000000000.5', 8);
    t('4100000', '4100000.5', 8);
    t('694900000000000', '694900000000000.5', 8);
    t('-2668752017792840000000000000000000000000000000000000000000000000000000000001', '-2668752017792840000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('5138303265393258323808903836559900000000000000000', '5138303265393258323808903836559900000000000000000.5', 8);
    t('-61599472977109463564534878506157000000000000000000000000000000001', '-61599472977109463564534878506157000000000000000000000000000000000.5', 8);
    t('-706579187032566119663804136174620000000000000000001', '-706579187032566119663804136174620000000000000000000.5', 8);
    t('10', '10.5', 8);
    t('903942469098308100136613765218227935820271584538277900367517300000000000000000', '903942469098308100136613765218227935820271584538277900367517300000000000000000.5', 8);
    t('7259090494485579718152684060350290000000000000000000000000000000000000000000000000000000000', '7259090494485579718152684060350290000000000000000000000000000000000000000000000000000000000.5', 8);
    t('50', '50.5', 8);
    t('-336051', '-336050.5', 8);
    t('-359851700000000000000000000000000000000000000000000000000000000000000000000000000000001', '-359851700000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('8300', '8300.5', 8);
    t('-5230000000000000000000000000000000000000000000000001', '-5230000000000000000000000000000000000000000000000000.5', 8);
    t('-832637338257576743857980873934631876338729504830820328750000000000000000000000000000000001', '-832637338257576743857980873934631876338729504830820328750000000000000000000000000000000000.5', 8);
    t('2170', '2170.5', 8);
    t('7530960067300000', '7530960067300000.5', 8);
    t('-69761077704964968462722176065747076185154464193791511210000000000000000000000000000000000000001', '-69761077704964968462722176065747076185154464193791511210000000000000000000000000000000000000000.5', 8);
    t('8951', '8951.5', 8);
    t('-62816928986109317245318155705980721116645949369247436234018847714093151000000000001', '-62816928986109317245318155705980721116645949369247436234018847714093151000000000000.5', 8);
    t('-273450712038160000001', '-273450712038160000000.5', 8);
    t('870651868531859742020859624838429666098184966539196792393834098130000000000000000000', '870651868531859742020859624838429666098184966539196792393834098130000000000000000000.5', 8);
    t('7467352850531742535166203140000', '7467352850531742535166203140000.5', 8);
    t('-27739701695659117725205758576200000000000001', '-27739701695659117725205758576200000000000000.5', 8);
    t('-21575913877090003604970871615441236644670000000000001', '-21575913877090003604970871615441236644670000000000000.5', 8);
    t('-84684097772847106401567201003729899910000000000001', '-84684097772847106401567201003729899910000000000000.5', 8);
    t('5639766204672194231584787000000000000000000000000000000000000000000000000000000000000000000000', '5639766204672194231584787000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('15504935934287862095572900000000000000000000000', '15504935934287862095572900000000000000000000000.5', 8);
    t('-1400577619554361844128289470441748896959600001', '-1400577619554361844128289470441748896959600000.5', 8);
    t('-562508502033790781280182919180003097225841259584126808028780000001', '-562508502033790781280182919180003097225841259584126808028780000000.5', 8);
    t('462676260328650000000000000000000', '462676260328650000000000000000000.5', 8);
    t('80610550887265206737088350828621745655000000000000000000000000000000000000000000000000000000', '80610550887265206737088350828621745655000000000000000000000000000000000000000000000000000000.5', 8);
    t('-802119959634183327467393167608755000000000000000000000000000000001', '-802119959634183327467393167608755000000000000000000000000000000000.5', 8);
    t('78620000000000000000000000000000000000000', '78620000000000000000000000000000000000000.5', 8);
    t('810000000000000000000000000000000000', '810000000000000000000000000000000000.5', 8);
    t('-3676779349655275521581849548079800000000000000000000000000000000000000000000000000000000001', '-3676779349655275521581849548079800000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-81572917297627759505118569656972784453878787892500000000000001', '-81572917297627759505118569656972784453878787892500000000000000.5', 8);
    t('-198003769523621', '-198003769523620.5', 8);
    t('5306', '5306.5', 8);
    t('-4000000000000000000000000000000000000000001', '-4000000000000000000000000000000000000000000.5', 8);
    t('-6964300000000000000000000000000000001', '-6964300000000000000000000000000000000.5', 8);
    t('5993000000000000000000', '5993000000000000000000.5', 8);
    t('-58', '-57.5', 8);
    t('-314705009617969030000000000000000000000000000000000000000001', '-314705009617969030000000000000000000000000000000000000000000.5', 8);
    t('6', '6.5', 8);
    t('4588255716462680467926051688938462657411100947687253979336100000000000000', '4588255716462680467926051688938462657411100947687253979336100000000000000.5', 8);
    t('300', '300.5', 8);
    t('-8095535225924203212402801', '-8095535225924203212402800.5', 8);
    t('30838279901100000000000000000000000000000000000000000000', '30838279901100000000000000000000000000000000000000000000.5', 8);
    t('-644246265100306600000000000001', '-644246265100306600000000000000.5', 8);
    t('-7949436643533748183697968284137159105717056926000000000000000000000000000000001', '-7949436643533748183697968284137159105717056926000000000000000000000000000000000.5', 8);
    t('5961899981183314783500000000', '5961899981183314783500000000.5', 8);
    t('329320658758032654290364416883005342614380587029114737580920000000000000', '329320658758032654290364416883005342614380587029114737580920000000000000.5', 8);
    t('-70000000000000000000000000001', '-70000000000000000000000000000.5', 8);
    t('-773550954099185953800000000000000000000000000000001', '-773550954099185953800000000000000000000000000000000.5', 8);
    t('-71820001', '-71820000.5', 8);
    t('6824221413055181208880629906440161900000000000000000000000000000000000000000000', '6824221413055181208880629906440161900000000000000000000000000000000000000000000.5', 8);
    t('-293663761783147249577441951060981857156831085100000000000000000000000000000000000000000001', '-293663761783147249577441951060981857156831085100000000000000000000000000000000000000000000.5', 8);
    t('-805331726532205040286379140828190238797602490000000000000000000000000000000000000000001', '-805331726532205040286379140828190238797602490000000000000000000000000000000000000000000.5', 8);
    t('2627542620785923273450000000000000000000', '2627542620785923273450000000000000000000.5', 8);
    t('190055193848119613272218981746041232571688473321679545800000000000000000000000000000000000000000', '190055193848119613272218981746041232571688473321679545800000000000000000000000000000000000000000.5', 8);
    t('-61621063888036802203530857476475979818427083600000000000000000000000000000000000000000000000001', '-61621063888036802203530857476475979818427083600000000000000000000000000000000000000000000000000.5', 8);
    t('-652127458000489992750000000000000000000000000000000000000000000000001', '-652127458000489992750000000000000000000000000000000000000000000000000.5', 8);
    t('7000000000000000000000000', '7000000000000000000000000.5', 8);
    t('6533676582995433778209424794015171896011423181700000000000000', '6533676582995433778209424794015171896011423181700000000000000.5', 8);
    t('27456600275687693800000000000000000000000000000000000000000000000000000000000000000000000000000000', '27456600275687693800000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-194516150000000000000000001', '-194516150000000000000000000.5', 8);
    t('-176140000000001', '-176140000000000.5', 8);
    t('85012661713649137873180295487285610000000000000000000000000000000000000', '85012661713649137873180295487285610000000000000000000000000000000000000.5', 8);
    t('-37523700000000000000000000000000000000000000000000000000000000000000000000000000000001', '-37523700000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-15124465952376999197073295421688223628463231603610811', '-15124465952376999197073295421688223628463231603610810.5', 8);
    t('-5432100777163613256132439687382900000000000000000000000000000000000000000000000001', '-5432100777163613256132439687382900000000000000000000000000000000000000000000000000.5', 8);
    t('652000000', '652000000.5', 8);
    t('-8507892174157676630256932627461950536321146344662200000000000000000000000000000001', '-8507892174157676630256932627461950536321146344662200000000000000000000000000000000.5', 8);
    t('-5780286273116071692453993336522250324675081', '-5780286273116071692453993336522250324675080.5', 8);
    t('2885286893929064654337716318113439600000000000000000000000000000000000000000', '2885286893929064654337716318113439600000000000000000000000000000000000000000.5', 8);
    t('158421702922228687046272106697601990957751203100000000000000000000000000000000000000000000000000', '158421702922228687046272106697601990957751203100000000000000000000000000000000000000000000000000.5', 8);
    t('77845803952870686913386535517781103587210960982825399080400000000000000000000000', '77845803952870686913386535517781103587210960982825399080400000000000000000000000.5', 8);
    t('-65458261744123585088042880333251536411447000000000000000000000000000000000000000000000000001', '-65458261744123585088042880333251536411447000000000000000000000000000000000000000000000000000.5', 8);
    t('-5514810000000000001', '-5514810000000000000.5', 8);
    t('-3837413074391000000000000000000000000000001', '-3837413074391000000000000000000000000000000.5', 8);
    t('1891485546514380015490000000000000000000000000', '1891485546514380015490000000000000000000000000.5', 8);
    t('-670001', '-670000.5', 8);
    t('7365874638887372900000000000', '7365874638887372900000000000.5', 8);
    t('80464947550', '80464947550.5', 8);
    t('-913091300000000000000000000000000000000000000000000001', '-913091300000000000000000000000000000000000000000000000.5', 8);
    t('25911127696762546866341398944677062000000000000000000000000', '25911127696762546866341398944677062000000000000000000000000.5', 8);
    t('41448246317918170624465120157939322903000000000000000000000000000000000000000000000000000000000000', '41448246317918170624465120157939322903000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-800000000000000000000000000000000000000000000000000001', '-800000000000000000000000000000000000000000000000000000.5', 8);
    t('577447040809749466752886339212451433404458549958939957382201980490000000000000', '577447040809749466752886339212451433404458549958939957382201980490000000000000.5', 8);
    t('-57572272809174825641085924940000000000000000000000001', '-57572272809174825641085924940000000000000000000000000.5', 8);
    t('31581459400000000000000000000000000000000000000000000000000000000000000000000000000', '31581459400000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-27140000000000000000000000000000000000000000000000000000000000000001', '-27140000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('51603647184997719522882897042563846663899024307208600000000000000000000000', '51603647184997719522882897042563846663899024307208600000000000000000000000.5', 8);
    t('1644323000000', '1644323000000.5', 8);
    t('-9', '-8.5', 8);
    t('-902809102825706989238470350820560000000000000000000000000000000000000000000000000001', '-902809102825706989238470350820560000000000000000000000000000000000000000000000000000.5', 8);
    t('-6868477001', '-6868477000.5', 8);
    t('91714927731569000000000000000000000000000000000', '91714927731569000000000000000000000000000000000.5', 8);
    t('-8000000000000000000000000000000000000000000000000000000000000000001', '-8000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-233083085304449000000000000000000000000000000000000000000000000000000000000000000000000000001', '-233083085304449000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('46509073700000', '46509073700000.5', 8);
    t('-1973764439115071357288670461800000000000000001', '-1973764439115071357288670461800000000000000000.5', 8);
    t('-2568500000000000000000001', '-2568500000000000000000000.5', 8);
    t('-24809330903268410000000000000000000000000000000000000000000000000000001', '-24809330903268410000000000000000000000000000000000000000000000000000000.5', 8);
    t('7449994364098414600000000000000000000000000000000000000000000', '7449994364098414600000000000000000000000000000000000000000000.5', 8);
    t('2867488032797172375145449433748008158236530896489051275000000000000000000000000000000000', '2867488032797172375145449433748008158236530896489051275000000000000000000000000000000000.5', 8);
    t('-4900000000000000000000000000000000000000000000000000000000000000000000000000001', '-4900000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('4929000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4929000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('8267546457668258513715771480132070472862216176570000000000000000000000000', '8267546457668258513715771480132070472862216176570000000000000000000000000.5', 8);
    t('593747187059615072947679001676736984922713680000000000000000000000000000000000000000000000', '593747187059615072947679001676736984922713680000000000000000000000000000000000000000000000.5', 8);
    t('2483270425435993889589154244481827665973478012000000000', '2483270425435993889589154244481827665973478012000000000.5', 8);
    t('39543791425933526768174519227303827374238075020570198950767961859654305403206119372345770250000', '39543791425933526768174519227303827374238075020570198950767961859654305403206119372345770250000.5', 8);
    t('67', '67.5', 8);
    t('-219470058205534357335831751487551261713170336001', '-219470058205534357335831751487551261713170336000.5', 8);
    t('-183690000000000000000000000000000000000001', '-183690000000000000000000000000000000000000.5', 8);
    t('3450876609649939508104514905806066940000000000000000000000', '3450876609649939508104514905806066940000000000000000000000.5', 8);
    t('-80540000001', '-80540000000.5', 8);
    t('-62971517498372655372935973087516265926164530906763170287790', '-62971517498372655372935973087516265926164530906763170287789.5', 8);
    t('23409228118870000000', '23409228118870000000.5', 8);
    t('-7000000000000000000000000000000000000001', '-7000000000000000000000000000000000000000.5', 8);
    t('-9031', '-9030.5', 8);
    t('3492817610000000000000000000000000000000000000000000000000000', '3492817610000000000000000000000000000000000000000000000000000.5', 8);
    t('-57688287613641931000000000000000000000000001', '-57688287613641931000000000000000000000000000.5', 8);
    t('1984116330670000000000000000000000000000000000000000000000000000000000', '1984116330670000000000000000000000000000000000000000000000000000000000.5', 8);
    t('6512025216557000000000000000000000000000000000000000000000000000000', '6512025216557000000000000000000000000000000000000000000000000000000.5', 8);
    t('12182501403550393384353169696000000000000000000000000000000000000000000000000000000000000000000', '12182501403550393384353169696000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-7425931783170916885202085702943899676988400000000000000000000000000000000000000001', '-7425931783170916885202085702943899676988400000000000000000000000000000000000000000.5', 8);
    t('-4070000000000000000000000000000000000001', '-4070000000000000000000000000000000000000.5', 8);
    t('8647459874624869698149020230000000000000000000000000000000000000000000000000000000000', '8647459874624869698149020230000000000000000000000000000000000000000000000000000000000.5', 8);
    t('46599000', '46599000.5', 8);
    t('-5010000000000000000000000000000000000000000000000000000000000000000000000000000000000000001', '-5010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('39508000', '39508000.5', 8);
    t('603089057835800884421518508711417801', '603089057835800884421518508711417801.5', 8);
    t('-79471558363526485324781700384768609724195801918000000000000000000000000000001', '-79471558363526485324781700384768609724195801918000000000000000000000000000000.5', 8);
    t('69404910310759646801954680033844865865649000000000000000000000000000000000000000000000000', '69404910310759646801954680033844865865649000000000000000000000000000000000000000000000000.5', 8);
    t('1907980209926030560252116696113309208138159500662562258522400000000000000000000000000000000', '1907980209926030560252116696113309208138159500662562258522400000000000000000000000000000000.5', 8);
    t('329472654458195399229095412173156277813634317805761763485596051933469610596000', '329472654458195399229095412173156277813634317805761763485596051933469610596000.5', 8);
    t('276349000000000', '276349000000000.5', 8);
    t('787126144338547000000000000000000000000000000000000000000000', '787126144338547000000000000000000000000000000000000000000000.5', 8);
    t('1471099043691184058624460905988224213538031110904992774958270704956863896897997000000000000000', '1471099043691184058624460905988224213538031110904992774958270704956863896897997000000000000000.5', 8);
    t('370113010930715428293470648815487051030769793281763667800000000000000000000000', '370113010930715428293470648815487051030769793281763667800000000000000000000000.5', 8);
    t('2065922690807733136485062524396416121715082253637965000000000000000000000000000000000000000000', '2065922690807733136485062524396416121715082253637965000000000000000000000000000000000000000000.5', 8);
    t('-7382816558182407799471782300849610860474300000000000000000000000000000000000000000001', '-7382816558182407799471782300849610860474300000000000000000000000000000000000000000000.5', 8);
    t('-20097560535463220697432958047617508689581603082766010090000000000000000000001', '-20097560535463220697432958047617508689581603082766010090000000000000000000000.5', 8);
    t('628909746633683135845125129915144934623519151508200000', '628909746633683135845125129915144934623519151508200000.5', 8);
    t('-81192352241330340442871946386698010000000000000001', '-81192352241330340442871946386698010000000000000000.5', 8);
    t('6450', '6450.5', 8);
    t('37293916349890940922992467416776788689073657558307952866668243095547190629248951386344800000000', '37293916349890940922992467416776788689073657558307952866668243095547190629248951386344800000000.5', 8);
    t('1718544737998848', '1718544737998848.5', 8);
    t('87231983170377993831727615571315627607167084700000000000', '87231983170377993831727615571315627607167084700000000000.5', 8);
    t('-86061664983179049931693406385826804039300476826004230043833291400000000000001', '-86061664983179049931693406385826804039300476826004230043833291400000000000000.5', 8);
    t('-51891604836621651634017359747812499890000000000000000000000000000000000000000000000000001', '-51891604836621651634017359747812499890000000000000000000000000000000000000000000000000000.5', 8);
    t('8745224135400000000000000000000', '8745224135400000000000000000000.5', 8);
    t('-30620233581205628000000000000000001', '-30620233581205628000000000000000000.5', 8);
    t('14863144633163334418622520000000000000000', '14863144633163334418622520000000000000000.5', 8);
    t('-7820166128000000000000000000000000000000000000000000000000001', '-7820166128000000000000000000000000000000000000000000000000000.5', 8);
    t('490301854395822993260221610000000000000000000000000000000000000000000000000000000000000000000000000', '490301854395822993260221610000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('89435429126192621776997940079101579134477637123767783805081890000', '89435429126192621776997940079101579134477637123767783805081890000.5', 8);
    t('-8', '-7.5', 8);
    t('48028980775647633995683237409146256399109720500000000000000000000000000000000000000', '48028980775647633995683237409146256399109720500000000000000000000000000000000000000.5', 8);
    t('6335331194137189544436447913317232855829543152436874057500', '6335331194137189544436447913317232855829543152436874057500.5', 8);
    t('-628254630769343648758471830148888672377000000000000000000000000000000000000000000000001', '-628254630769343648758471830148888672377000000000000000000000000000000000000000000000000.5', 8);
    t('57839375650970125223236315959022043520630147028794000000000000000000000', '57839375650970125223236315959022043520630147028794000000000000000000000.5', 8);
    t('-771743548959352866414440497741988370454011151128027700000000000001', '-771743548959352866414440497741988370454011151128027700000000000000.5', 8);
    t('-53429592007706646032950000000000001', '-53429592007706646032950000000000000.5', 8);
    t('499380000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '499380000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-641898557705670000001', '-641898557705670000000.5', 8);
    t('600220992462460556877273974257288999278181263177202351996538136181825896120000', '600220992462460556877273974257288999278181263177202351996538136181825896120000.5', 8);
    t('4060100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4060100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('-8364590709247094923425000000000000000000000000001', '-8364590709247094923425000000000000000000000000000.5', 8);
    t('519409844212193540764606418404780000000000000000000000000000000000000000', '519409844212193540764606418404780000000000000000000000000000000000000000.5', 8);
    t('-3833671650388000000000000000000000000000000000000000000000000000000000000000000000000000000000000001', '-3833671650388000000000000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('48557176761800000000000000000000000000000000000000', '48557176761800000000000000000000000000000000000000.5', 8);
    t('-7711489971', '-7711489970.5', 8);
    t('75000000000000000000', '75000000000000000000.5', 8);
    t('3229580853248988018262985670791998740861538127939000', '3229580853248988018262985670791998740861538127939000.5', 8);
    t('5248316551608552493188922201763221117178358562272408864992232158101702807890000000000000000', '5248316551608552493188922201763221117178358562272408864992232158101702807890000000000000000.5', 8);
    t('-65467380782429716204000000000001', '-65467380782429716204000000000000.5', 8);
    t('-7205690650357445181000000001', '-7205690650357445181000000000.5', 8);
    t('-276409666783477000000000001', '-276409666783477000000000000.5', 8);
    t('-58428012013195193142046078040582490000000000000000000000000000000000001', '-58428012013195193142046078040582490000000000000000000000000000000000000.5', 8);
    t('-751950207597055427000000000000000000000000000001', '-751950207597055427000000000000000000000000000000.5', 8);
    t('-599626209998027538223544864628000000000000000000000000001', '-599626209998027538223544864628000000000000000000000000000.5', 8);
    t('493818820446671117064848278738045865755408340205207514513699818290870000000000000000000000000', '493818820446671117064848278738045865755408340205207514513699818290870000000000000000000000000.5', 8);
    t('72115246495423678068962420259406315700000', '72115246495423678068962420259406315700000.5', 8);
    t('8355575817405490000000000000000', '8355575817405490000000000000000.5', 8);
    t('-760420001', '-760420000.5', 8);
    t('-2637320151000000000000000000000000000000000000000001', '-2637320151000000000000000000000000000000000000000000.5', 8);
    t('365188058248000000000000000000000000000000000000000000000000000000000000000000000000000000', '365188058248000000000000000000000000000000000000000000000000000000000000000000000000000000.5', 8);
    t('24776070252572097097332452000000000', '24776070252572097097332452000000000.5', 8);
    t('-51423316645779393318000000000000001', '-51423316645779393318000000000000000.5', 8);
    t('623144979783067430949482841363102540602541201810000000000000000000000000000000000000000000000', '623144979783067430949482841363102540602541201810000000000000000000000000000000000000000000000.5', 8);
    t('-393701403238457991332600000000000000001', '-393701403238457991332600000000000000000.5', 8);
    t('255574900000000000000000000000000000000000000000000000', '255574900000000000000000000000000000000000000000000000.5', 8);
    t('361300718195583086529006200000000000000', '361300718195583086529006200000000000000.5', 8);
    t('61381935060212901891392491994620938159434302730207737183495000000000000', '61381935060212901891392491994620938159434302730207737183495000000000000.5', 8);
    t('17103494622240782188191669794453543590000000000000000000000000000000', '17103494622240782188191669794453543590000000000000000000000000000000.5', 8);

    t('13', '12.345', 2);
    t('12', '12.345', null);
    t('12', '12.345');
    t('13', '12.345', 0);
    t('13', '12.345', -0);


    Test.isException(function () {new BigNumber('12.345').integerValue(NaN)}, ".integerValue(NaN)");
    Test.isException(function () {new BigNumber('12.345').integerValue('NaN')}, ".integerValue('NaN')");
    Test.isException(function () {new BigNumber('12.345').integerValue([])}, ".integerValue([])");
    Test.isException(function () {new BigNumber('12.345').integerValue({})}, ".integerValue({})");
    Test.isException(function () {new BigNumber('12.345').integerValue('')}, ".integerValue('')");
    Test.isException(function () {new BigNumber('12.345').integerValue(' ')}, ".integerValue(' ')");
    Test.isException(function () {new BigNumber('12.345').integerValue('hello')}, ".integerValue('hello')");
    Test.isException(function () {new BigNumber('12.345').integerValue('\t')}, ".integerValue('\t')");
    Test.isException(function () {new BigNumber('12.345').integerValue(new Date)}, ".integerValue(new Date)");
    Test.isException(function () {new BigNumber('12.345').integerValue(new RegExp)}, ".integerValue(new RegExp)");
    Test.isException(function () {new BigNumber('12.345').integerValue(7.5)}, ".integerValue(7.5)");
    Test.isException(function () {new BigNumber('12.345').integerValue('-1.1e1')}, ".integerValue('-1.1e1')");

    Test.isException(function () {new BigNumber('12.345').integerValue('-1')}, ".integerValue(-1')");
    Test.isException(function () {new BigNumber('12.345').integerValue(-23)}, ".integerValue(-23)");
    Test.isException(function () {new BigNumber('12.345').integerValue(8.01)}, ".integerValue(8.01)");
    Test.isException(function () {new BigNumber('12.345').integerValue(9)}, ".integerValue(9)");
    Test.isException(function () {new BigNumber('12.345').integerValue(-1)}, ".integerValue(-1)");
    Test.isException(function () {new BigNumber('12.345').integerValue('-0.01')}, ".integerValue('-0.01')");
    Test.isException(function () {new BigNumber('12.345').integerValue('-1e-1')}, ".integerValue('-1e-1')");
    Test.isException(function () {new BigNumber('12.345').integerValue(Infinity)}, ".integerValue(Infinity)");
    Test.isException(function () {new BigNumber('12.345').integerValue('-Infinity')}, ".integerValue('-Infinity')");
});
