/*
 *
 *  Bluetooth low-complexity, subband codec (SBC) library
 *
 *  Copyright (C) 2004  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2004  Henryk Ploetz <henryk@ploetzli.ch>
 *  Copyright (C) 2004-2006  Brad Midgley <bmidgley@xmission.com>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __SBC_TABLES_H
#define __SBC_TABLES_H

#include "sbc_math.h"

/* A2DP specification: Appendix B, page 69 */
static const int sbc_offset4[4][4] = {
	{ -1, 0, 0, 0 },
	{ -2, 0, 0, 1 },
	{ -2, 0, 0, 1 },
	{ -2, 0, 0, 1 }
};

/* A2DP specification: Appendix B, page 69 */
static const int sbc_offset8[4][8] = {
	{ -2, 0, 0, 0, 0, 0, 0, 1 },
	{ -3, 0, 0, 0, 0, 0, 1, 2 },
	{ -4, 0, 0, 0, 0, 0, 1, 2 },
	{ -4, 0, 0, 0, 0, 0, 1, 2 }
};

#ifdef USE_FIXED

#define SP4(val) (((int32_t)val) >> SCALE_PROTO4_TBL)
#define SA4(val) (((int32_t)val) >> SCALE_ANA4_TBL)
#define SP8(val) (((int32_t)val) >> SCALE_PROTO8_TBL)
#define SA8(val) (((int32_t)val) >> SCALE_ANA8_TBL)
#define SS4(val) (((int32_t)val) >> SCALE_SPROTO4_TBL)
#define SS8(val) (((int32_t)val) >> SCALE_SPROTO8_TBL)
#define SN4(val) (((int32_t)val) >> SCALE_NPROTO4_TBL)
#define SN8(val) (((int32_t)val) >> SCALE_NPROTO8_TBL)

static const int32_t _sbc_proto_4[20] = {
	SP4(0x02cb3e8c), SP4(0x22b63dc0), SP4(0x002329cc), SP4(0x053b7548), SP4(0x31eab940), SP4(0xec1f5e60), SP4(0xff3773a8), SP4(0x0061c5a7),
	SP4(0x07646680), SP4(0x3f239480), SP4(0xf89f23a8), SP4(0x007a4737), SP4(0x00b32807), SP4(0x083ddc80), SP4(0x4825e480), SP4(0x0191e578),
	SP4(0x00ff11ca), SP4(0x00fb7991), SP4(0x069fdc58), SP4(0x4b584000)
};

static const int32_t  _anamatrix4[4] = {
	SA4(0x2d413cc0), SA4(0x3b20d780), SA4(0x40000000), SA4(0x187de2a0)
};

static const int32_t _sbc_proto_8[40] = {
	SP8(0x02e5cd20), SP8(0x22d0c200), SP8(0x006bfe27), SP8(0x07808930), SP8(0x3f1c8800), SP8(0xf8810d70), SP8(0x002cfdc6), SP8(0x055acf28),
	SP8(0x31f566c0), SP8(0xebfe57e0), SP8(0xff27c437), SP8(0x001485cc), SP8(0x041c6e58), SP8(0x2a7cfa80), SP8(0xe4c4a240), SP8(0xfe359e4c),
	SP8(0x0048b1f8), SP8(0x0686ce30), SP8(0x38eec5c0), SP8(0xf2a1b9f0), SP8(0xffe8904a), SP8(0x0095698a), SP8(0x0824a480), SP8(0x443b3c00),
	SP8(0xfd7badc8), SP8(0x00d3e2d9), SP8(0x00c183d2), SP8(0x084e1950), SP8(0x4810d800), SP8(0x017f43fe), SP8(0x01056dd8), SP8(0x00e9cb9f),
	SP8(0x07d7d090), SP8(0x4a708980), SP8(0x0488fae8), SP8(0x0113bd20), SP8(0x0107b1a8), SP8(0x069fb3c0), SP8(0x4b3db200), SP8(0x00763f48)
};

static const int32_t sbc_proto_4_40m0[] = {
	SS4(0x00000000), SS4(0xffa6982f), SS4(0xfba93848), SS4(0x0456c7b8),
        SS4(0x005967d1), SS4(0xfffb9ac7), SS4(0xff589157), SS4(0xf9c2a8d8),
        SS4(0x027c1434), SS4(0x0019118b), SS4(0xfff3c74c), SS4(0xff137330),
        SS4(0xf81b8d70), SS4(0x00ec1b8b), SS4(0xfff0b71a), SS4(0xffe99b00),
	SS4(0xfef84470), SS4(0xf6fb4370), SS4(0xffcdc351), SS4(0xffe01dc7)
};

static const int32_t sbc_proto_4_40m1[] = {
	SS4(0xffe090ce), SS4(0xff2c0475), SS4(0xf694f800), SS4(0xff2c0475),
        SS4(0xffe090ce), SS4(0xffe01dc7), SS4(0xffcdc351), SS4(0xf6fb4370),
        SS4(0xfef84470), SS4(0xffe99b00), SS4(0xfff0b71a), SS4(0x00ec1b8b),
        SS4(0xf81b8d70), SS4(0xff137330), SS4(0xfff3c74c), SS4(0x0019118b),
        SS4(0x027c1434), SS4(0xf9c2a8d8), SS4(0xff589157), SS4(0xfffb9ac7)
};

static const int32_t sbc_proto_8_80m0[] = {
	SS8(0x00000000), SS8(0xfe8d1970), SS8(0xee979f00), SS8(0x11686100), 
	SS8(0x0172e690), SS8(0xfff5bd1a), SS8(0xfdf1c8d4), SS8(0xeac182c0), 
	SS8(0x0d9daee0), SS8(0x00e530da), SS8(0xffe9811d), SS8(0xfd52986c), 
	SS8(0xe7054ca0), SS8(0x0a00d410), SS8(0x006c1de4), SS8(0xffdba705), 
	SS8(0xfcbc98e8), SS8(0xe3889d20), SS8(0x06af2308), SS8(0x000bb7db), 
	SS8(0xffca00ed), SS8(0xfc3fbb68), SS8(0xe071bc00), SS8(0x03bf7948), 
	SS8(0xffc4e05c), SS8(0xffb54b3b), SS8(0xfbedadc0), SS8(0xdde26200), 
	SS8(0x0142291c), SS8(0xff960e94), SS8(0xff9f3e17), SS8(0xfbd8f358), 
	SS8(0xdbf79400), SS8(0xff405e01), SS8(0xff7d4914), SS8(0xff8b1a31), 
	SS8(0xfc1417b8), SS8(0xdac7bb40), SS8(0xfdbb828c), SS8(0xff762170)
};

static const int32_t sbc_proto_8_80m1[] = {
	SS8(0xff7c272c), SS8(0xfcb02620), SS8(0xda612700), SS8(0xfcb02620),
        SS8(0xff7c272c), SS8(0xff762170), SS8(0xfdbb828c), SS8(0xdac7bb40),
        SS8(0xfc1417b8), SS8(0xff8b1a31), SS8(0xff7d4914), SS8(0xff405e01),
        SS8(0xdbf79400), SS8(0xfbd8f358), SS8(0xff9f3e17), SS8(0xff960e94),
        SS8(0x0142291c), SS8(0xdde26200), SS8(0xfbedadc0), SS8(0xffb54b3b),
        SS8(0xffc4e05c), SS8(0x03bf7948), SS8(0xe071bc00), SS8(0xfc3fbb68),
        SS8(0xffca00ed), SS8(0x000bb7db), SS8(0x06af2308), SS8(0xe3889d20),
        SS8(0xfcbc98e8), SS8(0xffdba705), SS8(0x006c1de4), SS8(0x0a00d410),
        SS8(0xe7054ca0), SS8(0xfd52986c), SS8(0xffe9811d), SS8(0x00e530da),
        SS8(0x0d9daee0), SS8(0xeac182c0), SS8(0xfdf1c8d4), SS8(0xfff5bd1a)
};

static const int32_t _anamatrix8[8] = {
	SA8(0x3b20d780), SA8(0x187de2a0), SA8(0x3ec52f80), SA8(0x3536cc40), SA8(0x238e7680), SA8(0x0c7c5c20), SA8(0x2d413cc0), SA8(0x40000000)
};

static const int32_t synmatrix4[8][4] =  {
        { SN4(0x05a82798), SN4(0xfa57d868), SN4(0xfa57d868), SN4(0x05a82798), 
	 },
        { SN4(0x030fbc54), SN4(0xf89be510), SN4(0x07641af0), SN4(0xfcf043ac), 
	 },
        { SN4(0x00000000), SN4(0x00000000), SN4(0x00000000), SN4(0x00000000), 
	 },
        { SN4(0xfcf043ac), SN4(0x07641af0), SN4(0xf89be510), SN4(0x030fbc54), 
	 },
        { SN4(0xfa57d868), SN4(0x05a82798), SN4(0x05a82798), SN4(0xfa57d868), 
	 },
        { SN4(0xf89be510), SN4(0xfcf043ac), SN4(0x030fbc54), SN4(0x07641af0), 
	 },
        { SN4(0xf8000000), SN4(0xf8000000), SN4(0xf8000000), SN4(0xf8000000), 
	 },
        { SN4(0xf89be510), SN4(0xfcf043ac), SN4(0x030fbc54), SN4(0x07641af0), 
	 }
};

static const int32_t synmatrix8[16][8] = {
        { SN8(0x05a82798), SN8(0xfa57d868), SN8(0xfa57d868), SN8(0x05a82798), 
	SN8(0x05a82798), SN8(0xfa57d868), SN8(0xfa57d868), SN8(0x05a82798), 
	 },
        { SN8(0x0471ced0), SN8(0xf8275a10), SN8(0x018f8b84), SN8(0x06a6d988), 
	SN8(0xf9592678), SN8(0xfe70747c), SN8(0x07d8a5f0), SN8(0xfb8e3130), 
	 },
        { SN8(0x030fbc54), SN8(0xf89be510), SN8(0x07641af0), SN8(0xfcf043ac), 
	SN8(0xfcf043ac), SN8(0x07641af0), SN8(0xf89be510), SN8(0x030fbc54), 
	 },
        { SN8(0x018f8b84), SN8(0xfb8e3130), SN8(0x06a6d988), SN8(0xf8275a10), 
	SN8(0x07d8a5f0), SN8(0xf9592678), SN8(0x0471ced0), SN8(0xfe70747c), 
	 },
        { SN8(0x00000000), SN8(0x00000000), SN8(0x00000000), SN8(0x00000000), 
	SN8(0x00000000), SN8(0x00000000), SN8(0x00000000), SN8(0x00000000), 
	 },
        { SN8(0xfe70747c), SN8(0x0471ced0), SN8(0xf9592678), SN8(0x07d8a5f0), 
	SN8(0xf8275a10), SN8(0x06a6d988), SN8(0xfb8e3130), SN8(0x018f8b84), 
	 },
        { SN8(0xfcf043ac), SN8(0x07641af0), SN8(0xf89be510), SN8(0x030fbc54), 
	SN8(0x030fbc54), SN8(0xf89be510), SN8(0x07641af0), SN8(0xfcf043ac), 
	 },
        { SN8(0xfb8e3130), SN8(0x07d8a5f0), SN8(0xfe70747c), SN8(0xf9592678), 
	SN8(0x06a6d988), SN8(0x018f8b84), SN8(0xf8275a10), SN8(0x0471ced0), 
	 },
        { SN8(0xfa57d868), SN8(0x05a82798), SN8(0x05a82798), SN8(0xfa57d868), 
	SN8(0xfa57d868), SN8(0x05a82798), SN8(0x05a82798), SN8(0xfa57d868), 
	 },
        { SN8(0xf9592678), SN8(0x018f8b84), SN8(0x07d8a5f0), SN8(0x0471ced0), 
	SN8(0xfb8e3130), SN8(0xf8275a10), SN8(0xfe70747c), SN8(0x06a6d988), 
	 },
        { SN8(0xf89be510), SN8(0xfcf043ac), SN8(0x030fbc54), SN8(0x07641af0), 
	SN8(0x07641af0), SN8(0x030fbc54), SN8(0xfcf043ac), SN8(0xf89be510), 
	 },
        { SN8(0xf8275a10), SN8(0xf9592678), SN8(0xfb8e3130), SN8(0xfe70747c), 
	SN8(0x018f8b84), SN8(0x0471ced0), SN8(0x06a6d988), SN8(0x07d8a5f0), 
	 },
        { SN8(0xf8000000), SN8(0xf8000000), SN8(0xf8000000), SN8(0xf8000000), 
	SN8(0xf8000000), SN8(0xf8000000), SN8(0xf8000000), SN8(0xf8000000), 
	 },
        { SN8(0xf8275a10), SN8(0xf9592678), SN8(0xfb8e3130), SN8(0xfe70747c), 
	SN8(0x018f8b84), SN8(0x0471ced0), SN8(0x06a6d988), SN8(0x07d8a5f0), 
	 },
        { SN8(0xf89be510), SN8(0xfcf043ac), SN8(0x030fbc54), SN8(0x07641af0), 
	SN8(0x07641af0), SN8(0x030fbc54), SN8(0xfcf043ac), SN8(0xf89be510), 
	 },
        { SN8(0xf9592678), SN8(0x018f8b84), SN8(0x07d8a5f0), SN8(0x0471ced0), 
	SN8(0xfb8e3130), SN8(0xf8275a10), SN8(0xfe70747c), SN8(0x06a6d988), 
	 }
};

#else // USE_FIXED

/* alternating blocks of 4 from 40m */
static const float sbc_proto_4_40m0[] = {
        -0.00000000e+00, -4.36550491e-02, -5.42373121e-01, 5.42373121e-01, 4.36550491e-02,
        -2.14619585e-03, -8.17540362e-02, -7.79951394e-01, 3.10585409e-01, 1.22404918e-02,
        -5.96753415e-03, -1.15502954e-01, -9.86546636e-01, 1.15286909e-01, -7.46326754e-03,
        -1.09348362e-02, -1.28775716e-01, -1.12731278e+00, -2.45298073e-02, -1.55682061e-02
};

static const float sbc_proto_4_40m1[] = {
        -1.53488079e-02, -1.03507124e-01, -1.17726135e+00, -1.03507124e-01, -1.53488079e-02,
        -1.55682061e-02, -2.45298073e-02, -1.12731278e+00, -1.28775716e-01, -1.09348362e-02,
        -7.46326754e-03, 1.15286909e-01, -9.86546636e-01, -1.15502954e-01, -5.96753415e-03,
        1.22404918e-02, 3.10585409e-01, -7.79951394e-01, -8.17540362e-02, -2.14619585e-03
};

/* alternating blocks of 8 from 80m[], reordered to eliminate state->X */
static const float sbc_proto_8_80m0[] = {
        -0.00000000e+00, -4.52759564e-02, -5.43991566e-01, 5.43991566e-01, 4.52759564e-02,
        -1.25260314e-03, -6.42352924e-02, -6.63878083e-01, 4.25498426e-01, 2.79773958e-02,
        -2.74605141e-03, -8.36675540e-02, -7.80603111e-01, 3.12601119e-01, 1.31978476e-02,
        -4.43696184e-03, -1.01977870e-01, -8.89573514e-01, 2.08879009e-01, 1.43044291e-03,
        -6.59135589e-03, -1.17220208e-01, -9.86116409e-01, 1.17123261e-01, -7.21723586e-03,
        -9.11940075e-03, -1.27236485e-01, -1.06611538e+00, 3.93262431e-02, -1.29325027e-02,
        -1.18112136e-02, -1.29766777e-01, -1.12602806e+00, -2.33926754e-02, -1.59563646e-02,
        -1.42697385e-02, -1.22547284e-01, -1.16311872e+00, -7.08606020e-02, -1.68297589e-02
};

/* alternating blocks of 8 from 80m[] (odds), reordered to eliminate state->X */
static const float sbc_proto_8_80m1[] = {
        -1.60946026e-02, -1.03497446e-01, -1.17564058e+00, -1.03497446e-01, -1.60946026e-02,
        -1.68297589e-02, -7.08606020e-02, -1.16311872e+00, -1.22547284e-01, -1.42697385e-02,
        -1.59563646e-02, -2.33926754e-02, -1.12602806e+00, -1.29766777e-01, -1.18112136e-02,
        -1.29325027e-02, 3.93262431e-02, -1.06611538e+00, -1.27236485e-01, -9.11940075e-03,
        -7.21723586e-03, 1.17123261e-01, -9.86116409e-01, -1.17220208e-01, -6.59135589e-03,
        1.43044291e-03, 2.08879009e-01, -8.89573514e-01, -1.01977870e-01, -4.43696184e-03,
        1.31978476e-02, 3.12601119e-01, -7.80603111e-01, -8.36675540e-02, -2.74605141e-03,
        2.79773958e-02, 4.25498426e-01, -6.63878083e-01, -6.42352924e-02, -1.25260314e-03,
};

/* Precomputed: synmatrix4[k][i] = cos( (i+0.5) * (k+2.0) * pi/4.0 ) */
static const float synmatrix4[8][4] =  {
	{  0.707106781186548, -0.707106781186547, -0.707106781186548,  0.707106781186547 },
	{  0.38268343236509,  -0.923879532511287,  0.923879532511287, -0.38268343236509  },
	{  0,                  0,                  0,                  0                 },
	{ -0.38268343236509,   0.923879532511287, -0.923879532511287,  0.382683432365091 },
	{ -0.707106781186547,  0.707106781186548,  0.707106781186547, -0.707106781186547 },
	{ -0.923879532511287, -0.38268343236509,   0.382683432365091,  0.923879532511288 },
	{ -1,                 -1,                 -1,                 -1                 },
	{ -0.923879532511287, -0.382683432365091,  0.38268343236509,   0.923879532511287 }
};

/* Precomputed: synmatrix8[k][i] = cos( (i+0.5) * (k+4.0) * pi/8.0 ) */
static const float synmatrix8[16][8] = {
	{  0.707106781186548, -0.707106781186547, -0.707106781186548,  0.707106781186547,
	   0.707106781186548, -0.707106781186547, -0.707106781186547,  0.707106781186547 },
	{  0.555570233019602, -0.98078528040323,   0.195090322016128,  0.831469612302545,
	  -0.831469612302545, -0.195090322016128,  0.980785280403231, -0.555570233019602 },
	{  0.38268343236509,  -0.923879532511287,  0.923879532511287, -0.38268343236509,
	  -0.382683432365091,  0.923879532511287, -0.923879532511286,  0.38268343236509  },
	{  0.195090322016128, -0.555570233019602,  0.831469612302545, -0.980785280403231,
	   0.98078528040323,  -0.831469612302545,  0.555570233019602, -0.195090322016129 },
	{  0,                  0,                  0,                  0,
	   0,                  0,                  0,                  0                 },
	{ -0.195090322016128,  0.555570233019602, -0.831469612302545,  0.98078528040323,
	  -0.980785280403231,  0.831469612302545, -0.555570233019603,  0.19509032201613  },
	{ -0.38268343236509,   0.923879532511287, -0.923879532511287,  0.382683432365091,
	   0.38268343236509,  -0.923879532511287,  0.923879532511288, -0.382683432365091 },
	{ -0.555570233019602,  0.98078528040323,  -0.195090322016128, -0.831469612302545,
	   0.831469612302545,  0.195090322016128, -0.98078528040323,   0.555570233019606 },
	{ -0.707106781186547,  0.707106781186548,  0.707106781186547, -0.707106781186547,
	  -0.707106781186546,  0.707106781186548,  0.707106781186546, -0.707106781186548 },
	{ -0.831469612302545,  0.195090322016129,  0.980785280403231,  0.555570233019602,
	  -0.555570233019603, -0.98078528040323,  -0.195090322016128,  0.831469612302547 },
	{ -0.923879532511287, -0.38268343236509,   0.382683432365091,  0.923879532511288,
	   0.923879532511287,  0.382683432365089, -0.382683432365091, -0.923879532511287 },
	{ -0.98078528040323,  -0.831469612302545, -0.555570233019602, -0.195090322016129,
	   0.19509032201613,   0.555570233019606,  0.831469612302547,  0.980785280403231 },
	{ -1,                 -1,                 -1,                 -1,
	  -1,                 -1,                 -1,                 -1                 },
	{ -0.98078528040323,  -0.831469612302546, -0.555570233019603, -0.19509032201613,
	   0.195090322016128,  0.555570233019604,  0.831469612302545,  0.98078528040323  },
	{ -0.923879532511287, -0.382683432365091,  0.38268343236509,   0.923879532511287,
	   0.923879532511288,  0.382683432365088, -0.382683432365089, -0.923879532511285 },
	{ -0.831469612302545,  0.195090322016127,  0.98078528040323,   0.555570233019603,
	  -0.555570233019601, -0.98078528040323,  -0.195090322016131,  0.831469612302545 }
};

static const float _anamatrix4[4] = {
	0.707106769084930, 0.923879504203796, 1.000000000000000, 0.382683426141739
};

static const float _sbc_proto_4[20] = {
	1.09137623e-02,	1.35593280e-01,	5.36548963e-04,	2.04385091e-02,	1.94987848e-01,	-7.76463524e-02,
	-3.06012295e-03, 1.49188354e-03, 2.88757384e-02, 2.46636659e-01, -2.88217273e-02, 1.86581688e-03,
	2.73370906e-03,	3.21939290e-02,	2.81828195e-01,	6.13245182e-03,	3.89205152e-03,	3.83720198e-03,
	2.58767810e-02,	2.94315338e-01
};

static const float _anamatrix8[8] = {
	0.923879532511287, 0.382683432365091, 0.980785280403231, 0.831469612302547, 0.555570233019602,
	0.195090322016128, 0.707106781186547, 1.000000000000000
};

static const float _sbc_proto_8[40] = {
	5.65949455E-03, 6.79989457E-02, 8.23919487E-04, 1.46525260E-02, 1.23264551E-01,-1.46404076E-02,
	3.43256426E-04, 1.04584442E-02, 9.75753888E-02, -3.90751399E-02, -1.64973096E-03, 1.56575392E-04,
	8.02941155E-03, 8.29847604E-02, -5.31873032E-02, -3.49717448E-03, 5.54620230E-04, 1.27472337E-02,
	1.11196689E-01, -2.61098761E-02, -1.78805363E-04, 1.13992509E-03, 1.59045607E-02, 1.33264422E-01,
	-4.91578039E-03, 1.61656283E-03, 1.47640170E-03, 1.62208471E-02, 1.40753508E-01, 2.92408443E-03,
	1.99454557E-03, 1.78371731E-03, 1.53184105E-02, 1.45389840E-01, 8.85757525E-03, 2.10371986E-03,
	2.01182533E-03, 1.29371807E-02, 1.46955073E-01, 9.02154483E-04
};
#endif // USE_FIXED

#endif /* __SBC_TABLES_H */

