//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Mask/MaskEditorActions.cpp
//! @brief     Implement class MaskEditorActions
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Mask/MaskEditorActions.h"
#include "GUI/Model/Data/MaskItems.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/Support/Util/ActionFactory.h"
#include <QAction>
#include <QItemSelectionModel>
#include <QMenu>

MaskEditorActions::MaskEditorActions(QWidget* parent)
    : QObject(parent)
    , m_toggleMaskValueAction(new QAction("Toggle mask value", parent))
    , m_bringToFrontAction(new QAction("Rise mask up", parent))
    , m_sendToBackAction(new QAction("Lower mask down", parent))
    , m_deleteMaskAction(new QAction("Remove mask", parent))
    , m_resetViewAction(new QAction(this))
    , m_savePlotAction(new QAction(this))
    , m_maskContainerModel(nullptr)
    , m_selectionModel(nullptr)
{
    connect(m_toggleMaskValueAction, &QAction::triggered, this,
            &MaskEditorActions::onToggleMaskValueAction);

    m_bringToFrontAction->setIcon(QIcon(":/Mask/images/maskeditor_bringtofront.svg"));
    m_bringToFrontAction->setToolTip("Rise selected mask one level up (PgUp)");
    m_bringToFrontAction->setShortcuts(QKeySequence::MoveToPreviousPage);
    connect(m_bringToFrontAction, &QAction::triggered, this,
            &MaskEditorActions::onBringToFrontAction);

    m_sendToBackAction->setIcon(QIcon(":/Mask/images/maskeditor_sendtoback.svg"));
    m_sendToBackAction->setToolTip("Lower selected mask one level down (PgDown)");
    m_sendToBackAction->setShortcuts(QKeySequence::MoveToNextPage);
    connect(m_sendToBackAction, &QAction::triggered, this, &MaskEditorActions::onSendToBackAction);

    m_deleteMaskAction->setToolTip("Remove selected mask (Del)");
    m_deleteMaskAction->setShortcuts(QKeySequence::Delete);
    parent->addAction(m_deleteMaskAction);
    connect(m_deleteMaskAction, &QAction::triggered, this, &MaskEditorActions::onDeleteMaskAction);

    // Actions for top toolbar
    m_resetViewAction->setText("Center view");
    m_resetViewAction->setIcon(QIcon(":/images/camera-metering-center.svg"));
    m_resetViewAction->setToolTip("Center View");
    connect(m_resetViewAction, &QAction::triggered, this, &MaskEditorActions::resetViewRequest);

    m_savePlotAction->setText("Save");
    m_savePlotAction->setIcon(QIcon(":/images/content-save-outline.svg"));
    m_savePlotAction->setToolTip("Save Plot");
    connect(m_savePlotAction, &QAction::triggered, this, &MaskEditorActions::savePlotRequest);

    m_togglePanelAction = ActionFactory::createTogglePropertiesPanelAction(this);
    connect(m_togglePanelAction, &QAction::triggered, this,
            &MaskEditorActions::propertyPanelRequest);
}

void MaskEditorActions::setModel(MaskContainerModel* maskContainerModel)
{
    ASSERT(maskContainerModel);
    m_maskContainerModel = maskContainerModel;
}

void MaskEditorActions::setSelectionModel(QItemSelectionModel* selectionModel)
{
    m_selectionModel = selectionModel;
}

QAction* MaskEditorActions::sendToBackAction()
{
    return m_sendToBackAction;
}

QAction* MaskEditorActions::bringToFrontAction()
{
    return m_bringToFrontAction;
}

QList<QAction*> MaskEditorActions::topToolbarActions()
{
    return QList<QAction*>() << m_resetViewAction << m_togglePanelAction;
}

//! Constructs MaskItem context menu following the request from MaskGraphicsScene
//! or MaskEditorInfoPanel
void MaskEditorActions::onItemContextMenuRequest(const QPoint& point)
{
    QMenu menu;
    initItemContextMenu(menu);
    menu.exec(point);
    setAllActionsEnabled(true);
}

void MaskEditorActions::onDeleteMaskAction()
{
    ASSERT(m_maskContainerModel);
    ASSERT(m_selectionModel);

    QModelIndexList indexes = m_selectionModel->selectedIndexes();
    while (!indexes.empty()) {
        m_maskContainerModel->removeMaskAt(indexes.back().row());
        indexes = m_selectionModel->selectedIndexes();
    }
    gProjectDocument.value()->setModified();
}

//! Performs switch of mask value for all selected items (true -> false, false -> true)
void MaskEditorActions::onToggleMaskValueAction()
{
    ASSERT(m_maskContainerModel);
    ASSERT(m_selectionModel);
    for (auto itemIndex : m_selectionModel->selectedIndexes()) {
        if (MaskItem* item = m_maskContainerModel->itemForIndex(itemIndex))
            item->setMaskValue(!item->maskValue());
    }
    gProjectDocument.value()->setModified();
}

void MaskEditorActions::onBringToFrontAction()
{
    changeMaskStackingOrder(MaskEditorFlags::BRING_TO_FRONT);
}

void MaskEditorActions::onSendToBackAction()
{
    changeMaskStackingOrder(MaskEditorFlags::SEND_TO_BACK);
}

//! Lower mask one level down or rise one level up in the masks stack
void MaskEditorActions::changeMaskStackingOrder(MaskEditorFlags::Stacking value)
{
    if (!m_maskContainerModel || !m_selectionModel)
        return;

    if (value == MaskEditorFlags::SEND_TO_BACK) {
        QModelIndexList indexes = m_selectionModel->selectedIndexes();
        for (const QModelIndex& itemIndex : indexes) {
            int row = itemIndex.row();
            int new_row = row + 1;
            if (new_row >= 0 && new_row < m_maskContainerModel->rowCount({})) {
                // Here we lower element from 'new_row' instead of rising element from 'row'
                // The result should be the same, but rising doesn't work for an unknown reason
                m_maskContainerModel->moveMask(new_row, row);
            }
        }
    }
    if (value == MaskEditorFlags::BRING_TO_FRONT) {
        QModelIndexList indexes = m_selectionModel->selectedIndexes();
        for (const QModelIndex& itemIndex : indexes) {
            int row = itemIndex.row();
            int new_row = row - 1;
            if (new_row >= 0 && new_row < m_maskContainerModel->rowCount({}))
                m_maskContainerModel->moveMask(row, new_row);
        }
    }
    gProjectDocument.value()->setModified();
}

//! Returns true if at least one of MaskItems in the selection can be moved one level up
//! (Naturally, it is always true, if selection contains more than one item. If selection contains
//! only one item, the result will depend on position of item on the stack.
//! Top item cannot be moved up. Used to disable corresponding context menu line.)
bool MaskEditorActions::isBringToFrontPossible() const
{
    bool result(false);
    QModelIndexList indexes = m_selectionModel->selectedIndexes();
    if (indexes.size() == 1 && indexes.front().row() != 0)
        result = true;
    return result;
}

//! Returns true if at least one of MaskItems in the selection can be moved one level down.
bool MaskEditorActions::isSendToBackPossible() const
{
    bool result(false);
    QModelIndexList indexes = m_selectionModel->selectedIndexes();
    if (indexes.size() == 1)
        if (indexes.front().row() != m_maskContainerModel->maskContainer->size() - 1)
            result = true;
    return result;
}

void MaskEditorActions::setAllActionsEnabled(bool value)
{
    m_sendToBackAction->setEnabled(value);
    m_bringToFrontAction->setEnabled(value);
    m_toggleMaskValueAction->setEnabled(value);
    m_deleteMaskAction->setEnabled(value);
}

//! Init external context menu with currently defined actions.
//! Triggered from MaskGraphicsScene of MaskEditorInfoPanel (QListView)
void MaskEditorActions::initItemContextMenu(QMenu& menu)
{
    ASSERT(m_maskContainerModel);
    ASSERT(m_selectionModel);

    if (m_selectionModel->selectedIndexes().isEmpty())
        setAllActionsEnabled(false);

    m_sendToBackAction->setEnabled(isSendToBackPossible());
    m_bringToFrontAction->setEnabled(isBringToFrontPossible());

    menu.addAction(m_toggleMaskValueAction);
    menu.addAction(m_bringToFrontAction);
    menu.addAction(m_sendToBackAction);
    menu.addSeparator();
    menu.addAction(m_deleteMaskAction);
}
