/**********************************************************************
 * cbf_ascii -- write plain ASCII values                              *
 *                                                                    *
 * Version 0.7.6 14 July 2006                                         *
 *                                                                    *
 *                          Paul Ellis and                            *
 *         Herbert J. Bernstein (yaya@bernstein-plus-sons.com)        *
 *                                                                    *
 * (C) Copyright 2006 Herbert J. Bernstein                            *
 *                                                                    *
 **********************************************************************/

/**********************************************************************
 *                                                                    *
 * YOU MAY REDISTRIBUTE THE CBFLIB PACKAGE UNDER THE TERMS OF THE GPL *
 *                                                                    *
 * ALTERNATIVELY YOU MAY REDISTRIBUTE THE CBFLIB API UNDER THE TERMS  *
 * OF THE LGPL                                                        *
 *                                                                    *
 **********************************************************************/

/*************************** GPL NOTICES ******************************
 *                                                                    *
 * This program is free software; you can redistribute it and/or      *
 * modify it under the terms of the GNU General Public License as     *
 * published by the Free Software Foundation; either version 2 of     *
 * (the License, or (at your option) any later version.               *
 *                                                                    *
 * This program is distributed in the hope that it will be useful,    *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 * GNU General Public License for more details.                       *
 *                                                                    *
 * You should have received a copy of the GNU General Public License  *
 * along with this program; if not, write to the Free Software        *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA           *
 * 02111-1307  USA                                                    *
 *                                                                    *
 **********************************************************************/

/************************* LGPL NOTICES *******************************
 *                                                                    *
 * This library is free software; you can redistribute it and/or      *
 * modify it under the terms of the GNU Lesser General Public         *
 * License as published by the Free Software Foundation; either       *
 * version 2.1 of the License, or (at your option) any later version. *
 *                                                                    *
 * This library is distributed in the hope that it will be useful,    *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of     *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU  *
 * Lesser General Public License for more details.                    *
 *                                                                    *
 * You should have received a copy of the GNU Lesser General Public   *
 * License along with this library; if not, write to the Free         *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,    *
 * MA  02110-1301  USA                                                *
 *                                                                    *
 **********************************************************************/

/**********************************************************************
 *                                                                    *
 *                    Stanford University Notices                     *
 *  for the CBFlib software package that incorporates SLAC software   *
 *                 on which copyright is disclaimed                   *
 *                                                                    *
 * This software                                                      *
 * -------------                                                      *
 * The term this software, as used in these Notices, refers to      *
 * those portions of the software package CBFlib that were created by *
 * employees of the Stanford Linear Accelerator Center, Stanford      *
 * University.                                                        *
 *                                                                    *
 * Stanford disclaimer of copyright                                   *
 * --------------------------------                                   *
 * Stanford University, owner of the copyright, hereby disclaims its  *
 * copyright and all other rights in this software.  Hence, anyone    *
 * may freely use it for any purpose without restriction.             *
 *                                                                    *
 * Acknowledgement of sponsorship                                     *
 * ------------------------------                                     *
 * This software was produced by the Stanford Linear Accelerator      *
 * Center, Stanford University, under Contract DE-AC03-76SFO0515 with *
 * the Department of Energy.                                          *
 *                                                                    *
 * Government disclaimer of liability                                 *
 * ----------------------------------                                 *
 * Neither the United States nor the United States Department of      *
 * Energy, nor any of their employees, makes any warranty, express or *
 * implied, or assumes any legal liability or responsibility for the  *
 * accuracy, completeness, or usefulness of any data, apparatus,      *
 * product, or process disclosed, or represents that its use would    *
 * not infringe privately owned rights.                               *
 *                                                                    *
 * Stanford disclaimer of liability                                   *
 * --------------------------------                                   *
 * Stanford University makes no representations or warranties,        *
 * express or implied, nor assumes any liability for the use of this  *
 * software.                                                          *
 *                                                                    *
 * Maintenance of notices                                             *
 * ----------------------                                             *
 * In the interest of clarity regarding the origin and status of this *
 * software, this and all the preceding Stanford University notices   *
 * are to remain affixed to any copy or derivative of this software   *
 * made or distributed by the recipient and are to be affixed to any  *
 * copy of software made or distributed by the recipient that         *
 * contains a copy or derivative of this software.                    *
 *                                                                    *
 * Based on SLAC Software Notices, Set 4                              *
 * OTT.002a, 2004 FEB 03                                              *
 **********************************************************************/



/**********************************************************************
 *                               NOTICE                               *
 * Creative endeavors depend on the lively exchange of ideas. There   *
 * are laws and customs which establish rights and responsibilities   *
 * for authors and the users of what authors create.  This notice     *
 * is not intended to prevent you from using the software and         *
 * documents in this package, but to ensure that there are no         *
 * misunderstandings about terms and conditions of such use.          *
 *                                                                    *
 * Please read the following notice carefully.  If you do not         *
 * understand any portion of this notice, please seek appropriate     *
 * professional legal advice before making use of the software and    *
 * documents included in this software package.  In addition to       *
 * whatever other steps you may be obliged to take to respect the     *
 * intellectual property rights of the various parties involved, if   *
 * you do make use of the software and documents in this package,     *
 * please give credit where credit is due by citing this package,     *
 * its authors and the URL or other source from which you obtained    *
 * it, or equivalent primary references in the literature with the    *
 * same authors.                                                      *
 *                                                                    *
 * Some of the software and documents included within this software   *
 * package are the intellectual property of various parties, and      *
 * placement in this package does not in any way imply that any       *
 * such rights have in any way been waived or diminished.             *
 *                                                                    *
 * With respect to any software or documents for which a copyright    *
 * exists, ALL RIGHTS ARE RESERVED TO THE OWNERS OF SUCH COPYRIGHT.   *
 *                                                                    *
 * Even though the authors of the various documents and software      *
 * found here have made a good faith effort to ensure that the        *
 * documents are correct and that the software performs according     *
 * to its documentation, and we would greatly appreciate hearing of   *
 * any problems you may encounter, the programs and documents any     *
 * files created by the programs are provided **AS IS** without any   *
 * warranty as to correctness, merchantability or fitness for any     *
 * particular or general use.                                         *
 *                                                                    *
 * THE RESPONSIBILITY FOR ANY ADVERSE CONSEQUENCES FROM THE USE OF    *
 * PROGRAMS OR DOCUMENTS OR ANY FILE OR FILES CREATED BY USE OF THE   *
 * PROGRAMS OR DOCUMENTS LIES SOLELY WITH THE USERS OF THE PROGRAMS   *
 * OR DOCUMENTS OR FILE OR FILES AND NOT WITH AUTHORS OF THE          *
 * PROGRAMS OR DOCUMENTS.                                             *
 **********************************************************************/

/**********************************************************************
 *                                                                    *
 *                           The IUCr Policy                          *
 *      for the Protection and the Promotion of the STAR File and     *
 *     CIF Standards for Exchanging and Archiving Electronic Data     *
 *                                                                    *
 * Overview                                                           *
 *                                                                    *
 * The Crystallographic Information File (CIF)[1] is a standard for   *
 * information interchange promulgated by the International Union of  *
 * Crystallography (IUCr). CIF (Hall, Allen & Brown, 1991) is the     *
 * recommended method for submitting publications to Acta             *
 * Crystallographica Section C and reports of crystal structure       *
 * determinations to other sections of Acta Crystallographica         *
 * and many other journals. The syntax of a CIF is a subset of the    *
 * more general STAR File[2] format. The CIF and STAR File approaches *
 * are used increasingly in the structural sciences for data exchange *
 * and archiving, and are having a significant influence on these     *
 * activities in other fields.                                        *
 *                                                                    *
 * Statement of intent                                                *
 *                                                                    *
 * The IUCr's interest in the STAR File is as a general data          *
 * interchange standard for science, and its interest in the CIF,     *
 * a conformant derivative of the STAR File, is as a concise data     *
 * exchange and archival standard for crystallography and structural  *
 * science.                                                           *
 *                                                                    *
 * Protection of the standards                                        *
 *                                                                    *
 * To protect the STAR File and the CIF as standards for              *
 * interchanging and archiving electronic data, the IUCr, on behalf   *
 * of the scientific community,                                       *
 *                                                                    *
 * * holds the copyrights on the standards themselves,                *
 *                                                                    *
 * * owns the associated trademarks and service marks, and            *
 *                                                                    *
 * * holds a patent on the STAR File.                                 *
 *                                                                    *
 * These intellectual property rights relate solely to the            *
 * interchange formats, not to the data contained therein, nor to     *
 * the software used in the generation, access or manipulation of     *
 * the data.                                                          *
 *                                                                    *
 * Promotion of the standards                                         *
 *                                                                    *
 * The sole requirement that the IUCr, in its protective role,        *
 * imposes on software purporting to process STAR File or CIF data    *
 * is that the following conditions be met prior to sale or           *
 * distribution.                                                      *
 *                                                                    *
 * * Software claiming to read files written to either the STAR       *
 * File or the CIF standard must be able to extract the pertinent     *
 * data from a file conformant to the STAR File syntax, or the CIF    *
 * syntax, respectively.                                              *
 *                                                                    *
 * * Software claiming to write files in either the STAR File, or     *
 * the CIF, standard must produce files that are conformant to the    *
 * STAR File syntax, or the CIF syntax, respectively.                 *
 *                                                                    *
 * * Software claiming to read definitions from a specific data       *
 * dictionary approved by the IUCr must be able to extract any        *
 * pertinent definition which is conformant to the dictionary         *
 * definition language (DDL)[3] associated with that dictionary.      *
 *                                                                    *
 * The IUCr, through its Committee on CIF Standards, will assist      *
 * any developer to verify that software meets these conformance      *
 * conditions.                                                        *
 *                                                                    *
 * Glossary of terms                                                  *
 *                                                                    *
 * [1] CIF:  is a data file conformant to the file syntax defined     *
 * at http://www.iucr.org/iucr-top/cif/spec/index.html                *
 *                                                                    *
 * [2] STAR File:  is a data file conformant to the file syntax       *
 * defined at http://www.iucr.org/iucr-top/cif/spec/star/index.html   *
 *                                                                    *
 * [3] DDL:  is a language used in a data dictionary to define data   *
 * items in terms of "attributes". Dictionaries currently approved    *
 * by the IUCr, and the DDL versions used to construct these          *
 * dictionaries, are listed at                                        *
 * http://www.iucr.org/iucr-top/cif/spec/ddl/index.html               *
 *                                                                    *
 * Last modified: 30 September 2000                                   *
 *                                                                    *
 * IUCr Policy Copyright (C) 2000 International Union of              *
 * Crystallography                                                    *
 **********************************************************************/

#ifdef __cplusplus

extern "C" {

#endif

#include "cbf.h"
#include "cbf_ascii.h"
#include "cbf_tree.h"
#include "cbf_file.h"

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

/* Format the next, possibly folded text line in fline, updating
   the pointer in string to be ready for the next pass.
   fline_size is the valid line length.  fline must be
   one longer to allow for termination.*/

int cbf_foldtextline(const char** string, char* fline, 
                                                int fline_size,
                                                int unfoldme,
                                                int foldme,
                                                char termc ) {
    const char *c;
    
    char *ofl;
    
    int ipos, left=fline_size;
    
    int savbpos, savbleft;
    
    c = *string;
    
    if (foldme && (termc == '\'' || termc == '\"') ) left -=2;

    ofl = fline;
    
    savbpos = -1;
    
	savbleft = left+1;
	
    /* protect folded lines that begin with ; */
      
    if (c[0] == ';' && termc == ';' && (isspace(c[1])|| !c[1]) ){

      *ofl++ = ';';
      
      *ofl++ = '\\';
      
      *ofl++ = '\n';
      
      *string = c+1;
      
      return 0;
   	
    }
    

    for (ipos=0; c[ipos]; ipos++) {
    

      /* save the last blank or tab to break on */
 
      if(( c[ipos] == ' ' || c[ipos] == '\t' ) && left < fline_size) {

        savbpos = ipos; savbleft = left;

      }
        
      /* If this is a bracketed construct, break on ',' or
         the terminating character not at the beginning */

        
      if ((termc==')' || termc==']' || termc==',') 
          && ((c[ipos]==termc || c[ipos]==',') &&  left-1 < fline_size)) {

          savbpos = ipos+1; savbleft = left-1;

      }

      /* check for a backslash */
      
      if ( foldme && c[ipos] == '\\') {
      
        /* if unfolding, ignore "\\\n" */
        
        if (unfoldme) {
        
           if (c[ipos+1] == '\n' || c[ipos+1] == '\0') {
           
             ipos++;
              
             continue;
           	
           }
        	
        }

        /* if the backslash would be at the end of the line
           insert "\\" and end the line */
 
        if ( foldme && left < 2) {
        
          *ofl++ = '\\';
          
          *ofl = '\0';
          
          *string = c+ipos;
          
          return 0;
        	
        }
      	
      }
      
      /* check if folding would bring "; " to the front of a line
         if so, end here */
      
      if ( foldme && left == 2 && c[ipos+1]==';' && isspace(c[ipos+2])  ) {
      
          *ofl++ = '\\';
          
          *ofl = '\0';
          
          *string = c+ipos;
          
          return 0;
      	
      }
      
      /* now, see if the line has ended by itself */
      
      if ( c[ipos+1] == '\n' || !c[ipos+1]) 
      {
      
         *ofl++ = c[ipos];
         
         *ofl = '\0';
         
         *string = c+ipos+1;
         
         if (c[ipos+1] == '\n') (*string)++;
         
         if (c[ipos+1]) return 0;
 
         return 1;
      	
      }
      
      /* see if we must fold */
      
      if ( foldme && left < 2)  {
      
        if (savbleft > left && savbleft < fline_size) {

           ipos = savbpos;
           
           ofl = ofl+left-savbleft;
        }
        
        *ofl++ = '\\';
        
        *ofl = '\0';
        
        *string = c+ipos;
        
        return 0;
      	
      }

      *ofl++ = c[ipos];
      
      left--;
    	
    }
    
    *ofl ='\0';
    
    *string = c+ipos;
    
    return 1;
    
    
}



  /* Write an ascii value */

int cbf_write_ascii (cbf_handle handle, const char *string, cbf_file *file)
{
  static const char missing [] = { CBF_TOKEN_WORD, '?', '\0' };

  int end, lw, lc, foldme=0, unfoldme=0;
  
  char initc=';', termc=';';

  unsigned int column;

  const char *c;
  
  char delim, adelim;
  
  char buffer[80];
  
  char fline[2049];


    /* Check the arguments */

  if (!string)

    string = missing;

  else

    if (*string != CBF_TOKEN_WORD       &&
        *string != CBF_TOKEN_SQSTRING   &&
        *string != CBF_TOKEN_DQSTRING   &&
        *string != CBF_TOKEN_SCSTRING   &&
        *string != CBF_TOKEN_TSQSTRING  &&
        *string != CBF_TOKEN_TDQSTRING  &&
        *string != CBF_TOKEN_BKTSTRING  &&
        *string != CBF_TOKEN_BRCSTRING  &&
        *string != CBF_TOKEN_PRNSTRING  &&
        *string != CBF_TOKEN_NULL)

      return CBF_ARGUMENT;


    /* Get the current column */

  cbf_failnez (cbf_get_filecoordinates (file, NULL, &column))


    /* Do we need to start a new line? */

  if (column) {

    if (*string == CBF_TOKEN_SCSTRING) {

      cbf_failnez (cbf_write_character (file, '\n'))

    } else {
    
      if (*string == CBF_TOKEN_WORD ||
          *string == CBF_TOKEN_NULL )

        end = column + 3;

      else if (*string == CBF_TOKEN_TSQSTRING ||
        *string == CBF_TOKEN_TDQSTRING)
        
        end = column + 6;

      else if (*string == CBF_TOKEN_PRNSTRING ||
        *string == CBF_TOKEN_BKTSTRING ||
        *string == CBF_TOKEN_BKTSTRING)
        
        end = column + 5;
        
      else

        end = column + 1;

      for (c = string + 1; *c && end <= (file->columnlimit); c++) {

        if (*c == '\t')

          end = (end & ~0x07) + 8;

        else

          end = end + 1;

      }

      if (end > (file->columnlimit))

        cbf_failnez (cbf_write_character (file, '\n'))
    }

  }


    /* Write the value */
 

  switch (*string)
  {
      /* Simple word? */

    case  CBF_TOKEN_WORD:
    case  CBF_TOKEN_NULL:
    
      if (strlen(string+1) <= file->columnlimit
        && *(string+1)!='"' && *(string+1)!='\''
        && !strpbrk(string+1," \t\n\r")
        && !(strlen(string+1) == file->columnlimit && *(string+1)==';') ) {

        if (strlen(string+1) != file->columnlimit)

          cbf_failnez (cbf_write_character (file, ' '))

        cbf_failnez (cbf_write_string (file, string + 1))

        break;
      
      }


      /* Single line? */

    case CBF_TOKEN_SQSTRING:
    case CBF_TOKEN_DQSTRING:
    
      if (*string == CBF_TOKEN_SQSTRING) {

        delim = '\'';
        
        adelim = '"';

      } else {

        delim = '"';

        adelim = '\'';
        
      }
        
      if (strchr(string+1,delim) && !strchr(string+1,adelim)) {

        delim = adelim;
      	  
      }

      if (strlen(string+1)+2 < file->columnlimit && !strchr(string+1,delim))  {

        if (strlen(string+1)+3 < file->columnlimit) {
        	
          cbf_failnez (cbf_write_character (file, ' '))
        }

        cbf_failnez (cbf_write_character (file, delim))

        cbf_failnez (cbf_write_string (file, string + 1))

        cbf_failnez (cbf_write_character (file, delim))

      } else {

        sprintf(buffer, "output line %u(%u) folded",1+file->line,1+file->column);

        cbf_log(handle, buffer, CBF_LOGWARNING|CBF_LOGSTARTLOC);
      
        if (file->column > 0) {
         
          cbf_failnez (cbf_write_character (file, '\n'))
        }
      
        cbf_failnez (cbf_write_string (file, ";\\\n"))

        end = 0;

        for (c = string + 1; *c; c++) {
        
          if (((file->column > file->columnlimit-10)&& (isspace(*c)||*c=='\\'))||
            file->column > file->columnlimit-2) {

          	cbf_failnez (cbf_write_string (file, "\\\n"))

          	end = 0;
          }

          cbf_failnez (cbf_write_character (file, *c))

          if (*c == ';' && end == 0 && (isspace(*(c+1))||!*(c+1))) 
          {
          	cbf_failnez (cbf_write_string (file, "\\\n"))
          	
          	end = 0;
          	
          	continue;
          }

          if (*c == '\n')

            end = 0;

          else

            end = 1;
        }

      cbf_failnez (cbf_write_string (file, "\\\n;\n"))

      end = 0;
      	
      }

      break;


      /* Multiple lines? */

    case CBF_TOKEN_SCSTRING:
    
      unfoldme = 0;
      
      foldme = 0;
      
      if (*(string+1)=='\\' && *(string+2)=='\n' ) unfoldme=2;
    
      lw = 0;
      
      lc = 1;

      end = 1;
      
      for (c = string +1+unfoldme; *c; c++) {

        if (*c == ';' && end == 0 && (isspace(*(c+1))|| !*(c+1))) foldme=1;
        
        if (*c == '\n') {

          if (!unfoldme || *(c-1) !='\\') {

            end = 0;
          
            if (lc > lw) lw = lc;
          
            lc = 0;
            
          } else  {
          
            lc--;
          	
          }

        } else {
        
          lc++;

          end = 1;
        }
      }
      
      if (lc > lw) lw = lc;

      if ( foldme || lw > file->columnlimit || (unfoldme && *(c-1)=='\\')) {

        sprintf(buffer, "output line %u(%u) folded",1+file->line,1+file->column);

        cbf_log(handle, buffer, CBF_LOGWARNING|CBF_LOGSTARTLOC);
      
        cbf_failnez (cbf_write_string (file, ";\\\n"))
        
        end = 0;
        
        foldme = 1;
      	
      } else {

        cbf_failnez (cbf_write_character (file, ';'))
        
        end = 1;
        
        foldme = 0;
      
      }

            
      for (c = string + 1+ unfoldme; *c; ) {
      
        int done;
      
        done = cbf_foldtextline(&c, fline, file->columnlimit, unfoldme, foldme, ';');
        
        cbf_failnez (cbf_write_string (file, fline))
        
        if ( !done ) cbf_failnez (cbf_write_character (file, '\n'))
        
      }
      

      if (unfoldme && ((c > string+1+unfoldme && *(c-1)=='\\') ||
          (c > string+2+unfoldme && *(c-1)=='\0' && *(c-2)=='\\'))) {

      	cbf_failnez (cbf_write_string (file, "\\\n;\n"))

      } else {

        if (file->column) {

      	  cbf_failnez (cbf_write_character (file, '\n'))

        }

      	cbf_failnez (cbf_write_string (file, ";\n"))

      }
      

      end = 0;

      break;

    case CBF_TOKEN_TSQSTRING:
    case CBF_TOKEN_TDQSTRING:
  	case CBF_TOKEN_PRNSTRING:
  	case CBF_TOKEN_BRCSTRING:
  	case CBF_TOKEN_BKTSTRING:
     
      unfoldme = 0;
      
      foldme = 0;
     
      switch (*string) {
        case CBF_TOKEN_TSQSTRING: initc = termc = '\'';
              if (!(file->write_headers & CBF_PARSE_TQ)) {initc = termc = ';'; foldme= 1;} break;
        case CBF_TOKEN_TDQSTRING: initc = termc = '"'; 
              if (!(file->write_headers & CBF_PARSE_TQ)) {initc = termc = ';'; foldme= 1;} break;
        case CBF_TOKEN_PRNSTRING: initc = '('; termc = ')';
              if (!(file->write_headers & CBF_PARSE_PRN)) {
                  if (file->write_headers & CBF_PARSE_BRC){
                      initc = '{'; termc = '}';
                  } else {
                     initc = termc = ';'; foldme= 1;
                  }
              } break;
        case CBF_TOKEN_BRCSTRING: initc = '{'; termc = '}'; 
              if (!(file->write_headers & CBF_PARSE_BRC)) {initc = termc = ';'; foldme= 1;} break;
  	    case CBF_TOKEN_BKTSTRING: initc = '['; termc = ']'; 
              if (!(file->write_headers & CBF_PARSE_PRN)) {
                  if (file->write_headers & CBF_PARSE_BRC){
                      initc = '{'; termc = '}'; 
                  } else {
                      initc = termc = ';'; foldme= 1;
                  }
              } break;
      }
      
      if (*(string+1)=='\\' && *(string+2)=='\n' ) unfoldme=2;
    
      lw = 0;
      
      lc = 1;

      end = 1;
      
      for (c = string +1+unfoldme; *c; c++) {

        if (termc==';' && *c == ';' && end == 0 && (isspace(*(c+1))|| !*(c+1))) foldme=1;
        
        if (*c == '\n') {

          if (!unfoldme || *(c-1) !='\\') {

            end = 0;
          
            if (lc > lw) lw = lc;
          
            lc = 0;
            
          } else  {
          
            lc--;
          	
          }

        } else {
        
          lc++;

          end = 1;
        }
      }
      
      if (lc > lw) lw = lc;

      if ( foldme || lw > file->columnlimit || (unfoldme && *(c-1)=='\\')) {

        sprintf(buffer, "output line %u(%u) folded",1+file->line,1+file->column);

        cbf_log(handle, buffer, CBF_LOGWARNING|CBF_LOGSTARTLOC);
      
        if (initc==';') {
          if (file->column) cbf_failnez (cbf_write_character (file, '\n'))
          cbf_failnez (cbf_write_string (file, ";\\\n"))
        } else {
          if (file->column) cbf_failnez (cbf_write_character (file, ' '))
          cbf_failnez (cbf_write_character (file, initc))
          cbf_failnez (cbf_write_string (file, "\\\n"))        	
        }
        
        end = 0;
        
        foldme = 1;
      	
      } else {

        if (file->column) cbf_failnez (cbf_write_character (file, ' '))

        cbf_failnez (cbf_write_character (file, initc))
        
        end = 1;
        
        foldme = 0;
      
      }

            
      for (c = string + 1+ unfoldme; *c; ) {
      
        int done;
      
        done = cbf_foldtextline(&c, fline, file->columnlimit
          -file->column, unfoldme, foldme, termc);
        
        cbf_failnez (cbf_write_string (file, fline))
        
        if ( !done ) cbf_failnez (cbf_write_character (file, '\n'))
        
      }
      

      if (unfoldme && *(c-1)=='\\') {

        if (termc == ';') {
          cbf_failnez (cbf_write_string (file, "\\\n;\n"))	
        }  else  {
          cbf_failnez (cbf_write_string (file, "\\\n"))	
          cbf_failnez (cbf_write_character (file, termc )) 
          if (*string==CBF_TOKEN_TSQSTRING 
            || *string==CBF_TOKEN_TDQSTRING ) {
            cbf_failnez (cbf_write_character (file, termc )) 
            cbf_failnez (cbf_write_character (file, termc ))           	
          }
        }

      } else {


        if (termc == ';') {
            if (file->column) {
                cbf_failnez (cbf_write_character (file, '\n'))
            }
            cbf_failnez (cbf_write_string (file, ";\n"))	
        }  else  {
          cbf_failnez (cbf_write_character (file, termc ))        	
          if (*string==CBF_TOKEN_TSQSTRING 
            || *string==CBF_TOKEN_TDQSTRING ) {
            cbf_failnez (cbf_write_character (file, termc )) 
            cbf_failnez (cbf_write_character (file, termc ))           	
          }
        }

      }
      

      end = 0;

      break;


  }


    /* Flush the buffer */

  return cbf_flush_characters (file);
}

#ifdef __cplusplus

}

#endif

