// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "crypto/openpgp_symmetric_encryption.h"

#include "testing/gtest/include/gtest/gtest.h"

namespace crypto {

// These test vectors were created by hand using either GPG or the Go OpenPGP
// library.

// AES-128, GPG
static const uint8 kTestMessage1[] = {
  0x8c, 0x0d, 0x04, 0x07, 0x03, 0x02, 0x69, 0x24, 0xaf, 0xbf, 0x0b, 0x31, 0x98,
  0x6d, 0x60, 0xd2, 0x3d, 0x01, 0xc4, 0x29, 0xab, 0xec, 0x1b, 0xdf, 0xfa, 0x90,
  0x86, 0x92, 0x94, 0xc7, 0xa5, 0xe7, 0xd8, 0x80, 0x0a, 0x55, 0x3e, 0xbd, 0x10,
  0xef, 0x40, 0xfe, 0xb7, 0x39, 0x83, 0x4e, 0x5e, 0x77, 0x9d, 0x57, 0x94, 0xb6,
  0xe4, 0x59, 0xa7, 0x32, 0x76, 0x22, 0x48, 0xed, 0x37, 0xe8, 0x6f, 0xf9, 0x80,
  0x04, 0xa1, 0xe4, 0xbf, 0x40, 0xa6, 0x9b, 0xd1, 0x3e, 0xba, 0xaa, 0x52, 0xd0,
};

// AES-256, GPG
static const uint8 kTestMessage2[] = {
  0x8c, 0x0d, 0x04, 0x09, 0x03, 0x02, 0xd6, 0xb0, 0x34, 0xa0, 0xb8, 0x6c, 0x15,
  0xb8, 0x60, 0xd2, 0x3d, 0x01, 0x97, 0xe4, 0x46, 0x56, 0xc5, 0xc9, 0xc7, 0x81,
  0xd1, 0x09, 0xf3, 0xa0, 0x5d, 0x3b, 0xa7, 0xe3, 0x68, 0xb4, 0x19, 0xd2, 0x76,
  0x83, 0x38, 0x13, 0x98, 0xb8, 0xaf, 0x54, 0x51, 0x52, 0xbb, 0xc5, 0x7d, 0x8a,
  0x70, 0x66, 0x40, 0x0b, 0xb5, 0x92, 0xc3, 0xd3, 0x51, 0x63, 0x5d, 0x99, 0x9b,
  0x96, 0x82, 0xe1, 0xfe, 0xac, 0xa8, 0xa5, 0x87, 0x8b, 0x3f, 0xd1, 0x90, 0x70,
};

// AES-128, Go
static const uint8 kTestMessage3[] = {
  0xc3, 0x1e, 0x04, 0x07, 0x03, 0x02, 0x74, 0x1e, 0x2d, 0x7d, 0x2e, 0xdf, 0x20,
  0xdb, 0x60, 0xb1, 0x22, 0xca, 0x39, 0x74, 0x2f, 0xe8, 0x2f, 0x09, 0xf8, 0xa4,
  0x13, 0x76, 0x14, 0x65, 0x6e, 0xb9, 0xd2, 0xe0, 0x01, 0xe4, 0x40, 0x83, 0x90,
  0x3c, 0x76, 0xac, 0x23, 0x3e, 0xd2, 0xb0, 0xb2, 0x1c, 0x9d, 0x24, 0xcb, 0x7b,
  0xe1, 0x0c, 0x6b, 0xe0, 0x86, 0xe0, 0x22, 0xe1, 0xf2, 0xcb, 0xe0, 0xf3, 0xe2,
  0xbf, 0xdd, 0x89, 0xda, 0xe0, 0x83, 0xe3, 0xd9, 0x34, 0xf9, 0xaa, 0x17, 0x96,
  0x6a, 0x28, 0xe0, 0xca, 0xe2, 0x78, 0xca, 0x0e, 0x7d, 0xe0, 0xeb, 0xe4, 0xca,
  0xe6, 0xf0, 0x90, 0x4f, 0x22, 0x91, 0xf9, 0xb2, 0xbb, 0x19, 0x0b, 0x45, 0xc0,
  0x16, 0x9e, 0xe2, 0xdb, 0x30, 0xad, 0xbb, 0xe1, 0xb8, 0x12, 0x00,
};

// AES-128, salted S2K, GPG
static const uint8 kTestMessage4[] = {
  0x8c, 0x0c, 0x04, 0x07, 0x01, 0x02, 0xf8, 0x42, 0x78, 0x07, 0x04, 0xaa, 0x54,
  0xcc, 0xd2, 0x3d, 0x01, 0xcf, 0xb7, 0x30, 0xe1, 0xed, 0xb2, 0x53, 0x6e, 0x4a,
  0xbc, 0x49, 0x27, 0x45, 0xde, 0x1d, 0x5b, 0xe2, 0x17, 0x43, 0x39, 0x79, 0xdc,
  0xa5, 0xb7, 0x1a, 0x1b, 0xb7, 0x29, 0x9c, 0xb5, 0x69, 0x2f, 0x42, 0xc5, 0xe5,
  0x0c, 0x78, 0x57, 0x16, 0xa6, 0x46, 0x22, 0x18, 0x0c, 0xa2, 0xb3, 0x8c, 0xee,
  0xa1, 0xde, 0x38, 0xf1, 0xca, 0x73, 0xd3, 0xd6, 0xa3, 0x61, 0x47, 0xe2,
};

TEST(OpenPGPSymmetricEncrytionTest, AES128GPG) {
  base::StringPiece encrypted(reinterpret_cast<const char*>(kTestMessage1),
                              sizeof(kTestMessage1));
  std::string out;

  OpenPGPSymmetricEncrytion::Result r =
      OpenPGPSymmetricEncrytion::Decrypt(encrypted, "testing", &out);
  EXPECT_EQ(OpenPGPSymmetricEncrytion::OK, r);
  EXPECT_EQ("Hello world\n", out);
}

TEST(OpenPGPSymmetricEncrytionTest, AES256GPG) {
  base::StringPiece encrypted(reinterpret_cast<const char*>(kTestMessage2),
                              sizeof(kTestMessage2));
  std::string out;

  OpenPGPSymmetricEncrytion::Result r =
      OpenPGPSymmetricEncrytion::Decrypt(encrypted, "testing", &out);
  EXPECT_EQ(OpenPGPSymmetricEncrytion::OK, r);
  EXPECT_EQ("Hello world\n", out);
}

TEST(OpenPGPSymmetricEncrytionTest, AES128Go) {
  base::StringPiece encrypted(reinterpret_cast<const char*>(kTestMessage3),
                              sizeof(kTestMessage3));
  std::string out;

  OpenPGPSymmetricEncrytion::Result r =
      OpenPGPSymmetricEncrytion::Decrypt(encrypted, "testing", &out);
  EXPECT_EQ(OpenPGPSymmetricEncrytion::OK, r);
  EXPECT_EQ("Hello world\n", out);
}

TEST(OpenPGPSymmetricEncrytionTest, SaltedS2K) {
  base::StringPiece encrypted(reinterpret_cast<const char*>(kTestMessage4),
                              sizeof(kTestMessage4));
  std::string out;

  OpenPGPSymmetricEncrytion::Result r =
      OpenPGPSymmetricEncrytion::Decrypt(encrypted, "testing", &out);
  EXPECT_EQ(OpenPGPSymmetricEncrytion::OK, r);
  EXPECT_EQ("Hello world\n", out);
}

TEST(OpenPGPSymmetricEncrytionTest, Encrypt) {
  for (unsigned i = 0; i < 16; i++) {
    std::string encrypted =
        OpenPGPSymmetricEncrytion::Encrypt("Hello world\n", "testing");

    std::string out;
    OpenPGPSymmetricEncrytion::Result r =
        OpenPGPSymmetricEncrytion::Decrypt(encrypted, "testing", &out);
    EXPECT_EQ(OpenPGPSymmetricEncrytion::OK, r);
    EXPECT_EQ("Hello world\n", out);
  }
}

}  // namespace crypto
