/* $Id: kmo_fit_profile-test.c,v 1.5 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.5 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmclipm_functions.h"

#include "kmo_priv_fit_profile.h"
#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe        = "kmo_fit_profile/";

const char      *valid_files[]      = {"gauss1d_f1i_wcs.fits",
                                       "gauss1d_f1i.fits",
                                       "moffat1d_f1i.fits",
                                       "moffat1d_f1i_wcs.fits",
                                       "moffat1d_f1i_noise.fits",
                                       "lorentz1d_f1i.fits",
                                       "gauss2d_f2i.fits",
                                       "gauss2d_f2i_noise.fits",
                                       "moffat2d_f2d.fits",
                                       "moffat1d_2_f1i_noise.fits"};

/**
    @defgroup kmo_fit_profile   kmo_fit_profile unit tests

    @{
*/

/**
 * @brief test with data and mask
 */
static void test_fit_profile(const char *path,
                             const char *method,
                             const char *range,
                             int ret_val)
{
    // create sof-file
    const char *sof_path = "test_fit_profile.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   DATA\n", path);
    fclose(fh);

    // create esorex-command
    char esorex_command[512], tmp[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_fit_profile --method=%s ",
            method);

    if (strcmp(range, "") != 0) {
        sprintf(tmp, " --range=%s ", range);
        strcat(esorex_command, tmp);
    }
    strcat(esorex_command, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_fit_profile.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
    @brief
        Generates test data for kmo_fit_profile.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_fit_profile(const char *path)
{
    char        out_path[256],
                cmd[512];

    int         size                = 256,
                i                   = 0,
                j                   = 0,
                nx                  = 100,
                ny                  = 70;

    cpl_vector  *data_out           = cpl_vector_new(size),
                *par                = NULL;

    cpl_image   *img_out            = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);

    double      xx[1],
                xxx[2],
                result              = 0.0,
                *pdata_out          = cpl_vector_get_data(data_out),
                *ppar               = NULL;

    float       *pimg_out           = NULL;

    char        *tmpstr             = NULL;
    
    FILE        *fh                 = NULL;

    /* ----- valid test data ----- */


    //
    // create simple 1d gauss curve
    //
    par = cpl_vector_new(4);
    cpl_vector_set(par, 0, 2);      // offset
    cpl_vector_set(par, 1, 3);      // intensity
    cpl_vector_set(par, 2, 129);    // center
    cpl_vector_set(par, 3, 21);     // sigma
    ppar = cpl_vector_get_data(par);

    for(i = 0; i < size; i++) {
        xx[0] = i;
        kmo_priv_gauss1d_fnc(xx, ppar, &result);
        pdata_out[i] = result;
    }
    cpl_vector_delete(par); par = NULL;

    // save gauss curve
    sprintf(out_path, "%s%s", path, "gauss1d.fits");
    cpl_vector_save(data_out, out_path, CPL_BPP_IEEE_FLOAT, NULL,CPL_IO_DEFAULT);

    // create sof file "stack.sof"
    sprintf(out_path, "%s%s", path, "stack.sof");
    fh = fopen(out_path, "w");
    fprintf (fh, "test_data/kmo_fit_profile/gauss1d.fits   STACK_DATA\n");
    fclose(fh);

    //
    // create 1d gauss curve with WCS and no noise
    //
    // stack gauss1d
    tmpstr = cpl_sprintf("%s", valid_files[0]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F1I "
            "--filename=%s "
            "--s=\"CRPIX1;double;1.0;CRVAL1;double;0.2;CDELT1;double;0.01\" %s",
            tmpstr, out_path);

    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    //
    // create 1d gauss curve without WCS and no noise
    //
    tmpstr = cpl_sprintf("%s", valid_files[1]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    // stack gauss1d
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F1I "
            "--filename=%s %s", tmpstr, out_path);

    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    //
    // create simple 1d moffat curve
    //
    par = cpl_vector_new(5);
    cpl_vector_set(par, 0, 2);      // offset
    cpl_vector_set(par, 1, 3);      // intensity
    cpl_vector_set(par, 2, 129);    // center
    cpl_vector_set(par, 3, 21);     // alpha
    cpl_vector_set(par, 4, 2);      // beta
    ppar = cpl_vector_get_data(par);

    for(i = 0; i < size; i++) {
        xx[0] = i;
        kmo_priv_moffat1d_fnc(xx, ppar, &result);
        pdata_out[i] = result;
    }
    cpl_vector_delete(par); par = NULL;

    // save moffat curve
    sprintf(out_path, "%s%s", path, "moffat1d.fits");
    cpl_vector_save(data_out, out_path, CPL_BPP_IEEE_FLOAT, NULL,CPL_IO_DEFAULT);
    // create sof file "stack.sof"
    sprintf(out_path, "%s%s", path, "stack.sof");
    fh = fopen(out_path, "w");
    fprintf (fh, "test_data/kmo_fit_profile/moffat1d.fits   STACK_DATA\n");
    fclose(fh);

    //
    // create 1d moffat curve with WCS and no noise
    //
    // stack moffat1d
    tmpstr = cpl_sprintf("%s", valid_files[3]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F1I --filename=%s "
            "--s=\"CRPIX1;double;1.0;CRVAL1;double;0.2;CDELT1;double;0.01\" %s",
            tmpstr, out_path);
    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    //
    // create 1d moffat curve without WCS and no noise
    //
    tmpstr = cpl_sprintf("%s", valid_files[2]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    // stack moffat1d
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F1I --filename=%s %s",
            tmpstr, out_path);
    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    //
    // create simple 1d moffat curve with noise
    //
    sprintf(out_path, "%s%s", path, "stack.sof");
    fh = fopen(out_path, "w");
    fprintf (fh, "test_data/kmo_fit_profile/moffat1d.fits   STACK_DATA\n");
    char *my_path = cpl_sprintf("%s/ref_data/noise1d.fits", getenv("srcdir"));
    fprintf (fh, "%s                     STACK_NOISE\n", my_path);
    fclose(fh);

    //
    // create 1d moffat curve without WCS and with noise
    //
    tmpstr = cpl_sprintf("%s", valid_files[4]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    // stack moffat1d
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F1I --filename=%s %s",
            tmpstr, out_path);
    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    // create sof file "stack.sof"
    sprintf(out_path, "%s%s", path, "stack.sof");
    fh = fopen(out_path, "w");
    fprintf (fh, "test_data/kmo_fit_profile/moffat1d.fits   STACK_DATA\n");
    fprintf (fh, "%s                     STACK_NOISE\n", my_path);
    fprintf (fh, "test_data/kmo_fit_profile/moffat1d.fits   STACK_DATA\n");
    fprintf (fh, "%s                     STACK_NOISE\n", my_path);
    fclose(fh);
    cpl_free(my_path);

    //
    // create 1d moffat curve without WCS and with noise
    //
    tmpstr = cpl_sprintf("%s", valid_files[9]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    // stack moffat1d with 2 extensions
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F1I --filename=%s %s",
            tmpstr, out_path);
    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    //
    // create simple 1d lorentz curve
    //
    par = cpl_vector_new(5);
    ppar = cpl_vector_get_data(par);
    ppar[0] = 2;      // offset
    ppar[1] = 1;      // intensity
    ppar[2] = 129;    // center
    ppar[3] = 10;     // scale
    ppar[4] = 0;     // slope

    for(i = 0; i < size; i++) {
        xx[0] = i;
        kmo_priv_lorentz1d_fnc(xx, ppar, &result);
        pdata_out[i] = result;
    }
    cpl_vector_delete(par); par = NULL;

    // save lorentz curve
    sprintf(out_path, "%s%s", path, "lorentz1d.fits");
    cpl_vector_save(data_out, out_path, CPL_BPP_IEEE_FLOAT, NULL,CPL_IO_DEFAULT);
    // create sof file "stack.sof"
    sprintf(out_path, "%s%s", path, "stack.sof");
    fh = fopen(out_path, "w");
    fprintf (fh, "test_data/kmo_fit_profile/lorentz1d.fits   STACK_DATA\n");
    fclose(fh);

    //
    // create 1d lorentz curve without WCS and without noise
    //
    tmpstr = cpl_sprintf("%s", valid_files[5]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    // stack lorentz1d
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F1I --filename=%s %s",
            tmpstr, out_path);
    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    //
    // create simple 2d gauss curve
    //
    par = cpl_vector_new(7);
    cpl_vector_set(par, 0, 2);      // offset
    cpl_vector_set(par, 1, 3);      // intensity
    cpl_vector_set(par, 2, 50);     // center x
    cpl_vector_set(par, 3, 30);     // center y
    cpl_vector_set(par, 4, 10);     // axis x
    cpl_vector_set(par, 5, 20);     // axis y
    cpl_vector_set(par, 6, 1.2);     // rotation
    ppar = cpl_vector_get_data(par);

    pimg_out = cpl_image_get_data_float(img_out);

    for (j = 0; j < ny; j++) {
        xxx[1] = j;
        for (i = 0; i < nx; i++) {
            xxx[0] = i;
            kmo_priv_gauss2d_fnc(xxx, ppar, &result);
            pimg_out[i+j*nx] = result;
        }
    }
    cpl_vector_delete(par); par = NULL;

    // save gauss 2d curve
    sprintf(out_path, "%s%s", path, "gauss2d.fits");
    cpl_image_save(img_out, out_path, CPL_BPP_IEEE_FLOAT, NULL,CPL_IO_DEFAULT);

    // create sof file "stack.sof"
    sprintf(out_path, "%s%s", path, "stack.sof");
    fh = fopen(out_path, "w");
    fprintf (fh, "test_data/kmo_fit_profile/gauss2d.fits   STACK_DATA\n");
    fclose(fh);

    //
    // create 2D gauss curve without WCS and without noise
    //
    tmpstr = cpl_sprintf("%s", valid_files[6]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    // stack gauss2D
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F2I --filename=%s %s",
            tmpstr, out_path);
    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    //
    // create 2D gauss curve without WCS and with noise
    //
    sprintf(out_path, "%s%s", path, "stack.sof");
    fh = fopen(out_path, "w");
    fprintf (fh, "test_data/kmo_fit_profile/gauss2d.fits   STACK_DATA\n");
    my_path = cpl_sprintf("%s/ref_data/noise2d.fits", getenv("srcdir"));
    fprintf (fh, "%s                    STACK_NOISE\n", my_path);
    cpl_free(my_path);
    fclose(fh);

    tmpstr = cpl_sprintf("%s", valid_files[7]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    // stack gauss2D
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F2I --filename=%s %s",
            tmpstr, out_path);
    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    //
    // create simple 2d moffat curve
    //
    par = cpl_vector_new(8);
    cpl_vector_set(par, 0, 2);      // offset
    cpl_vector_set(par, 1, 3);      // intensity
    cpl_vector_set(par, 2, 50);     // center x
    cpl_vector_set(par, 3, 30);     // center y
    cpl_vector_set(par, 4, 10);     // axis x
    cpl_vector_set(par, 5, 20);     // axis y
    cpl_vector_set(par, 6, 1.2);    // rotation
    cpl_vector_set(par, 7, 2);      // beta
    ppar = cpl_vector_get_data(par);

    pimg_out = cpl_image_get_data_float(img_out);

    for (j = 0; j < ny; j++) {
        xxx[1] = j;
        for (i = 0; i < nx; i++) {
            xxx[0] = i;
            kmo_priv_moffat2d_fnc(xxx, ppar, &result);
            pimg_out[i+j*nx] = result;
        }
    }
    cpl_vector_delete(par); par = NULL;

    // save moffat 2d curve
    sprintf(out_path, "%s%s", path, "moffat2d.fits");
    cpl_image_save(img_out, out_path, CPL_BPP_IEEE_FLOAT, NULL,CPL_IO_DEFAULT);

    // create sof file "stack.sof"
    sprintf(out_path, "%s%s", path, "stack.sof");
    fh = fopen(out_path, "w");
    fprintf (fh, "test_data/kmo_fit_profile/moffat2d.fits   STACK_DATA\n");
    fclose(fh);

    //
    // create 2D moffat curve without WCS and without noise
    //
    tmpstr = cpl_sprintf("%s", valid_files[8]);
    tmpstr[strlen(tmpstr)-5] = '\0';
    // stack moffat2D
    sprintf(cmd,
            "esorex --suppress-prefix=TRUE kmo_fits_stack --type=F2I --filename=%s %s",
            tmpstr, out_path);
    kmo_get_pipe_command(cmd,
                         "log_kmo_fit_profile.txt", NULL, TRUE);
    system(cmd);

    // move file to right place
    sprintf(cmd, "mv %s.fits %s", tmpstr, path);
    system(cmd);
    cpl_free(tmpstr);

    cpl_vector_delete(data_out); data_out = NULL;
    cpl_image_delete(img_out); img_out = NULL;

    /* ----- invalid test data ----- */

    return 0;
}

/**
  @brief    Test of kmo_fit_profile recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_fit_profile-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_fit_profile-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol     = 0.01;

    char    test_path[256],
            file_path[256],
            out_path[256];

    cpl_propertylist *h = NULL;

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    FILE *fd = fopen("log_kmo_fit_profile.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_fit_profile(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    strcpy(out_path, FIT_PROFILE);
    kmo_strlower(out_path);
    strcat(out_path, ".fits");

    /* ----- valid tests ----- */

    // gauss 1d
    kmo_test_cat_strings(file_path, test_path, valid_files[1]);
    test_fit_profile(file_path, "\"gauss\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.61928, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_SIGMA), 21, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_SIGMA), 0, tol);
    cpl_propertylist_delete(h); h = NULL;

    // gauss 1d, wcs
    kmo_test_cat_strings(file_path, test_path, valid_files[0]);
    test_fit_profile(file_path, "\"gauss\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.61687, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 1.49, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 1.49, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_SIGMA), 0.21, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_SIGMA), 0, tol);
    cpl_propertylist_delete(h); h = NULL;

    test_fit_profile(file_path, "\"gauss\"", "\"0.9,2.1\"", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 3.29988, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 1.49, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 1.49, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_SIGMA), 0.21, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_SIGMA), 0, tol);
    cpl_propertylist_delete(h); h = NULL;

    // moffat 1d
    kmo_test_cat_strings(file_path, test_path, valid_files[2]);
    test_fit_profile(file_path, "\"moffat\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.38737, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ALPHA), 21, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_BETA), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_ALPHA), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_BETA), 0, tol);
    cpl_propertylist_delete(h); h = NULL;

    // moffat 1d with wcs
    kmo_test_cat_strings(file_path, test_path, valid_files[3]);
    test_fit_profile(file_path, "\"moffat\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.38737, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 1.49, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 1.49, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ALPHA), 0.21, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_BETA), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_ALPHA), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_BETA), 0, tol);
    cpl_propertylist_delete(h); h = NULL;

    // moffat 1d with noise
    kmo_test_cat_strings(file_path, test_path, valid_files[4]);
    test_fit_profile(file_path, "\"moffat\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.38737, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ALPHA), 21, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_BETA), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_ALPHA), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_BETA), 0, tol);

    cpl_propertylist_delete(h); h = NULL;

    // lorentz 1d
    kmo_test_cat_strings(file_path, test_path, valid_files[5]);
    test_fit_profile(file_path, "\"lorentz\"", "", 0);
    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.00382, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 1, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_SCALE), 10, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_SCALE), 0, tol);
    cpl_propertylist_delete(h); h = NULL;

    // gauss2d
    kmo_test_cat_strings(file_path, test_path, valid_files[6]);
    test_fit_profile(file_path, "\"gauss\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.53135, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX_X), 51, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX_Y), 31, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID_X), 51, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID_Y), 31, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RADIUS_X), 20, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RADIUS_Y), 10, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ROTATION), -21.2451, tol);
    cpl_propertylist_delete(h); h = NULL;

    // gauss2d with noise
    kmo_test_cat_strings(file_path, test_path, valid_files[7]);
    test_fit_profile(file_path, "\"gauss\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.53135, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX_X), 51, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX_Y), 31, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID_X), 51, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID_Y), 31, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RADIUS_X), 20, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RADIUS_Y), 10, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ROTATION), -21.2451, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID_X), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID_Y), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_RADIUS_X), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_RADIUS_Y), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_ROTATION), 0.0, tol);
    cpl_propertylist_delete(h); h = NULL;

    // moffat2d
    kmo_test_cat_strings(file_path, test_path, valid_files[8]);
    test_fit_profile(file_path, "\"moffat\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 2.24427, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX_X), 51, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX_Y), 31, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID_X), 51, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID_Y), 31, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RADIUS_X), 20, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RADIUS_Y), 10, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ROTATION), -0.370796*180/CPL_MATH_PI, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_BETA), 2, tol);

    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID_X), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID_Y), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_RADIUS_X), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_RADIUS_Y), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_ROTATION), 0.0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_BETA), 0.0, tol);
    cpl_propertylist_delete(h); h = NULL;


    // moffat 1d with noise and two extensions
    kmo_test_cat_strings(file_path, test_path, valid_files[9]);
    test_fit_profile(file_path, "\"moffat\"", "", 0);

    cpl_test_abs(kmo_test_esorex_data(FIT_PROFILE, 0), 4.77473, tol);
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ALPHA), 21, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_BETA), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_ALPHA), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_BETA), 0, tol);
    cpl_propertylist_delete(h); h = NULL;

    h = kmclipm_propertylist_load(out_path, 2);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_MAX_PIX), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_CENTROID), 130, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_OFFSET), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_INTENSITY), 3, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ALPHA), 21, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_BETA), 2, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_RED_CHISQ), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_CENTROID), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_OFFSET), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_INTENSITY), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_ALPHA), 0, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, FIT_ERR_BETA), 0, tol);
    cpl_propertylist_delete(h); h = NULL;

    /* ----- invalid tests ----- */

    return cpl_test_end(0);
}

/** @} */
