#
# diffoscope: in-depth comparison of files, archives, and directories
#
# Copyright © 2016 Emanuel Bronshtein <e3amn2l@gmx.com>
# Copyright © 2016-2020, 2024 Chris Lamb <lamby@debian.org>
#
# diffoscope is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# diffoscope is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with diffoscope.  If not, see <https://www.gnu.org/licenses/>.

import pytest
import subprocess
import re

from diffoscope.config import Config
from diffoscope.comparators.openssh import PublicKeyFile
from diffoscope.comparators.missing_file import MissingFile

from ..utils.data import assert_diff, load_fixture
from ..utils.tools import (
    skip_unless_tools_exist,
    skip_unless_tool_is_at_least,
    skip_unless_tool_is_at_most,
)


# Generated by: ssh-keygen -t dsa -C "Test1"
opensshpubkey1 = load_fixture("test_openssh_pub_key1.pub")
# Generated by: ssh-keygen -t rsa -b 4096 -C "Test2"
opensshpubkey2 = load_fixture("test_openssh_pub_key2.pub")
# Generated by: ssh-keygen -t ed25519 -C Test3
opensshpubkey3 = load_fixture("test_openssh_pub_key3.pub")


def openssh_version():
    out = subprocess.check_output(("ssh", "-V"), stderr=subprocess.STDOUT)
    return re.match(r"OpenSSH_([a-zA-Z0-9._+-]+)", out.decode()).group(1)


def test_identification_dsa(opensshpubkey1):
    assert isinstance(opensshpubkey1, PublicKeyFile)


def test_identification_rsa(opensshpubkey1):
    assert isinstance(opensshpubkey1, PublicKeyFile)


def test_identification_ed25519(opensshpubkey3):
    assert isinstance(opensshpubkey3, PublicKeyFile)


def test_no_differences(opensshpubkey1):
    difference = opensshpubkey1.compare(opensshpubkey1)
    assert difference is None


@pytest.fixture
def differences12(opensshpubkey1, opensshpubkey2):
    return opensshpubkey1.compare(opensshpubkey2).details


@pytest.fixture
def differences23(opensshpubkey2, opensshpubkey3):
    return opensshpubkey2.compare(opensshpubkey3).details


@skip_unless_tool_is_at_least("ssh-keygen", openssh_version, "6.9")
@skip_unless_tool_is_at_most("ssh-keygen", openssh_version, "9.7p1")
def test_diff12(differences12):
    assert_diff(differences12[0], "openssh_pub_key_expected_diff_12")


@skip_unless_tool_is_at_least("ssh-keygen", openssh_version, "6.9")
def test_diff23(differences23):
    assert_diff(differences23[0], "openssh_pub_key_expected_diff_23")


@skip_unless_tools_exist("ssh-keygen")
def test_compare_non_existing(monkeypatch, opensshpubkey2):
    monkeypatch.setattr(Config(), "new_file", True)
    difference = opensshpubkey2.compare(
        MissingFile("/nonexisting", opensshpubkey1)
    )
    assert difference.source2 == "/nonexisting"
    assert len(difference.details) > 0
