%{
/*
 * Copyright (c) 2010 Mark Liversedge (liversedge@gmail.com)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "RideDB.h"

// we use stdio for reading from FILE *RideDBin
// because thats what lex likes to do, and since we're
// reading files that seems ok anyway
#include <stdio.h>

// The parser defines the token values for us
// so lets include them before declaring the
// token patterns
#include "RideDB_yacc.h"/* generated by the scanner */

// the options below tell flex to no bother with
// yywrap since we only ever read a single file
// anyway. And yyunput() isn't needed for our
// parser, we read in one pass with no swanky
// interactions
#define YYSTYPE QString

// Un-Escape special characters (JSON compliance)
static QString unprotect(char *string)
{
    // sending UTF-8 to FLEX demands symetric conversion back to QString
    QString string2 = QString::fromUtf8(string);

    // this is a lexer string so it will be enclosed
    // in quotes. Lets strip those first
    QString s = string2.mid(1,string2.length()-2);

    // does it end with a space (to avoid token conflict) ?
    if (s.endsWith(" ")) s = s.mid(0, s.length()-1);

    // now un-escape the control characters
    s.replace("\\t", "\t");  // tab
    s.replace("\\n", "\n");  // newline
    s.replace("\\r", "\r");  // carriage-return
    s.replace("\\b", "\b");  // backspace
    s.replace("\\f", "\f");  // formfeed
    s.replace("\\/", "/");   // solidus
    s.replace("\\\"", "\""); // quote
    s.replace("\\\\", "\\"); // backslash

    return s;
}

// we reimplement these to remove compiler warnings
// about unused parameter (scanner) in the default
// implementations, which may freak out developers
void *RideDBalloc (yy_size_t  size , yyscan_t /*scanner*/)
{
	return (void *) malloc( size );
}

void *RideDBrealloc  (void * ptr, yy_size_t  size , yyscan_t /*scanner*/)
{
	/* The cast to (char *) in the following accommodates both
	 * implementations that use char* generic pointers, and those
	 * that use void* generic pointers.  It works with the latter
	 * because both ANSI C and C++ allow castless assignment from
	 * any pointer type to void*, and deal with argument conversions
	 * as though doing an assignment.
	 */
	return (void *) realloc( (char *) ptr, size );
}

void RideDBfree (void * ptr , yyscan_t /*scanner*/)
{
	free( (char *) ptr );	/* see RideDBrealloc() for (char *) cast */
}

// replace this too, as a) it exits (!!) 
// cannot shutup compiler warning on yy_fatal_error function tho :(
#define YY_FATAL_ERROR(msg) qDebug()<<msg;

%}
%option prefix="RideDB"
%option never-interactive
%option noyyalloc
%option noyyrealloc
%option noyyfree
%option noyywrap
%option nounput
%option noinput
%option reentrant
%option bison-bridge

%%
\"RIDES\"                       return RIDES;
\"METRICS\"                     return METRICS;
\"TAGS\"                        return TAGS;
\"XDATA\"                       return XDATA;
\"VERSION\"                     return VERSION;
\"INTERVALS\"                   return INTERVALS;
\"([^\"]|\\\")*\"               { *yylval = unprotect(yytext); return STRING;  } /* contains non-quotes or escaped-quotes */
[ \n\t\r]                       ;               /* we just ignore whitespace */
.                               return yytext[0]; /* any other character, typically :, { or } */
%%

// Older versions of flex (prior to 2.5.9) do not have the destroy function
// Or We're not using GNU flex then we also won't have a destroy function
#if !defined(FLEX_SCANNER) || (YY_FLEX_MINOR_VERSION < 6 && YY_FLEX_SUBMINOR_VERSION < 9)
int RideDBlex_destroy(void*) { return 0; }
#endif

void RideDB_setString(QString p, void *scanner)
{
    // internally work with UTF-8 encoding
    // this works for FLEX, since the multi-byte characters only appear WITHIN a "String",
    // but not as part of the grammar - this is important since a char in UTF-8 can have up to 4 bytes
    RideDB_scan_string(p.toUtf8().data(), scanner);
}
