//*****************************************************************************
//                               SimnBase.cpp                                 *
//                              --------------                                *
// Started     : 01/05/2008                                                   *
// Last Update : 01/07/2011                                                   *
// Copyright   : (C) 2008 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "base/SimnBase.hpp"

//*****************************************************************************
// Constructor.

SimnBase::SimnBase( void )
{
  m_eSimEng = eSIMR_NONE;

  // Initialize all object attributes
  bClear( );
}

//*****************************************************************************
// Destructor.

SimnBase::~SimnBase( )
{
}

//*****************************************************************************
// Extract the source component.
//
// Several tasks are performed here :
//  - Search for a component which is being used as a signal source.
//  - Extract the source component's original definition.
//  - Set the attribute m_oCpntSwpSrc to the source component definition.
//  - Set the component in m_oaCpnts to it's original definition.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnBase::bLoadSigSrc( void )
{
  Component  oCpnt;
  wxString   os1;
  size_t     sz1;

  // Scan the circuit description for a source component
  for( sz1=0; sz1<m_osaNetLst.GetCount( )-1; sz1++, os1.Empty( ) )
  {
    os1 = m_osaNetLst.Item( sz1 );

    // Is this a signal source?
    if(   os1.IsEmpty( ) )                             continue; // No
    if( ! os1.StartsWith( wxT("* Signal source (") ) ) continue; // No
    if(   os1.GetChar( os1.Len( )-1 ) != wxT(')') )    continue; // No
    break; // Yes, found what looks like a signal source
  }
  if( os1.IsEmpty( ) )      return( FALSE );

  // Attempt to extract the original component definition
  os1 = os1.AfterFirst( wxT('(') );
  os1 = os1.BeforeLast( wxT(')') );
  oCpnt = os1;
  if( ! oCpnt.bIsValid( ) ) return( FALSE );

  // Set the source component
  m_oCpntSwpSrc = m_osaNetLst.Item( sz1 + 1 );

  // Scan the component list for the source component
  for( sz1=0; sz1<m_oaCpnts.GetCount( ); sz1++ )
  {
    if( m_oaCpnts.Item( sz1 ).m_osName == oCpnt.m_osName )
    {
      // Set the signal source component back to it's original definition
      m_oaCpnts.Item( sz1 ) = oCpnt.rosGetString( );
      break;
    }
  }

  return( TRUE );
}

//*****************************************************************************
// If a signal source has been defined save it's current and original
// definitions in the simulation file.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnBase::bSaveSigSrc( void )
{
  Component  oCpnt;
  wxString   os1;
  size_t     sz1, sz2;

  // If a signal source has been selected insert it in the simulation file
  if( ! m_oCpntSwpSrc.bIsValid( ) ) return( TRUE );

  // Find the original definition of the signal source component
  for( sz1=0; sz1<m_oaCpnts.GetCount( ); sz1++ )
    if( m_oaCpnts[ sz1 ].m_osName == m_oCpntSwpSrc.m_osName )
      break;

  // Open the simulation file
  wxTextFile  oFileCct( m_ofnSaveFile.GetFullPath( ) );
  if( ! oFileCct.Open( ) )          return( FALSE );

  // Find the source component and insert a comment before it
  for( sz2=0; sz2<oFileCct.GetLineCount( ); sz2++ )
  {
    // Is this a component definition?
    oCpnt = oFileCct[ sz2 ];
    if( ! oCpnt.bIsValid( ) ) continue;

    // Is this the signal source component?
    if( oCpnt.m_osName == m_oCpntSwpSrc.m_osName )
    {
      oFileCct[ sz2 ] = m_oCpntSwpSrc.rosGetString( );
      os1 = wxT("* Signal source (") + oCpnt.rosGetString( ) + wxT(')');
      oFileCct.InsertLine( os1, sz2 );
      break;
    }
  }

  // Save the changes to disk and close the simulation file
  oFileCct.Write( );
  oFileCct.Close( );

  return( TRUE );
}

//*****************************************************************************
// Clear all object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnBase::bClear( void )
{
  // Clear the sweep source component
  m_oCpntSwpSrc.bClear( );
  m_oCpntSwpSrc.m_osName = wxT("None");

  // Clear any error messages
  m_osErrMsg.Empty( );

  // Clear the base class
  return( NetList::bClear( ) );
}

//*****************************************************************************
// Do the current simulation settings constitute a valid simulation?
//
// Return Values:
//   TRUE  - Valid
//   FALSE - Not valid

bool  SimnBase::bValidate( void )
{
  // Validate the base class
  NetList::bValidate( );

  // Clear the error message
  m_osErrMsg.Clear( );

  // Check the simulation engine and analysis type
  if( m_eSimEng<eSIMR_FST || m_eSimEng>eSIMR_FST )
  {
    SetErrMsg( wxT("No simulation engine has been specified.") );
    return( FALSE );
  }
  if( eGetAnaType( )<eCMD_ANA_FST || eGetAnaType( )>eCMD_ANA_LST ) ;
  {
    SetErrMsg( wxT("No analysis type has been specified.") );
    return( FALSE );
  }

  // Have any test components or test nodes been specified?
  if( rosaGetTstNodes( ).GetCount( )<=0 || rosaGetTstCpnts( ).GetCount( )<=0 )
  {
    SetErrMsg( wxT("No nodes or components have been selected.") );
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Load (or reload) the simulation from file.
//
// Argument List :
//   rosFName - The name of the file to be loaded
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnBase::bLoadFile( const wxString & rosFName )
{
  // Load the basic netlist file
  if( ! NetList::bLoadFile( rosFName ) ) return( FALSE );

  // Check for a simulator signifier
  if( ! bLoadSimEng( ) )                 return( FALSE );

  // Extract the simulator instructions
  bLoadSimCmds( );
  bLoadSigSrc ( );

  return( TRUE );
}

//*****************************************************************************
// Save (or resave) the simulation to file.
//
// Argument List :
//   rosFName - The name of the file to be saved
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  SimnBase::bSaveFile( const wxString & rosFName )
{
  // Save the netlist to file
  if( ! NetList::bSaveFile( rosFName ) ) return( FALSE );

  // Save the signal source to the simulation file
  if( ! bSaveSigSrc( ) )                 return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Copy the contents of a SimnBase object.
//
// Argument List :
//   roSimn - A reference to a SimnBase object
//
// Return Values :
//   A reference to this object

SimnBase & SimnBase::operator = ( const SimnBase & roSimn )
{
  m_oCpntSwpSrc = roSimn.m_oCpntSwpSrc;

  return( *this );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  SimnBase::Print( const wxString & rosPrefix )
{
  NetList::Print( rosPrefix + wxT("NetList::") );

  cout << rosPrefix.mb_str( )  << "m_eSimEng  : ";
  switch( m_eSimEng )
  {
    case eSIMR_GNUCAP  : cout << "eSIMR_GNUCAP";  break;
    case eSIMR_NGSPICE : cout << "eSIMR_NGSPICE"; break;
    case eSIMR_NONE    : cout << "eSIMR_NONE";    break;
  }
  cout << '\n';

  cout << rosPrefix .mb_str( ) << "m_osErrMsg : "
       << m_osErrMsg.mb_str( ) << '\n';

  m_oCpntSwpSrc.Print( rosPrefix + wxT("m_oCpntSwpSrc::") );
}

//*****************************************************************************
