/* Copyright (C) 2009-2021 Greenbone Networks GmbH
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "networking.c"

#include <cgreen/cgreen.h>
#include <cgreen/mocks.h>

Describe (networking);
BeforeEach (networking)
{
  cgreen_mocks_are (loose_mocks);
}
AfterEach (networking)
{
}

Ensure (networking, validate_port_range)
{
  /* No port range provided. */
  assert_that (validate_port_range (NULL), is_equal_to (1));
  assert_that (validate_port_range (""), is_equal_to (1));

  /* '\0' on end. */
  assert_that (validate_port_range ("\0"), is_equal_to (1));
  assert_that (validate_port_range ("T:1-5,7,9,U:1-3,5,7,9,\\0"),
               is_equal_to (1));

  /* Newline in between range description.*/
  assert_that (validate_port_range ("\nT:1-\n5,7,9,\nU:1-3,5\n,7,9\n"),
               is_equal_to (1));

  /* Port <= 0 or Port > 65535. */
  assert_that (validate_port_range ("0"), is_equal_to (1));
  assert_that (validate_port_range ("-9"), is_equal_to (1));
  assert_that (validate_port_range ("1,0,6,7"), is_equal_to (1));
  assert_that (validate_port_range ("2,-9,4"), is_equal_to (1));
  assert_that (validate_port_range ("90000"), is_equal_to (1));

  /* Illegal Ranges. */
  assert_that (validate_port_range ("T:-"), is_equal_to (1));
  assert_that (validate_port_range ("T:-9"), is_equal_to (1));
  assert_that (validate_port_range ("T:0-"), is_equal_to (1));
  assert_that (validate_port_range ("T:0-9"), is_equal_to (1));
  assert_that (validate_port_range ("T:90000-"), is_equal_to (1));
  assert_that (validate_port_range ("T:90000-90010"), is_equal_to (1));
  assert_that (validate_port_range ("T:9-\\0"), is_equal_to (1));
  assert_that (validate_port_range ("T:9-0"), is_equal_to (1));
  assert_that (validate_port_range ("T:9-90000"), is_equal_to (1));
  assert_that (validate_port_range ("T:100-9"), is_equal_to (1));
  assert_that (validate_port_range ("0-"), is_equal_to (1));
  assert_that (validate_port_range ("0-9"), is_equal_to (1));
  assert_that (validate_port_range ("9-"), is_equal_to (1));
  assert_that (validate_port_range ("9-\\0"), is_equal_to (1));
  assert_that (validate_port_range ("9-8"), is_equal_to (1));
  assert_that (validate_port_range ("90000-90010"), is_equal_to (1));
  assert_that (validate_port_range ("100-9"), is_equal_to (1));
  assert_that (validate_port_range ("T,U"), is_equal_to (1));
  assert_that (validate_port_range ("T  :\n: 1-2,U"), is_equal_to (1));
  assert_that (validate_port_range ("T  :: 1-2,U"), is_equal_to (1));
  assert_that (validate_port_range ("T:2=2"), is_equal_to (1));
  assert_that (validate_port_range ("T:1.2-5,4.5"), is_equal_to (1));

  /* Legal single ports.*/
  assert_that (validate_port_range ("6,6,6,6,10,20"), is_equal_to (0));
  assert_that (validate_port_range ("T:7, U:7"), is_equal_to (0));
  assert_that (validate_port_range ("T:7, U:9"), is_equal_to (0));
  assert_that (validate_port_range ("9"), is_equal_to (0));
  assert_that (validate_port_range ("U:,T:"), is_equal_to (0)); /* is ignored */
  assert_that (validate_port_range ("1,2,,,,,,,\n\n\n\n\n\n,,,5"),
               is_equal_to (0));

  /* Example in Documentation. */
  assert_that (validate_port_range ("T:1-5,7,9,U:1-3,5,7,9"), is_equal_to (0));

  /* Treat newlines like commas. */
  assert_that (validate_port_range ("1,2,\n,\n4,6"), is_equal_to (0));
  assert_that (validate_port_range ("T:1-5,7,9,\nU:1-3,5\n,7,9"),
               is_equal_to (0));

  /* Ranges without type specifier. */
  assert_that (validate_port_range ("6-9,7,7,10-20,20"), is_equal_to (0));

  /* Allow whitespace after T/U and anywhere else. */
  assert_that (
    validate_port_range ("   T: 1 -5,  7   ,9, \nU   :1-  3,5  \n,7,9"),
    is_equal_to (0));
  assert_that (validate_port_range (
                 "   T: 1 -5,  7   ,9, \nU :1- 3,5  \n,7,9, T    :  5 -7"),
               is_equal_to (0));
  assert_that (validate_port_range ("   T  : 1"), is_equal_to (0));

  /* Some standard port lists.*/
  enum
  {
    ALL_IANA_ASSIGNED_TCP_2020_02_12 = 0,
    ALL_PRIVILEGED_TCP,
    ALL_PRIVILEGED_TCP_AND_UDP,
    ALL_TCP,
    OPENVAS_DEFAULT,
    ALL_IANA_ASSIGNED_TCP_AND_UDP_2020_02_12,
    ALL_TCP_AND_NMAP_5_51_TOP_100_UDP,
    ALL_TCP_AND_NMAP_5_51_TOP_1000_UDP,
    NMAP_5_51_TOP_2000_TCP_AND_TOP_100_UDP,
    WEB_SERVICES,
  };

  const char *portlists[] = {
    "T:1-50,52-80,82-99,101-113,115-224,242-248,256-257,259-271,280-284,286-"
    "287,308-324,333,344-584,586-658,660-702,704-707,709-715,729-731,741-742,"
    "744,747-754,758-765,767,769-777,780,800-802,810,828-833,847-848,853-854,"
    "860-862,873,886-888,900-903,910-913,953,989-1001,1010,1021-1027,1029,1033-"
    "1108,1110-1490,1492-1782,1784-2193,2197-2258,2260-2368,2370-2377,2379-"
    "2681,2683-2793,2795-2824,2826-2872,2874-2924,2926-3091,3093-3125,3127-"
    "3300,3302-3321,3326-3366,3372-3402,3405-3545,3547-3693,3695-3993,3995-"
    "4047,4049-4143,4145-4193,4197,4199,4300-4314,4316,4320-4323,4325-4336,"
    "4340-4362,4366,4368-4379,4389-4396,4400-4423,4425-4433,4441-4458,4484-"
    "4488,4500,4534-4538,4545-4559,4563,4566-4570,4573,4590-4605,4621,4646,"
    "4658-4692,4700-4704,4711,4725-4733,4737-4747,4749-4756,4774,4784-4791,"
    "4800-4804,4827,4837-4851,4867-4871,4876-4885,4888-4889,4894,4899-4902,"
    "4912-4915,4936-4937,4940-4942,4949-4953,4969-4971,4980,4984-4991,4999-"
    "5015,5020-5034,5042-5075,5078-5087,5092-5094,5099-5107,5111-5112,5114-"
    "5117,5120,5133-5137,5145-5146,5150-5157,5161-5168,5172,5190-5197,5200-"
    "5203,5209,5215,5221-5237,5245-5254,5264-5265,5269-5272,5280-5282,5298-"
    "5310,5312-5318,5320-5321,5343-5344,5349-5364,5397-5437,5443,5445,5450,"
    "5453-5456,5461-5465,5470-5475,5500-5507,5550,5553-5557,5565-5569,5573-"
    "5575,5579-5586,5597-5605,5618,5627-5639,5646,5666,5670-5684,5687-5689,"
    "5693,5696,5700,5705,5713-5730,5741-5748,5750,5755,5757,5766-5771,5777,"
    "5780-5787,5793-5794,5813-5814,5841-5842,5859,5863,5868,5883,5900,5910-"
    "5913,5963,5968-5969,5984-5993,5999,6064-6066,6068-6077,6080-6088,6099-"
    "6118,6121-6124,6130,6133,6140-6149,6159-6163,6200-6201,6209,6222,6241-"
    "6244,6251-6253,6267-6269,6300-6301,6306,6315-6317,6320-6322,6324-6326,"
    "6343-6344,6346-6347,6350,6355,6360,6363,6370,6379,6382,6389-6390,6417-"
    "6421,6432,6442-6446,6455-6456,6464,6471,6480-6489,6500-6503,6505-6511,"
    "6513-6515,6543-6544,6547-6551,6558,6566,6568,6579-6583,6600-6602,6619-"
    "6629,6632-6636,6640,6653,6655-6657,6670-6673,6678-6679,6687-6690,6696-"
    "6697,6701-6706,6714-6716,6767-6771,6777-6778,6784-6791,6801,6817,6831,"
    "6841-6842,6850,6868,6888,6900-6901,6924,6935-6936,6946,6951,6961-6966,"
    "6969-6970,6997-7026,7030-7031,7040,7070-7073,7080,7088,7095,7099-7101,"
    "7107,7117,7121,7128-7129,7161-7174,7181,7200-7202,7215-7216,7227-7229,"
    "7235-7237,7244,7262,7272-7283,7365,7391-7395,7397,7400-7402,7410-7411,"
    "7420-7421,7426-7431,7437,7443,7471,7473-7474,7478,7491,7500-7501,7508-"
    "7511,7542-7551,7560,7563,7566,7569-7570,7574,7588,7606,7624,7626-7631,"
    "7633,7648,7663,7672-7677,7680,7683,7687,7689,7697,7700-7701,7707-7708,"
    "7720,7724-7728,7734,7738,7741-7744,7747,7775,7777-7779,7781,7784,7786-"
    "7787,7789,7794,7797-7802,7810,7845-7847,7869-7872,7878,7880,7887,7900-"
    "7903,7913,7932-7933,7962,7967,7979-7982,7997-8009,8015-8016,8019-8023,"
    "8025-8026,8032-8034,8040-8044,8051-8060,8066-8067,8070,8074,8077,8080-"
    "8084,8086-8088,8090-8091,8097,8100-8102,8111,8115-8118,8121-8122,8128-"
    "8132,8140,8148-8149,8153,8160-8162,8181-8184,8190-8192,8194-8195,8199-"
    "8202,8204-8208,8211,8230-8232,8243,8266,8270,8276,8280,8282,8292-8294,"
    "8300-8301,8313,8320-8322,8351,8376-8380,8383-8384,8400-8405,8415-8417,"
    "8423,8442-8445,8450,8457,8470-8474,8500-8503,8554-8555,8567,8600,8609-"
    "8615,8665-8666,8675,8686,8688,8699,8710-8711,8732-8733,8750,8763-8770,"
    "8778,8786-8787,8793,8800,8804-8805,8807-8809,8873,8880-8881,8883,8888-"
    "8894,8899-8901,8910-8913,8937,8953-8954,8980-8981,8989-8991,8997-9002,"
    "9005,9007-9011,9020-9026,9050-9051,9060,9080-9081,9083-9093,9100-9107,"
    "9111,9119,9122-9123,9131,9160-9164,9191,9200-9217,9222,9255,9277-9287,"
    "9292-9295,9300,9306,9312,9318,9321,9339,9343-9346,9374,9380,9387-9390,"
    "9396-9397,9400-9402,9418,9443-9445,9450,9500,9522,9535-9536,9555,9592-"
    "9600,9612,9614,9616-9618,9628-9632,9640,9666-9668,9694-9695,9700,9747,"
    "9750,9753,9762,9800-9802,9875-9878,9888-9889,9898-9900,9903,9909,9911,"
    "9925,9950-9956,9966,9978-9979,9981,9987-9988,9990-10010,10020,10023,10050-"
    "10051,10055,10080-10081,10100-10104,10107,10110-10111,10113-10117,10125,"
    "10128-10129,10160-10162,10200-10201,10252-10253,10260-10261,10288,10321,"
    "10439,10443,10500,10540-10544,10548,10631,10800,10805,10809-10810,10860,"
    "10880,10933,10990,11000-11001,11095,11103-11106,11108-11112,11161-11165,"
    "11171-11175,11201-11202,11208,11211,11319-11321,11367,11371,11430,11489,"
    "11600,11623,11720,11723,11751,11796,11876-11877,11967,11971,12000-12010,"
    "12012-12013,12109,12121,12168,12172,12300,12302,12321-12322,12345,12753,"
    "12865,13160,13216-13218,13223-13224,13400,13720-13722,13724,13782-13783,"
    "13785-13786,13818-13823,13894,13929-13930,14000-14002,14033-14034,14141-"
    "14143,14145,14149-14150,14154,14250,14414,14500,14936-14937,15000,15002,"
    "15118,15345,15363,15555,15660,15740,15998-16003,16020-16021,16161-16162,"
    "16309-16311,16360-16361,16367-16368,16384-16385,16619,16665-16666,16789,"
    "16900,16950,16991-16995,17007,17184-17185,17219-17225,17234-17235,17500,"
    "17555,17729,17754-17756,17777,18000,18104,18136,18181-18187,18241-18243,"
    "18262,18463,18634-18635,18668,18769,18881,18888,19000,19007,19020,19191,"
    "19194,19220,19283,19315,19398,19410-19412,19539-19541,19788,19998-20003,"
    "20005,20012-20014,20034,20046,20048-20049,20057,20167,20202,20222,20480,"
    "20670,20999-21000,21010,21212-21213,21221,21553-21554,21590,21800,21845-"
    "21849,22000-22005,22125,22128,22222,22273,22305,22335,22343,22347,22350-"
    "22351,22537,22555,22763,22800,22951,23000-23005,23053,23272,23294,23333,"
    "23400-23402,23456-23457,23546,24000-24006,24242,24249,24321-24323,24386,"
    "24465,24554,24577,24666,24676-24678,24680,24754,24850,24922,25000-25009,"
    "25576,25604,25793,25900-25903,25954-25955,26000,26133,26208,26257,26260-"
    "26263,26486-26487,26489,27010,27017,27345,27442,27504,27782,27876,27999-"
    "28001,28010,28119,28200,28240,28589,29118,29167-29168,29999-30004,30100,"
    "30260,30400,30832,30999,31016,31020,31029,31400,31416,31457,31620,31685,"
    "31765,31948-31949,32034,32249,32400,32483,32635-32636,32767-32777,32801,"
    "32811,32896,33000,33060,33123,33331,33333-33334,33434-33435,33656,34249,"
    "34378-34379,34567,34962-34964,34980,35000-35006,35100,35354-35357,36001,"
    "36411-36412,36422,36462,36524,36602,36700,36865,37475,37483,37601,37654,"
    "38000-38002,38201-38203,38412,38422,38462,38472,38800,38865,39681,40000,"
    "40023,40404,40841-40843,40853,41111,41121,41230,41794-41797,42508-42510,"
    "43000,43188-43191,43210,43438-43441,44123,44321-44323,44444-44445,44544,"
    "44553,44600,44818,44900,45000-45002,45045,45054,45514,45678,45824-45825,"
    "45966,46336,46998-47001,47100,47557,47624,47806,47808-47809,48000-48005,"
    "48048-48050,48128-48129,48556,48619,48653,49000-49001,49150,6000-6063,"
    "6665-6669",
    "T:1-1023",
    "T:1-1023, U:1-1023",
    "T:1-65535",
    "T:1-5,7-7,9-9,11-11,13-13,15-15,17-25,27-27,29-29,31-31,33-33,35-35,37-39,"
    "41-59,61-224,242-248,256-268,280-287,308-322,333-333,344-700,702-702,704-"
    "707,709-711,721-721,723-723,729-731,740-742,744-744,747-754,758-765,767-"
    "767,769-777,780-783,786-787,799-801,808-808,810-810,828-829,847-848,860-"
    "860,871-871,873-873,886-888,898-898,900-904,911-913,927-927,950-950,953-"
    "953,975-975,989-1002,1005-1005,1008-1008,1010-1010,1023-1027,1029-1036,"
    "1040-1040,1042-1042,1045-1045,1047-1112,1114-1117,1119-1120,1122-1127,"
    "1139-1139,1154-1155,1161-1162,1168-1170,1178-1178,1180-1181,1183-1188,"
    "1194-1194,1199-1231,1233-1286,1288-1774,1776-2028,2030-2030,2032-2035,"
    "2037-2038,2040-2065,2067-2083,2086-2087,2089-2152,2155-2155,2159-2167,"
    "2170-2177,2180-2181,2190-2191,2199-2202,2213-2213,2220-2223,2232-2246,"
    "2248-2255,2260-2260,2273-2273,2279-2289,2294-2311,2313-2371,2381-2425,"
    "2427-2681,2683-2824,2826-2854,2856-2924,2926-3096,3098-3299,3302-3321,"
    "3326-3366,3372-3403,3405-3545,3547-3707,3709-3765,3767-3770,3772-3800,"
    "3802-3802,3845-3871,3875-3876,3885-3885,3900-3900,3928-3929,3939-3939,"
    "3959-3959,3970-3971,3984-3987,3999-4036,4040-4042,4045-4045,4080-4080,"
    "4096-4100,4111-4111,4114-4114,4132-4134,4138-4138,4141-4145,4154-4154,"
    "4160-4160,4199-4200,4242-4242,4300-4300,4321-4321,4333-4333,4343-4351,"
    "4353-4358,4369-4369,4400-4400,4442-4457,4480-4480,4500-4500,4545-4547,"
    "4555-4555,4557-4557,4559-4559,4567-4568,4600-4601,4658-4662,4672-4672,"
    "4752-4752,4800-4802,4827-4827,4837-4839,4848-4849,4868-4869,4885-4885,"
    "4894-4894,4899-4899,4950-4950,4983-4983,4987-4989,4998-4998,5000-5011,"
    "5020-5025,5031-5031,5042-5042,5050-5057,5060-5061,5064-5066,5069-5069,"
    "5071-5071,5081-5081,5093-5093,5099-5102,5137-5137,5145-5145,5150-5152,"
    "5154-5154,5165-5165,5190-5193,5200-5203,5222-5222,5225-5226,5232-5232,"
    "5236-5236,5250-5251,5264-5265,5269-5269,5272-5272,5282-5282,5300-5311,"
    "5314-5315,5351-5355,5400-5432,5435-5435,5454-5456,5461-5463,5465-5465,"
    "5500-5504,5510-5510,5520-5521,5530-5530,5540-5540,5550-5550,5553-5556,"
    "5566-5566,5569-5569,5595-5605,5631-5632,5666-5666,5673-5680,5688-5688,"
    "5690-5690,5713-5717,5720-5720,5729-5730,5741-5742,5745-5746,5755-5755,"
    "5757-5757,5766-5768,5771-5771,5800-5803,5813-5813,5858-5859,5882-5882,"
    "5888-5889,5900-5903,5968-5969,5977-5979,5987-5991,5997-6010,6050-6051,"
    "6064-6073,6085-6085,6100-6112,6123-6123,6141-6150,6175-6177,6200-6200,"
    "6253-6253,6255-6255,6270-6270,6300-6300,6321-6322,6343-6343,6346-6347,"
    "6373-6373,6382-6382,6389-6389,6400-6400,6455-6456,6471-6471,6500-6503,"
    "6505-6510,6543-6543,6547-6550,6558-6558,6566-6566,6580-6582,6588-6588,"
    "6620-6621,6623-6623,6628-6628,6631-6631,6665-6670,6672-6673,6699-6701,"
    "6714-6714,6767-6768,6776-6776,6788-6790,6831-6831,6841-6842,6850-6850,"
    "6881-6889,6891-6891,6901-6901,6939-6939,6961-6966,6969-6970,6998-7015,"
    "7020-7021,7030-7030,7070-7070,7099-7100,7121-7121,7161-7161,7170-7170,"
    "7174-7174,7200-7201,7210-7210,7269-7269,7273-7273,7280-7281,7283-7283,"
    "7300-7300,7320-7320,7326-7326,7391-7392,7395-7395,7426-7431,7437-7437,"
    "7464-7464,7491-7491,7501-7501,7510-7511,7544-7545,7560-7560,7566-7566,"
    "7570-7570,7575-7575,7588-7588,7597-7597,7624-7624,7626-7627,7633-7634,"
    "7648-7649,7666-7666,7674-7676,7743-7743,7775-7779,7781-7781,7786-7786,"
    "7797-7798,7800-7801,7845-7846,7875-7875,7902-7902,7913-7913,7932-7933,"
    "7967-7967,7979-7980,7999-8005,8007-8010,8022-8022,8032-8033,8044-8044,"
    "8074-8074,8080-8082,8088-8089,8098-8098,8100-8100,8115-8116,8118-8118,"
    "8121-8122,8130-8132,8160-8161,8181-8194,8199-8201,8204-8208,8224-8225,"
    "8245-8245,8311-8311,8351-8351,8376-8380,8400-8403,8416-8417,8431-8431,"
    "8443-8444,8450-8450,8473-8473,8554-8555,8649-8649,8733-8733,8763-8765,"
    "8786-8787,8804-8804,8863-8864,8875-8875,8880-8880,8888-8894,8900-8901,"
    "8910-8911,8954-8954,8989-8989,8999-9002,9006-9006,9009-9009,9020-9026,"
    "9080-9080,9090-9091,9100-9103,9110-9111,9131-9131,9152-9152,9160-9164,"
    "9200-9207,9210-9211,9217-9217,9281-9285,9287-9287,9292-9292,9321-9321,"
    "9343-9344,9346-9346,9374-9374,9390-9390,9396-9397,9400-9400,9418-9418,"
    "9495-9495,9500-9500,9535-9537,9593-9595,9600-9600,9612-9612,9704-9704,"
    "9747-9747,9753-9753,9797-9797,9800-9802,9872-9872,9875-9876,9888-9889,"
    "9898-9901,9909-9909,9911-9911,9950-9952,9990-10005,10007-10008,10012-"
    "10012,10080-10083,10101-10103,10113-10116,10128-10128,10252-10252,10260-"
    "10260,10288-10288,10607-10607,10666-10666,10752-10752,10990-10990,11000-"
    "11001,11111-11111,11201-11201,11223-11223,11319-11321,11367-11367,11371-"
    "11371,11600-11600,11720-11720,11751-11751,11965-11965,11967-11967,11999-"
    "12006,12076-12076,12109-12109,12168-12168,12172-12172,12223-12223,12321-"
    "12321,12345-12346,12361-12362,12468-12468,12701-12701,12753-12753,13160-"
    "13160,13223-13224,13701-13702,13705-13706,13708-13718,13720-13722,13724-"
    "13724,13782-13783,13818-13822,14001-14001,14033-14034,14141-14141,14145-"
    "14145,14149-14149,14194-14194,14237-14237,14936-14937,15000-15000,15126-"
    "15126,15345-15345,15363-15363,16360-16361,16367-16368,16384-16384,16660-"
    "16661,16959-16959,16969-16969,16991-16991,17007-17007,17185-17185,17219-"
    "17219,17300-17300,17770-17772,18000-18000,18181-18187,18190-18190,18241-"
    "18241,18463-18463,18769-18769,18888-18888,19191-19191,19194-19194,19283-"
    "19283,19315-19315,19398-19398,19410-19412,19540-19541,19638-19638,19726-"
    "19726,20000-20001,20005-20005,20011-20012,20034-20034,20200-20200,20202-"
    "20203,20222-20222,20670-20670,20999-21000,21490-21490,21544-21544,21590-"
    "21590,21800-21800,21845-21849,22000-22001,22222-22222,22273-22273,22289-"
    "22289,22305-22305,22321-22321,22370-22370,22555-22555,22800-22800,22951-"
    "22951,23456-23456,24000-24006,24242-24242,24249-24249,24345-24347,24386-"
    "24386,24554-24554,24677-24678,24922-24922,25000-25009,25378-25378,25544-"
    "25544,25793-25793,25867-25867,25901-25901,25903-25903,26000-26000,26208-"
    "26208,26260-26264,27000-27010,27345-27345,27374-27374,27504-27504,27665-"
    "27665,27999-27999,28001-28001,29559-29559,29891-29891,30001-30002,30100-"
    "30102,30303-30303,30999-30999,31337-31337,31339-31339,31416-31416,31457-"
    "31457,31554-31554,31556-31556,31620-31620,31765-31765,31785-31787,32261-"
    "32261,32666-32666,32768-32780,32786-32787,32896-32896,33270-33270,33331-"
    "33331,33434-33434,33911-33911,34249-34249,34324-34324,34952-34952,36865-"
    "36865,37475-37475,37651-37651,38037-38037,38201-38201,38292-38293,39681-"
    "39681,40412-40412,40841-40843,41111-41111,41508-41508,41794-41795,42508-"
    "42510,43118-43118,43188-43190,44321-44322,44333-44334,44442-44443,44818-"
    "44818,45000-45000,45054-45054,45678-45678,45966-45966,47000-47000,47557-"
    "47557,47624-47624,47806-47806,47808-47808,47891-47891,48000-48003,48556-"
    "48556,49400-49400,50000-50004,50505-50505,50776-50776,51210-51210,53001-"
    "53001,54320-54321,57341-57341,59595-59595,60177-60177,60179-60179,61439-"
    "61441,61446-61446,65000-65000,65301-65301",
    "T:1-50,52-80,82-99,101-113,115-224,242-248,256-257,259-271,280-284,286-"
    "287,308-324,333,344-584,586-658,660-702,704-707,709-715,729-731,741-742,"
    "744,747-754,758-765,767,769-777,780,800-802,810,828-833,847-848,853-854,"
    "860-862,873,886-888,900-903,910-913,953,989-1001,1010,1021-1027,1029,1033-"
    "1108,1110-1490,1492-1782,1784-2193,2197-2258,2260-2368,2370-2377,2379-"
    "2681,2683-2793,2795-2824,2826-2872,2874-2924,2926-3091,3093-3125,3127-"
    "3300,3302-3321,3326-3366,3372-3402,3405-3545,3547-3693,3695-3993,3995-"
    "4047,4049-4143,4145-4193,4197,4199,4300-4314,4316,4320-4323,4325-4336,"
    "4340-4362,4366,4368-4379,4389-4396,4400-4423,4425-4433,4441-4458,4484-"
    "4488,4500,4534-4538,4545-4559,4563,4566-4570,4573,4590-4605,4621,4646,"
    "4658-4692,4700-4704,4711,4725-4733,4737-4747,4749-4756,4774,4784-4791,"
    "4800-4804,4827,4837-4851,4867-4871,4876-4885,4888-4889,4894,4899-4902,"
    "4912-4915,4936-4937,4940-4942,4949-4953,4969-4971,4980,4984-4991,4999-"
    "5015,5020-5034,5042-5075,5078-5087,5092-5094,5099-5107,5111-5112,5114-"
    "5117,5120,5133-5137,5145-5146,5150-5157,5161-5168,5172,5190-5197,5200-"
    "5203,5209,5215,5221-5237,5245-5254,5264-5265,5269-5272,5280-5282,5298-"
    "5310,5312-5318,5320-5321,5343-5344,5349-5364,5397-5437,5443,5445,5450,"
    "5453-5456,5461-5465,5470-5475,5500-5507,5550,5553-5557,5565-5569,5573-"
    "5575,5579-5586,5597-5605,5618,5627-5639,5646,5666,5670-5684,5687-5689,"
    "5693,5696,5700,5705,5713-5730,5741-5748,5750,5755,5757,5766-5771,5777,"
    "5780-5787,5793-5794,5813-5814,5841-5842,5859,5863,5868,5883,5900,5910-"
    "5913,5963,5968-5969,5984-5993,5999,6064-6066,6068-6077,6080-6088,6099-"
    "6118,6121-6124,6130,6133,6140-6149,6159-6163,6200-6201,6209,6222,6241-"
    "6244,6251-6253,6267-6269,6300-6301,6306,6315-6317,6320-6322,6324-6326,"
    "6343-6344,6346-6347,6350,6355,6360,6363,6370,6379,6382,6389-6390,6417-"
    "6421,6432,6442-6446,6455-6456,6464,6471,6480-6489,6500-6503,6505-6511,"
    "6513-6515,6543-6544,6547-6551,6558,6566,6568,6579-6583,6600-6602,6619-"
    "6629,6632-6636,6640,6653,6655-6657,6670-6673,6678-6679,6687-6690,6696-"
    "6697,6701-6706,6714-6716,6767-6771,6777-6778,6784-6791,6801,6817,6831,"
    "6841-6842,6850,6868,6888,6900-6901,6924,6935-6936,6946,6951,6961-6966,"
    "6969-6970,6997-7026,7030-7031,7040,7070-7073,7080,7088,7095,7099-7101,"
    "7107,7117,7121,7128-7129,7161-7174,7181,7200-7202,7215-7216,7227-7229,"
    "7235-7237,7244,7262,7272-7283,7365,7391-7395,7397,7400-7402,7410-7411,"
    "7420-7421,7426-7431,7437,7443,7471,7473-7474,7478,7491,7500-7501,7508-"
    "7511,7542-7551,7560,7563,7566,7569-7570,7574,7588,7606,7624,7626-7631,"
    "7633,7648,7663,7672-7677,7680,7683,7687,7689,7697,7700-7701,7707-7708,"
    "7720,7724-7728,7734,7738,7741-7744,7747,7775,7777-7779,7781,7784,7786-"
    "7787,7789,7794,7797-7802,7810,7845-7847,7869-7872,7878,7880,7887,7900-"
    "7903,7913,7932-7933,7962,7967,7979-7982,7997-8009,8015-8016,8019-8023,"
    "8025-8026,8032-8034,8040-8044,8051-8060,8066-8067,8070,8074,8077,8080-"
    "8084,8086-8088,8090-8091,8097,8100-8102,8111,8115-8118,8121-8122,8128-"
    "8132,8140,8148-8149,8153,8160-8162,8181-8184,8190-8192,8194-8195,8199-"
    "8202,8204-8208,8211,8230-8232,8243,8266,8270,8276,8280,8282,8292-8294,"
    "8300-8301,8313,8320-8322,8351,8376-8380,8383-8384,8400-8405,8415-8417,"
    "8423,8442-8445,8450,8457,8470-8474,8500-8503,8554-8555,8567,8600,8609-"
    "8615,8665-8666,8675,8686,8688,8699,8710-8711,8732-8733,8750,8763-8770,"
    "8778,8786-8787,8793,8800,8804-8805,8807-8809,8873,8880-8881,8883,8888-"
    "8894,8899-8901,8910-8913,8937,8953-8954,8980-8981,8989-8991,8997-9002,"
    "9005,9007-9011,9020-9026,9050-9051,9060,9080-9081,9083-9093,9100-9107,"
    "9111,9119,9122-9123,9131,9160-9164,9191,9200-9217,9222,9255,9277-9287,"
    "9292-9295,9300,9306,9312,9318,9321,9339,9343-9346,9374,9380,9387-9390,"
    "9396-9397,9400-9402,9418,9443-9445,9450,9500,9522,9535-9536,9555,9592-"
    "9600,9612,9614,9616-9618,9628-9632,9640,9666-9668,9694-9695,9700,9747,"
    "9750,9753,9762,9800-9802,9875-9878,9888-9889,9898-9900,9903,9909,9911,"
    "9925,9950-9956,9966,9978-9979,9981,9987-9988,9990-10010,10020,10023,10050-"
    "10051,10055,10080-10081,10100-10104,10107,10110-10111,10113-10117,10125,"
    "10128-10129,10160-10162,10200-10201,10252-10253,10260-10261,10288,10321,"
    "10439,10443,10500,10540-10544,10548,10631,10800,10805,10809-10810,10860,"
    "10880,10933,10990,11000-11001,11095,11103-11106,11108-11112,11161-11165,"
    "11171-11175,11201-11202,11208,11211,11319-11321,11367,11371,11430,11489,"
    "11600,11623,11720,11723,11751,11796,11876-11877,11967,11971,12000-12010,"
    "12012-12013,12109,12121,12168,12172,12300,12302,12321-12322,12345,12753,"
    "12865,13160,13216-13218,13223-13224,13400,13720-13722,13724,13782-13783,"
    "13785-13786,13818-13823,13894,13929-13930,14000-14002,14033-14034,14141-"
    "14143,14145,14149-14150,14154,14250,14414,14500,14936-14937,15000,15002,"
    "15118,15345,15363,15555,15660,15740,15998-16003,16020-16021,16161-16162,"
    "16309-16311,16360-16361,16367-16368,16384-16385,16619,16665-16666,16789,"
    "16900,16950,16991-16995,17007,17184-17185,17219-17225,17234-17235,17500,"
    "17555,17729,17754-17756,17777,18000,18104,18136,18181-18187,18241-18243,"
    "18262,18463,18634-18635,18668,18769,18881,18888,19000,19007,19020,19191,"
    "19194,19220,19283,19315,19398,19410-19412,19539-19541,19788,19998-20003,"
    "20005,20012-20014,20034,20046,20048-20049,20057,20167,20202,20222,20480,"
    "20670,20999-21000,21010,21212-21213,21221,21553-21554,21590,21800,21845-"
    "21849,22000-22005,22125,22128,22222,22273,22305,22335,22343,22347,22350-"
    "22351,22537,22555,22763,22800,22951,23000-23005,23053,23272,23294,23333,"
    "23400-23402,23456-23457,23546,24000-24006,24242,24249,24321-24323,24386,"
    "24465,24554,24577,24666,24676-24678,24680,24754,24850,24922,25000-25009,"
    "25576,25604,25793,25900-25903,25954-25955,26000,26133,26208,26257,26260-"
    "26263,26486-26487,26489,27010,27017,27345,27442,27504,27782,27876,27999-"
    "28001,28010,28119,28200,28240,28589,29118,29167-29168,29999-30004,30100,"
    "30260,30400,30832,30999,31016,31020,31029,31400,31416,31457,31620,31685,"
    "31765,31948-31949,32034,32249,32400,32483,32635-32636,32767-32777,32801,"
    "32811,32896,33000,33060,33123,33331,33333-33334,33434-33435,33656,34249,"
    "34378-34379,34567,34962-34964,34980,35000-35006,35100,35354-35357,36001,"
    "36411-36412,36422,36462,36524,36602,36700,36865,37475,37483,37601,37654,"
    "38000-38002,38201-38203,38412,38422,38462,38472,38800,38865,39681,40000,"
    "40023,40404,40841-40843,40853,41111,41121,41230,41794-41797,42508-42510,"
    "43000,43188-43191,43210,43438-43441,44123,44321-44323,44444-44445,44544,"
    "44553,44600,44818,44900,45000-45002,45045,45054,45514,45678,45824-45825,"
    "45966,46336,46998-47001,47100,47557,47624,47806,47808-47809,48000-48005,"
    "48048-48050,48128-48129,48556,48619,48653,49000-49001,49150,6000-6063,"
    "6665-6669, "
    "U:1-50,52-80,82-99,101-113,115-224,242-248,256-257,259-271,280-284,286-"
    "287,308-324,333,344-584,586-658,660-702,704-707,709-716,729-731,741-742,"
    "744,747-754,758-765,767,769-777,780,800-802,810,828-833,847-848,853-854,"
    "860-862,873,886-888,900-903,910-913,953,989-1001,1008,1010,1021-1027,1029,"
    "1033-1108,1110-1490,1492-1782,1784-2193,2197-2258,2260-2368,2370-2375,"
    "2377,2379-2681,2683-2793,2795-2824,2826-2872,2874-2924,2926-3091,3093-"
    "3125,3127-3300,3302-3321,3326-3366,3372-3402,3405-3545,3547-3693,3695-"
    "3993,3995-4047,4049-4143,4145-4193,4197,4199,4300-4314,4316,4320-4323,"
    "4325-4336,4340-4362,4366,4368-4379,4389-4396,4400-4423,4425-4433,4441-"
    "4458,4484-4488,4500,4534-4538,4545-4559,4563,4566-4570,4573,4590-4605,"
    "4621,4646,4658-4692,4700-4704,4711,4725-4733,4737-4747,4749-4756,4774,"
    "4784-4791,4800-4804,4827,4837-4851,4867-4871,4876-4885,4888-4889,4894,"
    "4899-4902,4912,4914-4915,4936-4937,4940-4942,4949-4953,4969-4971,4980,"
    "4984-4991,4999-5015,5020-5034,5042-5075,5078-5087,5092-5094,5099-5107,"
    "5111-5112,5114-5117,5120,5133-5137,5145-5146,5150-5157,5161-5168,5172,"
    "5190-5197,5200-5203,5209,5215,5221-5237,5245-5254,5264-5265,5269-5272,"
    "5280-5282,5298-5310,5312-5318,5320-5321,5343-5344,5349-5364,5397-5437,"
    "5443,5445,5450,5453-5456,5461-5465,5470-5475,5500-5507,5550,5553-5557,"
    "5565-5569,5573-5575,5579-5586,5597-5605,5618,5627-5639,5646,5666,5670-"
    "5684,5687-5689,5693,5696,5700,5705,5713-5730,5741-5748,5750,5755,5757,"
    "5766-5771,5777,5780-5787,5793-5794,5813-5814,5841-5842,5859,5863,5868,"
    "5900,5910-5913,5963,5968-5969,5984-5993,5999,6064-6066,6068-6077,6080-"
    "6088,6099-6118,6121-6124,6130,6133,6140-6149,6159-6163,6200-6201,6209,"
    "6222,6241-6244,6251-6253,6267-6269,6300-6301,6306,6315-6317,6320-6322,"
    "6324-6326,6343-6344,6346-6347,6350,6355,6360,6363,6370,6379,6382,6389-"
    "6390,6417-6421,6432,6442-6446,6455-6456,6464,6471,6480-6489,6500-6503,"
    "6505-6511,6513-6515,6543-6544,6547-6551,6558,6566,6568,6579-6583,6600-"
    "6602,6619-6629,6632-6636,6640,6653,6655-6657,6670-6673,6678-6679,6687-"
    "6690,6696-6697,6701-6706,6714-6716,6767-6771,6777-6778,6784-6791,6801,"
    "6817,6831,6841-6842,6850,6868,6888,6900-6901,6924,6935-6936,6946,6951,"
    "6961-6966,6969-6970,6997-7026,7030-7031,7040,7070-7073,7080,7088,7095,"
    "7099-7101,7107,7117,7121,7128-7129,7161-7174,7181,7200-7201,7215-7216,"
    "7227-7229,7235-7237,7244,7262,7272-7283,7365,7391-7395,7397,7400-7402,"
    "7410-7411,7420-7421,7426-7431,7437,7443,7471,7473-7474,7478,7491,7500-"
    "7501,7508-7511,7542-7551,7560,7563,7566,7569-7570,7574,7588,7606,7624,"
    "7626-7631,7633,7648,7663,7672-7677,7680,7683,7687,7689,7697,7700-7702,"
    "7707-7708,7720,7724-7728,7734,7738,7741-7744,7747,7775,7777-7779,7781,"
    "7784,7786-7787,7789,7794,7797-7802,7810,7845-7847,7869-7872,7878,7880,"
    "7887,7900-7903,7913,7932-7933,7962,7967,7979-7982,7997-8009,8015-8016,"
    "8019-8023,8025-8026,8032-8034,8040-8044,8051-8060,8066-8067,8070,8074,"
    "8077,8080-8084,8086-8088,8090-8091,8097,8100-8102,8111,8115-8118,8121-"
    "8122,8128-8132,8140,8148-8149,8153,8160-8162,8181-8184,8190-8192,8194-"
    "8195,8199-8202,8204-8208,8211,8230-8232,8243,8266,8270,8276,8280,8282,"
    "8292-8294,8300-8301,8313,8320-8322,8351,8376-8380,8383-8384,8400-8405,"
    "8415-8417,8423,8442-8445,8450,8457,8470-8474,8500-8503,8554-8555,8567,"
    "8600,8609-8615,8665-8666,8675,8686,8688,8699,8710-8711,8732-8733,8750,"
    "8763-8770,8778,8786-8787,8793,8800,8804-8805,8807-8809,8873,8880-8881,"
    "8883,8888-8894,8899-8901,8910-8913,8937,8953-8954,8980-8981,8989-8991,"
    "8997-9002,9005,9007-9011,9020-9026,9050-9051,9060,9080-9081,9083-9093,"
    "9100-9107,9111,9119,9122-9123,9131,9160-9164,9191,9200-9217,9222,9255,"
    "9277-9287,9292-9295,9300,9306,9312,9318,9321,9339,9343-9346,9374,9380,"
    "9387-9390,9396-9397,9400-9402,9418,9443-9445,9450,9500,9522,9535-9536,"
    "9555,9592-9600,9612,9614,9616-9618,9628-9632,9640,9666-9668,9694-9695,"
    "9700,9747,9750,9753,9762,9800-9802,9875,9877-9878,9888-9889,9898-9901,"
    "9903,9909,9911,9925,9950-9956,9966,9978-9979,9981,9987-9988,9990-10010,"
    "10020,10023,10050-10051,10055,10080-10081,10100-10104,10107,10110-10111,"
    "10113-10117,10125,10128-10129,10160-10162,10200-10201,10252-10253,10260-"
    "10261,10288,10321,10439,10443,10500,10540-10544,10548,10631,10800,10805,"
    "10809-10810,10860,10880,10933,10990,11000-11001,11095,11103-11106,11108-"
    "11112,11161-11165,11171-11175,11201-11202,11208,11211,11319-11321,11367,"
    "11371,11430,11489,11600,11623,11720,11723,11751,11796,11876-11877,11967,"
    "11971,12000-12010,12012-12013,12109,12121,12168,12172,12300,12302,12321-"
    "12322,12345,12753,12865,13160,13216-13218,13223-13224,13400,13720-13722,"
    "13724,13782-13783,13785-13786,13818-13823,13894,13929-13930,14000-14002,"
    "14033-14034,14141-14143,14145,14149-14150,14154,14250,14414,14500,14936-"
    "14937,15000,15002,15118,15345,15363,15555,15660,15740,15998-16003,16020-"
    "16021,16161-16162,16309-16311,16360-16361,16367-16368,16384-16385,16619,"
    "16665-16666,16789,16900,16950,16991-16995,17007,17184-17185,17219-17225,"
    "17234-17235,17500,17555,17729,17754-17756,17777,18000,18104,18136,18181-"
    "18187,18241-18243,18262,18463,18634-18635,18668,18769,18881,18888,19000,"
    "19007,19020,19191,19194,19220,19283,19315,19398,19410-19412,19539-19541,"
    "19788,19998-20003,20005,20012-20014,20034,20046,20048-20049,20057,20167,"
    "20202,20222,20480,20670,20999-21000,21010,21212-21213,21221,21553-21554,"
    "21590,21800,21845-21849,22000-22005,22125,22128,22222,22273,22305,22335,"
    "22343,22347,22350-22351,22537,22555,22763,22800,22951,23000-23005,23053,"
    "23272,23294,23333,23400-23402,23456-23457,23546,24000-24006,24242,24249,"
    "24321-24323,24386,24465,24554,24577,24666,24676-24678,24680,24754,24850,"
    "24922,25000-25009,25576,25604,25793,25900-25903,25954-25955,26000,26133,"
    "26208,26257,26260-26263,26486-26487,26489,27010,27017,27345,27442,27504,"
    "27782,27876,27999-28001,28010,28119,28200,28240,28589,29118,29167-29168,"
    "29999-30004,30100,30260,30400,30832,30999,31016,31020,31029,31400,31416,"
    "31457,31620,31685,31765,31948-31949,32034,32249,32400,32483,32635-32636,"
    "32767-32777,32801,32811,32896,33000,33060,33123,33331,33333-33334,33434-"
    "33435,33656,34249,34378-34379,34567,34962-34964,34980,35000-35006,35100,"
    "35354-35357,36001,36411-36412,36422,36462,36524,36602,36700,36865,37475,"
    "37483,37601,37654,38000-38002,38201-38203,38412,38422,38462,38472,38800,"
    "38865,39681,40000,40023,40404,40841-40843,40853,41111,41121,41230,41794-"
    "41797,42508-42510,43000,43188-43191,43210,43438-43441,44123,44321-44323,"
    "44444-44445,44544,44553,44600,44818,44900,45000-45002,45045,45054,45514,"
    "45678,45824-45825,45966,46336,46998-47001,47100,47557,47624,47806,47808-"
    "47809,48000-48005,48048-48050,48128-48129,48556,48619,48653,49000-49001,"
    "49150,6000-6063,6665-6669",
    "T:1-65535, "
    "U:7-7,9-9,17-17,19-19,49-49,53-53,67-69,80-80,88-88,111-111,120-120,123-"
    "123,135-139,158-158,161-162,177-177,427-427,443-443,445-445,497-497,500-"
    "500,514-515,518-518,520-520,593-593,623-623,626-626,631-631,996-999,1022-"
    "1023,1025-1030,1433-1434,1645-1646,1701-1701,1718-1719,1812-1813,1900-"
    "1900,2000-2000,2048-2049,2222-2223,3283-3283,3456-3456,3703-3703,4444-"
    "4444,4500-4500,5000-5000,5060-5060,5353-5353,5632-5632,9200-9200,10000-"
    "10000,17185-17185,20031-20031,30718-30718,31337-31337,32768-32769,32771-"
    "32771,32815-32815,33281-33281,49152-49154,49156-49156,49181-49182,49185-"
    "49186,49188-49188,49190-49194,49200-49201",
    "T:1-65535, "
    "U:2-3,7-7,9-9,13-13,17-17,19-23,37-38,42-42,49-49,53-53,67-69,80-80,88-88,"
    "111-113,120-120,123-123,135-139,158-158,161-162,177-177,192-192,199-199,"
    "207-207,217-217,363-363,389-389,402-402,407-407,427-427,434-434,443-443,"
    "445-445,464-464,497-497,500-500,502-502,512-515,517-518,520-520,539-539,"
    "559-559,593-593,623-623,626-626,631-631,639-639,643-643,657-657,664-664,"
    "682-689,764-764,767-767,772-776,780-782,786-786,789-789,800-800,814-814,"
    "826-826,829-829,838-838,902-903,944-944,959-959,965-965,983-983,989-990,"
    "996-1001,1007-1008,1012-1014,1019-1051,1053-1060,1064-1070,1072-1072,1080-"
    "1081,1087-1088,1090-1090,1100-1101,1105-1105,1124-1124,1200-1200,1214-"
    "1214,1234-1234,1346-1346,1419-1419,1433-1434,1455-1455,1457-1457,1484-"
    "1485,1524-1524,1645-1646,1701-1701,1718-1719,1761-1761,1782-1782,1804-"
    "1804,1812-1813,1885-1886,1900-1901,1993-1993,2000-2000,2002-2002,2048-"
    "2049,2051-2051,2148-2148,2160-2161,2222-2223,2343-2343,2345-2345,2362-"
    "2362,2967-2967,3052-3052,3130-3130,3283-3283,3296-3296,3343-3343,3389-"
    "3389,3401-3401,3456-3457,3659-3659,3664-3664,3702-3703,4000-4000,4008-"
    "4008,4045-4045,4444-4444,4500-4500,4666-4666,4672-4672,5000-5003,5010-"
    "5010,5050-5050,5060-5060,5093-5093,5351-5351,5353-5353,5355-5355,5500-"
    "5500,5555-5555,5632-5632,6000-6002,6004-6004,6050-6050,6346-6347,6970-"
    "6971,7000-7000,7938-7938,8000-8001,8010-8010,8181-8181,8193-8193,8900-"
    "8900,9000-9001,9020-9020,9103-9103,9199-9200,9370-9370,9876-9877,9950-"
    "9950,10000-10000,10080-10080,11487-11487,16086-16086,16402-16402,16420-"
    "16420,16430-16430,16433-16433,16449-16449,16498-16498,16503-16503,16545-"
    "16545,16548-16548,16573-16573,16674-16674,16680-16680,16697-16697,16700-"
    "16700,16708-16708,16711-16711,16739-16739,16766-16766,16779-16779,16786-"
    "16786,16816-16816,16829-16829,16832-16832,16838-16839,16862-16862,16896-"
    "16896,16912-16912,16918-16919,16938-16939,16947-16948,16970-16970,16972-"
    "16972,16974-16974,17006-17006,17018-17018,17077-17077,17091-17091,17101-"
    "17101,17146-17146,17184-17185,17205-17205,17207-17207,17219-17219,17236-"
    "17237,17282-17282,17302-17302,17321-17321,17331-17332,17338-17338,17359-"
    "17359,17417-17417,17423-17424,17455-17455,17459-17459,17468-17468,17487-"
    "17487,17490-17490,17494-17494,17505-17505,17533-17533,17549-17549,17573-"
    "17573,17580-17580,17585-17585,17592-17592,17605-17605,17615-17616,17629-"
    "17629,17638-17638,17663-17663,17673-17674,17683-17683,17726-17726,17754-"
    "17754,17762-17762,17787-17787,17814-17814,17823-17824,17836-17836,17845-"
    "17845,17888-17888,17939-17939,17946-17946,17989-17989,18004-18004,18081-"
    "18081,18113-18113,18134-18134,18156-18156,18228-18228,18234-18234,18250-"
    "18250,18255-18255,18258-18258,18319-18319,18331-18331,18360-18360,18373-"
    "18373,18449-18449,18485-18485,18543-18543,18582-18582,18605-18605,18617-"
    "18617,18666-18666,18669-18669,18676-18676,18683-18683,18807-18807,18818-"
    "18818,18821-18821,18830-18830,18832-18832,18835-18835,18869-18869,18883-"
    "18883,18888-18888,18958-18958,18980-18980,18985-18985,18987-18987,18991-"
    "18991,18994-18994,18996-18996,19017-19017,19022-19022,19039-19039,19047-"
    "19047,19075-19075,19096-19096,19120-19120,19130-19130,19140-19141,19154-"
    "19154,19161-19161,19165-19165,19181-19181,19193-19193,19197-19197,19222-"
    "19222,19227-19227,19273-19273,19283-19283,19294-19294,19315-19315,19322-"
    "19322,19332-19332,19374-19374,19415-19415,19482-19482,19489-19489,19500-"
    "19500,19503-19504,19541-19541,19600-19600,19605-19605,19616-19616,19624-"
    "19625,19632-19632,19639-19639,19647-19647,19650-19650,19660-19660,19662-"
    "19663,19682-19683,19687-19687,19695-19695,19707-19707,19717-19719,19722-"
    "19722,19728-19728,19789-19789,19792-19792,19933-19933,19935-19936,19956-"
    "19956,19995-19995,19998-19998,20003-20004,20019-20019,20031-20031,20082-"
    "20082,20117-20117,20120-20120,20126-20126,20129-20129,20146-20146,20154-"
    "20154,20164-20164,20206-20206,20217-20217,20249-20249,20262-20262,20279-"
    "20279,20288-20288,20309-20309,20313-20313,20326-20326,20359-20360,20366-"
    "20366,20380-20380,20389-20389,20409-20409,20411-20411,20423-20425,20445-"
    "20445,20449-20449,20464-20465,20518-20518,20522-20522,20525-20525,20540-"
    "20540,20560-20560,20665-20665,20678-20679,20710-20710,20717-20717,20742-"
    "20742,20752-20752,20762-20762,20791-20791,20817-20817,20842-20842,20848-"
    "20848,20851-20851,20865-20865,20872-20872,20876-20876,20884-20884,20919-"
    "20919,21000-21000,21016-21016,21060-21060,21083-21083,21104-21104,21111-"
    "21111,21131-21131,21167-21167,21186-21186,21206-21207,21212-21212,21247-"
    "21247,21261-21261,21282-21282,21298-21298,21303-21303,21318-21318,21320-"
    "21320,21333-21333,21344-21344,21354-21354,21358-21358,21360-21360,21364-"
    "21364,21366-21366,21383-21383,21405-21405,21454-21454,21468-21468,21476-"
    "21476,21514-21514,21524-21525,21556-21556,21566-21566,21568-21568,21576-"
    "21576,21609-21609,21621-21621,21625-21625,21644-21644,21649-21649,21655-"
    "21655,21663-21663,21674-21674,21698-21698,21702-21702,21710-21710,21742-"
    "21742,21780-21780,21784-21784,21800-21800,21803-21803,21834-21834,21842-"
    "21842,21847-21847,21868-21868,21898-21898,21902-21902,21923-21923,21948-"
    "21948,21967-21967,22029-22029,22043-22043,22045-22045,22053-22053,22055-"
    "22055,22105-22105,22109-22109,22123-22124,22341-22341,22692-22692,22695-"
    "22695,22739-22739,22799-22799,22846-22846,22914-22914,22986-22986,22996-"
    "22996,23040-23040,23176-23176,23354-23354,23531-23531,23557-23557,23608-"
    "23608,23679-23679,23781-23781,23965-23965,23980-23980,24007-24007,24279-"
    "24279,24511-24511,24594-24594,24606-24606,24644-24644,24854-24854,24910-"
    "24910,25003-25003,25157-25157,25240-25240,25280-25280,25337-25337,25375-"
    "25375,25462-25462,25541-25541,25546-25546,25709-25709,25931-25931,26407-"
    "26407,26415-26415,26720-26720,26872-26872,26966-26966,27015-27015,27195-"
    "27195,27444-27444,27473-27473,27482-27482,27707-27707,27892-27892,27899-"
    "27899,28122-28122,28369-28369,28465-28465,28493-28493,28543-28543,28547-"
    "28547,28641-28641,28840-28840,28973-28973,29078-29078,29243-29243,29256-"
    "29256,29810-29810,29823-29823,29977-29977,30263-30263,30303-30303,30365-"
    "30365,30544-30544,30656-30656,30697-30697,30704-30704,30718-30718,30975-"
    "30975,31059-31059,31073-31073,31109-31109,31189-31189,31195-31195,31335-"
    "31335,31337-31337,31365-31365,31625-31625,31681-31681,31731-31731,31891-"
    "31891,32345-32345,32385-32385,32528-32528,32768-32780,32798-32798,32815-"
    "32815,32818-32818,32931-32931,33030-33030,33249-33249,33281-33281,33354-"
    "33355,33459-33459,33717-33717,33744-33744,33866-33866,33872-33872,34038-"
    "34038,34079-34079,34125-34125,34358-34358,34422-34422,34433-34433,34555-"
    "34555,34570-34570,34577-34580,34758-34758,34796-34796,34855-34855,34861-"
    "34862,34892-34892,35438-35438,35702-35702,35777-35777,35794-35794,36108-"
    "36108,36206-36206,36384-36384,36458-36458,36489-36489,36669-36669,36778-"
    "36778,36893-36893,36945-36945,37144-37144,37212-37212,37393-37393,37444-"
    "37444,37602-37602,37761-37761,37783-37783,37813-37813,37843-37843,38037-"
    "38037,38063-38063,38293-38293,38412-38412,38498-38498,38615-38615,39213-"
    "39213,39217-39217,39632-39632,39683-39683,39714-39714,39723-39723,39888-"
    "39888,40019-40019,40116-40116,40441-40441,40539-40539,40622-40622,40708-"
    "40708,40711-40711,40724-40724,40732-40732,40805-40805,40847-40847,40866-"
    "40866,40915-40915,41058-41058,41081-41081,41308-41308,41370-41370,41446-"
    "41446,41524-41524,41638-41638,41702-41702,41774-41774,41896-41896,41967-"
    "41967,41971-41971,42056-42056,42172-42172,42313-42313,42431-42431,42434-"
    "42434,42508-42508,42557-42557,42577-42577,42627-42627,42639-42639,43094-"
    "43094,43195-43195,43370-43370,43514-43514,43686-43686,43824-43824,43967-"
    "43967,44101-44101,44160-44160,44179-44179,44185-44185,44190-44190,44253-"
    "44253,44334-44334,44508-44508,44923-44923,44946-44946,44968-44968,45247-"
    "45247,45380-45380,45441-45441,45685-45685,45722-45722,45818-45818,45928-"
    "45928,46093-46093,46532-46532,46836-46836,47624-47624,47765-47765,47772-"
    "47772,47808-47808,47915-47915,47981-47981,48078-48078,48189-48189,48255-"
    "48255,48455-48455,48489-48489,48761-48761,49152-49163,49165-49182,49184-"
    "49202,49204-49205,49207-49216,49220-49220,49222-49222,49226-49226,49259-"
    "49259,49262-49262,49306-49306,49350-49350,49360-49360,49393-49393,49396-"
    "49396,49503-49503,49640-49640,49968-49968,50099-50099,50164-50164,50497-"
    "50497,50612-50612,50708-50708,50919-50919,51255-51255,51456-51456,51554-"
    "51554,51586-51586,51690-51690,51717-51717,51905-51905,51972-51972,52144-"
    "52144,52225-52225,52503-52503,53006-53006,53037-53037,53571-53571,53589-"
    "53589,53838-53838,54094-54094,54114-54114,54281-54281,54321-54321,54711-"
    "54711,54807-54807,54925-54925,55043-55043,55544-55544,55587-55587,56141-"
    "56141,57172-57172,57409-57410,57813-57813,57843-57843,57958-57958,57977-"
    "57977,58002-58002,58075-58075,58178-58178,58419-58419,58631-58631,58640-"
    "58640,58797-58797,59193-59193,59207-59207,59765-59765,59846-59846,60172-"
    "60172,60381-60381,60423-60423,61024-61024,61142-61142,61319-61319,61322-"
    "61322,61370-61370,61412-61412,61481-61481,61550-61550,61685-61685,61961-"
    "61961,62154-62154,62287-62287,62575-62575,62677-62677,62699-62699,62958-"
    "62958,63420-63420,63555-63555,64080-64080,64481-64481,64513-64513,64590-"
    "64590,64727-64727",
    "T:1-1,3-4,6-7,9-9,13-13,17-17,19-27,30-30,32-33,37-37,42-43,49-49,53-53,"
    "55-55,57-57,59-59,70-70,77-77,79-90,98-100,102-102,106-106,109-111,113-"
    "113,119-119,123-123,125-125,127-127,135-135,139-139,143-144,146-146,157-"
    "157,161-161,163-163,179-179,199-199,210-212,220-220,222-223,225-225,250-"
    "252,254-257,259-259,264-264,280-280,301-301,306-306,311-311,333-333,340-"
    "340,366-366,388-389,406-407,411-411,416-417,419-419,425-425,427-427,441-"
    "445,447-447,458-458,464-465,475-475,481-481,497-497,500-500,502-502,512-"
    "515,523-524,540-541,543-545,548-548,554-557,563-563,587-587,593-593,600-"
    "600,602-602,606-606,610-610,616-617,621-621,623-623,625-625,631-631,636-"
    "636,639-639,641-641,646-646,648-648,655-655,657-657,659-660,666-669,674-"
    "674,683-684,687-687,690-691,700-701,705-705,709-711,713-715,720-720,722-"
    "722,725-726,728-732,740-740,748-749,754-754,757-758,765-765,777-778,780-"
    "780,782-783,786-787,790-790,792-792,795-795,800-803,805-806,808-808,822-"
    "823,825-825,829-829,839-840,843-843,846-846,856-856,859-859,862-862,864-"
    "864,873-874,878-878,880-880,888-888,898-898,900-905,911-913,918-918,921-"
    "922,924-924,928-928,930-931,943-943,953-953,969-969,971-971,980-981,987-"
    "987,990-990,992-993,995-996,998-1002,1004-1015,1020-1114,1116-1119,1121-"
    "1128,1130-1132,1134-1138,1141-1141,1143-1145,1147-1154,1156-1159,1162-"
    "1169,1173-1176,1179-1180,1182-1188,1190-1192,1194-1196,1198-1201,1204-"
    "1204,1207-1213,1215-1218,1220-1223,1228-1229,1233-1234,1236-1236,1239-"
    "1241,1243-1244,1247-1251,1259-1259,1261-1262,1264-1264,1268-1268,1270-"
    "1272,1276-1277,1279-1279,1282-1282,1287-1287,1290-1291,1296-1297,1299-"
    "1303,1305-1311,1314-1319,1321-1322,1324-1324,1327-1328,1330-1331,1334-"
    "1334,1336-1337,1339-1340,1347-1347,1350-1353,1357-1357,1413-1414,1417-"
    "1417,1433-1434,1443-1443,1455-1455,1461-1461,1494-1494,1500-1501,1503-"
    "1503,1516-1516,1521-1522,1524-1526,1533-1533,1547-1547,1550-1550,1556-"
    "1556,1558-1560,1565-1566,1569-1569,1580-1580,1583-1584,1592-1592,1594-"
    "1594,1598-1598,1600-1600,1605-1605,1607-1607,1615-1615,1620-1620,1622-"
    "1622,1632-1632,1635-1635,1638-1638,1641-1641,1645-1645,1658-1658,1666-"
    "1666,1677-1677,1683-1683,1687-1688,1691-1691,1694-1694,1699-1701,1703-"
    "1703,1707-1709,1711-1713,1715-1715,1717-1723,1730-1730,1735-1736,1745-"
    "1745,1750-1750,1752-1753,1755-1755,1761-1761,1782-1783,1791-1792,1799-"
    "1801,1805-1808,1811-1812,1823-1823,1825-1825,1835-1835,1839-1840,1858-"
    "1858,1861-1864,1871-1871,1875-1875,1900-1901,1911-1912,1914-1914,1918-"
    "1918,1924-1924,1927-1927,1935-1935,1947-1947,1954-1954,1958-1958,1971-"
    "1976,1981-1981,1984-1984,1998-2013,2020-2022,2025-2025,2030-2031,2033-"
    "2035,2038-2038,2040-2049,2062-2062,2065-2065,2067-2070,2080-2083,2086-"
    "2087,2095-2096,2099-2101,2103-2107,2111-2112,2115-2115,2119-2119,2121-"
    "2121,2124-2124,2126-2126,2134-2135,2142-2142,2144-2144,2148-2148,2150-"
    "2150,2160-2161,2170-2170,2179-2179,2187-2187,2190-2191,2196-2197,2200-"
    "2201,2203-2203,2222-2222,2224-2224,2232-2232,2241-2241,2250-2251,2253-"
    "2253,2260-2262,2265-2265,2269-2271,2280-2280,2288-2288,2291-2292,2300-"
    "2302,2304-2304,2312-2313,2323-2323,2325-2326,2330-2330,2335-2335,2340-"
    "2340,2366-2366,2371-2372,2381-2383,2391-2391,2393-2394,2399-2399,2401-"
    "2401,2418-2418,2425-2425,2433-2433,2435-2436,2438-2439,2449-2449,2456-"
    "2456,2463-2463,2472-2472,2492-2492,2500-2501,2505-2505,2522-2522,2525-"
    "2525,2531-2532,2550-2551,2557-2558,2567-2567,2580-2580,2583-2584,2598-"
    "2598,2600-2602,2604-2608,2622-2623,2628-2628,2631-2631,2638-2638,2644-"
    "2644,2691-2691,2700-2702,2706-2706,2710-2712,2717-2718,2723-2723,2725-"
    "2725,2728-2728,2734-2734,2800-2800,2804-2804,2806-2806,2809-2809,2811-"
    "2812,2847-2847,2850-2850,2869-2869,2875-2875,2882-2882,2888-2889,2898-"
    "2898,2901-2903,2908-2910,2920-2920,2930-2930,2957-2958,2967-2968,2973-"
    "2973,2984-2984,2987-2988,2991-2991,2997-2998,3000-3003,3005-3007,3011-"
    "3011,3013-3014,3017-3017,3023-3023,3025-3025,3030-3031,3050-3050,3052-"
    "3052,3057-3057,3062-3063,3071-3071,3077-3077,3080-3080,3089-3089,3102-"
    "3103,3118-3119,3121-3121,3128-3128,3146-3146,3162-3162,3167-3168,3190-"
    "3190,3200-3200,3210-3211,3220-3221,3240-3240,3260-3261,3263-3263,3268-"
    "3269,3280-3281,3283-3283,3291-3291,3299-3301,3304-3304,3306-3307,3310-"
    "3311,3319-3319,3322-3325,3333-3334,3351-3351,3362-3363,3365-3365,3367-"
    "3372,3374-3374,3376-3376,3388-3390,3396-3396,3399-3400,3404-3404,3410-"
    "3410,3414-3415,3419-3419,3425-3425,3430-3430,3439-3439,3443-3443,3456-"
    "3456,3476-3476,3479-3479,3483-3483,3485-3486,3493-3493,3497-3497,3503-"
    "3503,3505-3506,3511-3511,3513-3515,3517-3517,3519-3520,3526-3527,3530-"
    "3530,3532-3532,3546-3546,3551-3551,3577-3577,3580-3580,3586-3586,3599-"
    "3600,3602-3603,3621-3622,3632-3632,3636-3637,3652-3653,3656-3656,3658-"
    "3659,3663-3663,3669-3670,3672-3672,3680-3681,3683-3684,3689-3690,3697-"
    "3697,3700-3700,3703-3703,3712-3712,3728-3728,3731-3731,3737-3737,3742-"
    "3742,3749-3749,3765-3766,3784-3784,3787-3788,3790-3790,3792-3793,3795-"
    "3796,3798-3801,3803-3803,3806-3806,3808-3814,3817-3817,3820-3820,3823-"
    "3828,3830-3831,3837-3837,3839-3839,3842-3842,3846-3853,3856-3856,3859-"
    "3860,3863-3863,3868-3872,3876-3876,3878-3880,3882-3882,3888-3890,3897-"
    "3897,3899-3899,3901-3902,3904-3909,3911-3911,3913-3916,3918-3920,3922-"
    "3923,3928-3931,3935-3937,3940-3941,3943-3946,3948-3949,3952-3952,3956-"
    "3957,3961-3964,3967-3969,3971-3972,3975-3975,3979-3983,3986-3986,3989-"
    "4007,4009-4010,4016-4016,4020-4020,4022-4022,4024-4025,4029-4029,4035-"
    "4036,4039-4040,4045-4045,4056-4056,4058-4058,4065-4065,4080-4080,4087-"
    "4087,4090-4090,4096-4096,4100-4101,4111-4113,4118-4121,4125-4126,4129-"
    "4129,4135-4135,4141-4141,4143-4143,4147-4147,4158-4158,4161-4161,4164-"
    "4164,4174-4174,4190-4190,4192-4192,4200-4200,4206-4206,4220-4220,4224-"
    "4224,4234-4234,4242-4242,4252-4252,4262-4262,4279-4279,4294-4294,4297-"
    "4298,4300-4300,4302-4302,4321-4321,4325-4325,4328-4328,4333-4333,4342-"
    "4343,4355-4358,4369-4369,4374-4376,4384-4384,4388-4388,4401-4401,4407-"
    "4407,4414-4415,4418-4418,4430-4430,4433-4433,4442-4447,4449-4449,4454-"
    "4454,4464-4464,4471-4471,4476-4476,4516-4517,4530-4530,4534-4534,4545-"
    "4545,4550-4550,4555-4555,4558-4559,4567-4567,4570-4570,4599-4602,4606-"
    "4606,4609-4609,4644-4644,4649-4649,4658-4658,4662-4662,4665-4665,4687-"
    "4687,4689-4689,4700-4700,4712-4712,4745-4745,4760-4760,4767-4767,4770-"
    "4771,4778-4778,4793-4793,4800-4800,4819-4819,4848-4848,4859-4860,4875-"
    "4877,4881-4881,4899-4900,4903-4903,4912-4912,4931-4931,4949-4949,4998-"
    "5005,5009-5017,5020-5021,5023-5023,5030-5030,5033-5033,5040-5040,5050-"
    "5055,5060-5061,5063-5063,5066-5066,5070-5070,5074-5074,5080-5081,5087-"
    "5088,5090-5090,5095-5096,5098-5098,5100-5102,5111-5111,5114-5114,5120-"
    "5122,5125-5125,5133-5133,5137-5137,5147-5147,5151-5152,5190-5190,5200-"
    "5202,5212-5212,5214-5214,5219-5219,5221-5223,5225-5226,5233-5235,5242-"
    "5242,5250-5250,5252-5252,5259-5259,5261-5261,5269-5269,5279-5280,5291-"
    "5291,5298-5298,5339-5339,5347-5347,5353-5353,5357-5357,5370-5370,5377-"
    "5377,5405-5405,5414-5414,5423-5423,5431-5433,5440-5442,5444-5444,5457-"
    "5458,5473-5473,5475-5475,5500-5502,5510-5510,5520-5520,5544-5544,5550-"
    "5550,5552-5555,5557-5557,5560-5560,5566-5566,5631-5631,5633-5633,5666-"
    "5666,5678-5680,5718-5718,5730-5730,5800-5803,5807-5807,5810-5812,5815-"
    "5815,5818-5818,5822-5823,5825-5825,5850-5850,5859-5859,5862-5862,5868-"
    "5869,5877-5877,5899-5907,5909-5911,5914-5915,5918-5918,5922-5922,5925-"
    "5925,5938-5938,5940-5940,5950-5950,5952-5952,5959-5963,5968-5968,5981-"
    "5981,5987-5989,5998-6009,6017-6017,6025-6025,6050-6051,6059-6060,6068-"
    "6068,6100-6101,6103-6103,6106-6106,6112-6112,6123-6123,6129-6129,6156-"
    "6156,6203-6203,6222-6222,6247-6247,6346-6346,6389-6389,6481-6481,6500-"
    "6500,6502-6502,6504-6504,6510-6510,6520-6520,6543-6543,6547-6547,6550-"
    "6550,6565-6567,6580-6580,6600-6600,6646-6646,6662-6662,6666-6670,6689-"
    "6689,6692-6692,6699-6699,6711-6711,6732-6732,6779-6779,6788-6789,6792-"
    "6792,6839-6839,6881-6881,6896-6896,6901-6901,6969-6969,7000-7004,7007-"
    "7007,7010-7010,7019-7019,7024-7025,7050-7051,7070-7070,7080-7080,7100-"
    "7100,7103-7103,7106-7106,7123-7123,7200-7201,7241-7241,7272-7272,7278-"
    "7278,7281-7281,7402-7402,7435-7435,7438-7438,7443-7443,7496-7496,7512-"
    "7512,7625-7625,7627-7627,7676-7676,7725-7725,7741-7741,7744-7744,7749-"
    "7749,7770-7770,7777-7778,7800-7800,7878-7878,7900-7900,7911-7911,7913-"
    "7913,7920-7921,7929-7929,7937-7938,7999-8002,8007-8011,8015-8016,8019-"
    "8019,8021-8022,8031-8031,8042-8042,8045-8045,8050-8050,8080-8090,8093-"
    "8093,8095-8095,8097-8100,8118-8118,8180-8181,8189-8189,8192-8194,8200-"
    "8200,8222-8222,8254-8254,8290-8294,8300-8300,8333-8333,8383-8383,8385-"
    "8385,8400-8400,8402-8402,8443-8443,8481-8481,8500-8500,8540-8540,8600-"
    "8600,8648-8649,8651-8652,8654-8654,8675-8676,8686-8686,8701-8701,8765-"
    "8766,8800-8800,8873-8873,8877-8877,8888-8889,8899-8899,8987-8987,8994-"
    "8994,8996-8996,9000-9003,9009-9011,9040-9040,9050-9050,9071-9071,9080-"
    "9081,9090-9091,9098-9103,9110-9111,9152-9152,9191-9191,9197-9198,9200-"
    "9200,9207-9207,9220-9220,9290-9290,9409-9409,9415-9415,9418-9418,9443-"
    "9444,9485-9485,9500-9503,9535-9535,9575-9575,9593-9595,9600-9600,9618-"
    "9618,9621-9621,9643-9643,9666-9666,9673-9673,9815-9815,9876-9878,9898-"
    "9898,9900-9900,9914-9914,9917-9917,9929-9929,9941-9941,9943-9944,9968-"
    "9968,9988-9988,9992-9992,9998-10005,10008-10012,10022-10025,10034-10034,"
    "10058-10058,10082-10083,10160-10160,10180-10180,10215-10215,10243-10243,"
    "10566-10566,10616-10617,10621-10621,10626-10626,10628-10629,10778-10778,"
    "10873-10873,11110-11111,11967-11967,12000-12000,12006-12006,12021-12021,"
    "12059-12059,12174-12174,12215-12215,12262-12262,12265-12265,12345-12346,"
    "12380-12380,12452-12452,13456-13456,13722-13722,13724-13724,13782-13783,"
    "14000-14000,14238-14238,14441-14442,15000-15004,15402-15402,15660-15660,"
    "15742-15742,16000-16001,16012-16012,16016-16016,16018-16018,16080-16080,"
    "16113-16113,16705-16705,16800-16800,16851-16851,16992-16993,17595-17595,"
    "17877-17877,17988-17988,18000-18000,18018-18018,18040-18040,18101-18101,"
    "18264-18264,18988-18988,19101-19101,19283-19283,19315-19315,19350-19350,"
    "19780-19780,19801-19801,19842-19842,19900-19900,20000-20000,20002-20002,"
    "20005-20005,20031-20031,20221-20222,20828-20828,21571-21571,21792-21792,"
    "22222-22222,22939-22939,23052-23052,23502-23502,23796-23796,24444-24444,"
    "24800-24800,25734-25735,26000-26000,26214-26214,26470-26470,27000-27000,"
    "27352-27353,27355-27357,27715-27715,28201-28201,28211-28211,29672-29672,"
    "29831-29831,30000-30000,30005-30005,30704-30704,30718-30718,30951-30951,"
    "31038-31038,31337-31337,31727-31727,32768-32785,32791-32792,32803-32803,"
    "32816-32816,32822-32822,32835-32835,33354-33354,33453-33453,33554-33554,"
    "33899-33899,34571-34573,35500-35500,35513-35513,37839-37839,38037-38037,"
    "38185-38185,38188-38188,38292-38292,39136-39136,39376-39376,39659-39659,"
    "40000-40000,40193-40193,40811-40811,40911-40911,41064-41064,41511-41511,"
    "41523-41523,42510-42510,44176-44176,44334-44334,44442-44443,44501-44501,"
    "44709-44709,45100-45100,46200-46200,46996-46996,47544-47544,48080-48080,"
    "49152-49161,49163-49165,49167-49168,49171-49171,49175-49176,49186-49186,"
    "49195-49195,49236-49236,49400-49401,49999-50003,50006-50006,50050-50050,"
    "50300-50300,50389-50389,50500-50500,50636-50636,50800-50800,51103-51103,"
    "51191-51191,51413-51413,51493-51493,52660-52660,52673-52673,52710-52710,"
    "52735-52735,52822-52822,52847-52851,52853-52853,52869-52869,53211-53211,"
    "53313-53314,53535-53535,54045-54045,54328-54328,55020-55020,55055-55056,"
    "55555-55555,55576-55576,55600-55600,56737-56738,57294-57294,57665-57665,"
    "57797-57797,58001-58002,58080-58080,58630-58630,58632-58632,58838-58838,"
    "59110-59110,59200-59202,60020-60020,60123-60123,60146-60146,60443-60443,"
    "60642-60642,61532-61532,61613-61613,61900-61900,62078-62078,63331-63331,"
    "64623-64623,64680-64680,65000-65000,65129-65129,65310-65310, "
    "U:7-7,9-9,17-17,19-19,49-49,53-53,67-69,80-80,88-88,111-111,120-120,123-"
    "123,135-139,158-158,161-162,177-177,427-427,443-443,445-445,497-497,500-"
    "500,514-515,518-518,520-520,593-593,623-623,626-626,631-631,996-999,1022-"
    "1023,1025-1030,1433-1434,1645-1646,1701-1701,1718-1719,1812-1813,1900-"
    "1900,2000-2000,2048-2049,2222-2223,3283-3283,3456-3456,3703-3703,4444-"
    "4444,4500-4500,5000-5000,5060-5060,5353-5353,5632-5632,9200-9200,10000-"
    "10000,17185-17185,20031-20031,30718-30718,31337-31337,32768-32769,32771-"
    "32771,32815-32815,33281-33281,49152-49154,49156-49156,49181-49182,49185-"
    "49186,49188-49188,49190-49194,49200-49201",
    "T:80-80,443-443"};

  assert_that (
    validate_port_range (portlists[ALL_IANA_ASSIGNED_TCP_2020_02_12]),
    is_equal_to (0));
  assert_that (validate_port_range (portlists[ALL_PRIVILEGED_TCP]),
               is_equal_to (0));
  assert_that (validate_port_range (portlists[ALL_PRIVILEGED_TCP_AND_UDP]),
               is_equal_to (0));
  assert_that (validate_port_range (portlists[ALL_TCP]), is_equal_to (0));
  assert_that (validate_port_range (portlists[OPENVAS_DEFAULT]),
               is_equal_to (0));
  assert_that (
    validate_port_range (portlists[ALL_IANA_ASSIGNED_TCP_AND_UDP_2020_02_12]),
    is_equal_to (0));
  assert_that (
    validate_port_range (portlists[ALL_TCP_AND_NMAP_5_51_TOP_100_UDP]),
    is_equal_to (0));
  assert_that (
    validate_port_range (portlists[ALL_TCP_AND_NMAP_5_51_TOP_1000_UDP]),
    is_equal_to (0));
  assert_that (
    validate_port_range (portlists[NMAP_5_51_TOP_2000_TCP_AND_TOP_100_UDP]),
    is_equal_to (0));
  assert_that (validate_port_range (portlists[WEB_SERVICES]), is_equal_to (0));
}

Ensure (networking, port_range_ranges)
{
  const gchar *valid_portrange1;
  array_t *valid_portrange1_ranges;
  range_t *valid_portrange1_range1;
  range_t *valid_portrange1_range2;
  range_t *valid_portrange1_range3;
  range_t *valid_portrange1_range4;
  range_t *valid_portrange1_range5;

  /* Valid port ranges. U:,T: are empty ranges which are ignored. */
  valid_portrange1 = "1,10-13,10-10,T  :1-2,U : 10-12,U:,T:";
  assert_that (validate_port_range (valid_portrange1), is_equal_to (0));

  assert_that (port_range_ranges (NULL), is_null);
  assert_that (port_range_ranges (valid_portrange1), is_not_null);

  valid_portrange1_ranges = port_range_ranges (valid_portrange1);
  assert_that (valid_portrange1_ranges, is_not_null);
  assert_that (valid_portrange1_ranges->len, is_equal_to (5));

  assert_that (valid_portrange1_ranges->len, is_equal_to (5));
  valid_portrange1_range1 =
    (range_t *) g_ptr_array_index (valid_portrange1_ranges, 0);
  assert_that (valid_portrange1_range1->start, is_equal_to (1));
  assert_that (valid_portrange1_range1->end, is_equal_to (1));
  assert_that (valid_portrange1_range1->exclude, is_equal_to (0));
  assert_that (valid_portrange1_range1->type, is_equal_to (PORT_PROTOCOL_TCP));

  valid_portrange1_range2 =
    (range_t *) g_ptr_array_index (valid_portrange1_ranges, 1);
  assert_that (valid_portrange1_range2->start, is_equal_to (10));
  assert_that (valid_portrange1_range2->end, is_equal_to (13));
  assert_that (valid_portrange1_range2->exclude, is_equal_to (0));
  assert_that (valid_portrange1_range2->type, is_equal_to (PORT_PROTOCOL_TCP));

  valid_portrange1_range3 =
    (range_t *) g_ptr_array_index (valid_portrange1_ranges, 2);
  assert_that (valid_portrange1_range3->start, is_equal_to (10));
  assert_that (valid_portrange1_range3->end, is_equal_to (10));
  assert_that (valid_portrange1_range3->exclude, is_equal_to (0));
  assert_that (valid_portrange1_range3->type, is_equal_to (PORT_PROTOCOL_TCP));

  valid_portrange1_range4 =
    (range_t *) g_ptr_array_index (valid_portrange1_ranges, 3);
  assert_that (valid_portrange1_range4->start, is_equal_to (1));
  assert_that (valid_portrange1_range4->end, is_equal_to (2));
  assert_that (valid_portrange1_range4->exclude, is_equal_to (0));
  assert_that (valid_portrange1_range4->type, is_equal_to (PORT_PROTOCOL_TCP));

  valid_portrange1_range5 =
    (range_t *) g_ptr_array_index (valid_portrange1_ranges, 4);
  assert_that (valid_portrange1_range5->start, is_equal_to (10));
  assert_that (valid_portrange1_range5->end, is_equal_to (12));
  assert_that (valid_portrange1_range5->exclude, is_equal_to (0));
  assert_that (valid_portrange1_range5->type, is_equal_to (PORT_PROTOCOL_UDP));
}

Ensure (networking, port_in_port_ranges)
{
  const gchar *portrange;
  array_t *portrange_ranges;

  /* Valid port ranges. U:,T: are empty ranges which are ignored. */
  portrange = "1,10-12,10-10,T:1-2,U:10-14,U:,T:";
  assert_that (validate_port_range (portrange), is_equal_to (0));

  portrange_ranges = port_range_ranges (portrange);
  assert_that (portrange_ranges, is_not_null);
  assert_that (portrange_ranges->len, is_equal_to (5));

  assert_that (port_in_port_ranges (1, PORT_PROTOCOL_TCP, portrange_ranges),
               is_true);
  assert_that (port_in_port_ranges (10, PORT_PROTOCOL_TCP, portrange_ranges),
               is_true);
  assert_that (port_in_port_ranges (11, PORT_PROTOCOL_TCP, portrange_ranges),
               is_true);
  assert_that (port_in_port_ranges (12, PORT_PROTOCOL_TCP, portrange_ranges),
               is_true);
  assert_that (port_in_port_ranges (10, PORT_PROTOCOL_UDP, portrange_ranges),
               is_true);
  assert_that (port_in_port_ranges (11, PORT_PROTOCOL_UDP, portrange_ranges),
               is_true);
  assert_that (port_in_port_ranges (12, PORT_PROTOCOL_UDP, portrange_ranges),
               is_true);
  assert_that (port_in_port_ranges (13, PORT_PROTOCOL_UDP, portrange_ranges),
               is_true);

  assert_that (port_in_port_ranges (-1, PORT_PROTOCOL_TCP, portrange_ranges),
               is_false);
  assert_that (port_in_port_ranges (0, PORT_PROTOCOL_TCP, portrange_ranges),
               is_false);
  assert_that (port_in_port_ranges (90000, PORT_PROTOCOL_TCP, portrange_ranges),
               is_false);
  assert_that (port_in_port_ranges (1, PORT_PROTOCOL_UDP, portrange_ranges),
               is_false);
  assert_that (port_in_port_ranges (13, PORT_PROTOCOL_TCP, portrange_ranges),
               is_false);
  assert_that (port_in_port_ranges (13, PORT_PROTOCOL_TCP, portrange_ranges),
               is_false);
  assert_that (port_in_port_ranges (12, PORT_PROTOCOL_OTHER, portrange_ranges),
               is_false);
}

/* Test suite. */

Ensure (networking, ip_islocalhost)
{
  /* IPv4 */
  struct in_addr addr;
  struct sockaddr_storage storage;
  struct sockaddr_in sin;
  memset (&sin, 0, sizeof (struct sockaddr_in));
  sin.sin_family = AF_INET;

  inet_pton (AF_INET, "127.0.0.1", &(addr.s_addr));
  sin.sin_addr.s_addr = addr.s_addr;
  memcpy (&storage, &sin, sizeof (sin));
  assert_that (ip_islocalhost (&storage), is_true);

  inet_pton (AF_INET, "0.0.0.0", &(addr.s_addr));
  sin.sin_addr.s_addr = addr.s_addr;
  memcpy (&storage, &sin, sizeof (sin));
  assert_that (ip_islocalhost (&storage), is_true);

  inet_pton (AF_INET, "127.100.5.99", &(addr.s_addr));
  sin.sin_addr.s_addr = addr.s_addr;
  memcpy (&storage, &sin, sizeof (sin));
  assert_that (ip_islocalhost (&storage), is_true);

  /* dependent on local environment */
  // inet_pton (AF_INET, <some local interface address>, &(addr.s_addr));
  // sin.sin_addr.s_addr = addr.s_addr;
  // memcpy (&storage, &sin, sizeof (sin));
  // assert_that (ip_islocalhost (&storage), is_true);

  /* example.com */
  inet_pton (AF_INET, "93.184.216.34", &(addr.s_addr));
  sin.sin_addr.s_addr = addr.s_addr;
  memcpy (&storage, &sin, sizeof (sin));
  assert_that (ip_islocalhost (&storage), is_false);

  /* IPv6 */
  struct in6_addr addr_6;
  struct sockaddr_in6 sin6;
  memset (&sin6, 0, sizeof (struct sockaddr_in6));
  sin6.sin6_family = AF_INET6;

  inet_pton (AF_INET6, "::FFFF:127.0.0.1", &(addr_6));
  sin6.sin6_addr = addr_6;
  memcpy (&storage, &sin6, sizeof (sin6));
  assert_that (ip_islocalhost (&storage), is_true);

  inet_pton (AF_INET6, "::FFFF:0.0.0.0", &(addr_6));
  sin6.sin6_addr = addr_6;
  memcpy (&storage, &sin6, sizeof (sin6));
  assert_that (ip_islocalhost (&storage), is_true);

  inet_pton (AF_INET6, "::FFFF:127.100.5.99", &(addr_6));
  sin6.sin6_addr = addr_6;
  memcpy (&storage, &sin6, sizeof (sin6));
  assert_that (ip_islocalhost (&storage), is_true);

  /* loopback address */
  inet_pton (AF_INET6, "0:0:0:0:0:0:0:1", &(addr_6));
  sin6.sin6_addr = addr_6;
  memcpy (&storage, &sin6, sizeof (sin6));
  assert_that (ip_islocalhost (&storage), is_true);

  /* dependent on local environment */
  // inet_pton (AF_INET6, <some local interface address>, &(addr_6));
  // sin6.sin6_addr = addr_6;
  // memcpy (&storage, &sin6, sizeof (sin6));
  // assert_that (ip_islocalhost (&storage), is_true);

  /* example.com */
  inet_pton (AF_INET6, "2606:2800:220:1:248:1893:25c8:1946", &(addr_6));
  sin6.sin6_addr = addr_6;
  memcpy (&storage, &sin6, sizeof (sin6));
  assert_that (ip_islocalhost (&storage), is_false);
}

__attribute__ ((weak)) GIOChannel *
__real_g_io_channel_new_file (const char *filename, const char *mode,
                              GError **error);

gboolean g_g_io_channel_new_file_use_real = TRUE;
GIOChannel *
__wrap_g_io_channel_new_file (const char *filename, const char *mode,
                              GError **error)
{
  if (g_g_io_channel_new_file_use_real)
    return __real_g_io_channel_new_file (filename, mode, error);

  return (GIOChannel *) mock (filename, mode, error);
}

__attribute__ ((weak)) GIOStatus
__real_g_io_channel_shutdown (GIOChannel *channel, gboolean flush,
                              GError **err);

gboolean g_g_io_channel_shutdown_use_real = TRUE;
GIOStatus
__wrap_g_io_channel_shutdown (GIOChannel *channel, gboolean flush, GError **err)
{
  if (g_g_io_channel_shutdown_use_real)
    return __real_g_io_channel_shutdown (channel, flush, err);

  return (GIOStatus) mock (channel, flush, err);
}

Ensure (networking, get_routes)
{
  int status;
  int ret;
  GError *err = NULL;
  GIOChannel *file_channel;
  GSList *list = NULL;
  int list_len;

  /* Content for mocked file. */
  gchar *procnetroute_testfile =
    "\nIface\tDestination\tGateway "
    "\tFlags\tRefCnt\tUse\tMetric\tMask\t\tMTU\tWindow\tIRTT                   "
    "                                    "
    "\nenp0s9\t00000000\t01B2A8C0\t0003\t0\t0\t100\t00000000\t0\t0\t0          "
    "                                                                 "
    "\nenp0s8\t0038A8C0\t00000000\t0001\t0\t0\t0\t00FFFFFF\t0\t0\t0            "
    "                                                                 "
    "\nenp0s9\t00B2A8C0\t00000000\t0001\t0\t0\t100\t00FFFFFF\t0\t0\t0          "
    "                                                                 \n";

  /* Create mock file. */
  file_channel = g_io_channel_new_file ("./myfile", "w+", NULL);
  if (!file_channel)
    g_warning ("Not my channel");
  g_io_channel_write_chars (file_channel, procnetroute_testfile, -1, NULL,
                            NULL);
  g_io_channel_seek_position (file_channel, 0, G_SEEK_SET, NULL);

  /* Test get_routes with mocked "/proc/net/route". */
  g_g_io_channel_new_file_use_real = false;
  g_g_io_channel_shutdown_use_real = false;
  expect (__wrap_g_io_channel_new_file,
          when (filename, is_equal_to_string ("/proc/net/route")),
          will_return (file_channel));
  expect (__wrap_g_io_channel_shutdown, will_return (G_IO_STATUS_NORMAL));
  list = get_routes ();
  list_len = g_slist_length (list);
  assert_that (list, is_not_null);
  assert_that (list_len, is_equal_to (3));
  g_g_io_channel_new_file_use_real = true;
  g_g_io_channel_shutdown_use_real = true;

  /* Close channel and delete mock file. */
  status = g_io_channel_shutdown (file_channel, TRUE, &err);
  if ((G_IO_STATUS_NORMAL != status) || err)
    g_warning ("%s: Could not shutdown channel.", __func__);
  ret = unlink ("./myfile");
  if (0 != ret)
    g_warning ("%s: Could not delete file \"./myfile\");", __func__);
}

Ensure (networking, gvm_routethrough_v4)
{
  struct in_addr dst;
  struct sockaddr_storage storage_src;
  struct sockaddr_storage storage_dst;
  struct sockaddr_in sin_src;
  struct sockaddr_in sin_dst;
  struct sockaddr_in sin_always_empty;
  memset (&sin_src, 0, sizeof (struct sockaddr_in));
  memset (&sin_dst, 0, sizeof (struct sockaddr_in));
  memset (&sin_always_empty, 0, sizeof (struct sockaddr_in));
  sin_src.sin_family = AF_INET;
  sin_dst.sin_family = AF_INET;
  gchar *interface = NULL;

  /* No destination address. */
  interface = gvm_routethrough (NULL, &storage_src);
  assert_that (interface, is_null);

  /* Destination address localhost and no source address. */
  inet_pton (AF_INET, "127.0.0.1", &(dst.s_addr));
  sin_dst.sin_addr.s_addr = dst.s_addr;
  memcpy (&storage_dst, &sin_dst, sizeof (sin_dst));
  interface = gvm_routethrough (&storage_dst, NULL);
  assert_that (interface, is_not_null);
  /* Dependent on local environment. */
  // assert_that (interface, is_equal_to_string ("lo"));

  /* Destination address not localhost and no source address. */
  inet_pton (AF_INET, "93.184.216.34", &(dst.s_addr)); // example.com
  sin_dst.sin_addr.s_addr = dst.s_addr;
  memcpy (&storage_dst, &sin_dst, sizeof (sin_dst));
  interface = gvm_routethrough (&storage_dst, NULL);
  assert_that (interface, is_not_null);
  /* Dependent on local environment. */
  // assert_that (interface, is_equal_to_string ("enp0s9"));

  /* Destination address localhost and source address */
  inet_pton (AF_INET, "127.0.0.1", &(dst.s_addr));
  sin_dst.sin_addr.s_addr = dst.s_addr;
  memcpy (&storage_dst, &sin_dst, sizeof (sin_dst));
  interface = gvm_routethrough (&storage_dst, &storage_src);
  assert_that (interface, is_not_null);
  assert_that (((struct sockaddr_in *) (&storage_src))->sin_addr.s_addr
               == htonl (0x7F000001));
  /* Dependent on local environment. */
  // assert_that (interface, is_equal_to_string ("lo"));

  /* Dst address not localhost and src address */
  inet_pton (AF_INET, "93.184.216.34", &(dst.s_addr));
  sin_dst.sin_addr.s_addr = dst.s_addr;
  memcpy (&storage_dst, &sin_dst, sizeof (sin_dst));
  memcpy (&storage_src, &sin_always_empty, sizeof (struct sockaddr_in));
  interface = gvm_routethrough (&storage_dst, &storage_src);
  assert_that (interface, is_not_null);
  assert_that (((struct sockaddr_in *) (&storage_src))->sin_addr.s_addr
               != htonl (0x7F000001));
  /* Dependent on local environment. */
  // assert_that (((struct sockaddr_in *) (&storage_src))->sin_addr.s_addr !=
  // 0); assert_that (interface, is_equal_to_string ("enp0s9"));
}

Ensure (networking, gvm_source_addr)
{
  struct in_addr src;

  /* global source address not set */
  gvm_source_iface_init (NULL);
  gvm_source_addr (&src);
  assert_that ((src.s_addr == INADDR_ANY));

  /* global source address */
  gvm_source_iface_init ("lo");
  gvm_source_addr (&src);
  assert_that ((src.s_addr != INADDR_ANY));
}

TestSuite *
gvm_routethough ()
{
  TestSuite *suite = create_test_suite ();
  add_test_with_context (suite, networking, ip_islocalhost);
  add_test_with_context (suite, networking, get_routes);
  add_test_with_context (suite, networking, gvm_routethrough_v4);

  return suite;
}

int
main (int argc, char **argv)
{
  TestSuite *suite;

  suite = create_test_suite ();
  add_suite (suite, gvm_routethough ());

  add_test_with_context (suite, networking, gvm_source_addr);
  add_test_with_context (suite, networking, validate_port_range);
  add_test_with_context (suite, networking, port_range_ranges);
  add_test_with_context (suite, networking, port_in_port_ranges);

  if (argc > 1)
    return run_single_test (suite, argv[1], create_text_reporter ());

  return run_test_suite (suite, create_text_reporter ());
}
