package latexDraw.figures;

import java.awt.*;
import java.awt.geom.*;
import java.io.IOException;
import java.io.ObjectInputStream;

import latexDraw.figures.properties.BordersMovable;
import latexDraw.figures.properties.DoubleBoundaryable;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.util.LaTeXDrawNumber;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines an Ellipse.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Ellipse extends Figure implements BordersMovable, DoubleBoundaryable
{
	private static final long serialVersionUID = 1L;


	/**
	 * The constructor by default
	 */
	public Ellipse(boolean increaseMeter)
	{
		this(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), increaseMeter);
	}
	
	
		
	/**
	 * The constructor using four points
	 * @param pt1 The top left point of rectangle containing the ellipse
	 * @param pt2 The top right point of the rectangle containing the ellipse
	 * @param pt3 The bottom left point
	 * @param pt4 The bottom right point
	 */
	public Ellipse(LaTeXDrawPoint2D pt1, LaTeXDrawPoint2D pt2, LaTeXDrawPoint2D pt3, LaTeXDrawPoint2D pt4, boolean increaseMeter)
	{
		super(increaseMeter);
		borders = new LaTeXDrawRectangle(pt1, pt2, pt3, pt4, false);
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	/**
	 * @param f The figure to copy
	 * @param sameNumber True if the figure will have the same number of the copy.
	 */
	public Ellipse(Figure f, boolean sameNumber)
	{
		super(f, sameNumber);
		
		try 
		{ 
			borders = new LaTeXDrawRectangle((LaTeXDrawPoint2D)f.getBordersPoint(0).clone(),
								(LaTeXDrawPoint2D)f.getBordersPoint(-1).clone(), false);
			updateGravityCenter();
			updateShape();
			LaTeXDrawPoint2D nw = borders.getTheNWPoint();
			LaTeXDrawPoint2D cg = getGravityCenter();
			double c = Math.abs(cg.x-nw.x)*3./2.25;
			double a = (Math.sqrt((cg.x-nw.x+c)*(cg.x-nw.x+c)+((cg.y-nw.y)*(cg.y-nw.y)))+
						Math.sqrt((cg.x-nw.x-c)*(cg.x-nw.x-c)+((cg.y-nw.y)*(cg.y-nw.y))))/2.;
			double b = Math.sqrt(a*a-c*c);
			borders.setFirstPoint(cg.x-a, cg.y-b);
			borders.setLastPoint(cg.x+a, cg.y+b);
		} 
		catch (Exception e)
		{ borders = new LaTeXDrawRectangle(false); }
		
		updateGravityCenter();
		updateShape();
	}



	@Override
	public void onDragged(Point formerPt, Point newPt) 
	{
		if(formerPt.equals(newPt)) return;

		borders.onDragged(formerPt, newPt);
		rotationAngle = borders.getRotationAngle();//update the angle(when rotation)
		updateGravityCenter();//update centre of gravity when reshaping
		shape = getInsideOutsideOrMiddleBorders();
	}

	
	
	
	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		LaTeXDrawPoint2D NW = borders.getTheNWPoint(), SE = borders.getTheSEPoint();
		Color formerCol = g.getColor();
		double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;
		double c2x = Math.cos(rotationAngle)*cx - Math.sin(rotationAngle)*cy;
		double c2y = Math.sin(rotationAngle)*cx + Math.cos(rotationAngle)*cy;
		double c3x = Math.cos(-rotationAngle)*(cx-c2x) - Math.sin(-rotationAngle)*(cy-c2y);
		double c3y = Math.sin(-rotationAngle)*(cx-c2x) + Math.cos(-rotationAngle)*(cy-c2y);
		double dx=0, dy=0;
		boolean changeFillStyle = false;

		if(rotationAngle%(Math.PI*2)!=0)
		{		
			g.rotate(rotationAngle);
			g.translate(c3x,c3y);
		}
		
		if(hasShadow)
		{
			LaTeXDrawPoint2D cg = getGravityCenter();
			LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
			shadowCg.setLocation(cg.x+shadowSize, cg.y);
			shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
			dx = shadowCg.x-cg.x;
			dy = cg.y-shadowCg.y;
		}

		if(hasDoubleBoundary)
		{
			Shape s0 = shape;
			Shape s[] = getDbleBoundariesOutInOrMiddle(s0);
			Shape s1, s2, s3;

			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			{
				s1 = s0;
				s2 = s[0];
				s3 = s[1];
			}
			else
				if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				{
					s1 = s[0];
					s2 = s0;
					s3 = s[1];
				}
				else
				{
					s1 = s[0];
					s2 = s[1];
					s3 = s0;
				}
			
			Shape sTooSmall = getTooSmallShape(s1);

			if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
			{
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				if(hasShadow)
				{
					g.translate(dx, dy);
					g.setColor(shadowColor);
					g.fill(s1);
					
					if(sTooSmall==null)
						g.draw(s1);
					else
						g.draw(sTooSmall);
					
					g.translate(-dx, -dy);
					
					if(!isFilled)
					{
						changeFillStyle = true;
						isFilled = true;
					}
				}
				
				g.setColor(doubleColor);
				g.setStroke(new BasicStroke((float)(doubleSep + thickness), 
								BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.draw(s2);
				fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, s3);
				g.setColor(linesColor);
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				
				if(sTooSmall==null)
					g.draw(s1);
				else
					g.draw(sTooSmall);
				
				g.draw(s3);
			}
			else
			{
				if(hasShadow)
				{
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.translate(dx, dy);
					g.setColor(shadowColor);
					g.fill(s1);
					
					if(sTooSmall==null)
						g.draw(s1);
					else
						g.draw(sTooSmall);
					
					g.translate(-dx, -dy);
					g.setColor(interiorColor);
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep), 
							BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.draw(s2);
					
					if(!isFilled)
					{
						changeFillStyle = true;
						isFilled = true;
					}
				}
				
				if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep), BasicStroke.CAP_ROUND,
							BasicStroke.JOIN_MITER, 1.f, new float[] { 0, (float)(thickness*2+doubleSep + dotSep) }, 0));
				else
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep),
							BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.f,
							new float[] { blackDashLength, whiteDashLength }, 0));
					
				fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, s2);
				g.setColor(linesColor);
				
				Shape sTooSmall2 = getTooSmallShape(s2);
				if(sTooSmall2==null)
					g.draw(s2);
				else
					g.draw(sTooSmall2);
				
				g.setStroke(new BasicStroke((float)doubleSep, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.setColor(doubleColor);
				
				if(sTooSmall2==null)
					g.draw(s2);
				else
					g.draw(sTooSmall2);
			}				
		}
		else
		{
			Shape sTooSmall = getTooSmallShape(shape);
			
			if(hasShadow)
			{
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.translate(dx, dy);
				g.setColor(shadowColor);
				g.fill(shape);
				
				if(sTooSmall==null)
					g.draw(shape);
				else
					g.draw(sTooSmall);
				
				g.translate(-dx, -dy);
				if(!isFilled)
				{
					changeFillStyle = true;
					isFilled = true;
				}
				g.setColor(interiorColor);
				g.draw(shape);
			}
			
			if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			else
				if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
				{
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_ROUND,
								BasicStroke.JOIN_MITER, 1.f, new float[] { 0, thickness + dotSep }, 0));
				}
				else
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.f,
								new float[] { blackDashLength, whiteDashLength }, 0));
			
			fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, shape);
			g.setColor(linesColor);
			
			if(sTooSmall==null)
				g.draw(shape);
			else
				g.draw(sTooSmall);
		}
		 
		if(changeFillStyle) isFilled = false;
		g.setColor(formerCol);

		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.translate(-c3x, -c3y);
			g.rotate(-rotationAngle);
		}
		
		if(isSelected)
			borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
	}

	
		
	
	@Override
	public synchronized void setLastPoint(double x, double y)
	{
		borders.setLastPoint(x, y);
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}

	
	
	@Override
	public synchronized void setFirstPoint(double x, double y)
	{
		borders.setFirstPoint(x, y);
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	@Override
	public synchronized void setBordersPosition(String doubleLinePosition)
	{
		super.setBordersPosition(doubleLinePosition);
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	@Override
	public boolean isIn(LaTeXDrawPoint2D p) 
	{		
		LaTeXDrawPoint2D pt = rotateInvertPoint(p);
		
		if(isSelected && (borders.dNE.isIn(pt) || borders.dNW.isIn(pt) || borders.dSE.isIn(pt) || 
			borders.dSW.isIn(pt) || borders.dS.isIn(pt)  || borders.dN.isIn(pt)   || borders.dE.isIn(pt)  || borders.dW.isIn(pt)))
			return true;
		
		LaTeXDrawPoint2D NW = getTheNWNonRotatedBoundPoint();
		LaTeXDrawPoint2D SE = getTheSENonRotatedBoundPoint();
		Shape s = new Ellipse2D.Double(NW.x, NW.y, Math.abs(NW.x - SE.x), Math.abs(NW.y - SE.y));

		Rectangle2D bounds = s.getBounds2D();
	
		if(bounds.getHeight()<=1)
		{
			Line l;
			if(bounds.getWidth()<=1)
				l = new Line(bounds.getMaxX(), Math.max(1, bounds.getMaxY()), bounds.getMaxX(), Math.max(1, bounds.getMaxY()), false);
			else
				l = new Line(bounds.getMinX(), Math.max(1, bounds.getMinY()), bounds.getMaxX(), Math.max(1, bounds.getMaxY()), false);
			
			return l.isIn(pt);
		}
		
		if(bounds.getWidth()<=1)
		{
			Line l = new Line(Math.max(1, bounds.getMaxX()), bounds.getMinY(), Math.max(1, bounds.getMaxX()), bounds.getMaxY(), false);
			
			return l.isIn(pt);
		}
		
		if(!s.contains(pt))
			return false;

		if(isFilled || hasShadow || hasGradient())
			return true;

		Shape s2;

		if(hasDoubleBoundary)
			s2 = new Ellipse2D.Double(NW.x + thickness * 2 + doubleSep, NW.y + thickness * 2 + doubleSep, 
						Math.abs(NW.x - SE.x)- 4 * thickness - 2 * doubleSep, Math.abs(NW.y- SE.y) - 4 * thickness - 2 * doubleSep);
		else
			s2 = new Ellipse2D.Double(NW.x + thickness, NW.y + thickness, Math.abs(NW.x - SE.x) - 2 * thickness, 
									Math.abs(NW.y- SE.y)- 2 * thickness);

		return !s2.contains(pt);
	}

	
	
	
	@Override
	public String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		LaTeXDrawPoint2D pt1 = borders.getPoint(0), pt2 = borders.getPoint(1);
		LaTeXDrawPoint2D pt3 = borders.getPoint(2);
		String add = "", addBegin="", addEnd="", fillType=""; //$NON-NLS-1$ //$NON-NLS-3$ //$NON-NLS-2$ //$NON-NLS-4$
		double x = (pt1.x+pt2.x)/2. - d.x, y = d.y - (pt1.y+pt3.y)/2. ;
		double width  = Math.abs(pt1.x-pt2.x);
		double height = Math.abs(pt1.y-pt3.y);
		boolean isFilledWasChanged = false;
		double threshold = 0.001;
		
		if(hasShadow)
		{
			fillType+=",shadow=true";//$NON-NLS-1$
			if(Math.toDegrees(shadowAngle)!=PSTricksConstants.DEFAULT_SHADOW_ANGLE)
				fillType+=",shadowangle="+(float)Math.toDegrees(shadowAngle);//$NON-NLS-1$
			
			if(((float)shadowSize)!=((float)DEFAULT_SHADOW_SIZE))
				fillType+=",shadowsize="+LaTeXDrawNumber.getCutNumber((float)(shadowSize/PPC),threshold);//$NON-NLS-1$
			
			if(!shadowColor.equals(PSTricksConstants.DEFAULT_SHADOW_COLOR))
			{
				String name = DviPsColors.getColourName(shadowColor);
				if(name==null)
				{
					name = "color"+number+'e';//$NON-NLS-1$
					DviPsColors.addUserColour(shadowColor, name); 
				}
				add += ",shadowcolor=" + name; //$NON-NLS-1$
			}
			if(!isFilled)
			{
				isFilled = true;
				isFilledWasChanged = true;
			}
		}
		
		String str = getPSTricksCodeFilling(ppc);
		if(str.length()>0) fillType=fillType+','+str;
		
		str = getPSTricksCodeLine(ppc);
		if(str.length()>0) add=add+','+str;
		
		if(rotationAngle%(Math.PI*2)!=0.)
		{
			double angle = -Math.toDegrees(rotationAngle);
			double cx = (gravityCenter.x-d.x)/ppc;
			double cy = (d.y-gravityCenter.y)/ppc;
			double x2 = -Math.cos(-rotationAngle)*cx+
						Math.sin(-rotationAngle)*cy+cx;
			double y2 =  -Math.sin(-rotationAngle)*cx-
						Math.cos(-rotationAngle)*cy+cy;

			addBegin +="\\rput{"+(float)angle+ "}("+LaTeXDrawNumber.getCutNumber((float)x2,threshold)+','+LaTeXDrawNumber.getCutNumber((float)y2,threshold)+"){"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			addEnd = "}"; //$NON-NLS-1$
		}
		
		add+=",dimen="+bordersPosition;  //$NON-NLS-1$
		
		if(hasDoubleBoundary)
		{
			add+=",doubleline=true,doublesep="+LaTeXDrawNumber.getCutNumber((float)(doubleSep/ppc),threshold); //$NON-NLS-1$
			
			if(doubleColor!=PSTricksConstants.DEFAULT_DOUBLE_COLOR)
			{
				String name = DviPsColors.getColourName(doubleColor);
				if(name==null)
				{
					name = "color"+number+'d';//$NON-NLS-1$
					DviPsColors.addUserColour(doubleColor, name); 
				}
				add+= ",doublecolor="+name; //$NON-NLS-1$
			}
		}
		
		if(isFilledWasChanged) isFilled = false;
		
		return addBegin+"\\psellipse[linewidth=" + (thickness/ppc) +  //$NON-NLS-1$
			add + fillType + "](" + LaTeXDrawNumber.getCutNumber((float)(x/ppc),threshold) + ',' + LaTeXDrawNumber.getCutNumber((float)(y/ppc),threshold) + ")("  //$NON-NLS-1$ //$NON-NLS-2$
				+ LaTeXDrawNumber.getCutNumber((float)((width/2.)/ppc),threshold) + ',' + LaTeXDrawNumber.getCutNumber((float)((height/2.)/ppc),threshold) + ')'+addEnd;
	}



	@Override
	public void shift(double shiftX,double shiftY)
	{
		if(shiftX==0 && shiftY==0) return ;

		borders.shift(shiftX, shiftY);
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}




	@Override
	public Object clone() throws CloneNotSupportedException
	{
		Ellipse e = (Ellipse) super.clone();
		e.borders = (LaTeXDrawRectangle) borders.clone();
		e.gravityCenter = (LaTeXDrawPoint2D)gravityCenter.clone();
		e.shape = getInsideOutsideOrMiddleBorders();
		
		return e;
	}

	
	
	@Override
	public Shape createShape2D()
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();
		Shape area = createNonRotatedShape2D();

		if(rotationAngle%(Math.PI*2)!=0)
		{
			double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;

			double c2x = Math.cos(rotationAngle)*cx-Math.sin(rotationAngle)*cy;
			double c2y = Math.sin(rotationAngle)*cx+Math.cos(rotationAngle)*cy;

			AffineTransform at = AffineTransform.getTranslateInstance(cx-c2x, cy-c2y);
			at.rotate(rotationAngle);
			area = at.createTransformedShape(area);
		}

		return area;
	}




	@Override
	public void rescaleX(double formerX, double newX, double percent, LaTeXDrawRectangle bound) 
	{
		if(percent==1.) return ;
		
		borders.rescaleX(formerX, newX, percent, bound);
		updateGravityCenter();//update centre of gravity when reshaping
		shape = getInsideOutsideOrMiddleBorders();
	}




	@Override
	public void rescaleY(double formerY, double newY, double percent, LaTeXDrawRectangle bound) 
	{
		if(percent==1.) return ;
		
		borders.rescaleY(formerY, newY, percent, bound);
		updateGravityCenter();//update centre of gravity when reshaping
		shape = getInsideOutsideOrMiddleBorders();
	}

	
	
	
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		canHaveShadow = true;
		interiorColor = (Color) ois.readObject();
		lineStyle = (String) ois.readObject();
		rotationAngle = ois.readDouble();
		thickness = ois.readFloat();
		isFilled = ois.readBoolean();
		isSelected = ois.readBoolean();
		isOnRotation = ois.readBoolean();
		linesColor = (Color) ois.readObject();
		blackDashLength = ois.readFloat();
		dotSep = ois.readFloat();
		whiteDashLength = ois.readFloat();
		borders = (LaTeXDrawRectangle) ois.readObject();

		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.5")>=0)//$NON-NLS-1$
		{
			hasDoubleBoundary = ois.readBoolean();
			doubleColor = (Color)ois.readObject();
			doubleSep = ois.readDouble();
			bordersPosition = (String)ois.readObject();
			if(!(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")>=0)) //$NON-NLS-1$
				ois.readBoolean();
			hatchingAngle = ois.readDouble();
			hatchingColor = (Color)ois.readObject();
			hatchingStyle = (String)ois.readObject();
			hatchingWidth = ois.readFloat();
			
			if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6") < 0)//$NON-NLS-1$
			{
				if(hatchingStyle.equals(DECREPETED_FILL_CROSS))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_CROSSHATCH;
				else if(hatchingStyle.equals(DECREPETED_FILL_HORIZ))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_HLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_VERT))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_VLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_NO))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_NONE;
			}
		}
		else
		{
			hasDoubleBoundary  = DEFAULT_HAS_DOUBLE_BOUNDARY;
			doubleColor = DEFAULT_DOUBLE_COLOR;
			doubleSep   = DEFAULT_DOUBLESEP;
			bordersPosition = DEFAULT_BORDERS_POSITION;
			hatchingAngle = DEFAULT_HATCH_ANGLE;
			hatchingColor = DEFAULT_HATCH_COL;
			hatchingStyle = DEFAULT_HATCH_STYLE;
			hatchingWidth = DEFAULT_HATCH_WIDTH;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.7")>=0) //$NON-NLS-1$
		{
			hasShadow 	= ois.readBoolean();
			shadowAngle = ois.readDouble();
			shadowSize	= ois.readDouble();
			shadowColor	= (Color)ois.readObject();
			gradientEndColor = (Color)ois.readObject();
			gradientStartColor = (Color)ois.readObject();
			gradientAngle = ois.readDouble();
			gradientMidPoint = ois.readDouble();
		}
		else
		{
			hasShadow 	= DEFAULT_SHADOW_HAS;
			shadowAngle	= DEFAULT_SHADOW_ANGLE;
			shadowSize	= DEFAULT_SHADOW_SIZE;
			shadowColor	= DEFAULT_SHADOW_COLOR;
			gradientEndColor = PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
			gradientStartColor = PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
			gradientAngle = DEFAULT_GRADIENT_ANGLE;
			gradientMidPoint = DEFAULT_GRADIENT_MID_POINT;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.8")>=0) //$NON-NLS-1$
			hatchingSep = ois.readDouble();
		else
			hatchingSep = DEFAULT_HATCH_SEP;
		
		updateGravityCenter();
		updateStyleOfDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	/**
	 * Allows to create the tangent to the ellipse at the point point
	 * @param angle The position of the tangent point in radian
	 * @param orientation Change the orientation of the tangent
	 * @return The tangent
	 */
	public Line2D.Double getTangenteAt(double angle, boolean orientation)
	{
		Line2D.Double l = new Line2D.Double();
		LaTeXDrawPoint2D NW = getTheNWPoint();
		LaTeXDrawPoint2D SE = getTheSEPoint();
		LaTeXDrawPoint2D grav = getGravityCenter();
		LaTeXDrawPoint2D pt = rotatePoint(new LaTeXDrawPoint2D(SE.x, (SE.y+NW.y)/2.), -angle);
		double a = Math.abs(NW.x-grav.x);
		double b = Math.abs(NW.y-grav.y);
		double dec = 5.;
		
		l.x1 = pt.x;
		l.y1 = pt.y;
	
		if(((float)angle)%((float)Math.PI)<=0.01)//0. is not possible
		{
			l.x2 = pt.x;
			if(orientation)
				l.y2 = pt.y - dec;
			else
				l.y2 = pt.y + dec;
		}
		else
		{
			if(orientation)
				l.x2 = pt.x-dec;
			else
				l.x2 = pt.x+dec;
			
			if(((float)angle)%((float)Math.PI/2f)<=0.01)//0. is not possible
				l.y2 = pt.y;
			else
				l.y2 = - (b*(pt.x-grav.x)*(l.x2-pt.x))/(a*(pt.y-grav.y)) + pt.y;
		}
		
		return l;
	}




	public Shape getInsideBorders()
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();

		return new Ellipse2D.Double(NW.x + thickness / 2., NW.y + thickness/ 2., 
							Math.max(Math.abs(NW.x - SE.x) - thickness, 1), Math.max(Math.abs(NW.y - SE.y) - thickness, 1));
	}

	


	public Shape getOutsideBorders()
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();

		return new Ellipse2D.Double(NW.x - thickness / 2., NW.y - thickness/ 2., 
							Math.max(Math.abs(NW.x - SE.x) + thickness, 1), Math.max(Math.abs(NW.y - SE.y) + thickness, 1));
	}

	


	public Shape getMiddleBorders()
	{
		LaTeXDrawPoint2D NW = getTheNWPoint(), SE = getTheSEPoint();

		return new Ellipse2D.Double(NW.x, NW.y, Math.max(Math.abs(NW.x - SE.x), 1), Math.max(Math.abs(NW.y - SE.y), 1));
	}




	public Shape getInsideOutsideOrMiddleBorders()
	{
		Shape s;

		if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			s = getInsideBorders();
		else
			if(bordersPosition.equals(PSTricksConstants.BORDERS_OUTSIDE))
				s = getOutsideBorders();
			else
				s = getMiddleBorders();

		return s;
	}





	public Shape[] getDbleBoundariesOutside(Shape classicBord)
	{
		if(classicBord == null)
			return null;

		Ellipse2D.Double r = (Ellipse2D.Double)classicBord;
		Shape[] s = new Shape[2];

		s[0] = new Ellipse2D.Double(r.x - doubleSep - thickness, r.y- doubleSep - thickness, 
									r.width + 2*(doubleSep + thickness), r.height + 2*(doubleSep + thickness));
		s[1] = new Ellipse2D.Double(r.x - (doubleSep + thickness) / 2., r.y - (doubleSep + thickness) / 2.,
									r.width + doubleSep+ thickness, r.height + doubleSep + thickness);

		return s;
	}




	public Shape[] getDbleBoundariesInside(Shape classicBord)
	{
		if(classicBord == null)
			return null;

		Ellipse2D.Double r = (Ellipse2D.Double)classicBord;
		Shape[] s = new Shape[2];

		s[0] = new Ellipse2D.Double(r.x + (doubleSep + thickness) / 2., r.y + (doubleSep + thickness) / 2., 
									r.width - doubleSep- thickness, r.height - doubleSep - thickness);
		s[1] = new Ellipse2D.Double(r.x + doubleSep + thickness, r.y + doubleSep + thickness, 
					r.width - 2*(doubleSep + thickness), r.height - 2 * (doubleSep + thickness));

		return s;
	}




	public Shape[] getDbleBoundariesMiddle(Shape classicBord)
	{
		if(classicBord == null)
			return null;

		Ellipse2D.Double r = (Ellipse2D.Double)classicBord;
		Shape[] s = new Shape[2];
		double add = (doubleSep + thickness) / 2.;

		s[0] = new Ellipse2D.Double(r.x - add, r.y - add, r.width + doubleSep + thickness, r.height + doubleSep + thickness);
		s[1] = new Ellipse2D.Double(r.x + add, r.y + add, r.width - doubleSep - thickness, r.height - doubleSep - thickness);

		return s;
	}




	public Shape[] getDbleBoundariesOutInOrMiddle(Shape classicBord)
	{
		Shape[] s;

		if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			s = getDbleBoundariesInside(classicBord);
		else
			if(bordersPosition.equals(PSTricksConstants.BORDERS_OUTSIDE))
				s = getDbleBoundariesOutside(classicBord);
			else
				s = getDbleBoundariesMiddle(classicBord);

		return s;
	}


	


	@Override
	public Shape createNonRotatedShape2D()
	{
		Shape area;
		Shape s = getInsideOutsideOrMiddleBorders();

		if(hasDoubleBoundary)
		{
			Shape[] s2 = getDbleBoundariesOutInOrMiddle(s);
			Shape min;
			Shape max;

			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			{
				max = s;
				min = s2[1];
			}
			else
				if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				{
					max = s2[0];
					min = s2[1];
				}
				else
				{
					max = s2[0];
					min = s;
				}

			area = new Area(max);
			((Area)area).exclusiveOr(new Area(min));
			
			Shape tooSmallShape = getTooSmallShape(area);
			
			if(tooSmallShape!=null)
			{
				Rectangle2D bounds = max.getBounds2D();
				area = new Line2D.Double(Math.max(1, bounds.getMinX()), Math.max(1, bounds.getMinY()),
						Math.max(1, bounds.getMaxX()), Math.max(1, bounds.getMaxY()));
			}
		}
		else
			area = s;

		return area;
	}




	@Override
	public synchronized void setThickness(float value)
	{
		super.setThickness(value);
		shape = getInsideOutsideOrMiddleBorders();
	}



	@Override
	public void updateShape()
	{
		shape = getInsideOutsideOrMiddleBorders();
	}



	
	@Override
	public boolean isTooSmallToBeRescaled()
	{
		return borders.isTooSmallToBeRescaled();
	}



	
	
	@Override
	public Shape createShadowShape()
	{
		if(!canHaveShadow || !hasShadow) return shape;
		
		Rectangle2D b = createShape2D().getBounds2D();
		double dx=0, dy=0;
		LaTeXDrawPoint2D cg = getGravityCenter();
		LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
		
		shadowCg.setLocation(cg.x+shadowSize, cg.y);
		shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
		dx = shadowCg.x-cg.x;
		dy = cg.y-shadowCg.y;
		
		return new Ellipse2D.Double(b.getX()+dx-thickness/2.,b.getY()+dy-thickness/2.,
									b.getWidth()+thickness, b.getHeight()+thickness);
	}



	@Override
	public void mirrorHorizontal(LaTeXDrawPoint2D origin)
	{
		borders.mirrorHorizontal(origin);
		updateShape();
	}



	@Override
	public void mirrorVertical(LaTeXDrawPoint2D origin)
	{
		borders.mirrorVertical(origin);
		updateShape();
	}



	@Override
	public synchronized LaTeXDrawPoint2D getLastPoint()
	{
		return borders.getLastPoint();
	}



	@Override
	public void updateToGrid(MagneticGrid grid)
	{
		borders.updateToGrid(grid);
		updateShape();
	}



	
	@Override
	public int getSelectedDelimitorOrientation()
	{
		return borders.getSelectedDelimitorOrientation();
	}
	
	
	
	@Override
	public int hashCode()
	{
		return (int)(super.hashCode()*1.5);
	}
}
