# Copyright (C) 2010 Canonical
#
# Authors:
#  Jean-Baptiste Lallement
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; version 3.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

"""Config file management

This is an extend for SafeConfigParser. The configuration file has a structure
similar to what you would find on Microsoft Windows INI files.

[section]
key = value


Additional configuration files can be included with the special 'include' key
example
[section]
key1 = value1
include = ~/config01

Only one include and one configuration file can be specified by include directive
"""

import ConfigParser
import os

MAGO_CONFIG_FILE = os.path.join(os.path.expanduser('~'), '.magorc')

class MagoConfig(ConfigParser.SafeConfigParser):
    """This class loads one or many configuration files"""

    # List of configuration files loaded
    config_files = []

    def __init__(self, file):
        """Constructor

        :param config: Configuration file name
        """
        ConfigParser.SafeConfigParser.__init__(self)
        if not os.path.exists(os.path.dirname(file)):
            os.makedirs(os.path.dirname(file))
        self.config_files = []
        self.load_config(file)

    def load_config(self, file):
        """Loads a configuration file

        This method loads a configuration file using the ConfigParser.read()
        method.  Addional configuration files can be added with the 'include'
        directive.

        [section]
        key1 = value1
        include = path_to_file

        Only one include directive is allowed, the last one will be used and
        only one include file can be specified.

        Relative path in the include directive are relative to the current
        directory not the directory of the parent configuration file.
        """
        files = []
        try:
            files = self.read(file)
        except:
            # don't crash on a corrupted config file
            pass

        if not files:
            return

        self.config_files.extend(files)
        for section in self.sections():
            if self.has_option(section, 'include'):
                inc = os.path.abspath(os.path.expanduser(self.get(section, 'include')))
                if not inc in self.config_files:
                    self.load_config(inc)


def get_config(configfile = None):
    """ get the global config class

    :param configFile: Path to the configuration file. If no configuration file
                       is specificied then the value MAGO_CONFIG_FILE will be
                       used (default to ~/.magorc)"""
    if not configfile:
        configfile = MAGO_CONFIG_FILE
    return MagoConfig(configfile)
