/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2009 Sun Microsystems, Inc.
 */

package org.netbeans.modules.remote.impl.fs;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.StringWriter;
import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.net.ConnectException;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.logging.Level;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.netbeans.api.annotations.common.NonNull;
import org.netbeans.modules.dlight.libs.common.PathUtilities;
import org.netbeans.modules.nativeexecution.api.ExecutionEnvironment;
import org.netbeans.modules.nativeexecution.api.HostInfo;
import org.netbeans.modules.nativeexecution.api.util.CommonTasksSupport;
import org.netbeans.modules.nativeexecution.api.util.ConnectionManager;
import org.netbeans.modules.nativeexecution.api.util.FileInfoProvider.StatInfo.FileType;
import org.netbeans.modules.nativeexecution.api.util.HostInfoUtils;
import org.netbeans.modules.nativeexecution.api.util.ProcessUtils;
import org.netbeans.modules.remote.impl.RemoteLogger;
import org.netbeans.modules.remote.impl.fileoperations.spi.FilesystemInterceptorProvider;
import org.netbeans.modules.remote.impl.fileoperations.spi.FilesystemInterceptorProvider.FilesystemInterceptor;
import org.netbeans.modules.remote.spi.FileSystemProvider;
import org.openide.filesystems.FileChangeListener;
import org.openide.filesystems.FileEvent;
import org.openide.filesystems.FileLock;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileRenameEvent;
import org.openide.filesystems.FileUtil;
import org.openide.util.Exceptions;
import org.openide.util.NbBundle;
import org.openide.util.Parameters;

/**
 *
 * @author Vladimir Kvashin
 */
public class RemoteDirectory extends RemoteFileObjectBase {

    private static final boolean trace = Boolean.getBoolean("cnd.remote.directory.trace"); //NOI18N

    private Reference<DirectoryStorage> storageRef = new SoftReference<>(null);
    private Reference<MagicCache> magicCache = new SoftReference<>(null);

    private static final class RefLock {}
    private final Object refLock = new RefLock();

    private static final class MagicLock {}
    private final Object magicLock = new MagicLock();

    private volatile RemoteFileSystemTransport.Warmup warmup;

    /*package*/ RemoteDirectory(RemoteFileObject wrapper, RemoteFileSystem fileSystem, ExecutionEnvironment execEnv,
            RemoteDirectory parent, String remotePath, File cache) {
        super(wrapper, fileSystem, execEnv, parent, remotePath, cache);
        if (getStorageFile().exists()) {
            RemoteFileSystemTransport.registerDirectory(this);
        }
    }

    @Override
    public boolean isFolder() {
        return true;
    }

    @Override
    public boolean isData() {
        return false;
    }

    @Override
    public RemoteFileObject getFileObject(String name, String ext,  @NonNull Set<String> antiLoop) {
         return getFileObject(composeName(name, ext), antiLoop);
    }

    public final FileSystemProvider.Stat getStat(String childNameExt) throws IOException {
        DirEntry entry = getEntry(childNameExt);
        return FileSystemProvider.Stat.create(entry.getDevice(), entry.getINode());
    }

    private DirEntry getEntry(String childNameExt) throws IOException {
        try {
            DirectoryStorage storage = getDirectoryStorage(childNameExt);
            DirEntry entry = storage.getValidEntry(childNameExt);
            return entry;
        } catch (ConnectException ex) {
            throw ex;
        } catch (InterruptedIOException | ExecutionException | InterruptedException ex) {
            RemoteLogger.finest(ex, this);
            return null; // don't report
        } catch (CancellationException ex) {
            return null; // don't report
        }
    }

    /*package*/ boolean canWrite(String childNameExt) throws IOException, ConnectException {
            DirEntry entry = getEntry(childNameExt);
            return entry != null && entry.canWrite(); //TODO:rfs - check groups
    }

    /*package*/ boolean canRead(String childNameExt) throws IOException {
        DirEntry entry = getEntry(childNameExt);
        return entry != null && entry.canRead();
    }

    /*package*/ boolean canExecute(String childNameExt) throws IOException {
        DirEntry entry = getEntry(childNameExt);
        return entry != null && entry.canExecute();
    }

    @Override
    public RemoteFileObject createDataImpl(String name, String ext, RemoteFileObjectBase orig) throws IOException {
        return create(composeName(name, ext), false, orig);
    }

    @Override
    public RemoteFileObject createFolderImpl(String name, RemoteFileObjectBase orig) throws IOException {
        return create(name, true, orig);
    }

    /**
     * Called after child creation (sometimes - for now only when copying or moving) or removal.
     * TODO: call after child creation via createData/createFolder
     * @param child is NULL if the file was created (creation is always external => we don't know file object yet),
     * not null if the file was deleted
     */
    @Override
    protected void postDeleteOrCreateChild(RemoteFileObject child, DirEntryList entryList) {
        // leave old implementation for a while (under a flag, by default use new impl.)
        String childNameExt = (child == null) ? null : child.getNameExt();
        if (RemoteFileSystemUtils.getBoolean("remote.fast.delete", true)) {
            Lock writeLock = RemoteFileSystem.getLock(getCache()).writeLock();
            writeLock.lock();
            boolean sendEvents = true;
            try {
                DirectoryStorage storage = getExistingDirectoryStorage();
                if (child != null && storage == DirectoryStorage.EMPTY) {
                    Exceptions.printStackTrace(new IllegalStateException("postDeleteOrCreateChild stat is called but remote directory cache does not exist")); // NOI18N
                }
                List<DirEntry> entries;
                if (entryList == null) {
                    entries = storage.listValid(childNameExt);
                    DirectoryStorage newStorage = new DirectoryStorage(getStorageFile(), entries);
                    try {
                        newStorage.store();
                    } catch (IOException ex) {
                        Exceptions.printStackTrace(ex); // what else can we do?..
                    }
                    synchronized (refLock) {
                        storageRef = new SoftReference<>(newStorage);
                    }
                    if (child != null) {
                        getFileSystem().getFactory().invalidate(child.getPath());
                    }
                } else {
                    try {
                        updateChildren(toMap(entryList), storage, true, childNameExt, null, false);
                    } catch (IOException ex) {
                        RemoteLogger.finest(ex, this);
                    }
                    sendEvents = false;
                }
            } finally {
                writeLock.unlock();
            }
            if (sendEvents) {
                if (child != null) {
                    RemoteLogger.assertTrue(!child.isValid(), "Calling postDelete ob valid child " + child);
                    fireDeletedEvent(this.getOwnerFileObject(), child, false, true);
                }
            }
            //RemoteFileSystemTransport.scheduleRefresh(getExecutionEnvironment(), Arrays.asList(getPath()));
        } else {
            try {
                DirectoryStorage ds = refreshDirectoryStorage(childNameExt, false); // it will fire events itself
            } catch (ConnectException ex) {
                RemoteLogger.getInstance().log(Level.INFO, "Error post removing/creating child " + child, ex);
            } catch (IOException | ExecutionException ex) {
                RemoteLogger.finest(ex, this);
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                RemoteLogger.finest(ex, this);
            } catch (CancellationException ex) {
                // too late
            }
        }
    }

    @Override
    protected DirEntryList deleteImpl(FileLock lock) throws IOException {
        return RemoteFileSystemTransport.delete(getExecutionEnvironment(), getPath(), true);
    }

    private RemoteFileObject create(String name, boolean directory, RemoteFileObjectBase orig) throws IOException {
        // Have to comment this out since NB does lots of stuff in the UI thread and I have no way to control this :(
        // RemoteLogger.assertNonUiThread("Remote file operations should not be done in UI thread");
        String path = getPath() + '/' + name;
        if (name.contains("\\") || name.contains("/")) { //NOI18N
            throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                    "EXC_CannotCreateFile", getDisplayName(path))); //NOI18N
        }
        if (!ConnectionManager.getInstance().isConnectedTo(getExecutionEnvironment())) {
            throw RemoteExceptions.createConnectException(NbBundle.getMessage(RemoteDirectory.class,
                    "EXC_CantCreateNoConnect", getDisplayName(path))); //NOI18N
        }
        if (USE_VCS) {
            FilesystemInterceptorProvider.FilesystemInterceptor interceptor = FilesystemInterceptorProvider.getDefault().getFilesystemInterceptor(getFileSystem());
            if (interceptor != null) {
                try {
                    getFileSystem().setInsideVCS(true);
                    interceptor.beforeCreate(FilesystemInterceptorProvider.toFileProxy(orig.getOwnerFileObject()), name, directory);
                } finally {
                    getFileSystem().setInsideVCS(false);
                }
            }
        }
        ProcessUtils.ExitStatus res;
        if (directory) {
            res = ProcessUtils.execute(getExecutionEnvironment(), "mkdir", path); //NOI18N
        } else {
            String script = String.format("ls ./\"%s\" || touch ./\"%s\"", name, name); // NOI18N
            res = ProcessUtils.executeInDir(getPath(), getExecutionEnvironment(), "sh", "-c", script); // NOI18N
            if (res.isOK() && res.error.length() == 0) {
                creationFalure(name, directory, orig);
                throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                        "EXC_AlreadyExists", getDisplayName(path))); // NOI18N
            }
        }
        if (res.isOK()) {
            try {
                refreshDirectoryStorage(name, false);
                RemoteFileObject fo = getFileObject(name, new HashSet<String>());
                if (fo == null) {
                    creationFalure(name, directory, orig);
                    throw RemoteExceptions.createFileNotFoundException(NbBundle.getMessage(RemoteDirectory.class,
                            "EXC_CannotCreateFile", getDisplayName(path))); //NOI18N
                }
                if (USE_VCS) {
                    try {
                        getFileSystem().setInsideVCS(true);
                        getFileSystem().setBeingCreated(fo.getImplementor());
                        FilesystemInterceptorProvider.FilesystemInterceptor interceptor = FilesystemInterceptorProvider.getDefault().getFilesystemInterceptor(getFileSystem());
                        if (interceptor != null) {
                            if (this == orig) {
                                interceptor.createSuccess(FilesystemInterceptorProvider.toFileProxy(fo));
                            } else {
                                RemoteFileObject originalFO = orig.getFileObject(name, new HashSet<String>());
                                if (originalFO == null) {
                                    throw RemoteExceptions.createFileNotFoundException(NbBundle.getMessage(RemoteDirectory.class,
                                            "EXC_CannotCreateFile", getDisplayName(path))); //NOI18N
                                }
                                interceptor.createSuccess(FilesystemInterceptorProvider.toFileProxy(originalFO));
                            }
                        }
                    } finally {
                        getFileSystem().setInsideVCS(false);
                        getFileSystem().setBeingCreated(null);
                    }
                }
                return fo;
            } catch (ConnectException ex) {
                creationFalure(name, directory, orig);
                throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                        "EXC_CannotCreateFileWithReason", getDisplayName(path), "not connected"), ex); // NOI18N
            } catch (InterruptedIOException ex) {
                creationFalure(name, directory, orig);
                throw RemoteExceptions.createInterruptedIOException(NbBundle.getMessage(RemoteDirectory.class,
                        "EXC_CannotCreateFileWithReason", getDisplayName(path), "interrupted"), ex); // NOI18N
            } catch (IOException ex) {
                creationFalure(name, directory, orig);
                throw ex;
            } catch (ExecutionException ex) {
                creationFalure(name, directory, orig);
                throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                        "EXC_CannotCreateFileWithReason2", getDisplayName(path), //NOI18N
                        "exception occurred", ex.getLocalizedMessage()), ex); // NOI18N
            } catch (InterruptedException ex) {
                creationFalure(name, directory, orig);
                throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                        "EXC_CannotCreateFileWithReason", getDisplayName(path), "interrupted"), ex); // NOI18N
            } catch (CancellationException ex) {
                creationFalure(name, directory, orig);
                throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                        "EXC_CannotCreateFileWithReason", getDisplayName(path), "cancelled"), ex); // NOI18N
            }
        } else {
            creationFalure(name, directory, orig);
            throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                    "EXC_CannotCreateFileWithReason", getDisplayName(path), res.error)); // NOI18N
        }
    }

    private void creationFalure(String name, boolean directory, RemoteFileObjectBase orig) {
        if (USE_VCS) {
            try {
                getFileSystem().setInsideVCS(true);
                FilesystemInterceptorProvider.FilesystemInterceptor interceptor = FilesystemInterceptorProvider.getDefault().getFilesystemInterceptor(getFileSystem());
                if (interceptor != null) {
                    interceptor.createFailure(FilesystemInterceptorProvider.toFileProxy(getOwnerFileObject()), name, directory);
                }
            } finally {
                getFileSystem().setInsideVCS(false);
            }
        }
    }

    @Override
    public RemoteFileObject getFileObject(String relativePath, @NonNull Set<String> antiLoop) {
        Parameters.notNull("path", relativePath);
        relativePath = PathUtilities.normalizeUnixPath(relativePath);
        if ("".equals(relativePath)) { // NOI18N
            return getOwnerFileObject();
        }
        if (relativePath.startsWith("..")) { //NOI18N
            String absPath = getPath() + '/' + relativePath;
            absPath = PathUtilities.normalizeUnixPath(absPath);
            return getFileSystem().findResource(absPath, antiLoop);
        }
        if (relativePath.length()  > 0 && relativePath.charAt(0) == '/') { //NOI18N
            relativePath = relativePath.substring(1);
        }
        if (relativePath.endsWith("/")) { // NOI18N
            relativePath = relativePath.substring(0,relativePath.length()-1);
        }
        int slashPos = relativePath.lastIndexOf('/');
        if (slashPos > 0) { // can't be 0 - see the check above
            // relative path contains '/' => delegate to direct parent
            String parentRemotePath = getPath() + '/' + relativePath.substring(0, slashPos); //TODO:rfs: process ../..
            if (antiLoop != null) {
                String absPath = getPath() + '/' + relativePath;
                if (antiLoop.contains(absPath)) {
                    return null;
                }
                antiLoop.add(absPath);
            }
            String childNameExt = relativePath.substring(slashPos + 1);
            RemoteFileObject parentFileObject = getFileSystem().findResource(parentRemotePath, antiLoop);
            if (parentFileObject != null &&  parentFileObject.isFolder()) {
                RemoteFileObject result = parentFileObject.getFileObject(childNameExt, antiLoop);
                return result;
            } else {
                return null;
            }
        }
        RemoteLogger.assertTrue(slashPos == -1);
        try {
            DirectoryStorage storage = getDirectoryStorage(relativePath);
            DirEntry entry = storage.getValidEntry(relativePath);
            if (entry == null) {
                return null;
            }
            return getFileSystem().getFactory().createFileObject(this, entry).getOwnerFileObject();
        } catch (InterruptedException | InterruptedIOException | CancellationException 
                | ExecutionException | FileNotFoundException ex) {
            RemoteLogger.finest(ex, this);
            return null;
        } catch (ConnectException ex) {
            // don't report, this just means that we aren't connected
            setFlag(CONNECTION_ISSUES, true);
            RemoteLogger.finest(ex, this);
            return null;
        } catch (IOException ex) {
            Exceptions.printStackTrace(ex);
            //RemoteLogger.finest(ex);
            return null;
        }
    }

    private void fireRemoteFileObjectCreated(RemoteFileObject fo) {
        FileEvent e = new FileEvent(this.getOwnerFileObject(), fo);
        RemoteFileObjectBase delegate = fo.getImplementor();
        if (delegate instanceof RemoteDirectory) { // fo.isFolder() very slow if it is a link
            fireFileFolderCreatedEvent(getListeners(), e);
        } else if (delegate instanceof RemotePlainFile) {
            fireFileDataCreatedEvent(getListeners(), e);
        } else {
            if (delegate instanceof RemoteLinkBase) {
                RemoteLogger.warning("firing fireFileDataCreatedEvent for a link {0} [{1}]", delegate, delegate.getClass().getSimpleName());
            }
            fireFileDataCreatedEvent(getListeners(), e);
        }
//            if (fo.isFolder()) { // fo.isFolder() very slow if it is a link
//                fireFileFolderCreatedEvent(getListeners(), e);
//            } else {
//                fireFileDataCreatedEvent(getListeners(), e);
//            }
    }

    @Override
    protected RemoteFileObjectBase[] getExistentChildren() {
        return getExistentChildren(getExistingDirectoryStorage());
    }

    private DirectoryStorage getExistingDirectoryStorage() {

        DirectoryStorage storage;
        synchronized (refLock) {
            storage = storageRef.get();
        }
        if (storage == null) {
            File storageFile = getStorageFile();
            if (storageFile.exists()) {
                Lock readLock = RemoteFileSystem.getLock(getCache()).readLock();
                readLock.lock();
                try {
                    storage = DirectoryStorage.load(storageFile, getExecutionEnvironment());
                } catch (FormatException e) {
                    FormatException.reportIfNeeded(e);
                    storageFile.delete();
                } catch (InterruptedIOException e) {
                    // nothing
                } catch (FileNotFoundException e) {
                    // this might happen if we switch to different DirEntry implementations, see storageFile.delete() above
                    RemoteLogger.finest(e, this);
                } catch (IOException e) {
                    RemoteLogger.finest(e, this);
                } finally {
                    readLock.unlock();
                }
            }
        }
        return  storage == null ? DirectoryStorage.EMPTY : storage;
    }

    private RemoteFileObjectBase[] getExistentChildren(DirectoryStorage storage) {
        List<DirEntry> entries = storage.listValid();
        List<RemoteFileObjectBase> result = new ArrayList<>(entries.size());
        for (DirEntry entry : entries) {
            String path = getPath() + '/' + entry.getName();
            RemoteFileObjectBase fo = getFileSystem().getFactory().getCachedFileObject(path);
            if (fo != null) {
                result.add(fo);
            }
        }
        return result.toArray(new RemoteFileObjectBase[result.size()]);
    }

    @Override
    public RemoteFileObject[] getChildren() {
        try {
            DirectoryStorage storage = getDirectoryStorage(null);
            List<DirEntry> entries = storage.listValid();
            RemoteFileObject[] childrenFO = new RemoteFileObject[entries.size()];
            for (int i = 0; i < entries.size(); i++) {
                DirEntry entry = entries.get(i);
                childrenFO[i] = getFileSystem().getFactory().createFileObject(this, entry).getOwnerFileObject();
            }
            return childrenFO;
        } catch (InterruptedException | InterruptedIOException | 
                FileNotFoundException | CancellationException | ExecutionException ex ) {
            // InterruptedException:
            //      don't report, this just means that we aren't connected
            //      or just interrupted (for example by FileChooser UI)
            //      or cancelled
            // ExecutionException: should we report it?
            RemoteLogger.finest(ex, this);
        } catch (ConnectException ex) {
            RemoteLogger.finest(ex, this);
            // don't report, this just means that we aren't connected
            setFlag(CONNECTION_ISSUES, true);
        } catch (IOException ex) {
            RemoteLogger.info(ex, this); // undo won't show a red brick dialog, but print
        }
        return new RemoteFileObject[0];
    }

    private DirectoryStorage getDirectoryStorage(String childName) throws
            ConnectException, IOException, InterruptedException, CancellationException, ExecutionException {
        long time = System.currentTimeMillis();
        try {
            return getDirectoryStorageImpl(false, null, childName, false);
        } catch (StackOverflowError soe) { // workaround for #130929
            String text = "StackOverflowError when accessing " + getPath(); //NOI18N
            Exceptions.printStackTrace(new Exception(text, soe));
            throw new IOException(text, soe); // new IOException sic! this should never happen
        } finally {
            if (trace) {
                trace("getDirectoryStorage for {1} took {0} ms", this, System.currentTimeMillis() - time); // NOI18N
            }
        }
    }

    private DirectoryStorage refreshDirectoryStorage(String expectedName, boolean expected) throws
            ConnectException, IOException, InterruptedException, CancellationException, ExecutionException {
        long time = System.currentTimeMillis();
        try {
            return getDirectoryStorageImpl(true, expectedName, null, expected);
        } finally {
            if (trace) {
                trace("refreshDirectoryStorage for {1} took {0} ms", this, System.currentTimeMillis() - time); // NOI18N
            }
        }
    }

    private boolean isProhibited() {
        final String path = getPath();
        if (path.equals("/proc") || getPath().equals("/dev")) { //NOI18N
            return true;
        }
        if (path.equals("/run")) { //NOI18N
        if (HostInfoUtils.isHostInfoAvailable(getExecutionEnvironment())) {
                try {
                    HostInfo hi = HostInfoUtils.getHostInfo(getExecutionEnvironment());
                    if (hi.getOSFamily() == HostInfo.OSFamily.LINUX) {
                        return true;
                    }
                } catch (IOException | ConnectionManager.CancellationException ex) {
                    Exceptions.printStackTrace(ex); // should never be the case if isHostInfoAvailable retured true
                }
            }
        }
        return false;
    }

    private void warmupDirs() {
        if (RemoteFileSystemUtils.getBoolean("remote.warmup", true)) {
            setFlag(MASK_WARMUP, true);
        }
    }

    @Override
    public RemoteDirectory getParent() {
        return (RemoteDirectory) super.getParent(); // see constructor
    }

    private boolean isFlaggedForWarmup() {
        if(getFlag(MASK_WARMUP)) {
            return true;
        } else {
            RemoteDirectory p = getParent();
            if (p != null) {
                return p.isFlaggedForWarmup();
            }
        }
        return false;
    }

    private RemoteFileSystemTransport.Warmup getWarmup() {
        RemoteFileSystemTransport.Warmup w = warmup;
        if (w == null) {
            RemoteDirectory p = getParent();
            if (p != null) {
                return p.getWarmup();
            }
        }
        return w;
    }

    private Map<String, DirEntry> toMap(DirEntryList entryList) {
        Map<String, DirEntry> map = new HashMap<>();
        for (DirEntry entry : entryList.getEntries()) {
            map.put(entry.getName(), entry);
        }
        return map;
    }

    private static final AtomicInteger warmupHints = new AtomicInteger();
    private static final AtomicInteger warmupReqs = new AtomicInteger();
    private static final AtomicInteger readEntryReqs = new AtomicInteger();

    private Map<String, DirEntry> readEntries(DirectoryStorage oldStorage, boolean forceRefresh, String childName) throws IOException, InterruptedException, ExecutionException, CancellationException {
        if (isProhibited()) {
            return Collections.<String, DirEntry>emptyMap();
        }
        readEntryReqs.incrementAndGet();
        try {
            if (isFlaggedForWarmup()) {
                RemoteFileSystemTransport.Warmup w = getWarmup();
                if (forceRefresh) {
                    if (w != null) {
                        w.remove(getPath());
                    }
                } else {
                    warmupReqs.incrementAndGet();
                    DirEntryList entryList = null;
                    if (w == null) {
                        warmup = RemoteFileSystemTransport.createWarmup(getExecutionEnvironment(), getPath());
                        if (warmup != null) {
                            entryList = warmup.getAndRemove(getPath());
                        }
                    } else {
                        entryList = w.tryGetAndRemove(getPath());
                    }
                    if (entryList != null) {
                        warmupHints.incrementAndGet();
                        return toMap(entryList);
                    }
                }
            }
        } finally {
            if (RemoteLogger.getInstance().isLoggable(Level.FINEST)) {
                RemoteLogger.finest("Warmup hits: {0} of {1} (total {2} dir.read reqs)", warmupHints.get(), warmupReqs.get(), readEntryReqs.get());
            }
        }
        Map<String, DirEntry> newEntries = new HashMap<>();
        boolean canLs = canLs();
        if (canLs) {
            DirEntryList entryList = RemoteFileSystemTransport.readDirectory(getExecutionEnvironment(), getPath());
            newEntries = toMap(entryList);
        }
        if (canLs && !isAutoMount()) {
            return newEntries;
        }
        if (childName != null) {
            String absPath = getPath() + '/' + childName;
            RemoteLogger.assertTrueInConsole(!oldStorage.isKnown(childName) || forceRefresh, "should not get here: " + absPath); //NOI18N
            if (!newEntries.containsKey(childName)) {
                DirEntry entry = getSpecialDirChildEntry(absPath, childName);
                newEntries.put(entry.getName(), entry);
            }
        }
        for (DirEntry oldEntry : oldStorage.listAll()) {
            String oldChildName = oldEntry.getName();
            if (!newEntries.containsKey(oldChildName)) {
                if (forceRefresh) {
                    if (oldEntry.isValid()) {
                        String absPath = getPath() + '/' + oldChildName;
                        DirEntry newEntry = getSpecialDirChildEntry(absPath, oldChildName);
                        newEntries.put(oldChildName, newEntry);
                    }
                } else {
                    newEntries.put(oldChildName, oldEntry);
                }
            }
        }
        return newEntries;
    }

    private DirEntry getSpecialDirChildEntry(String absPath, String childName) 
            throws ConnectException, IOException, InterruptedException, ExecutionException {
        DirEntry entry;
        try {
            entry = RemoteFileSystemTransport.lstat(getExecutionEnvironment(), absPath);
        } catch (ExecutionException e) {
            if (RemoteFileSystemUtils.isFileNotFoundException(e)) {
                entry = null;
            } else {
                throw e;
            }
        }
        return (entry != null) ? entry : new DirEntryInvalid(childName);
    }

    private boolean isAutoMount() {
        return getFileSystem().isAutoMount(getPath());
    }

    private boolean canLs() {
        return canRead();
    }

    private boolean isSpecialDirectory() {
        return isAutoMount() || !canLs();
    }

    private boolean isAlreadyKnownChild(DirectoryStorage storage, String childName) {
        if (childName != null && storage != null) {
            if (!storage.isKnown(childName)) {
                if (ConnectionManager.getInstance().isConnectedTo(getExecutionEnvironment())) {
                    if (isSpecialDirectory()) {
                        return false;
                    }
                }
            }
        }
        return true;
    }

    @Override
    protected final void renameChild(FileLock lock, RemoteFileObjectBase directChild2Rename, String newNameExt, RemoteFileObjectBase orig) throws
            ConnectException, IOException, InterruptedException, CancellationException, ExecutionException {
        String nameExt2Rename = directChild2Rename.getNameExt();
        String name2Rename = directChild2Rename.getName();
        String ext2Rename = directChild2Rename.getExt();
        String path2Rename = directChild2Rename.getPath();

        checkConnection(this, true);

        Lock writeLock = RemoteFileSystem.getLock(getCache()).writeLock();
        if (trace) {trace("waiting for lock");} // NOI18N
        writeLock.lock();
        try {
            DirectoryStorage storage = getExistingDirectoryStorage();
            if (storage.getValidEntry(nameExt2Rename) == null) {
                throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                        "EXC_NotExistingChild", nameExt2Rename, getDisplayName())); // NOI18N
            }
            if (!getCache().exists()) {
                getCache().mkdirs();
                if (!getCache().exists()) {
                    throw new IOException("Can not create cache directory " + getCache()); // NOI18N   new IOException sic - should never happen
                }
            }
            if (trace) {trace("renaming");} // NOI18N
            boolean isRenamed = false;
            if (USE_VCS) {
                try {
                    getFileSystem().setInsideVCS(true);
                    FilesystemInterceptor interceptor = FilesystemInterceptorProvider.getDefault().getFilesystemInterceptor(getFileSystem());
                    if (interceptor != null) {
                        FilesystemInterceptorProvider.IOHandler renameHandler = interceptor.getRenameHandler(FilesystemInterceptorProvider.toFileProxy(orig.getOwnerFileObject()), newNameExt);
                        if (renameHandler != null) {
                            renameHandler.handle();
                            isRenamed = true;
                        }
                    }
                } finally {
                    getFileSystem().setInsideVCS(false);
                }
            }
            if (!isRenamed) {
                ProcessUtils.ExitStatus ret = ProcessUtils.executeInDir(getPath(), getExecutionEnvironment(), "mv", nameExt2Rename, newNameExt);// NOI18N
                if (!ret.isOK()) {
                    throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                            "EXC_CanNotRename", ret.error)); //NOI18N
                }
            }

            if (trace) {trace("synchronizing");} // NOI18N
            Exception problem = null;
            Map<String, DirEntry> newEntries = Collections.emptyMap();
            try {
                newEntries = readEntries(storage, true, newNameExt);
            } catch (FileNotFoundException ex) {
                throw ex;
            } catch (IOException | ExecutionException ex) {
                problem = ex;
            }
            if (problem != null) {
                if (!ConnectionManager.getInstance().isConnectedTo(getExecutionEnvironment())) {
                    // connection was broken while we read directory content - add notification
                    getFileSystem().addPendingFile(this);
                    throw RemoteExceptions.createConnectException(problem.getMessage());
                } else {
                    boolean fileNotFoundException = RemoteFileSystemUtils.isFileNotFoundException(problem);
                    if (fileNotFoundException) {
                        getFileSystem().getFactory().invalidate(this);
                        synchronized (refLock) {
                            storageRef = new SoftReference<>(DirectoryStorage.EMPTY);
                        }
                    }
                    if (!fileNotFoundException) {
                        if (problem instanceof IOException) {
                            throw (IOException) problem;
                        } else if (problem instanceof ExecutionException) {
                            throw (ExecutionException) problem;
                        } else {
                            throw new IllegalStateException("Unexpected exception class: " + problem.getClass().getName(), problem); //NOI18N
                        }
                    }
                }
            }
            getFileSystem().incrementDirSyncCount();
            Map<String, List<DirEntry>> dupLowerNames = new HashMap<>();
            boolean hasDups = false;
            boolean changed = true;
            Set<DirEntry> keepCacheNames = new HashSet<>();
            List<DirEntry> entriesToFireChanged = new ArrayList<>();
            List<DirEntry> entriesToFireChangedRO = new ArrayList<>();
            List<DirEntry> entriesToFireCreated = new ArrayList<>();
            List<RemoteFileObject> filesToFireDeleted = new ArrayList<>();
            for (DirEntry newEntry : newEntries.values()) {
                if (newEntry.isValid()) {
                    String cacheName;
                    DirEntry oldEntry = storage.getValidEntry(newEntry.getName());
                    if (oldEntry == null) {
                        cacheName = RemoteFileSystemUtils.escapeFileName(newEntry.getName());
                        if (newEntry.getName().equals(newNameExt)) {
                            DirEntry renamedEntry = storage.getValidEntry(nameExt2Rename);
                            RemoteLogger.assertTrueInConsole(renamedEntry != null, "original DirEntry is absent for " + path2Rename + " in " + this); // NOI18N
                            // reuse cache from original file
                            if (renamedEntry != null) {
                                cacheName = renamedEntry.getCache();
                                newEntry.setCache(cacheName);
                                keepCacheNames.add(newEntry);
                            }
                        } else {
                            entriesToFireCreated.add(newEntry);
                        }
                    } else {
                        if (oldEntry.isSameType(newEntry)) {
                            cacheName = oldEntry.getCache();
                            keepCacheNames.add(newEntry);
                            boolean fire = false;
                            if (!newEntry.isSameLastModified(oldEntry) || newEntry.getSize() != oldEntry.getSize()) {
                                if (newEntry.isPlainFile()) {
                                    changed = fire = true;
                                    File entryCache = new File(getCache(), oldEntry.getCache());
                                    if (entryCache.exists()) {
                                        if (trace) {trace("removing cache for updated file {0}", entryCache.getAbsolutePath());} // NOI18N
                                        entryCache.delete(); // TODO: We must just mark it as invalid instead of physically deleting cache file...
                                    }
                                }
                            }
                            if (!equals(newEntry.getLinkTarget(), oldEntry.getLinkTarget())) {
                                changed = fire = true; // TODO: we forgot old link path, probably should be passed to change event
                                getFileSystem().getFactory().setLink(this, getPath() + '/' + newEntry.getName(), newEntry.getLinkTarget());
                            }
                            if (! newEntry.isSameAccess(oldEntry)) {
                                entriesToFireChangedRO.add(newEntry);
                                changed = fire = true;
                            }
                            if (!newEntry.isDirectory() && (newEntry.getSize() != oldEntry.getSize())) {
                                changed = fire = true;// TODO: shouldn't it be the same as time stamp change?
                            }
                            if (fire) {
                                entriesToFireChanged.add(newEntry);
                            }
                        } else {
                            changed = true;
                            getFileSystem().getFactory().changeImplementor(this, oldEntry, newEntry);
                            entriesToFireChanged.add(newEntry);
                            cacheName = null; // unchanged
                        }
                    }
                    if (cacheName !=null) {
                        newEntry.setCache(cacheName);
                    }
                    String lowerCacheName = RemoteFileSystemUtils.isSystemCaseSensitive() ? newEntry.getCache() : newEntry.getCache().toLowerCase();
                    List<DirEntry> dupEntries = dupLowerNames.get(lowerCacheName);
                    if (dupEntries == null) {
                        dupEntries = new ArrayList<>();
                        dupLowerNames.put(lowerCacheName, dupEntries);
                    } else {
                        hasDups = true;
                    }
                    dupEntries.add(newEntry);
                } else {
                    changed = true;
                }
            }
            if (changed) {
                // Check for removal
                for (DirEntry oldEntry : storage.listValid()) {
                    if (!oldEntry.getName().equals(nameExt2Rename)) {
                        DirEntry newEntry = newEntries.get(oldEntry.getName());
                        if (newEntry == null || !newEntry.isValid()) {
                            RemoteFileObject removedFO = invalidate(oldEntry);
                            if (removedFO != null) {
                                filesToFireDeleted.add(removedFO);
                            }
                        }
                    }
                }
                if (hasDups) {
                    for (Map.Entry<String, List<DirEntry>> mapEntry :
                            new ArrayList<>(dupLowerNames.entrySet())) {

                        List<DirEntry> dupEntries = mapEntry.getValue();
                        if (dupEntries.size() > 1) {
                            for (int i = 0; i < dupEntries.size(); i++) {
                                DirEntry entry = dupEntries.get(i);
                                if (keepCacheNames.contains(entry)) {
                                    continue; // keep the one that already exists
                                }
                                // all duplicates will have postfix
                                for (int j = 0; j < Integer.MAX_VALUE; j++) {
                                    String cacheName = mapEntry.getKey() + '_' + j;
                                    String lowerCacheName = cacheName.toLowerCase();
                                    if (!dupLowerNames.containsKey(lowerCacheName)) {
                                        if (trace) {
                                            trace("resolving cache names conflict in {0}: {1} -> {2}", // NOI18N
                                                    getCache().getAbsolutePath(), entry.getCache(), cacheName);
                                        }
                                        entry.setCache(cacheName);
                                        dupLowerNames.put(lowerCacheName, Collections.singletonList(entry));
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
                storage = new DirectoryStorage(getStorageFile(), newEntries.values());
                storage.store();
            } else {
                storage.touch();
            }
            // always put new content in cache
            // do it before firing events, to give liseners real content
            synchronized (refLock) {
                storageRef = new SoftReference<>(storage);
            }
            // fire all event under lockImpl
            if (changed) {
                dropMagic();
                for (FileObject deleted : filesToFireDeleted) {
                    fireFileDeletedEvent(getListeners(), new FileEvent(this.getOwnerFileObject(), deleted));
                }
                for (DirEntry entry : entriesToFireCreated) {
                    RemoteFileObjectBase fo = getFileSystem().getFactory().createFileObject(this, entry);
                    fireRemoteFileObjectCreated(fo.getOwnerFileObject());
                }
                for (DirEntry entry : entriesToFireChanged) {
                    RemoteFileObjectBase fo = getFileSystem().getFactory().getCachedFileObject(getPath() + '/' + entry.getName());
                    if (fo != null) {
                        RemoteFileObject ownerFileObject = fo.getOwnerFileObject();
                        fireFileChangedEvent(getListeners(), new FileEvent(ownerFileObject, ownerFileObject, false, ownerFileObject.lastModified().getTime()));
                    }
                }
                // rename itself
                String newPath = getPath() + '/' + newNameExt;
                getFileSystem().getFactory().rename(path2Rename, newPath, directChild2Rename);
                // fire rename
                fireFileRenamedEvent(directChild2Rename.getListeners(),
                        new FileRenameEvent(directChild2Rename.getOwnerFileObject(), directChild2Rename.getOwnerFileObject(), name2Rename, ext2Rename));
                fireFileRenamedEvent(this.getListeners(),
                        new FileRenameEvent(this.getOwnerFileObject(), directChild2Rename.getOwnerFileObject(), name2Rename, ext2Rename));
                fireReadOnlyChangedEventsIfNeed(entriesToFireChangedRO);
            }
        } finally {
            writeLock.unlock();
        }
    }

    /*package */ DirEntry getDirEntry(String childName) {
        Lock writeLock = RemoteFileSystem.getLock(getCache()).writeLock();
        if (trace) {trace("waiting for lock");} // NOI18N
        writeLock.lock();
        try {
            DirectoryStorage storage = getExistingDirectoryStorage();
            if (storage == DirectoryStorage.EMPTY) {
                return null;
            }
            return storage.getValidEntry(childName);
        } finally {
            writeLock.unlock();
        }        
    }

    /*package */void updateStat(RemotePlainFile fo, DirEntry entry) {
        RemoteLogger.assertTrue(fo.getNameExt().equals(entry.getName()));
        RemoteLogger.assertTrue(fo.getParent() == this);
        RemoteLogger.assertFalse(entry.isDirectory());
        RemoteLogger.assertFalse(entry.isLink());
        Lock writeLock = RemoteFileSystem.getLock(getCache()).writeLock();
        if (trace) {trace("waiting for lock");} // NOI18N
        writeLock.lock();
        try {
            DirectoryStorage storage = getExistingDirectoryStorage();
            if (storage == DirectoryStorage.EMPTY) {
                Exceptions.printStackTrace(new IllegalStateException("Update stat is called but remote directory cache does not exist")); // NOI18N
            } else {
                List<DirEntry> entries = storage.listValid(fo.getNameExt());
                entry.setCache(fo.getCache().getName());
                entries.add(entry);
                DirectoryStorage newStorage = new DirectoryStorage(getStorageFile(), entries);
                try {
                    newStorage.store();
                } catch (IOException ex) {
                    Exceptions.printStackTrace(ex); // what else can we do?..
                }
                synchronized (refLock) {
                    storageRef = new SoftReference<>(newStorage);
                }
                fo.setPendingRemoteDelivery(false);
            }
        } finally {
            writeLock.unlock();
        }
    }

    private DirectoryStorage getDirectoryStorageImpl(final boolean forceRefresh, final String expectedName, final String childName, final boolean expected) throws
            ConnectException, IOException, InterruptedException, CancellationException, ExecutionException {

        if (forceRefresh && ! ConnectionManager.getInstance().isConnectedTo(getExecutionEnvironment())) {
            //RemoteLogger.getInstance().warning("refreshDirectoryStorage is called while host is not connected");
            //force = false;
            throw RemoteExceptions.createConnectException(RemoteFileSystemUtils.getConnectExceptionMessage(getExecutionEnvironment()));
        }

        DirectoryStorage storage;

        File storageFile = getStorageFile();

        // check whether it is cached in memory
        synchronized (refLock) {
            storage = storageRef.get();
        }
        boolean fromMemOrDiskCache;

        if (storage == null) {
            // try loading from disk
            fromMemOrDiskCache = false;
            storage = DirectoryStorage.EMPTY;
            if (storageFile.exists()) {
                Lock readLock = RemoteFileSystem.getLock(getCache()).readLock();
                try {
                    readLock.lock();
                    try {
                        storage = DirectoryStorage.load(storageFile, getExecutionEnvironment());
                        fromMemOrDiskCache = true;
                        // try to keep loaded cache in memory
                        synchronized (refLock) {
                            DirectoryStorage s = storageRef.get();
                            // it could be cache put in memory by writer (the best content)
                            // or by previous reader => it's the same as loaded
                            if (s != null) {
                                if (trace) { trace("using storage that was kept by other thread"); } // NOI18N
                                storage = s;
                            } else {
                                storageRef = new SoftReference<>(storage);
                            }
                        }
                    } catch (FormatException e) {
                        FormatException.reportIfNeeded(e);
                        storageFile.delete();
                    } catch (InterruptedIOException e) {
                        throw e;
                    } catch (FileNotFoundException e) {
                        // this might happen if we switch to different DirEntry implementations, see storageFile.delete() above
                        RemoteLogger.finest(e, this);
                    } catch (IOException e) {
                        Exceptions.printStackTrace(e);
                    }
                } finally {
                    readLock.unlock();
                }
            }
        } else {
            if (trace) { trace("use memory cached storage"); } // NOI18N
            fromMemOrDiskCache = true;
        }

        if (fromMemOrDiskCache && !forceRefresh && isAlreadyKnownChild(storage, childName)) {
            RemoteLogger.assertTrue(storage != null);
            if (trace) { trace("returning cached storage"); } // NOI18N
            return storage;
        }
        if (childName != null && RemoteFileSystem.isSniffing(childName)) {
            if (isAutoMount() || getFileSystem().isDirectAutoMountChild(getPath())) {
                return DirectoryStorage.EMPTY;
            }
        }
        // neither memory nor disk cache helped or was request to force refresh
        // proceed with reading remote content

        checkConnection(this, true);

        Lock writeLock = RemoteFileSystem.getLock(getCache()).writeLock();
        if (trace) { trace("waiting for lock"); } // NOI18N
        writeLock.lock();
        try {
            // in case another writer thread already synchronized content while we were waiting for lockImpl
            // even in refresh mode, we need this content, otherwise we'll generate events twice
            synchronized (refLock) {
                DirectoryStorage s = storageRef.get();
                if (s != null) {
                    if (trace) { trace("got storage from mem cache after waiting on writeLock: {0} expectedName={1}", getPath(), expectedName); } // NOI18N
                    if (forceRefresh || !isAlreadyKnownChild(s, childName)) {
                        storage = s;
                    } else {
                        return s;
                    }
                }
            }
            if (!getCache().exists()) {
                getCache().mkdirs();
                if (!getCache().exists()) {
                    throw new IOException("Can not create cache directory " + getCache()); // NOI18N // new IOException sic - should never happen
                }
            }
            if (trace) { trace("synchronizing"); } // NOI18N

            if (childName != null && RemoteLogger.isLoggable(Level.FINEST)) {
                RemoteLogger.finest("{0} is asked for child {1} while not having cache", getPath(), childName);
            }

            Exception problem = null;
            Map<String, DirEntry> newEntries = Collections.emptyMap();
            try {
                newEntries = readEntries(storage, forceRefresh, childName);
            }  catch (FileNotFoundException ex) {
                throw ex;
            }  catch (IOException | ExecutionException ex) {
                problem = ex;
            }
            if (problem != null) {
                if (!ConnectionManager.getInstance().isConnectedTo(getExecutionEnvironment())) {
                    // connection was broken while we read directory content - add notification
                    getFileSystem().addPendingFile(this);
                    throw RemoteExceptions.createConnectException(problem.getMessage());
                } else {
                    boolean fileNotFoundException = RemoteFileSystemUtils.isFileNotFoundException(problem);
                    if (fileNotFoundException) {
                        synchronized (refLock) {
                            storageRef = new SoftReference<>(DirectoryStorage.EMPTY);
                        }
                    }
                    if (!fileNotFoundException) {
                        if (problem instanceof IOException) {
                            throw (IOException) problem;
                        } else if (problem instanceof ExecutionException) {
                            throw (ExecutionException) problem;
                        } else {
                            throw new IllegalStateException("Unexpected exception class: " + problem.getClass().getName(), problem); //NOI18N
                        }
                    }
                }
            }
            storage = updateChildren(newEntries, storage, fromMemOrDiskCache, expectedName, childName, expected);
        } finally {
            writeLock.unlock();
        }
        return storage;
    }

    private DirectoryStorage updateChildren(Map<String, DirEntry> newEntries, DirectoryStorage storage,
            boolean fromMemOrDiskCache, final String expectedName, final String childName,
            final boolean expected) throws IOException {

        getFileSystem().incrementDirSyncCount();
        Map<String, List<DirEntry>> dupLowerNames = new HashMap<>();
        boolean hasDups = false;
        boolean changed = (newEntries.size() != storage.listAll().size()) || (storage == DirectoryStorage.EMPTY);
        Set<DirEntry> keepCacheNames = new HashSet<>();
        List<DirEntry> entriesToFireChanged = new ArrayList<>();
        List<DirEntry> entriesToFireChangedRO = new ArrayList<>();
        List<DirEntry> entriesToFireCreated = new ArrayList<>();
        DirEntry expectedCreated = null;
        List<RemoteFileObject> filesToFireDeleted = new ArrayList<>();
        for (DirEntry newEntry : newEntries.values()) {
            if (newEntry.isValid()) {
                String cacheName;
                DirEntry oldEntry = storage.getValidEntry(newEntry.getName());
                if (oldEntry == null || !oldEntry.isValid()) {
                    changed = true;
                    cacheName = RemoteFileSystemUtils.escapeFileName(newEntry.getName());
                    if (fromMemOrDiskCache || newEntry.getName().equals(expectedName) || getFlag(CONNECTION_ISSUES)) {
                        entriesToFireCreated.add(newEntry);
                        expectedCreated = newEntry;
                    }
                } else {
                    if (oldEntry.isSameType(newEntry)) {
                        cacheName = oldEntry.getCache();
                        keepCacheNames.add(newEntry);
                        boolean fire = false;
                        if (!newEntry.isSameLastModified(oldEntry) || newEntry.getSize() != oldEntry.getSize()) {
                            if (newEntry.isPlainFile()) {
                                changed = fire = true;
                                File entryCache = new File(getCache(), oldEntry.getCache());
                                if (entryCache.exists()) {
                                    if (trace) { trace("removing cache for updated file {0}", entryCache.getAbsolutePath()); } // NOI18N
                                    entryCache.delete(); // TODO: We must just mark it as invalid instead of physically deleting cache file...
                                }
                            }

                        }
                        if (!equals(newEntry.getLinkTarget(), oldEntry.getLinkTarget())) {
                            changed = fire = true; // TODO: we forgot old link path, probably should be passed to change event
                            getFileSystem().getFactory().setLink(this, getPath() + '/' + newEntry.getName(), newEntry.getLinkTarget());
                        }
                        if (!newEntry.isSameAccess(oldEntry)) {
                            entriesToFireChangedRO.add(newEntry);
                            changed = fire = true;
                        }
                        if (!newEntry.isDirectory() && (newEntry.getSize() != oldEntry.getSize())) {
                            changed = fire = true;// TODO: shouldn't it be the same as time stamp change?
                        }
                        if (fire) {
                            entriesToFireChanged.add(newEntry);
                        }
                    } else {
                        changed = true;
                        getFileSystem().getFactory().changeImplementor(this, oldEntry, newEntry);
                        if (oldEntry.isLink() && newEntry.isPlainFile() && newEntry.canWrite()) {
                            entriesToFireChangedRO.add(newEntry);
                        } else {
                            entriesToFireChanged.add(newEntry);
                        }
                        cacheName = null; // unchanged
                    }
                }
                if (cacheName !=null) {
                    newEntry.setCache(cacheName);
                }
                String lowerCacheName = RemoteFileSystemUtils.isSystemCaseSensitive() ? newEntry.getCache() : newEntry.getCache().toLowerCase();
                List<DirEntry> dupEntries = dupLowerNames.get(lowerCacheName);
                if (dupEntries == null) {
                    dupEntries = new ArrayList<>();
                    dupLowerNames.put(lowerCacheName, dupEntries);
                } else {
                    hasDups = true;
                }
                dupEntries.add(newEntry);
            } else {
                if (!storage.isKnown(childName)) {
                    changed = true;
                }
            }
        }
        if (changed) {
            // Check for removal
            for (DirEntry oldEntry : storage.listValid()) {
                DirEntry newEntry = newEntries.get(oldEntry.getName());
                if (newEntry == null || !newEntry.isValid()) {
                    RemoteFileObject removedFO = invalidate(oldEntry);
                    if (removedFO != null) {
                        filesToFireDeleted.add(removedFO);
                    }
                }
            }
            if (hasDups) {
                for (Map.Entry<String, List<DirEntry>> mapEntry :
                        new ArrayList<>(dupLowerNames.entrySet())) {

                    List<DirEntry> dupEntries = mapEntry.getValue();
                    if (dupEntries.size() > 1) {
                        for (int i = 0; i < dupEntries.size(); i++) {
                            DirEntry entry = dupEntries.get(i);
                            if (keepCacheNames.contains(entry)) {
                                continue; // keep the one that already exists
                            }
                            // all duplicates will have postfix
                            for (int j = 0; j < Integer.MAX_VALUE; j++) {
                                String cacheName = mapEntry.getKey() + '_' + j;
                                String lowerCacheName = cacheName.toLowerCase();
                                if (!dupLowerNames.containsKey(lowerCacheName)) {
                                    if (trace) { trace("resolving cache names conflict in {0}: {1} -> {2}", // NOI18N
                                            getCache().getAbsolutePath(), entry.getCache(), cacheName); }
                                    entry.setCache(cacheName);
                                    dupLowerNames.put(lowerCacheName, Collections.singletonList(entry));
                                    break;
                                }
                            }
                        }
                    }
                }
            }
            storage = new DirectoryStorage(getStorageFile(), newEntries.values());
            storage.store();
        } else {
            storage.touch();
        }
        setFlag(CONNECTION_ISSUES, false);
        // always put new content in cache
        // do it before firing events, to give liseners real content
        synchronized (refLock) {
            storageRef = new SoftReference<>(storage);
        }
        // fire all event under lockImpl
        if (changed) {
            dropMagic();
            for (RemoteFileObject deleted : filesToFireDeleted) {
                fireDeletedEvent(this.getOwnerFileObject(), deleted, expected, true);
            }

            FilesystemInterceptorProvider.FilesystemInterceptor interceptor =
                    USE_VCS ? FilesystemInterceptorProvider.getDefault().getFilesystemInterceptor(getFileSystem()) : null;

            for (DirEntry entry : entriesToFireCreated) {
                RemoteFileObject fo = getFileSystem().getFactory().createFileObject(this, entry).getOwnerFileObject();
                if (interceptor != null && expectedCreated != null && !expectedCreated.equals(entry)) {
                    try {
                        getFileSystem().setInsideVCS(true);
                        interceptor.createdExternally(FilesystemInterceptorProvider.toFileProxy(fo));
                    } finally {
                        getFileSystem().setInsideVCS(false);
                    }
                }
                fireRemoteFileObjectCreated(fo);
            }
            for (DirEntry entry : entriesToFireChanged) {
                RemoteFileObjectBase fo = getFileSystem().getFactory().getCachedFileObject(getPath() + '/' + entry.getName());
                if (fo != null) {
                    if (fo.isPendingRemoteDelivery()) {
                        RemoteLogger.getInstance().log(Level.FINE, "Skipping change event for pending file {0}", fo);
                    } else {
                        final long time = fo.lastModified().getTime();
                        fo.fireFileChangedEvent(fo.getListeners(), new FileEvent(fo.getOwnerFileObject(), fo.getOwnerFileObject(), expected, time));
                        this.fireFileChangedEvent(this.getListeners(), new FileEvent(this.getOwnerFileObject(), fo.getOwnerFileObject(), expected, time));
                    }
                }
            }
            fireReadOnlyChangedEventsIfNeed(entriesToFireChangedRO);
            //fireFileChangedEvent(getListeners(), new FileEvent(this));
        }
        return storage;
    }

    private void fireReadOnlyChangedEventsIfNeed(List<DirEntry> entriesToFireChangedRO) {
        for (DirEntry entry : entriesToFireChangedRO) {
            RemoteFileObjectBase fo = getFileSystem().getFactory().getCachedFileObject(getPath() + '/' + entry.getName());
            if (fo != null) {
                if (fo.isPendingRemoteDelivery()) {
                    RemoteLogger.getInstance().log(Level.FINE, "Skipping change r/o event for pending file {0}", fo);
                } else {
                    fo.fireReadOnlyChangedEvent();
                }
            }
        }
    }

    private void fireDeletedEvent(RemoteFileObject parent, RemoteFileObject fo, boolean expected, boolean recursive) {

        FilesystemInterceptorProvider.FilesystemInterceptor interceptor =
                USE_VCS ? FilesystemInterceptorProvider.getDefault().getFilesystemInterceptor(getFileSystem()) : null;

        if (recursive) {
            RemoteFileObjectBase[] children = fo.getImplementor().getExistentChildren(true);
            for (RemoteFileObjectBase c : children) {
                Enumeration<FileChangeListener> listeners = c.getListeners();
                RemoteFileObject childFO = c.getOwnerFileObject();
                if (interceptor != null) {
                    try {
                        getFileSystem().setInsideVCS(true);
                        getFileSystem().setExternallyRemoved(childFO.getImplementor());
                        try {
                            interceptor.deletedExternally(FilesystemInterceptorProvider.toFileProxy(childFO));
                        } finally {
                            getFileSystem().setExternallyRemoved(null);
                        }
                    } finally {
                        getFileSystem().setInsideVCS(false);
                    }
                }
                c.fireFileDeletedEvent(listeners, new FileEvent(childFO, childFO, expected));
                RemoteFileObjectBase p = c.getParent();
                p.fireFileDeletedEvent(p.getListeners(), new FileEvent(p.getOwnerFileObject(), childFO, expected));
            }
        }
        if (interceptor != null) {
            getFileSystem().setExternallyRemoved(fo.getImplementor());
            try {
                getFileSystem().setInsideVCS(true);
                interceptor.deletedExternally(FilesystemInterceptorProvider.toFileProxy(fo));
            } finally {
                getFileSystem().setInsideVCS(false);
                getFileSystem().setExternallyRemoved(null);
            }
        }
        fo.fireFileDeletedEvent(fo.getImplementor().getListeners(), new FileEvent(fo, fo, expected));
        parent.fireFileDeletedEvent(parent.getImplementor().getListeners(), new FileEvent(parent, fo, expected));
    }

//    InputStream _getInputStream(RemotePlainFile child) throws
//            ConnectException, IOException, InterruptedException, CancellationException, ExecutionException {
//        Lock lock = RemoteFileSystem.getLock(child.getCache()).readLock();
//        lock.lock();
//        try {
//            if (child.getCache().exists()) {
//                return new FileInputStream(child.getCache());
//            }
//        } finally {
//            lock.unlock();
//        }
//        checkConnection(child, true);
//        DirectoryStorage storage = getDirectoryStorage(child.getNameExt()); // do we need this?
//        return new CachedRemoteInputStream(child, getExecutionEnvironment());
//    }

    @Override
    public void warmup(FileSystemProvider.WarmupMode mode, Collection<String> extensions) {
        switch(mode) {
            case FILES_CONTENT:
                warmupFiles(extensions);
                break;
            case RECURSIVE_LS:
                warmupDirs();
                break;
            default:
                Exceptions.printStackTrace(new IllegalAccessException("Unexpected warmup mode: " + mode)); //NOI18N
        }
    }

    private void warmupFiles(Collection<String> extensions) {
        if (ConnectionManager.getInstance().isConnectedTo(getExecutionEnvironment())) {
            File zipFile = new File(getCache(), RemoteFileSystem.CACHE_ZIP_FILE_NAME);
            if (!zipFile.exists()) {
                File zipPartFile = new File(getCache(), RemoteFileSystem.CACHE_ZIP_PART_NAME);
                getFileSystem().getZipper().schedule(zipFile, zipPartFile, getPath(), extensions);
            }
        }
    }

    private boolean ensureChildSyncFromZip(RemotePlainFile child) {
        File file = new File(getCache(), RemoteFileSystem.CACHE_ZIP_FILE_NAME);
        if (file.exists()) {
            ZipFile zipFile = null;
            InputStream is = null;
            OutputStream os = null;
            boolean ok = false;
            try {
                zipFile = new ZipFile(file);
                String path = child.getPath();
                RemoteLogger.assertTrue(path.startsWith("/")); //NOI18N
                path = path.substring(1); // remove starting '/'
                ZipEntry zipEntry = zipFile.getEntry(path);
                if (zipEntry != null) {
                    if (zipEntry.getSize() != child.getSize()) {
                        return false;
                    }
                    long zipTime = zipEntry.getTime();
                    long childTime = child.lastModified().getTime() - TimeZone.getDefault().getRawOffset();
                    zipTime /= 1000;
                    childTime /= 1000;
                    if (childTime%2 == 1 && zipTime%2 == 0) {
                        childTime ++; // zip rounds up to 2 seconds
                    }
                    long delta = zipTime - childTime;
                    boolean same;
                    if (delta == 0) {
                        same = true;
                    } else {
                        // on some servers (e.g. townes) timezone for /usr/include and /export/home differs
                        // below is a temporary workaround
                        if (delta%3600 == 0) {
                            long hours = delta / 3600;
                            same = -23 <= hours && hours <= 23;
                        } else {
                            same = false;
                        }
                    }
                    if (same) {
                        is = zipFile.getInputStream(zipEntry);
                        os = new FileOutputStream(child.getCache());
                        FileUtil.copy(is, os);
                        ok = true;
                    } else {
                        RemoteLogger.finest("Zip timestamp differ for {0}", child); //NOI18N
                    }
                }
            } catch (IOException ex) {
                RemoteLogger.fine(ex);
            } finally {
                if (os != null) {
                    try {
                        os.close();
                    } catch (IOException ex) {
                        ok = false;
                        RemoteLogger.fine(ex);
                    }
                }
                if (is != null) {
                    try {
                        is.close();
                    } catch (IOException ex) {
                        RemoteLogger.fine(ex);
                    }
                }
                if (zipFile != null) {
                    try {
                        zipFile.close();
                    } catch (IOException ex) {
                        RemoteLogger.fine(ex);
                    }
                }
                return ok;
            }
        } else {
            RemoteDirectory parent = getParent();
            if (parent != null) {
                return parent.ensureChildSyncFromZip(child);
            }
        }
        return false;
    }

    private static boolean isLoadingInEditor() {
        for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
            if ("org.openide.text.DocumentOpenClose$DocumentLoad".equals(element.getClassName())) { //NOI18N
                if ("atomicLockedRun".equals(element.getMethodName())) { //NOI18N
                    return true;
                }
            }
        }
        return false;
    }

    private boolean cacheExists(RemotePlainFile child) {
        Lock lock = RemoteFileSystem.getLock(child.getCache()).readLock();
        lock.lock();
        try {
            return child.getCache().exists();
        } finally {
            lock.unlock();
        }        
    }
    
    /*package*/ void ensureChildSync(RemotePlainFile child) throws
            ConnectException, IOException, InterruptedException, CancellationException, ExecutionException {

        if (cacheExists(child)) {
            if(isLoadingInEditor() && ConnectionManager.getInstance().isConnectedTo(getExecutionEnvironment())) {
                child.refreshImpl(false, null, false, RefreshMode.DEFAULT);
            }
            if (cacheExists(child)) {
                return;
            }
        }        
        checkConnection(child, true);
        DirectoryStorage storage = getDirectoryStorage(child.getNameExt()); // do we need this?
        Lock lock = RemoteFileSystem.getLock(child.getCache()).writeLock();
        lock.lock();
        try {
            if (child.getCache().exists()) {
                return;
            }
            final File cacheParentFile = child.getCache().getParentFile();
            if (!cacheParentFile.exists()) {
                cacheParentFile.mkdirs();
                if (!cacheParentFile.exists()) {
                    throw new IOException("Unable to create parent firectory " + cacheParentFile.getAbsolutePath()); //NOI18N // new IOException sic - should never happen
                }
            }
            if (ensureChildSyncFromZip(child)) {
                return; // cleanup is in finally block
            }
            StringWriter errorWriter = new StringWriter();
            Future<Integer> task = CommonTasksSupport.downloadFile(child.getPath(), getExecutionEnvironment(), child.getCache().getAbsolutePath(), errorWriter);
            int rc = task.get().intValue();
            if (rc == 0) {
                getFileSystem().incrementFileCopyCount();
            } else {
                throw RemoteExceptions.createIOException(NbBundle.getMessage(RemoteDirectory.class,
                        "EXC_CanNotDownload", getDisplayName(child.getPath()), errorWriter.toString())); //NOI18N
            }
        } catch (InterruptedException | ExecutionException ex) {
            child.getCache().delete();
            throw ex;
        } finally {
            lock.unlock();
        }
    }

    private void checkConnection(RemoteFileObjectBase fo, boolean throwConnectException) throws ConnectException {
        if (!ConnectionManager.getInstance().isConnectedTo(getExecutionEnvironment())) {
            getFileSystem().addPendingFile(fo);
            if (throwConnectException) {
                throw RemoteExceptions.createConnectException(RemoteFileSystemUtils.getConnectExceptionMessage(getExecutionEnvironment()));
            }
        }
    }

    @Override
    public FileType getType() {
        return FileType.Directory;
    }

    @Override
    public final InputStream getInputStream(boolean checkLock) throws FileNotFoundException {
        throw new FileNotFoundException(getPath()); // new IOException sic!- should never be called
    }

    public byte[] getMagic(RemoteFileObjectBase file) {
        return getMagicCache().get(file.getNameExt());
    }

    private MagicCache getMagicCache() {
        MagicCache magic;
        synchronized (magicLock) {
            magic = magicCache.get();
            if (magic == null) {
                magic = new MagicCache(this);
                magicCache = new SoftReference<>(magic);
            }
        }
        return magic;
    }

    private void dropMagic() {
        synchronized (magicLock) {
            MagicCache magic = magicCache.get();
            if (magic != null) {
                magic.clean(null);
                magicCache = new SoftReference<>(null);
            } else {
                new MagicCache(this).clean(null);
            }
        }
    }

    @Override
    protected final OutputStream getOutputStreamImpl(final FileLock lock, RemoteFileObjectBase orig) throws IOException {
        throw new IOException("Can not write into a directory " + getDisplayName()); // new IOException sic!- should never be called // NOI18N
    }

    private RemoteFileObject invalidate(DirEntry oldEntry) {
        RemoteFileObject fo = getFileSystem().getFactory().invalidate(getPath() + '/' + oldEntry.getName());
        File oldEntryCache = new File(getCache(), oldEntry.getCache());
        removeFile(oldEntryCache);
        return fo;
    }

    private void removeFile(File cache) {
        if (cache.isDirectory()) {
            File[] children = cache.listFiles();
            if (children != null) {
                for (File child : children) {
                    removeFile(child);
                }
            }
        }
        cache.delete();
    }

    @Override
    public void refreshImpl(boolean recursive, Set<String> antiLoop, boolean expected, RefreshMode refreshMode) throws ConnectException, IOException, InterruptedException, CancellationException, ExecutionException {
        if (antiLoop != null) {
            if (antiLoop.contains(getPath())) {
                return;
            } else {
                antiLoop.add(getPath());
            }
        }
        DirectoryStorage storage = getExistingDirectoryStorage();
        if (storage ==  null ||storage == DirectoryStorage.EMPTY) {
            if (!getFlag(CONNECTION_ISSUES)) {
                return;
            }
        }
        // unfortunately we can't skip refresh if there is a storage but no children exists
        // in this case we have to reafresh just storage - but for the time being only RemoteDirectory can do that
        // TODO: revisit this after refactoring cache into a separate class(es)
        try {
            DirectoryStorage refreshedStorage = refreshDirectoryStorage(null, expected);
            if (recursive) {
                for (RemoteFileObjectBase child : getExistentChildren(refreshedStorage)) {
                    child.refreshImpl(true, antiLoop, expected, RefreshMode.FROM_PARENT);
                }
            }
        } catch (FileNotFoundException ex) {
            final RemoteDirectory parent = getParent();
            if (parent != null) {
                parent.refreshImpl(false, antiLoop, expected, refreshMode);
            } else {
                throw ex;
            }
        }
    }

    private void trace(String message, Object... args) {
        if (trace) {
            message = "SYNC [" + getPath() + "][" + System.identityHashCode(this) + "][" + Thread.currentThread().getId() + "]: " + message; // NOI18N
            RemoteLogger.getInstance().log(Level.FINEST, message, args);
        }
    }

    private static boolean equals(String s1, String s2) {
        return (s1 == null) ? (s2 == null) : s1.equals(s2);
    }

    private DirEntry getChildEntry(RemoteFileObjectBase child) {
        try {
            DirectoryStorage directoryStorage = getDirectoryStorage(child.getNameExt());
            if (directoryStorage != null) {
                DirEntry entry = directoryStorage.getValidEntry(child.getNameExt());
                if (entry != null) {
                    return entry;
                } else {
                    RemoteLogger.getInstance().log(Level.INFO, "Not found entry for file {0}", child); // NOI18N
                }
            }
        } catch (ConnectException ex) {
            RemoteLogger.finest(ex, this);
        } catch (IOException ex) {
            RemoteLogger.finest(ex, this);
        } catch (ExecutionException | InterruptedException | CancellationException ex) {
            RemoteLogger.finest(ex, this);
        }
        return null;
    }

    long getSize(RemoteFileObjectBase child) {
        DirEntry childEntry = getChildEntry(child);
        if (childEntry != null) {
            return childEntry.getSize();
        }
        return 0;
    }

    /*package*/ Date lastModified(RemoteFileObjectBase child) {
        DirEntry childEntry = getChildEntry(child);
        if (childEntry != null) {
            return childEntry.getLastModified();
        }
        return new Date(0); // consistent with File.lastModified(), which returns 0 for inexistent file
    }

    /** for tests ONLY! */
    /*package*/ DirectoryStorage testGetExistingDirectoryStorage() {
        return getExistingDirectoryStorage();
    }

    private File getStorageFile() {
        return new File(getCache(), RemoteFileSystem.CACHE_FILE_NAME);
    }

    @Override
    public boolean hasCache() {
        return getStorageFile().exists();
    }

    @Override
    public void diagnostics(boolean recursive) {
        RemoteFileObjectBase[] existentChildren = getExistentChildren();
        System.err.printf("\nRemoteFS diagnostics for %s\n", this); //NOI18N
        System.err.printf("Existing children count: %d\n", existentChildren.length); //NOI18N
        File cache = getStorageFile();
        System.err.printf("Cache file: %s\n", cache.getAbsolutePath()); //NOI18N
        System.err.printf("Cache content: \n"); //NOI18N
        printFile(cache, System.err);
        System.err.printf("Existing children:\n"); //NOI18N
        for (RemoteFileObjectBase fo : existentChildren) {
            System.err.printf("\t%s [%s] %d\n",  //NOI18N
                    fo.getNameExt(), fo.getCache().getName(), fo.getCache().length());
        }
        if (recursive) {
            for (RemoteFileObjectBase fo : existentChildren) {
                fo.diagnostics(recursive);
            }
        }
    }

    private static void printFile(File file, PrintStream out) {
        BufferedReader rdr = null;
        try {
            rdr = new BufferedReader(new FileReader(file));
            try {
                String line;
                while ((line = rdr.readLine()) != null) {
                    out.printf("%s\n", line); // NOI18N
                }
            } finally {
                try {
                    rdr.close();
                } catch (IOException ex) {
                    ex.printStackTrace(System.err);
                }
            }
        } catch (IOException ex) {
            ex.printStackTrace(System.err);
        } finally {
            try {
                if (rdr != null) {
                    rdr.close();
                }
            } catch (IOException ex) {
                ex.printStackTrace(System.err);
            }
        }
    }

}
