# -*- coding: UTF-8 -*-
#
#  NtfsConfig.py : manage main interface
#  Copyright (C) 2007 Mertens Florent <flomertens@gmail.com>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import os
import time
from xml.sax.saxutils import escape as escape_mkup

import pygtk
pygtk.require("2.0")
import gtk
import gtk.glade
from SimpleGladeApp import SimpleGladeApp
from gettext import gettext as _

from Utility import *
from config import *
from Fstab.FstabHandler import *
from AddWizard import *


gtk.glade.bindtextdomain("ntfs-config",localedir)
gtk.glade.textdomain("ntfs-config")


class NtfsConfig(SimpleGladeApp) :
    ''' Class that manage the main window '''
    
    def __init__(self) :
        
        SimpleGladeApp.__init__(self, GLADEFILE, "dialog_main", domain = PACKAGE)
        
        self.dialog_main.set_title(_("NTFS Configuration Tool"))

        # Delete broken link. and create hal fdi directory if it don't exist
        for value in (TARGET_NTFS_WRITE, TARGET_NTFS_READ) :
            if not os.path.exists(value) and os.path.islink(value) :
                os.unlink(value)
        if not os.path.exists(HAL_CONFIG_DIR) :
            os.mkdir(HAL_CONFIG_DIR)

        # Create FstabHandler object
        self.disk = FstabHandler(FSTAB, parent = self.dialog_main, external_change_watch = True)
        self.ntfs_entries = [ k for k in self.disk.get_configured() if k["FS_TYPE"] == "ntfs" ]

        # Look at new ntfs device
        new = []
        for entry in self.disk.get_new() :
            if entry["FS_TYPE"] == "ntfs" :
                new.append(entry["DEVICE"])

        # Draw stuff
        self.handlers = []
        self.set_buttons()
        self.setup_treeview()
        self.update_main()
        self.dialog_main.show()

        # If new, configure them
        if new :
            app = AddWizard(new, self.dialog_main, self.disk)
            app.run()
            self.update_main()

        # Connect event
        self.handlers.append((self.checkbutton_internal, \
            self.checkbutton_internal.connect("toggled", self.on_internal_toggled)))
        self.checkbutton_external.connect("toggled", self.on_external_toggled)
        self.close_button.connect("clicked", self.on_close_clicked)
        self.undo_button.connect("clicked", self.on_undo_clicked)
        self.dialog_main.connect("destroy", self.on_close_clicked)
        self.treeview.connect("row-activated", self.on_row_activated)
        self.expander.connect("activate", self.on_expander_activate)
        self.disk.connect("any_changed", gobject.idle_add, self.update_main)

        gtk.gdk.threads_init()

    def set_buttons(self) :

        if self.get_ntfs_rw_device() :
            self.checkbutton_internal.set_active(True)
        if os.path.exists(TARGET_NTFS_WRITE) :
            self.checkbutton_external.set_active(True)
        
    def setup_treeview(self) :
    
        renderer= gtk.CellRendererToggle()
        renderer.set_property("activatable", True)
        renderer.connect("toggled", self.on_writable_toggled)
        column = gtk.TreeViewColumn(_("Writable"), renderer, active=0)
        self.treeview.append_column(column)
        
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Device"), renderer, text=1)
        self.treeview.append_column(column)
        
        renderer = gtk.CellRendererText()
        renderer.set_property("editable", True)
        renderer.connect("edited", self.on_mount_point_edit)
        column = gtk.TreeViewColumn(_("Mount point"), renderer, text=2)
        self.treeview.append_column(column) 
        
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Total"), renderer, text=3)
        self.treeview.append_column(column)
        
        renderer = gtk.CellRendererProgress()
        column = gtk.TreeViewColumn(_("Used"), renderer, text=4, value=5)
        renderer.set_property("ypad", 2)
        renderer.set_property("xpad", 3)
        self.treeview.append_column(column)
        
        self.tree_store = gtk.ListStore(bool, str, str, str, str, int)
        self.tree_store.set_sort_column_id(1, gtk.SORT_ASCENDING)
        self.treeview.set_model(self.tree_store)
        
    def update_main(self) :
    
        focus_path = self.treeview.get_cursor()[0]
        if focus_path :
            l = self.tree_store[focus_path]
            current_focus = (l[1], l[2])
        self.ntfs_entries = [ k for k in self.disk.get_configured() if k["FS_TYPE"] == "ntfs" ]
        self.checkbutton_internal.set_sensitive(True)
        if not self.ntfs_entries :
            self.checkbutton_internal.set_sensitive(False)
        self.ntfs_rw_entries = self.get_ntfs_rw_device()
        self.undo_button.set_sensitive(self.disk.original_has_changed())
        self.checkbutton_internal.set_inconsistent( \
                bool(self.ntfs_rw_entries and self.get_ntfs_ro_device()))
        self.block_handlers()
        self.checkbutton_internal.set_active(bool(self.ntfs_rw_entries))
        self.unblock_handlers()
        self.tree_store.clear()
        for entry in self.ntfs_entries :
            writable = bool(entry in self.ntfs_rw_entries)
            self.tree_store.append( ( writable, entry["DEVICE"], entry["FSTAB_PATH"], \
                    size_renderer(entry.get_size()), size_renderer(entry.get_used_size()), \
                    entry.get_used_size()/float(entry.get_size()) * 100 ) )         
        if focus_path :
            for path in range(len(self.tree_store)) :
                l = self.tree_store[path]
                if current_focus == (l[1], l[2]) :
                    self.treeview.set_cursor(path)
                    break
                elif current_focus[0] == l[1] and len(self.disk.search(l[1], keys = ["DEVICE"])) < 2 :
                    self.treeview.set_cursor(path)
                    break

    def block_handlers(self):
          
        for (widget, handler) in self.handlers :
            widget.handler_block(handler)
     
    def unblock_handlers(self):
          
        for (widget, handler) in self.handlers :
            widget.handler_unblock(handler)

                    
    def on_expander_activate(self, expander) :
    
        if not self.expander.get_expanded() :
            height = self.dialog_main.get_size()[1]
            if len(self.tree_store) < 5 :
                height += 60 + len(self.tree_store) * 25
            else :
                height += 200
            self.dialog_main.set_property("height-request", height)
        else :
            self.dialog_main.set_property("height-request", -1)
                
    def get_ntfs_ro_device(self) :
    
        entries = []
        for entry in self.ntfs_entries :
            if entry["FSTAB_TYPE"] == "ntfs" or entry.hasopt("ro") :
                entries.append(entry)
        return entries
        
    def get_ntfs_rw_device(self) :
    
        entries = []
        for entry in self.ntfs_entries :
            if entry["FSTAB_TYPE"] == "ntfs-3g" and not entry.hasopt("ro") :
                entries.append(entry)
        return entries
            
    def set_external_ntfs_policy(self, policy) :
        ''' Set external NTFS policy '''
    
        if policy == "rw" :
            if not os.path.exists(TARGET_NTFS_WRITE) :
                os.symlink(NTFS_WRITE, TARGET_NTFS_WRITE)
            if os.path.exists(TARGET_NTFS_READ) :
                os.unlink(TARGET_NTFS_READ)   
        else : 
            if not os.path.exists(TARGET_NTFS_READ) :
                os.symlink(NTFS_READ, TARGET_NTFS_READ)
            if os.path.exists(TARGET_NTFS_WRITE) :
                os.unlink(TARGET_NTFS_WRITE)
                
    def set_internal_ntfs_policy(self, entries, policy) :
    
        self.disk.set_mode("delay")
        for entry in entries :
            if policy == "rw" :
                self.disk.set(entry, type="ntfs-3g", option="-ro")
            else :
                if "ntfs" in self.disk.info.get_drivers("ntfs")["all"].keys() :
                    self.disk.set(entry, type="ntfs")
                else :
                    self.disk.set(entry, type="ntfs-3g", option="+ro")
        self.disk.set_mode("real")

    def on_internal_toggled(self, button) :
        ''' Set NTFS internal policy corresponding to the button status '''

        self.checkbutton_internal.set_inconsistent(False)
        if self.checkbutton_internal.get_active() :
            entries = self.get_ntfs_ro_device()
            policy = "rw"
        else :
            entries = self.get_ntfs_rw_device()
            policy = "ro"
        self.set_internal_ntfs_policy(entries, policy)
        
    def on_external_toggled(self, button) :

         if self.checkbutton_external.get_active() :
            self.set_external_ntfs_policy("rw")
         else :
            self.set_external_ntfs_policy("ro")
            
    def on_writable_toggled(self, renderer, path) :

        if self.tree_store[path][0] :
            self.set_internal_ntfs_policy([self.disk[self.tree_store[path][2]]], "ro")
        else :
            self.set_internal_ntfs_policy([self.disk[self.tree_store[path][2]]], "rw")

    def on_mount_point_edit(self, renderer, path_string, path) :

        entry = self.disk[self.tree_store[path_string][2]]
        if path == "" :
            return
        if not path[:len("/media/")] == "/media/" :
            if path[0] == "/" :
                dialog("warning", _("Formatting error"), \
                _("<i>%s</i> contains an invalid character.\n" \
                "Mount point should be in /media/. Alternatively, you can " \
                "enter a simple name, like <i>disk</i>.") % escape_mkup(path), \
                parent = self.dialog_main)
                return
            path = "/media/" + path
        self.disk.set(entry, path = path)
        self.tree_store[path_string][2] = entry["FSTAB_PATH"]

    def on_row_activated(self, treeview, path, view_column) :
        ''' Start browser on double click. '''
    
        dev = self.tree_store[path][2]
        if not self.disk.mount(dev) :
            open_url(self.disk[dev]["FSTAB_PATH"])
            
    def on_undo_clicked(self, button) :
    
        ret = dialog("question", _("Reverting to previous version?"), \
            _("All changes that you just made will be lost.\n"
            "Do you want to continue?"), parent = self.dialog_main)
        if ret[0] == gtk.RESPONSE_YES :
            self.disk.undo()
            
    def on_close_clicked(self, button) :
    
        self.dialog_main.hide()
        while gtk.events_pending() :
            gtk.main_iteration()
        self.disk.shutdown()
        gtk.main_quit()

