function [A, out] = python_ipc_system(what, cmd, mktmpfile, varargin)
% "out" is provided for debugging

  persistent show_msg

  if (strcmp(what, 'reset'))
    show_msg = [];
    A = true;
    out = [];
    return
  end

  if ~(strcmp(what, 'run'))
    error('unsupported command')
  end

  vstr = sympref('version');

  if (isempty(show_msg))
    disp(['OctSymPy v' vstr ': this is free software without warranty, see source.'])
    disp('You are using the slower system() communications with SymPy.')
    disp('Warning: this will be *SLOW*.  Every round-trip involves executing a')
    disp('new python process and many operations involve several round-trips.')
    show_msg = true;
  end

  newl = sprintf('\n');

  %% Headers
  headers = python_header();

  %% load all the inputs into python as pickles
  s1 = python_copy_vars_to('_ins', true, varargin{:});

  %% The actual command
  % cmd will be a snippet of python code that does something
  % with _ins and produce _outs.
  s2 = python_format_cmd(cmd);

  %% output, or perhaps a thrown error
  s3 = python_copy_vars_from('_outs');

  % join all the cell arrays with newlined
  s = mystrjoin([s1 s2 s3], newl);

  pyexec = sympref('python');
  if (isempty(pyexec))
    pyexec = 'python';
  end

  %% FIXME: Issue #63: with new regexp code on Matlab
  % workaround:
  % sympref python 'LD_LIBRARY_PATH="" python'
  % to prevent a conflict with the expat shipped with Matlab 2014a
  % See here with oracle
  % https://bugzilla.redhat.com/show_bug.cgi?id=821337
  % FIXME: make this the default on Matlab install?

  bigs = [headers s];

  if (~mktmpfile)
    %% paste all the commands into the system() command line
    % escaping: cmd or inputs might have \"
    % " -> \"
    % \" -> \\\"
    % \n -> \\n
    bigs = strrep(bigs, '\', '\\');
    bigs = strrep(bigs, '"', '\"');
    [status,out] = system([pyexec ' -c "' bigs '"']);
  else
    %% Generate a temp .py file then execute it with system()
    % can be useful for debugging, or if "python -c" fails for you
    fname = 'tmp_python_cmd.py';
    fd = fopen(fname, 'w');
    fprintf(fd, '# temporary autogenerated code\n\n');
    fputs(fd, bigs);
    fclose(fd);
    [status,out] = system([pyexec ' ' fname]);
  end

  if status ~= 0
    status
    out
    error('system() call failed!');
  end

  % there should be two blocks
  ind = strfind(out, '<output_block>');
  assert(length(ind) == 2)
  out1 = out(ind(1):(ind(2)-1));
  % could extractblock here, but just search for keyword instead
  if (isempty(strfind(out1, 'successful')))
    error('failed to import variables to python?')
  end
  A = extractblock(out(ind(2):end));
