/*                                                                            */
/* CDDL HEADER START                                                          */
/*                                                                            */
/* The contents of this file are subject to the terms of the Common           */
/* Development and Distribution License Version 1.0 (the "License").          */
/*                                                                            */
/* You can obtain a copy of the license at                                    */
/* http://www.opensource.org/licenses/CDDL-1.0.  See the License for the      */
/* specific language governing permissions and limitations under the License. */
/*                                                                            */
/* When distributing Covered Code, include this CDDL HEADER in each file and  */
/* include the License file in a prominent location with the name             */
/* LICENSE.CDDL.  If applicable, add the following below this CDDL HEADER,    */
/* with the fields enclosed by brackets "[]" replaced with your own           */
/* identifying information:                                                   */
/*                                                                            */
/* Portions Copyright (c) [yyyy] [name of copyright owner].                   */
/* All rights reserved.                                                       */
/*                                                                            */
/* CDDL HEADER END                                                            */
/*                                                                            */

/*                                                                            */
/* Portions Copyright (c) 2019, Regents of the University of Minnesota.       */
/* All rights reserved.                                                       */
/*                                                                            */
/* Contributors:                                                              */
/*    Daniel S. Karls                                                         */
/*    Ellad B. Tadmor                                                         */
/*    Ryan S. Elliott                                                         */
/*                                                                            */
/* <REPLACE: Add copyright information as needed.>                            */
/* Portions Copyright (c) Year, Organization                                  */
/* All rights reserved.                                                       */
/* </REPLACE>                                                                 */
/*                                                                            */
/* <REPLACE: Add contributors as needed.>                                     */
/* Contributors:                                                              */
/* </REPLACE>                                                                 */
/*                                                                            */

#include <math.h>

/*                                                                            */
/* Auxiliary files for the Stillinger-Weber model driver                      */
/*                                                                            */
/* Functions in this file are only used by cluster.inc. They are not required */
/* by ThreeBodyCluster.c.                                                     */
/*                                                                            */

double safe_acos(double theta)
{
  if (theta < -1.0)
  {
    theta = -1.0;
  }
  else if (theta > 1.0)
  {
    theta = 1.0;
  }

  return acos(theta);
}

/*                                                                            */
/* Define functions used in two-body calculations                             */
/*                                                                            */
static void f2_df2(double const * const params,
                   double const r,
                   double * const f2,
                   double * const df2_dr)
{
  /* Unpack parameters */
  double const A = params[PARAM_A];
  double const B = params[PARAM_B];
  double const a = params[PARAM_a];
  double const p = params[PARAM_p];
  double const q = params[PARAM_q];

  double const r_pow_minus_p = pow(r, -p);
  double const r_pow_minus_q = pow(r, -q);
  double const r_minus_a = (r - a);
  double const r_minus_a_sq = r_minus_a * r_minus_a;
  double const exp_inv_r_minus_a = exp(1.0 / r_minus_a);

  if (r < a)
  {
    *f2 = A * (B * r_pow_minus_p - r_pow_minus_q) * exp_inv_r_minus_a;

    if (df2_dr != NULL)
    {
      *df2_dr = (*f2) * (-1. / r_minus_a_sq)
                + A * exp_inv_r_minus_a
                      * (-p * B * r_pow_minus_p + q * r_pow_minus_q) / r;
    }
  }
  else
  {
    *f2 = 0.0;
    if (df2_dr != NULL) { *df2_dr = 0.0; }
  }
}

/*                                                                            */
/* Define functions used in three-body calculations                           */
/*                                                                            */
static double g(double const theta)
{
  double costheta = cos(theta);

  return (costheta + 1.0 / 3.0) * (costheta + 1.0 / 3.0);
}

static double dg_dcostheta(double const theta)
{
  double costheta = cos(theta);

  return 2.0 * (costheta + 1.0 / 3.0);
}

static double h(double const * const params,
                double const r,
                double const s,
                double const theta)
{
  /* Unpack parameters */
  double const lambda = params[PARAM_lambda];
  double const gamma = params[PARAM_gamma];
  double const a = params[PARAM_a];

  if (r < a && s < a)
  {
    return lambda * exp(gamma / (r - a) + gamma / (s - a)) * g(theta);
  }
  else
  {
    return 0.0;
  }
}

static void dh_drdsdt(double const * const params,
                      double const r,
                      double const s,
                      double const theta,
                      double * const dh_dr,
                      double * const dh_ds,
                      double * const dh_dt)
{
  /* Unpack parameters */
  double const lambda = params[PARAM_lambda];
  double const gamma = params[PARAM_gamma];
  double const a = params[PARAM_a];

  double costheta;
  double t;
  double gval;
  double dgdcos;
  double expterm;

  double dcostheta_dr;
  double dcostheta_ds;
  double dcostheta_dt;

  if (r < a && s < a)
  {
    /* Compute third distance using law of cosines */
    costheta = cos(theta);
    t = sqrt(r * r + s * s - 2 * r * s * costheta);

    dcostheta_dr = (r * r - s * s + t * t) / (2 * r * r * s);
    dcostheta_ds = (s * s - r * r + t * t) / (2 * r * s * s);
    dcostheta_dt = -t / (r * s);

    gval = g(theta);
    expterm = exp(gamma / (r - a) + gamma / (s - a));
    dgdcos = dg_dcostheta(theta);

    *dh_dr = lambda
             * (expterm * dgdcos * dcostheta_dr
                + gval * expterm * (-gamma / ((r - a) * (r - a))));
    *dh_ds = lambda
             * (expterm * dgdcos * dcostheta_ds
                + gval * expterm * (-gamma / ((s - a) * (s - a))));
    *dh_dt = lambda * expterm * dgdcos * dcostheta_dt;
  }
  else
  {
    *dh_dr = 0.0;
    *dh_ds = 0.0;
    *dh_dt = 0.0;
  }
}

static void f3_df3(double const * const params,
                   double const Rij,
                   double const Rik,
                   double const Rjk,
                   double * const f3,
                   double * const df3_dRij,
                   double * const df3_dRik,
                   double * const df3_dRjk)
{
  double theta_jik;
  double theta_ijk;
  double theta_ikj;

  double dh1_dRij;
  double dh1_dRik;
  double dh1_dRjk;

  double dh2_dRij;
  double dh2_dRik;
  double dh2_dRjk;

  double dh3_dRij;
  double dh3_dRik;
  double dh3_dRjk;

  /* Law of cosines to get angles from distances */
  theta_jik = safe_acos((Rij * Rij + Rik * Rik - Rjk * Rjk) / (2.0 * Rij * Rik));
  theta_ijk = safe_acos((Rij * Rij + Rjk * Rjk - Rik * Rik) / (2.0 * Rij * Rjk));
  theta_ikj = safe_acos((Rik * Rik + Rjk * Rjk - Rij * Rij) / (2.0 * Rik * Rjk));

  *f3 = h(params, Rij, Rik, theta_jik) + h(params, Rij, Rjk, theta_ijk)
        + h(params, Rik, Rjk, theta_ikj);

  if (df3_dRij != NULL)
  {
    dh_drdsdt(params, Rij, Rik, theta_jik, &dh1_dRij, &dh1_dRik, &dh1_dRjk);
    dh_drdsdt(params, Rij, Rjk, theta_ijk, &dh2_dRij, &dh2_dRjk, &dh2_dRik);
    dh_drdsdt(params, Rik, Rjk, theta_ikj, &dh3_dRik, &dh3_dRjk, &dh3_dRij);

    *df3_dRij = dh1_dRij + dh2_dRij + dh3_dRij;
    *df3_dRik = dh1_dRik + dh2_dRik + dh3_dRik;
    *df3_dRjk = dh1_dRjk + dh2_dRjk + dh3_dRjk;
  }

  return;
}
