/*****************************************************************************
 * adddialog.cpp - QStarDict, a dictionary for learning foreign languages    *
 * Copyright (C) 2023 Alexander Rodin                                        *
 *                                                                           *
 * This program is free software; you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation; either version 2 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License along   *
 * with this program; if not, write to the Free Software Foundation, Inc.,   *
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.               *
 *****************************************************************************/

#include "adddialog.h"
#include <QDir>
#include <QFileInfo>
#include <QMessageBox>
#include "kiwix.h"

namespace
{
enum class Operation
{
    Move,
    Symlink
};

QString prettifyFileName(QString fileName)
{
    QString homePath = QDir::homePath();
    if (fileName.startsWith(homePath))
        fileName.replace(0, homePath.size(), "~");
    return fileName;
}
}

AddDialog::AddDialog(QWidget *parent, const QString &fileName):
    QDialog(parent)
{
    setupUi(this);

    srcFileName = fileName;
    targetFileName = Kiwix::dataDirPath() + "/" + QFileInfo(srcFileName).fileName();

    operationComboBox->addItem(tr("Move"), static_cast<int>(Operation::Move));
    operationComboBox->addItem(tr("Symlink"), static_cast<int>(Operation::Symlink));
    fileLabel->setText(tr("the file <b>%1</b><br>to <b>%2</b>").arg(
                prettifyFileName(srcFileName),
                prettifyFileName(targetFileName)));
}

AddDialog::~AddDialog()
{

}

void AddDialog::accept()
{
    switch (static_cast<Operation>(operationComboBox->currentData().toInt()))
    {
        case Operation::Move:
            if (moveFile())
                QDialog::accept();
            else
                QMessageBox::warning(parentWidget(), tr("Cannot add a dictionary"),
                        tr("The file <b>%1</b> cannot be moved to <b>%2</b>").arg(
                            prettifyFileName(srcFileName),
                            prettifyFileName(targetFileName)));
            break;
        case Operation::Symlink:
            if (symlinkFile())
                QDialog::accept();
            else
                QMessageBox::warning(parentWidget(), tr("Cannot add a dictionary"),
                        tr("The symlink to file <b>%1</b> cannot be createdat <b>%2</b>").arg(
                            prettifyFileName(srcFileName),
                            prettifyFileName(targetFileName)));
            break;
    }
}

bool AddDialog::moveFile()
{
    if (QFile::exists(targetFileName))
        QFile::rename(targetFileName, targetFileName + ".old");
    bool result = QFile::rename(srcFileName, targetFileName);

    if (result)
        QFile::remove(targetFileName + ".old");
    else
        QFile::rename(targetFileName + ".old", targetFileName);

    return result;
}

bool AddDialog::symlinkFile()
{
    if (QFile::exists(targetFileName))
        QFile::rename(targetFileName, targetFileName + ".old");
    bool result = QFile::link(srcFileName, targetFileName);

    if (result)
        QFile::remove(targetFileName + ".old");
    else
        QFile::rename(targetFileName + ".old", targetFileName);

    return result;
}
