% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sheet_write.R
\name{sheet_write}
\alias{sheet_write}
\alias{write_sheet}
\title{(Over)write new data into a Sheet}
\usage{
sheet_write(data, ss = NULL, sheet = NULL)

write_sheet(data, ss = NULL, sheet = NULL)
}
\arguments{
\item{data}{A data frame. If it has zero rows, we send one empty pseudo-row
of data, so that we can apply the usual table styling. This empty row goes
away (gets filled, actually) the first time you send more data with
\code{\link[=sheet_append]{sheet_append()}}.}

\item{ss}{Something that identifies a Google Sheet:
\itemize{
\item its file id as a string or \code{\link[googledrive:as_id]{drive_id}}
\item a URL from which we can recover the id
\item a one-row \code{\link[googledrive:dribble]{dribble}}, which is how googledrive
represents Drive files
\item an instance of \code{googlesheets4_spreadsheet}, which is what \code{\link[=gs4_get]{gs4_get()}}
returns
}

Processed through \code{\link[=as_sheets_id]{as_sheets_id()}}.}

\item{sheet}{Sheet to write into, in the sense of "worksheet" or "tab". You can identify a sheet by name, with a string, or by position, with a number.}
}
\value{
The input \code{ss}, as an instance of \code{\link{sheets_id}}
}
\description{
This is one of the main ways to write data with googlesheets4. This function
writes a data frame into a (work)sheet inside a (spread)Sheet. The target
sheet is styled as a table:
\itemize{
\item Special formatting is applied to the header row, which holds column
names.
\item The first row (header row) is frozen.
\item The sheet's dimensions are set to "shrink wrap" the \code{data}.
}

If no existing Sheet is specified via \code{ss}, this function delegates to
\code{\link[=gs4_create]{gs4_create()}} and the new Sheet's name is randomly generated. If that's
undesirable, call \code{\link[=gs4_create]{gs4_create()}} directly to get more control.

If no \code{sheet} is specified or if \code{sheet} doesn't identify an existing sheet,
a new sheet is added to receive the \code{data}. If \code{sheet} specifies an existing
sheet, it is effectively overwritten! All pre-existing values, formats, and
dimensions are cleared and the targeted sheet gets new values and dimensions
from \code{data}.

This function goes by two names, because we want it to make sense in two
contexts:
\itemize{
\item \code{write_sheet()} evokes other table-writing functions, like
\code{readr::write_csv()}. The \code{sheet} here technically refers to an individual
(work)sheet (but also sort of refers to the associated Google
(spread)Sheet).
\item \code{sheet_write()} is the right name according to the naming convention used
throughout the googlesheets4 package.
}

\code{write_sheet()} and \code{sheet_write()} are synonyms and you can use either one.
}
\examples{
if (gs4_has_token()) {
  df <- data.frame(
    x = 1:3,
    y = letters[1:3]
  )

  # specify only a data frame, get a new Sheet, with a random name
  ss <- write_sheet(df)
  read_sheet(ss)

  # clean up
  googledrive::drive_trash(ss)

  # create a Sheet with some initial, placeholder data
  ss <- gs4_create(
    "sheet-write-demo",
    sheets = list(alpha = data.frame(x = 1), omega = data.frame(x = 1))
  )

  # write df into its own, new sheet
  sheet_write(df, ss = ss)

  # write mtcars into the sheet named "omega"
  sheet_write(mtcars, ss = ss, sheet = "omega")

  # get an overview of the sheets
  sheet_properties(ss)

  # view your magnificent creation in the browser
  gs4_browse(ss)

  # clean up
  gs4_find("sheet-write-demo") \%>\%
    googledrive::drive_trash()
}
}
\seealso{
Other write functions: 
\code{\link{gs4_create}()},
\code{\link{gs4_formula}()},
\code{\link{range_delete}()},
\code{\link{range_flood}()},
\code{\link{range_write}()},
\code{\link{sheet_append}()}

Other worksheet functions: 
\code{\link{sheet_add}()},
\code{\link{sheet_append}()},
\code{\link{sheet_copy}()},
\code{\link{sheet_delete}()},
\code{\link{sheet_properties}()},
\code{\link{sheet_relocate}()},
\code{\link{sheet_rename}()},
\code{\link{sheet_resize}()}
}
\concept{worksheet functions}
\concept{write functions}
