//roarcat.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2019
 *
 *  This file is part of roarclients a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>

#define BUFSIZE 1024

void usage (void) {
 printf("roarcat [OPTIONS]... [FILE]\n");

 printf("\nOptions:\n\n");

 printf("  --server    SERVER    - Set server hostname\n"
        "  --rate  -R  RATE      - Set sample rate\n"
        "  --bits  -B  BITS      - Set bits per sample\n"
        "  --chans -C  CHANNELS  - Set number of channels\n"
        "  --codec -E  CODEC     - Set the codec\n"
        "  --aiprofile PROFILE   - Set audio profile\n"
        "  --wave                - Use Wave Audio (PCM) as input\n"
        "  --midi                - Use MIDI Audio as input\n"
        "  --light               - Use light control input\n"
        "  --raw                 - Use raw input\n"
        "  --complex             - Use complex input\n"
        "  --rdtcs               - Use Radio Data and Transmitter Control System input\n"
        "  --rel-id ID           - Set ID of relative stream\n"
        "  --role ROLE           - Set stream role\n"
        "  --help                - Show this help\n"
       );

}

int main (int argc, char * argv[]) {
 struct roar_audio_info info = {
  .rate = ROAR_AUDIO_INFO_INVALID,
  .bits = ROAR_AUDIO_INFO_INVALID,
  .channels = ROAR_AUDIO_INFO_INVALID,
  .codec = ROAR_AUDIO_INFO_INVALID};
 int    dir      = ROAR_DIR_PLAY;
 int    rel_id   = -1;
 int    role     = ROAR_ROLE_UNKNOWN;
 const char * server   = NULL;
 const char * k;
 int    i;
 const char * name = "roarcat";
 struct roar_connection    con;
 struct roar_stream        s;
 struct roar_vio_calls     file, * stream;
 struct roar_vio_defaults  def;
 int file_opened = 0;

 // TODO: make use of roar_stdin.
 if ( roar_vio_open_fh(&file, ROAR_STDIN) == -1 )
  return 1;

 if ( roar_vio_dstr_init_defaults(&def, ROAR_VIO_DEF_TYPE_NONE, O_RDONLY, 0644) == -1 )
  return 1;

 for (i = 1; i < argc; i++) {
  k = argv[i];

  if ( !strcmp(k, "--server") || !strcmp(k, "-s") ) {
   ROAR_CKHAVEARGS(1);
   server = argv[++i];
  } else if ( !strcmp(k, "-n") ) {
   ROAR_CKHAVEARGS(1);
   name = argv[++i];
  } else if ( !strcmp(k, "--rate") || !strcmp(k, "-r") || !strcmp(k, "-R") ) {
   ROAR_CKHAVEARGS(1);
   info.rate = roar_str2rate(argv[++i]);
  } else if ( !strcmp(k, "--bits") || !strcmp(k, "-B") ) {
   ROAR_CKHAVEARGS(1);
   info.bits = roar_str2bits(argv[++i]);
  } else if ( !strcmp(k, "-b") ) {
   info.bits = 8;
  } else if ( !strcmp(k, "--channels") || !strcmp(k, "--chans") || !strcmp(k, "-C") ) {
   ROAR_CKHAVEARGS(1);
   info.channels = roar_str2channels(argv[++i]);
  } else if ( !strcmp(k, "-m") ) {
   info.channels = 1;
  } else if ( !strcmp(k, "--codec") || !strcmp(k, "-E") ) {
   ROAR_CKHAVEARGS(1);
   if ( (info.codec = roar_str2codec(argv[++i])) == ROAR_AUDIO_INFO_INVALID ) {
    fprintf(stderr, "Error: Unknown codec: %s\n", argv[i]);
    return 1;
   }
  } else if ( !strcmp(k, "--aiprofile") ) {
   ROAR_CKHAVEARGS(1);
   if ( roar_profile2info(&info, argv[++i]) == -1 ) {
    fprintf(stderr, "Error: Can not load audio profile: %s: %s\n", argv[i], roar_error2str(roar_error));
    return 1;
   }

  } else if ( !strcmp(k, "--wave") ) {
   dir   = ROAR_DIR_PLAY;
  } else if ( !strcmp(k, "--midi") ) {
   dir   = ROAR_DIR_MIDI_IN;
  } else if ( !strcmp(k, "--light") ) {
   dir   = ROAR_DIR_LIGHT_IN;
  } else if ( !strcmp(k, "--raw") ) {
   dir   = ROAR_DIR_RAW_IN;
  } else if ( !strcmp(k, "--complex") ) {
   dir   = ROAR_DIR_COMPLEX_IN;
  } else if ( !strcmp(k, "--rdtcs") ) {
   dir   = ROAR_DIR_RDTCS_IN;

  } else if ( !strcmp(k, "--rel-id") ) {
   ROAR_CKHAVEARGS(1);
   rel_id = atoi(argv[++i]);
  } else if ( !strcmp(k, "--role") ) {
   ROAR_CKHAVEARGS(1);
   role = roar_str2role(argv[++i]);

  } else if ( !strcmp(k, "--help") || !strcmp(k, "-h") ) {
   usage();
   return 0;
  } else if ( !file_opened ) {
   file_opened = 1;
   if ( roar_vio_open_dstr(&file, k, &def, 1) == -1 ) {
    fprintf(stderr, "Error: can not open file: %s: %s\n", k, roar_error2str(roar_error));
    return 1;
   }
  } else {
   fprintf(stderr, "Error: unknown argument: %s\n", k);
   usage();
   return 1;
  }
 }

 switch (dir) {
  case ROAR_DIR_PLAY:
    if ( info.rate     == ROAR_AUDIO_INFO_INVALID ) info.rate     = ROAR_RATE_DEFAULT;
    if ( info.bits     == ROAR_AUDIO_INFO_INVALID ) info.bits     = ROAR_BITS_DEFAULT;
    if ( info.channels == ROAR_AUDIO_INFO_INVALID ) info.channels = ROAR_CHANNELS_DEFAULT;
    if ( info.codec    == ROAR_AUDIO_INFO_INVALID ) info.codec    = ROAR_CODEC_DEFAULT;
   break;
  case ROAR_DIR_MIDI_IN:
    if ( info.rate     == ROAR_AUDIO_INFO_INVALID ) info.rate     = 0;
    if ( info.bits     == ROAR_AUDIO_INFO_INVALID ) info.bits     = ROAR_MIDI_BITS;
    if ( info.channels == ROAR_AUDIO_INFO_INVALID ) info.channels = ROAR_MIDI_CHANNELS_DEFAULT;
    if ( info.codec    == ROAR_AUDIO_INFO_INVALID ) info.codec    = ROAR_CODEC_MIDI;
   break;
  case ROAR_DIR_LIGHT_IN:
    if ( info.rate     == ROAR_AUDIO_INFO_INVALID ) info.rate     = 0;
    if ( info.bits     == ROAR_AUDIO_INFO_INVALID ) info.bits     = ROAR_LIGHT_BITS;
    if ( info.channels == ROAR_AUDIO_INFO_INVALID ) info.channels = 0;
    if ( info.codec    == ROAR_AUDIO_INFO_INVALID ) info.codec    = ROAR_CODEC_DMX512;
   break;
  case ROAR_DIR_COMPLEX_IN:
    if ( info.rate     == ROAR_AUDIO_INFO_INVALID ) info.rate     = ROAR_COMPLEX_RATE;
    if ( info.bits     == ROAR_AUDIO_INFO_INVALID ) info.bits     = ROAR_COMPLEX_BITS;
    if ( info.channels == ROAR_AUDIO_INFO_INVALID ) info.channels = ROAR_COMPLEX_CHANNELS;
    if ( info.codec    == ROAR_AUDIO_INFO_INVALID ) info.codec    = ROAR_COMPLEX_CODEC;
   break;
  case ROAR_DIR_RDTCS_IN:
    if ( info.rate     == ROAR_AUDIO_INFO_INVALID ) info.rate     = ROAR_RDTCS_RATE;
    if ( info.bits     == ROAR_AUDIO_INFO_INVALID ) info.bits     = ROAR_RDTCS_BITS;
    if ( info.channels == ROAR_AUDIO_INFO_INVALID ) info.channels = ROAR_RDTCS_CHANNELS;
    if ( info.codec    == ROAR_AUDIO_INFO_INVALID ) info.codec    = ROAR_RDTCS_CODEC;
   break;
  case ROAR_DIR_RAW_IN:
  default:
    if ( info.rate     == ROAR_AUDIO_INFO_INVALID ) info.rate     = 0;
    if ( info.bits     == ROAR_AUDIO_INFO_INVALID ) info.bits     = 0;
    if ( info.channels == ROAR_AUDIO_INFO_INVALID ) info.channels = 0;
    if ( info.codec    == ROAR_AUDIO_INFO_INVALID ) info.codec    = ROAR_CODEC_DEFAULT;
   break;
 }

 if ( roar_simple_connect(&con, server, name) == -1 ) {
  fprintf(stderr, "Error: can not connect to server: %s\n", roar_error2str(roar_error));
  return 10;
 }

 if ( roar_stream_new_by_info(&s, &info) == -1 ) {
  fprintf(stderr, "Error: can not create stream\n");
  roar_disconnect(&con);
  return 20;
 }

 if ( rel_id != -1 ) {
  if ( roar_stream_set_rel_id(&s, rel_id) ) {
   fprintf(stderr, "Error: can not set id or realative stream\n");
   roar_disconnect(&con);
   return 21;
  }
 }

 if ( roar_stream_connect(&con, &s, dir, -1) == -1 ) {
  fprintf(stderr, "Error: can not connect stream to server\n");
  roar_disconnect(&con);
  return 11;
 }

 if ( role != ROAR_ROLE_UNKNOWN ) {
  if ( roar_stream_set_role(&con, &s, role) == -1 ) {
   fprintf(stderr, "Warning: can not set stream role\n");
  }
 }

 if ( roar_stream_exec(&con, &s) == -1 ) {
  fprintf(stderr, "Error: can not exec stream\n");
  roar_disconnect(&con);
  return 12;
 }

 if ( (stream = roar_get_connection_vio2(&con)) == NULL ) {
  fprintf(stderr, "Error: can not get stream vio\n");
  roar_disconnect(&con);
  return 13;
 }

 if ( roar_vio_copy_data(stream, &file) == -1 ) {
  fprintf(stderr, "Error: can not copy data from source to server.\n");
  roar_vio_close(stream);
  roar_vio_close(&file);
  return 14;
 }

 roar_vio_close(stream);
 roar_vio_close(&file);

 return 0;
}

//ll
