/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was readten as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/
#include <klib/rc.h>
#include <vdb/table.h>
#include <vdb/cursor.h>

#include "reader-cmn.h"
#include "debug.h"
#include <sysalloc.h>

#include <stdlib.h>
#include <string.h>

struct TableReader {
    const VTable* tbl;
    const VCursor* curs;
    const TableReaderColumn* cols;
    int64_t curr;
};

rc_t CC TableReader_Make(const TableReader** cself, const VTable* table, TableReaderColumn* cols, size_t cache)
{
    rc_t rc = 0;
    TableReader* obj = NULL;

    if( cself == NULL || table == NULL || cols == NULL || cols->name == NULL ) {
        rc = RC(rcAlign, rcType, rcConstructing, rcParam, rcInvalid);
    } else if( (obj = calloc(1, sizeof(*obj))) == NULL ) {
        rc = RC(rcAlign, rcType, rcConstructing, rcMemory, rcExhausted);
    } else {
        obj->tbl = table;
        VTableAddRef(table);
        obj->cols = cols;
        if( (rc = VTableCreateCachedCursorRead(obj->tbl, &obj->curs, cache)) == 0 ) {
            while(cols->name != NULL) {
                if( !(cols->flags & ercol_Skip) ) {
                    if( (rc = VCursorAddColumn(obj->curs, &cols->idx, cols->name)) != 0) {
                        if( GetRCState(rc) == rcNotFound && (cols->flags & ercol_Optional) ) {
                            rc = 0;
                        }
                    }
                }
                cols++;
            }
            if( rc == 0 ) {
                rc = VCursorOpen(obj->curs);
            }
        }
    }
    if( rc == 0 ) {
        *cself = obj;
        ALIGN_DBG("ok%c", '!');
    } else {
        TableReader_Whack(obj);
        ALIGN_DBGERRP("table", rc, 0);
    }
    return rc;
}

void CC TableReader_Whack(const TableReader* cself)
{
    if( cself != NULL ) {
        VCursorRelease(cself->curs);
        VTableRelease(cself->tbl);
        free((TableReader*)cself);
    }
}

rc_t CC TableReader_ReadRow(const TableReader* cself, int64_t rowid)
{
    rc_t rc = 0;
    TableReaderColumn* c = NULL;

    if( cself == NULL ) {
        rc = RC(rcAlign, rcType, rcOpening, rcSelf, rcNull);
    } else if( cself->curr != rowid &&
               (rc = VCursorCloseRow(cself->curs)) == 0 &&
               (rc = VCursorSetRowId(cself->curs, rowid)) == 0 &&
               (rc = VCursorOpenRow(cself->curs)) == 0 ) {
        uint32_t boff = 0;
        c = (TableReaderColumn*)(cself->cols);
        while(c->name != NULL) {
            if (c->idx != 0) {
                rc = VCursorCellData(cself->curs, c->idx, NULL, (const void**)&c->base, &boff, &c->len);
                if (rc) {
                    if (c->flags & ercol_Optional) {
                        rc = 0;
                    }
                }
                else if (boff != 0) {
                    rc = RC(rcAlign, rcType, rcReading, rcData, rcUnsupported);
                }
            }
            c++;
        }
    }
    if( rc != 0 ) {
        ALIGN_DBGERRP("column %s row %ld", rc, c ? c->name : "<none>", rowid);
    } else {
        ((TableReader*)cself)->curr = rowid;
    }
    return rc;
}

rc_t CC TableReader_IdRange(const TableReader* cself, int64_t* first, uint64_t* count)
{
    return VCursorIdRange(cself->curs, 0, first, count);
}
