/*==============================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include "sra-stat.vers.h"

#include <kapp/main.h>

#include <sra/sradb.h>
#include <sra/sradb-priv.h>
#include <sra/types.h>

#include <kdb/database.h> /* KDatabase */
#include <kdb/table.h>
#include <kdb/meta.h> /* KMetadata */
#include <kdb/namelist.h> /* KMDataNodeListChild */
#include <kdb/kdb-priv.h>

#include <kfs/directory.h>

#include <klib/container.h>
#include <klib/namelist.h>
#include <klib/log.h>
#include <klib/out.h>
#include <klib/debug.h> /* DBGMSG */
#include <klib/rc.h>

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define DISP_RC(rc, msg) (void)((rc == 0) ? 0 : LOGERR(klogInt, rc, msg))

#define DISP_RC2(rc, name, msg) (void)((rc == 0) ? 0 : \
    PLOGERR(klogInt, (klogInt, rc, \
        "$(name): $(msg)", "name=%s,msg=%s", name, msg)))

#define DISP_RC_Read(rc, name, spot, msg) (void)((rc == 0) ? 0 : \
    PLOGERR(klogInt, (klogInt, rc, "column $(name), spot $(spot): $(msg)", \
        "name=%s,spot=%lu,msg=%s", name, msg)))

#define DESTRUCT(type, obj) do { rc_t rc2 = type##Release(obj); \
    if (rc2 && !rc) { rc = rc2; } obj = NULL; } while (false)

#define MAX_NREADS 2*1024
typedef struct srastat_parms {
    const char* table_path;

    bool xml; /* output format (txt or xml) */
    bool printMeta;
    bool quick; /* quick mode: stats from meta */

    spotid_t start, stop;

    bool hasSPOT_GROUP;
    bool variableReadLength;
} srastat_parms;
typedef struct SraStatsMeta {
    bool found;
    uint64_t BASE_COUNT;
    uint64_t BIO_BASE_COUNT;
    uint64_t CMP_BASE_COUNT; /* compressed base count when present */
    uint64_t spot_count;
    char* spot_group; /* optional */
} SraStatsMeta;
typedef struct MetaDataStats {
    bool found;
    SraStatsMeta table;
    uint32_t spotGroupN;
    SraStatsMeta *spotGroup;
} MetaDataStats;
typedef struct SraStats {
    BSTNode n;
    char     spot_group[1024]; /* SPOT_GROUP Column */
    uint64_t spot_count; /*** spot count ***/
    uint64_t spot_count_mates; /*** spots with mates ***/
    uint64_t bio_len; /** biological len **/
    uint64_t bio_len_mates; /** biological len when mates are present**/
    uint64_t total_len; /** total len **/
    uint64_t bad_spot_count; /** number of spots flagged with rd_filter=2 **/
    uint64_t bad_bio_len;        /** biological length of bad spots ***/
    uint64_t filtered_spot_count; /* number of spots flagged with rd_filter=2 */
    uint64_t filtered_bio_len;   /** biological length of filtered spots **/
    uint64_t total_cmp_len; /* CMP_READ : compressed */
} SraStats;
typedef struct SraStatsTotal {
    uint64_t BASE_COUNT;
    uint64_t BIO_BASE_COUNT;
    uint64_t spot_count;
    uint64_t total_cmp_len; /* CMP_READ : compressed */
} SraStatsTotal;
typedef struct SraSizeStats { uint64_t size; } SraSizeStats;
typedef struct Formatter {
    char name[256];
    char vers[256];
} Formatter;
typedef struct Loader {
    char date[256];
    char name[256];
    char vers[256];
} Loader;
typedef struct SraMeta {
    uint32_t metaVersion;
    int32_t tblVersion;
    time_t loadTimestamp;
    Formatter formatter;
    Loader loader;
} SraMeta;
static
void SraStatsMetaDestroy(SraStatsMeta* self)
{
    if (self == NULL)
    {   return; }
    free(self->spot_group);
}
static
rc_t CC visitor(const KDirectory* dir,
    uint32_t type, const char* name, void* data)
{
    rc_t rc = 0;

    SraSizeStats* sizes = (SraSizeStats*) data;

    if (type & kptAlias)
    {   return rc; }

    assert(sizes);

    switch (type) {
        case kptFile: {
            uint64_t size = 0;
            rc = KDirectoryFileSize(dir, &size, name);
            DISP_RC2(rc, name, "while calling KDirectoryFileSize");
            if (rc == 0) {
                sizes->size += size;
                DBGMSG(DBG_APP, DBG_COND_1,
                    ("File '%s', size %lu\n", name, size));
            }
            break;
        }
        case kptDir: 
            DBGMSG(DBG_APP, DBG_COND_1, ("Dir '%s'\n", name));
            rc = KDirectoryVisit(dir, false, visitor, sizes, name);
            DISP_RC2(rc, name, "while calling KDirectoryVisit");
            break;
        default:
            rc = RC(rcExe, rcDirectory, rcVisiting, rcType, rcUnexpected);
            DISP_RC2(rc, name, "during KDirectoryVisit");
            break;
    }

    return rc;
}

static
rc_t get_size(const SRATable* tbl, SraSizeStats* sizes)
{
    rc_t rc = 0;

    const KDatabase* kDb = NULL;
    const KTable* kTbl = NULL;
    const KDirectory* dir = NULL;

    assert(tbl && sizes);

    rc = SRATableGetKTableRead(tbl, &kTbl);
    DISP_RC(rc, "while calling SRATableGetKTableRead");

    if (rc == 0) {
        rc = KTableOpenParentRead(kTbl, &kDb);
        DISP_RC(rc, "while calling KTableOpenParentRead");
    }

    if (rc == 0 && kDb) {
        while (rc == 0) {
            const KDatabase* par = NULL;
            rc = KDatabaseOpenParentRead(kDb, &par);
            DISP_RC(rc, "while calling KDatabaseOpenParentRead");
            if (par == NULL)
            {   break; }
            else {
                DESTRUCT(KDatabase, kDb);
                kDb = par;
            }
        }
        if (rc == 0) {
            rc = KDatabaseOpenDirectoryRead(kDb, &dir);
            DISP_RC(rc, "while calling KDatabaseOpenDirectoryRead");
        }
    }

    if (rc == 0 && dir == NULL) {
        rc = KTableOpenDirectoryRead(kTbl, &dir);
        DISP_RC(rc, "while calling KTableOpenDirectoryRead");
    }

    memset(sizes, 0, sizeof *sizes);

    if (rc == 0) {
        rc = KDirectoryVisit(dir, false, visitor, sizes, NULL);
        DISP_RC(rc, "while calling KDirectoryVisit");
    }

    DESTRUCT(KDirectory, dir);
    DESTRUCT(KTable, kTbl);
    DESTRUCT(KDatabase, kDb);

    return rc;
}

static
rc_t readTxtAttr(const KMDataNode* self, const char* name, const char* attrName,
    char* buf, size_t buf_size)
{
    rc_t rc = 0;
    size_t num_read = 0;
    if (self == NULL) {
        rc = RC(rcExe, rcMetadata, rcReading, rcSelf, rcNull);
        return rc;
    }
    assert(self && name && attrName && buf && buf_size);
    rc = KMDataNodeReadAttr(self, attrName, buf, buf_size, &num_read);
    if (rc != 0) {
        PLOGERR(klogErr, (klogErr, rc, "Cannot read '$(name)@$(attr)'",
            "name=%s,attr=%s", name, attrName));
    }
    else { DBGMSG(DBG_APP, DBG_COND_1, ("%s@%s = %s\n", name, attrName, buf)); }
    return rc;
}
static
rc_t get_load_info(const KMetadata* meta, SraMeta* info)
{
    rc_t rc = 0;
    const KMDataNode* node = NULL;
    assert(meta && info);
    memset(info, 0, sizeof *info);
    info->metaVersion = 99;
    info->tblVersion = -1;
    if (rc == 0) {
        rc = KMetadataVersion(meta, &info->metaVersion);
        DISP_RC(rc, "While calling KMetadataVersion");
        if (rc == 0) {
            DBGMSG(DBG_APP, DBG_COND_1,
                ("KMetadataVersion=%d\n",info->metaVersion));
            switch (info->metaVersion) {
                case 1:
                    info->tblVersion = 0;
                    break;
                case 2: {
                    const char name[] = "SOFTWARE/loader";
                    rc = KMetadataOpenNodeRead(meta, &node, name);
                    if (rc != 0) {
                        if (GetRCState(rc) == rcNotFound) {
                            DBGMSG(DBG_APP,DBG_COND_1,("%s: not found\n",name));
                            rc = 0;
                        }
                        DISP_RC2(rc, name,
                            "while calling KMetadataOpenNodeRead");
                    }
                    else {
                        rc = readTxtAttr(node, name, "vers",
                            info->loader.vers, sizeof info->loader.vers);
                        if (rc == 0) {
                            if (strncmp("2.", info->loader.vers, 2) == 0)
                            {      info->tblVersion = 2; }
                            else { info->tblVersion = 1; }
                        }
                        else if (GetRCState(rc) == rcNotFound) {
                            DBGMSG(DBG_APP,DBG_COND_1,
                                ("%s/@%s: not found\n", name, "vers"));
                            rc = 0;
                        }
                        if (rc == 0) {
                            rc = readTxtAttr(node, name, "date",
                                info->loader.date, sizeof info->loader.date);
                            if (GetRCState(rc) == rcNotFound) {
                                DBGMSG(DBG_APP,DBG_COND_1,
                                    ("%s/@%s: not found\n", name, "date"));
                                rc = 0;
                            }
                        }
                        if (rc == 0) {
                            rc = readTxtAttr(node, name, "name",
                                info->loader.name, sizeof info->loader.name);
                            if (GetRCState(rc) == rcNotFound) {
                                DBGMSG(DBG_APP,DBG_COND_1,
                                    ("%s/@%s: not found\n", name, "name"));
                                rc = 0;
                            }
                        }
                    }
                    break;
                }
                default:
                    rc = RC(rcExe, rcMetadata, rcReading, rcData, rcUnexpected);
                    PLOGERR(klogErr, (klogErr, rc,
                        "Unexpected MetadataVersion: $(version)",
                        "version=%d", info->metaVersion));
                    break;
            }
        }
    }
    if (rc == 0) {
        const char name[] = "SOFTWARE/formatter";
        DESTRUCT(KMDataNode, node);
        rc = KMetadataOpenNodeRead(meta, &node, name);
        if (rc != 0) {
            if (GetRCState(rc) == rcNotFound ) {
                DBGMSG(DBG_APP,DBG_COND_1,("%s: not found\n",name));
                rc = 0;
            }
            DISP_RC2(rc, name, "while calling KMetadataOpenNodeRead");
        }
        else {
            if (rc == 0) {
                rc = readTxtAttr(node, name, "vers",
                    info->formatter.vers, sizeof info->formatter.vers);
                if (GetRCState(rc) == rcNotFound) {
                    DBGMSG(DBG_APP,DBG_COND_1,
                        ("%s/@%s: not found\n", name, "vers"));
                    rc = 0;
                }
            }
            if (rc == 0) {
                rc = readTxtAttr(node, name, "name",
                    info->formatter.name, sizeof info->formatter.name);
                if (GetRCState(rc) == rcNotFound) {
                    DBGMSG(DBG_APP,DBG_COND_1,
                        ("%s/@%s: not found\n", name, "name"));
                    rc = 0;
                }
            }
        }
    }
    if (rc == 0) {
        const char name[] = "LOAD/timestamp";
        DESTRUCT(KMDataNode, node);
        rc = KMetadataOpenNodeRead(meta, &node, name);
        if (rc != 0) {
            if (GetRCState(rc) == rcNotFound ) {
                DBGMSG(DBG_APP,DBG_COND_1,("%s: not found\n",name));
                rc = 0;
            }
            DISP_RC2(rc, name, "while calling KMetadataOpenNodeRead");
        }
        else {
            size_t num_read = 0;
            size_t remaining = 0;
            rc = KMDataNodeRead(node, 0,
                &info->loadTimestamp, sizeof info->loadTimestamp,
                &num_read, &remaining);
            if (rc == 0) {
                if (remaining || num_read != sizeof info->loadTimestamp) {
                    rc = RC(rcExe, rcMetadata, rcReading, rcData, rcIncorrect);
                }
            }
            else {
                if (GetRCState(rc) == rcNotFound) {
                    DBGMSG(DBG_APP, DBG_COND_1, ("%s: not found\n", name));
                    rc = 0;
                }
            }
            DISP_RC2(rc, name, "while calling KMetadataOpenNodeRead");
        }
    }
    DESTRUCT(KMDataNode, node);
    return rc;
}

static
rc_t readStatsMetaNode(const KMDataNode* parent, const char* parentName,
    const char* name, uint64_t* result, bool quick, bool optional)
{
    rc_t rc = 0;
    const KMDataNode* node = NULL;
    assert(parent && parentName && name && result);
    rc = KMDataNodeOpenNodeRead (parent, &node, name);
    if (rc != 0)
    {
        if (GetRCState(rc) == rcNotFound && optional)
        {
            *result = 0;
            rc = 0;
        }
        else if ( quick )
        {
            PLOGERR(klogInt, (klogInt, rc, "while opening $(parent)/$(child)",
                "parent=%s,child=%s", parentName, name));
        }
    }
    else
    {
        rc = KMDataNodeReadAsU64(node, result);
        if ( rc != 0 && quick )
        {
            PLOGERR(klogInt, (klogInt, rc, "while reading $(parent)/$(child)",
                "parent=%s,child=%s", parentName, name));
        }
    }
    return rc;
}
static
rc_t readStatsMetaNodes(const KMDataNode* parent, const char* name,
                        SraStatsMeta* stats, bool quick)
{
    rc_t rc = 0;
    const char* parentName = name;
    assert(parent && stats);
    if (parentName == NULL)
    {   parentName = "STATS/TABLE"; }
    if (rc == 0 && name) {
        stats->spot_group = strdup(name);
        if (stats->spot_group == NULL)
        {   rc = RC(rcExe, rcStorage, rcAllocating, rcMemory, rcExhausted); }
    }
    if (rc == 0) {
        rc = readStatsMetaNode
            (parent, parentName, "BASE_COUNT", &stats->BASE_COUNT, quick, false);
    }
    if (rc == 0) {
        rc = readStatsMetaNode
            (parent, parentName, "BIO_BASE_COUNT", &stats->BIO_BASE_COUNT, quick, false);
    }
    if (rc == 0) {
        rc = readStatsMetaNode
            (parent, parentName, "SPOT_COUNT", &stats->spot_count, quick, false);
    }
    if (rc == 0) {
        rc = readStatsMetaNode
            (parent, parentName, "CMP_BASE_COUNT", &stats->CMP_BASE_COUNT, quick, true);
    }
    if (rc == 0)
    {   stats->found = true; }
    else { SraStatsMetaDestroy(stats); }
    return rc;
}
static
rc_t get_stats_meta(const KMetadata* meta, MetaDataStats* stats, bool quick)
{
    rc_t rc = 0;
    assert(meta && stats);
    memset(stats, 0, sizeof *stats);
    if (rc == 0) {
        const char name[] = "STATS/TABLE";
        const KMDataNode* node = NULL;
        rc = KMetadataOpenNodeRead(meta, &node, name);
        if (rc != 0) {
            if (GetRCState(rc) == rcNotFound) {
                DBGMSG(DBG_APP,DBG_COND_1, ("%s: not found\n", name));
                rc = 0;
            }
            DISP_RC2(rc, name, "while calling KMetadataOpenNodeRead");
        }
        if (rc == 0 && node)
        {   rc = readStatsMetaNodes(node, NULL, &stats->table, quick); }
        DESTRUCT(KMDataNode, node);
    }
    if (rc == 0) {
        const char name[] = "STATS/SPOT_GROUP";
        const KMDataNode* parent = NULL;
        rc = KMetadataOpenNodeRead(meta, &parent, name);
        if (rc != 0) {
            if (GetRCState(rc) == rcNotFound) {
                DBGMSG(DBG_APP,DBG_COND_1, ("%s: not found\n", name));
                rc = 0;
            }
            DISP_RC2(rc, name, "while calling KMetadataOpenNodeRead");
        }
        if (rc == 0 && parent) {
            uint32_t count = 0, i = 0;
            KNamelist* names = NULL;
            rc = KMDataNodeListChild(parent, &names);
            DISP_RC2(rc, name, "while calling KMDataNodeListChild");
            if (rc == 0) {
                rc = KNamelistCount(names, &count);
                DISP_RC2(rc, name, "while calling KNamelistCount");
                if (rc == 0) {
                    stats->spotGroupN = count;
                    stats->spotGroup = calloc(count, sizeof *stats->spotGroup);
                    if (stats->spotGroup == NULL) {
                        rc = RC(rcExe,
                            rcStorage, rcAllocating, rcMemory, rcExhausted);
                    }
                }
            }
            for (i = 0; i < count && rc == 0; ++i) {
                const KMDataNode* node = NULL;
                const char* child = NULL;
                rc = KNamelistGet(names, i, &child);
                if (rc != 0) {
                    PLOGERR(klogInt, (klogInt, rc,
                        "while calling STATS/SPOT_GROUP::KNamelistGet($(idx))",
                        "idx=%i", i));
                }
                else {
                    rc = KMDataNodeOpenNodeRead(parent, &node, child);
                    DISP_RC2(rc, child, "while calling KMDataNodeOpenNodeRead");
                }
                if (rc == 0) {
                    rc = readStatsMetaNodes(node, child, &stats->spotGroup[i], quick);
                }
                DESTRUCT(KMDataNode, node);
            }
            DESTRUCT(KNamelist, names);
        }
        DESTRUCT(KMDataNode, parent);
    }
    if (rc == 0) {
        uint32_t i = 0;
        bool found = stats->table.found;
        for (i = 0; i < stats->spotGroupN && found; ++i)
        {   found = stats->spotGroup[i].found; }
        stats->found = found;
    }
    return rc;
}
static
void srastatmeta_print(const MetaDataStats* meta,
    const srastat_parms *pb)
{
    uint32_t i = 0;
    assert(pb && meta);
    for (i = 0; i < meta->spotGroupN; ++i) {
        const SraStatsMeta *ss = &meta->spotGroup[i];
        if (pb->xml) {
            if (pb->hasSPOT_GROUP) {
                OUTMSG(("  <Member member_name=\"%s\"", ss->spot_group));
            }
            OUTMSG((" spot_count=\"%ld\" base_count=\"%ld\"",
                ss->spot_count, ss->BASE_COUNT));
            OUTMSG((" base_count_bio=\"%ld\"", ss->BIO_BASE_COUNT));
            if (ss->CMP_BASE_COUNT > 0)
            {   OUTMSG((" cmp_base_count=\"%ld\"", ss->CMP_BASE_COUNT)); }
            if (pb->hasSPOT_GROUP)
            {   OUTMSG(("/>\n")); }
        }
        else {
            const char* spot_group = "";
            if (strcmp("default", ss->spot_group) != 0)
            {   spot_group = ss->spot_group; }
            OUTMSG(("%s|%s|%ld:%ld:%ld|:|:|:\n",
                pb->table_path, spot_group, ss->spot_count, ss->BASE_COUNT,
                ss->BIO_BASE_COUNT));
        }
    }
}
static
void CC srastat_print( BSTNode* n, void* data )
{
   const srastat_parms *pb = (const srastat_parms*) data;
   const SraStats *ss = ( const SraStats* ) n;
   assert(pb && ss);
   if (pb->xml) {
        if (pb->hasSPOT_GROUP) {
            OUTMSG(("  <Member member_name=\"%s\"", ss->spot_group));
        }
        OUTMSG((" spot_count=\"%ld\" base_count=\"%ld\"", ss->spot_count, ss->total_len));
        OUTMSG((" base_count_bio=\"%ld\"", ss->bio_len));
        OUTMSG((" spot_count_mates=\"%ld\" base_count_bio_mates=\"%ld\"", ss->spot_count_mates, ss->bio_len_mates));
        OUTMSG((" spot_count_bad=\"%ld\" base_count_bio_bad=\"%ld\"", ss->bad_spot_count, ss->bad_bio_len));
        OUTMSG((" spot_count_filtered=\"%ld\" base_count_bio_filtered=\"%ld\"", ss->filtered_spot_count, ss->filtered_bio_len));
        if (ss->total_cmp_len > 0)
        {   OUTMSG((" cmp_base_count=\"%ld\"", ss->total_cmp_len)); }
        if (pb->hasSPOT_GROUP) {
            OUTMSG(("/>\n"));
        }
    }
    else {
        OUTMSG(("%s|%s|%ld:%ld:%ld|%ld:%ld|%ld:%ld|%ld:%ld\n",
            pb->table_path,ss->spot_group,ss->spot_count,ss->total_len,ss->bio_len,ss->spot_count_mates,ss->bio_len_mates,
            ss->bad_spot_count,ss->bad_bio_len,ss->filtered_spot_count,ss->filtered_bio_len));
    }
}

static
int CC srastats_cmp ( const void *item, const BSTNode *n )
{
    const char *sg = item;
    const SraStats *ss = ( const SraStats* ) n;

    return strcmp(sg,ss->spot_group);
}

static
void print_results(srastat_parms* pb, const BSTree* tr,
    const SraSizeStats* sizes, const SraMeta* info,
    const MetaDataStats* meta, const SraStatsTotal* total)
{
    assert(pb && tr && sizes && info && meta && total);

    if (pb->quick) {
        if ( ! meta -> found )
            LOGMSG(klogWarn, "Statistics metadata not found");
        else
        {
            pb->hasSPOT_GROUP = meta->spotGroupN > 1 ||
                (meta->spotGroupN == 1
                 && strcmp("default", meta->spotGroup[0].spot_group) != 0);
        }
    }
    if (meta->found)
    {
        bool mismatch = false;
        SraStats* ss = (SraStats*)BSTreeFind(tr, "", srastats_cmp);
        const SraStatsMeta* m = &meta->table;
        if (total->BASE_COUNT != m->BASE_COUNT
            || total->BIO_BASE_COUNT != m->BIO_BASE_COUNT
            || total->spot_count != m->spot_count
            || total->total_cmp_len != m->CMP_BASE_COUNT)
        { mismatch = true; }
        if (ss != NULL) {
            const SraStatsMeta* m = &meta->table;
            if (ss->total_len != m->BASE_COUNT
                || ss->bio_len != m->BIO_BASE_COUNT
                || ss->spot_count != m->spot_count
                || ss->total_cmp_len != m->CMP_BASE_COUNT)
            { mismatch = true; }
            else {
                uint32_t i = 0;
                for (i = 0; i < meta->spotGroupN && !mismatch; ++i) {
                    const char* spot_group = "";
                    m = &meta->spotGroup[i];
                    assert(m);
                    if (strcmp("default", m->spot_group) != 0)
                    {   spot_group = m->spot_group; }
                    ss = (SraStats*)BSTreeFind(tr, spot_group, srastats_cmp);
                    if (ss == NULL) {
                        mismatch = true;
                        break;
                    }
                    if (ss->total_len != m->BASE_COUNT
                        || ss->bio_len != m->BIO_BASE_COUNT
                        || ss->spot_count != m->spot_count
                        || ss->total_cmp_len != m->CMP_BASE_COUNT)
                    {
                        mismatch = true;
                        break;
                    }
                }
            }
        }
        if (mismatch) {
            LOGMSG(klogWarn,
                "Mismatch between calculated and recorded statistics");
        }
    }

    if (pb->xml) {
        OUTMSG(("<Run accession=\"%s\"", pb->table_path));
        if (!pb->quick || ! meta->found) {
            OUTMSG((" read_length=\"%s\"",
                pb->variableReadLength ? "variable" : "fixed"));
        }
        if (pb->hasSPOT_GROUP)
        {   OUTMSG((">\n")); }
    }

    if (pb->quick && meta->found)
    {   srastatmeta_print(meta, pb); }
    else { BSTreeForEach(tr, false, srastat_print, pb); }

    if (pb->xml) {
        if (!pb->hasSPOT_GROUP)
        {   OUTMSG((">")); }
        if (sizes) {
            if (!pb->hasSPOT_GROUP)
            {   OUTMSG(("\n")); }
            OUTMSG(("  <Size value=\"%lu\" units=\"bytes\"/>\n", sizes->size));
        }
        if (pb->printMeta && info->tblVersion >= 0) {
            OUTMSG(("  <Table vers=\"%d\">\n    <Meta vers=\"%d\">\n",
                info->tblVersion, info->metaVersion));
            if (info->formatter.name[0] || info->formatter.vers[0] ||
                info->loader.date[0] || info->loader.name[0] ||
                info->loader.vers[0])
            {
                OUTMSG(("      <SOFTWARE>\n"));
                if (info->formatter.name[0] || info->formatter.vers[0]) {
                    OUTMSG(("        <formatter"));
                    if (info->formatter.name[0])
                    {   OUTMSG((" name=\"%s\"", info->formatter.name)); }
                    if (info->formatter.vers[0])
                    {   OUTMSG((" vers=\"%s\"", info->formatter.vers)); }
                    OUTMSG(("/>\n"));
                }
                if (info->loader.date[0] || info->loader.name[0] ||
                    info->loader.vers[0])
                {
                    OUTMSG(("        <loader"));
                    if (info->loader.date[0])
                    {   OUTMSG((" date=\"%s\"", info->loader.date)); }
                    if (info->loader.name[0])
                    {   OUTMSG((" name=\"%s\"", info->loader.name)); }
                    if (info->loader.vers[0])
                    {   OUTMSG((" vers=\"%s\"", info->loader.vers)); }
                    OUTMSG(("/>\n"));
                }
                OUTMSG(("      </SOFTWARE>\n"));
            }
            if (info->loadTimestamp) {
                char       buf[80];
                struct tm* ts = localtime(&info->loadTimestamp);
                strftime(buf, sizeof(buf), "%a %Y-%m-%d %H:%M:%S %Z", ts);
                OUTMSG(("      <LOAD timestamp=\"%lX\">%s</LOAD>\n",
                    info->loadTimestamp, buf));
            }
            OUTMSG(("    </Meta>\n  </Table>\n"));
        }
        OUTMSG(("</Run>\n"));
    }
}

static
void CC bst_whack_free ( BSTNode *n, void *ignore )
{
    free ( n );
}

static 
int CC srastats_sort ( const BSTNode *item, const BSTNode *n )
{
    const SraStats *ss = ( const SraStats* ) item;
    return srastats_cmp(ss->spot_group,n);
}

static
rc_t sra_stat(srastat_parms* pb, const SRATable* tbl,
    BSTree* tr, SraStatsTotal* total)
{
    rc_t rc = 0;

/*  const char CMP_READ  [] = "CMP_READ"; */
    const char PRIMARY_ALIGNMENT_ID[] = "PRIMARY_ALIGNMENT_ID";
    const char RD_FILTER [] = "RD_FILTER";
    const char READ_LEN  [] = "READ_LEN";
    const char READ_TYPE [] = "READ_TYPE";
    const char SPOT_GROUP[] = "SPOT_GROUP";

/*  const SRAColumn* cCMP_READ = NULL; */
    const SRAColumn* cPRIMARY_ALIGNMENT_ID = NULL;
    const SRAColumn* cRD_FILTER = NULL;
    const SRAColumn* cREAD_LEN = NULL;
    const SRAColumn* cREAD_TYPE = NULL;
    const SRAColumn* cSPOT_GROUP = NULL;

    assert(pb && tbl && tr && total);

    if (rc == 0) {
        const char* name = READ_LEN;
        rc = SRATableOpenColumnRead(tbl, &cREAD_LEN, name, vdb_uint32_t);
        DISP_RC2(rc, name, "while calling SRATableOpenColumnRead");
    }
    if (rc == 0) {
        const char* name = READ_TYPE;
        rc = SRATableOpenColumnRead(tbl, &cREAD_TYPE, name, sra_read_type_t);
        DISP_RC2(rc, name, "while calling SRATableOpenColumnRead");
    }

    if (rc == 0) {
        {
            const char* name = SPOT_GROUP;
            rc = SRATableOpenColumnRead(tbl, &cSPOT_GROUP, name, vdb_ascii_t);
            if (GetRCState(rc) == rcNotFound)
            {   rc = 0; }
            DISP_RC2(rc, name, "while calling SRATableOpenColumnRead");
        }
        if (rc == 0) {
            if (rc == 0) {
                const char* name = RD_FILTER;
                rc = SRATableOpenColumnRead
                    (tbl, &cRD_FILTER, name, sra_read_filter_t);
                if (GetRCState(rc) == rcNotFound)
                {   rc = 0; }
                DISP_RC2(rc, name, "while calling SRATableOpenColumnRead");
            }
/*          if (rc == 0) {
                const char* name = CMP_READ;
                rc = SRATableOpenColumnRead
                    (tbl, &cCMP_READ, name, "INSDC:dna:text");
                if (GetRCState(rc) == rcNotFound)
                {   rc = 0; }
                DISP_RC2(rc, name, "while calling SRATableOpenColumnRead");
            } */
            if (rc == 0) {
                const char* name = PRIMARY_ALIGNMENT_ID;
                rc = SRATableOpenColumnRead
                    (tbl, &cPRIMARY_ALIGNMENT_ID, name, "I64");
                if (GetRCState(rc) == rcNotFound)
                {   rc = 0; }
                DISP_RC2(rc, name, "while calling SRATableOpenColumnRead");
            }
            if (rc == 0) {
                spotid_t spotid;
                pb->hasSPOT_GROUP = 0;
                rc = SRATableMaxSpotId(tbl, &spotid);
                DISP_RC(rc, "failed to read max spot id");
                if (rc == 0) {
                    bool fixedReadLength = true;
                    int g_nreads = 0;
                    uint32_t g_dREAD_LEN[MAX_NREADS];

                    memset(g_dREAD_LEN, 0, sizeof g_dREAD_LEN);

                    if (pb->start == 0)
                    {   pb->start = 1; }
                    if (pb->stop == 0 || pb -> stop > spotid)
                    {   pb->stop = spotid; }

                    for (spotid = pb->start; spotid <= pb->stop && rc == 0;
                        ++spotid)
                    {
                        SraStats* ss;
                        uint32_t dREAD_LEN  [MAX_NREADS];
                        uint8_t  dREAD_TYPE [MAX_NREADS];
                        uint8_t  dRD_FILTER [MAX_NREADS];
                        char     dSPOT_GROUP[MAX_NREADS] = "NULL";

                        const void* base;
                        bitsz_t boff, row_bits;
                        int nreads;

                        rc = Quitting();
                        if (rc)
                        {   LOGMSG(klogWarn, "Interrupted"); }

                        if (rc == 0) {
                            rc = SRAColumnRead(cREAD_LEN, spotid, &base, &boff, &row_bits);
                            DISP_RC_Read(rc, READ_LEN, spotid, "while calling SRAColumnRead");
                        }
                        if (rc == 0) {
                            if (boff & 7)
                            {   rc = RC(rcExe, rcColumn, rcReading, rcOffset, rcInvalid); }
                            if (row_bits & 7)
                            {   rc = RC(rcExe, rcColumn, rcReading, rcSize, rcInvalid); }
                            if ((row_bits >> 3) > sizeof(dREAD_LEN))
                            {   rc = RC(rcExe, rcColumn, rcReading, rcBuffer, rcInsufficient); }
                            DISP_RC_Read(rc, READ_LEN, spotid, "after calling SRAColumnRead");
                        }
                        if (rc == 0) {
                            int i, bio_len, bio_count, bad_cnt, filt_cnt;
                            memcpy(dREAD_LEN, ((const char*)base) + (boff>>3), row_bits>>3);
                            nreads = (row_bits >> 3) / sizeof(*dREAD_LEN);
                            if (spotid == pb->start) {
                                g_nreads = nreads;
                            }
#if 0
                            else if (g_nreads != nreads) {
                                rc = RC(rcExe, rcTable, rcReading, rcData, rcInconsistent);
                                PLOGERR(klogInt, (klogInt, rc,
                                    "spot=$(spot), ReadNumber=$(n), previous=$(prev)", "spot=%lu,n=%d,prev=%d", spotid, nreads, g_nreads));
                            }
#endif

                            if (rc == 0) {
                                rc = SRAColumnRead(cREAD_TYPE, spotid, &base, &boff, &row_bits);
                                DISP_RC_Read(rc, READ_TYPE, spotid, "while calling SRAColumnRead");
                                if (rc == 0) {
                                    if (boff & 7)
                                    {   rc = RC(rcExe, rcColumn, rcReading, rcOffset, rcInvalid); }
                                    if (row_bits & 7)
                                    {   rc = RC(rcExe, rcColumn, rcReading, rcSize, rcInvalid); }
                                    if ((row_bits >> 3) > sizeof(dREAD_TYPE))
                                    {   rc = RC(rcExe, rcColumn, rcReading, rcBuffer, rcInsufficient); }
                                    if ((row_bits >> 3) !=  nreads)
                                    {   rc = RC(rcExe, rcColumn, rcReading, rcData, rcIncorrect); }
                                    DISP_RC_Read(rc, READ_TYPE, spotid, "after calling SRAColumnRead");
                                }
                            }
                            if (rc == 0) {
                                memcpy(dREAD_TYPE, ((const char*)base) + (boff >> 3), row_bits >> 3);
                                if (cSPOT_GROUP) {
                                    rc = SRAColumnRead(cSPOT_GROUP, spotid, &base, &boff, &row_bits);
                                    DISP_RC_Read(rc, SPOT_GROUP, spotid, "while calling SRAColumnRead");
                                    if (rc == 0) {
                                        if (row_bits > 0) {
                                            pb -> hasSPOT_GROUP = 1;
                                            if (boff & 7)
                                            {   rc = RC(rcExe, rcColumn, rcReading, rcOffset, rcInvalid); }
                                            if (row_bits & 7)
                                            {   rc = RC(rcExe, rcColumn, rcReading, rcSize, rcInvalid); }
                                            if ((row_bits >> 3) > sizeof(dSPOT_GROUP))
                                            {   rc = RC(rcExe, rcColumn, rcReading, rcBuffer, rcInsufficient); }
                                            DISP_RC_Read(rc, SPOT_GROUP, spotid, "after calling SRAColumnRead");
                                            if (rc == 0) {
                                                memcpy(dSPOT_GROUP,((const char*)base) + (boff>>3),row_bits>>3);
                                                dSPOT_GROUP[row_bits>>3]='\0';
                                            }
                                        }
                                        else {  dSPOT_GROUP[0]='\0'; }
                                    } else { break; }
                                }
                            }
                            if (rc == 0) {
                                uint64_t cmp_len = 0; /* CMP_READ */
                                if (cRD_FILTER) {
                                    rc = SRAColumnRead(cRD_FILTER, spotid, &base, &boff, &row_bits);
                                    DISP_RC_Read(rc, RD_FILTER, spotid, "while calling SRAColumnRead");
                                    if (rc == 0) {
                                        if (boff & 7)
                                        {   rc = RC(rcExe, rcColumn, rcReading, rcOffset, rcInvalid); }
                                        if (row_bits & 7)
                                        {   rc = RC(rcExe, rcColumn, rcReading, rcSize, rcInvalid); }
                                        if ((row_bits >> 3) > sizeof(dRD_FILTER))
                                        {   rc = RC(rcExe, rcColumn, rcReading, rcBuffer, rcInsufficient); }
                                        DISP_RC_Read(rc, RD_FILTER, spotid, "after calling SRAColumnRead");
                                        if (rc == 0)
                                        {   memcpy(dRD_FILTER,((const char*)base) + (boff>>3),row_bits>>3); }
                                    } else { break; }
                                }

                                if (cPRIMARY_ALIGNMENT_ID) {
                                    rc = SRAColumnRead(cPRIMARY_ALIGNMENT_ID, spotid, &base, &boff, &row_bits);
                                    DISP_RC_Read(rc, PRIMARY_ALIGNMENT_ID, spotid, "while calling SRAColumnRead");
                                    if (boff & 7)
                                    {   rc = RC(rcExe, rcColumn, rcReading, rcOffset, rcInvalid); }
                                    if (row_bits & 7)
                                    {   rc = RC(rcExe, rcColumn, rcReading, rcSize, rcInvalid); }
                                    DISP_RC_Read(rc, PRIMARY_ALIGNMENT_ID, spotid, "after calling calling SRAColumnRead");
                                    if (rc == 0) {
                                        int i = 0;
                                        const int64_t* pii = base;
                                        assert(nreads);
                                        for (i = 0; i < nreads; ++i) {
                                            if (pii[i] == 0) {
                                                cmp_len += dREAD_LEN[i];
                                            }
                                        }
                                    }
                                }
/*                              if (cCMP_READ) {
                                    rc = SRAColumnRead(cCMP_READ, spotid, &base, &boff, &row_bits);
                                    DISP_RC_Read(rc, CMP_READ, spotid, "while calling SRAColumnRead");
                                    if (boff & 7)
                                    {   rc = RC(rcExe, rcColumn, rcReading, rcOffset, rcInvalid); }
                                    if (row_bits & 7)
                                    {   rc = RC(rcExe, rcColumn, rcReading, rcSize, rcInvalid); }
                                    DISP_RC_Read(rc, CMP_READ, spotid, "after calling calling SRAColumnRead");
                                    if (rc == 0)
                                    {   assert(cmp_len == row_bits >> 3); }
                                } */

                                ss = (SraStats*)BSTreeFind(tr, dSPOT_GROUP, srastats_cmp);
                                if (ss == NULL) {
                                    ss = calloc(1, sizeof(*ss));
                                    if (ss == NULL) {
                                        rc = RC(rcExe, rcStorage, rcAllocating, rcMemory, rcExhausted);
                                        break;
                                    }
                                    else {
                                        strcpy(ss->spot_group, dSPOT_GROUP);
                                        BSTreeInsert(tr, (BSTNode*)ss, srastats_sort);
                                    }
                                }
                                ++ss->spot_count;
                                ++total->spot_count;

                                ss->total_cmp_len += cmp_len;
                                total->total_cmp_len += cmp_len;

                                for (bio_len = bio_count = i = bad_cnt = filt_cnt = 0; (i < nreads) && (rc == 0); i++) {
                                    if (spotid == pb->start) {
                                        g_dREAD_LEN[i] = dREAD_LEN[i];
                                    }
                                    else if (g_dREAD_LEN[i] != dREAD_LEN[i]) { fixedReadLength = false; }

                                    if (dREAD_LEN[i] > 0) {
                                        bool biological = false;
                                        ss->total_len += dREAD_LEN[i];
                                        total->BASE_COUNT += dREAD_LEN[i];
                                        if ((dREAD_TYPE[i] & SRA_READ_TYPE_BIOLOGICAL) != 0) {
                                            biological = true;
                                            bio_len += dREAD_LEN[i];
                                            bio_count++;
                                        }
                                        if (cRD_FILTER) {
                                            switch (dRD_FILTER[i]) {
                                                case SRA_READ_FILTER_PASS:
                                                    break;
                                                case SRA_READ_FILTER_REJECT:
                                                case SRA_READ_FILTER_CRITERIA:
                                                    if (biological)
                                                    {   ss->bad_bio_len += dREAD_LEN[i]; }
                                                    bad_cnt++;
                                                    break;
                                                case SRA_READ_FILTER_REDACTED:
                                                    if (biological)
                                                    {   ss->filtered_bio_len += dREAD_LEN[i]; }
                                                    filt_cnt++;
                                                    break;
                                                default:
                                                    rc = RC(rcExe, rcColumn, rcReading, rcData, rcUnexpected);
                                                    PLOGERR(klogInt, (klogInt, rc,
                                                        "spot=$(spot), read=$(read), READ_FILTER=$(val)", "spot=%lu,read=%d,val=%d",
                                                        spotid, i, dRD_FILTER[i]));
                                                    break;
                                            }
                                        }
                                    }
                                }
                                ss->bio_len += bio_len;
                                total->BIO_BASE_COUNT += bio_len;
                                if (bio_count > 1) {
                                    ss->spot_count_mates++;
                                    ss->bio_len_mates += bio_len;
                                }
                                if (bad_cnt)
                                {   ss->bad_spot_count++; }
                                if (filt_cnt)
                                {   ss->filtered_spot_count++; }
                            }
                        }
                    }
    /******* Output Results here ************/

                    if (rc == 0) {
                        pb->variableReadLength = !fixedReadLength;
                    }
                }
                SRAColumnRelease(cSPOT_GROUP);
            }
        }
    }

    DESTRUCT(SRAColumn, cPRIMARY_ALIGNMENT_ID);
    DESTRUCT(SRAColumn, cRD_FILTER);
    DESTRUCT(SRAColumn, cREAD_TYPE);
    DESTRUCT(SRAColumn, cREAD_LEN);

    return rc;
}

static
rc_t run(srastat_parms* pb)
{
    rc_t rc = 0;
    const SRAMgr* mgr = NULL;

    assert(pb && pb->table_path);

    rc = SRAMgrMakeRead(&mgr);

    if (rc != 0) {
        LOGERR(klogInt, rc, "failed to open SRAMgr");
    }
    else {
        SraSizeStats sizes;
        SraMeta info;
        const SRATable* tbl = NULL;

        rc = SRAMgrOpenTableRead(mgr, &tbl, pb->table_path);
        if (rc != 0) {
            PLOGERR(klogInt, (klogInt, rc,
                "'$(spec)'", "spec=%s", pb->table_path));
        }
        else {
            MetaDataStats stats;
            SraStatsTotal total;
            const KTable* ktbl = NULL;
            const KMetadata* meta = NULL;

            bool quick = pb -> quick;

            BSTree tr;
            BSTreeInit(&tr);

            memset(&total, 0, sizeof total);

            rc = SRATableGetKTableRead(tbl, &ktbl);
            DISP_RC(rc, "While calling SRATableGetKTableRead");
            if (rc == 0) {
                rc = KTableOpenMetadataRead(ktbl, &meta);
                DISP_RC(rc, "While calling KTableOpenMetadataRead");
            }
            if (rc == 0) {
                rc = get_stats_meta(meta, &stats, quick);
                if (rc == 0) {
                    if (quick && !stats.found) {
                        LOGMSG(klogWarn, "Statistics metadata not found: "
                            "performing full table scan");
                        quick = false;
                    }
                }
                rc = 0;
            }
            if (rc == 0)
            {   rc = get_size(tbl, &sizes); }
            if (rc == 0 && pb->printMeta)
            {   rc = get_load_info(meta, &info); }
            if (rc == 0 && !quick)
            {   rc = sra_stat(pb, tbl, &tr, &total); }
            if (rc == 0)
            {   print_results(pb, &tr, &sizes, &info, &stats, &total); }
            BSTreeWhack(&tr, bst_whack_free, NULL);
            DESTRUCT(KMetadata, meta);
            DESTRUCT(KTable, ktbl);
        }
        {
            rc_t rc2 = SRATableRelease(tbl);
            if (rc == 0)
            {   rc = rc2; }
        }
    }

    {
        rc_t rc2 = SRAMgrRelease(mgr);
        if (rc == 0)
        {   rc = rc2; }
    }

    return rc;
}

/* Version  EXTERN
 *  return 4-part version code: 0xMMmmrrrr, where
 *      MM = major release
 *      mm = minor release
 *    rrrr = bug-fix release
 */
ver_t CC KAppVersion ( void )
{
    return SRA_STAT_VERS;
}


/* Usage
 */
#define OPTION_SPT_D "spot-desc"
#define OPTION_META  "meta"
#define OPTION_QUICK "quick"
#define OPTION_START "start"
#define OPTION_STOP  "stop"
#define OPTION_XML   "xml"

#define ALIAS_SPT_D "d"
#define ALIAS_META  "m"
#define ALIAS_QUICK "q"
#define ALIAS_START "b"
#define ALIAS_STOP  "e"
#define ALIAS_XML   "x"


static const char * spt_d_usage[] = { "print table spot descriptor", NULL };
static const char * meta_usage[] = { "print load metadata", NULL };
static const char * start_usage[] = { "starting spot id ( default 1 )", NULL };
static const char * stop_usage[] = { "ending spot id ( default max )", NULL };
static const char * quick_usage[] ={ "quick mode: get statistics from metadata",
                                        "not to scan the table", NULL };
static const char * xml_usage[] = { "output as XML (default is text)", NULL };

OptDef Options[] =
{
      { OPTION_SPT_D, ALIAS_SPT_D, NULL, spt_d_usage, 1, false, false }
    , { OPTION_META,  ALIAS_META,  NULL, meta_usage,  1, false, false }
    , { OPTION_QUICK, ALIAS_QUICK, NULL, quick_usage, 1, false, false }
    , { OPTION_START, ALIAS_START, NULL, start_usage, 1, true,  false }
    , { OPTION_STOP,  ALIAS_STOP,  NULL, stop_usage,  1, true,  false }
    , { OPTION_XML,   ALIAS_XML,   NULL, xml_usage,   0, false, false }
};


rc_t CC UsageSummary (const char * progname)
{
    return KOutMsg (
        "\n"
        "Usage:\n"
        "  %s [options] table\n"
        "\n"
        "Summary:\n"
        "  Display table statistics\n"
        "\n", progname);
}

const char UsageDefaultName[] = "sra-stat";
rc_t CC Usage (const Args * args)
{
    const char * progname = UsageDefaultName;
    const char * fullpath = UsageDefaultName;
    rc_t rc;

    if (args == NULL)
        rc = RC (rcApp, rcArgv, rcAccessing, rcSelf, rcNull);
    else
        rc = ArgsProgram (args, &fullpath, &progname);
    if (rc)
        progname = fullpath = UsageDefaultName;

    UsageSummary (progname);

    KOutMsg ("Options:\n");

    HelpOptionLine (ALIAS_XML, OPTION_XML, NULL, xml_usage);
    HelpOptionLine (ALIAS_START, OPTION_START, "row-id", start_usage);
    HelpOptionLine (ALIAS_STOP, OPTION_STOP, "row-id", stop_usage);
    HelpOptionLine (ALIAS_META, OPTION_META, NULL, meta_usage);
    HelpOptionLine (ALIAS_QUICK, OPTION_QUICK, NULL, quick_usage);
    HelpOptionsStandard ();
    HelpVersion (fullpath, KAppVersion());
    return rc;
}


/* KMain - EXTERN
 *  executable entrypoint "main" is implemented by
 *  an OS-specific wrapper that takes care of establishing
 *  signal handlers, logging, etc.
 *
 *  in turn, OS-specific "main" will invoke "KMain" as
 *  platform independent main entrypoint.
 *
 *  "argc" [ IN ] - the number of textual parameters in "argv"
 *  should never be < 0, but has been left as a signed int
 *  for reasons of tradition.
 *
 *  "argv" [ IN ] - array of NUL terminated strings expected
 *  to be in the shell-native character set: ASCII or UTF-8
 *  element 0 is expected to be executable identity or path.
 */
rc_t CC KMain ( int argc, char *argv [] )
{
    Args* args = NULL;
    rc_t rc = 0;

    srastat_parms pb;
    memset(&pb, 0, sizeof pb);

    rc = ArgsMakeAndHandle(&args, argc, argv, 1,
        Options, sizeof Options / sizeof (OptDef));
    if (rc == 0)
    {
        do
        {
            uint32_t pcount;
            const char* pc;

            rc = ArgsOptionCount (args, OPTION_START, &pcount);
            if (rc)
                break;

            if (pcount == 1)
            {
                rc = ArgsOptionValue (args, OPTION_START, 0, &pc);
                if (rc)
                    break;

                pb.start = AsciiToU32 (pc, NULL, NULL);
            }

            rc = ArgsOptionCount (args, OPTION_STOP, &pcount);
            if (rc)
                break;

            if (pcount == 1)
            {
                rc = ArgsOptionValue (args, OPTION_STOP, 0, &pc);
                if (rc)
                    break;

                pb.stop = AsciiToU32 (pc, NULL, NULL);
            }

            rc = ArgsOptionCount (args, OPTION_XML, &pcount);
            if (rc)
                break;

            if (pcount)
                pb.xml = true;

            rc = ArgsOptionCount (args, OPTION_QUICK, &pcount);
            if (rc)
                break;

            if (pcount)
                pb.quick = true;

            rc = ArgsOptionCount (args, OPTION_META, &pcount);
            if (rc)
                break;

            if (pcount)
                pb.printMeta = true;

            rc = ArgsParamCount (args, &pcount);
            if (rc)
                break;

            if (pcount == 0)
                return MiniUsage (args);

            rc = ArgsParamValue (args, 0, &pb.table_path);
            if (rc)
                break;
        } while (0);
    }

    if (rc == 0)
    {   rc = run(&pb); }
    else {  DISP_RC(rc, "while processing command line"); }

    {
        rc_t rc2 = ArgsWhack(args);
        if (rc == 0)
        {   rc = rc2; }
    }

    return rc;
}

/* EOF */
