/*  vim:expandtab
 *  conversion.c
 *  Wcalc
 *
 *  Created by Kyle Wheeler on Wed Jul 31 2002.
 *  Copyright (c) 2002 Kyle Wheeler. All rights reserved.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdlib.h> /* for calloc() */
#include <string.h> /* for strcmp() */
#include "number.h"
#include "calculator.h"
#include "conversion.h"

/* The conversion tables */
const char *conversion_names[] = {
    "Lengths",
    "Areas",
    "Volumes",
    "Angles",
    "Masses",
    "Speeds",
    "Forces",
    "Accelerations",
    "Temperatures",
    "Angles",
    "Pressures",
    "Metric",
    NULL
};

/* based on Meters */
const struct conversion lengths[] = {
    {"1e-10",           "Angstroms",            {"Å","angstrom","angstroms",NULL}},
    {/* the following value is from wikipedia. An AU is 149,597,870,691 meters
        (plus or minus 30). */
     "149597870691",
                        "Astronomical Units",   {"AU","ua","astronomical_unit","astronomical_units",NULL}},
    {"5.291772108e-11", "Atomic Unit",          {"au","atomic_unit","a0","b",NULL}},
    {"10e-18",          "Attometers",           {"am","attometer","attometers",NULL}},
    {"0.008466667",     "Barley Corn",          {"barley_corn",NULL}},
    {/* 1 cable length = 720 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters.
        So, 1 cable lenght = 720*12*0.0254 meters */
     "219.456",         "Cable Lengths (U.S.)", {"cable_length","cable_lengths",NULL}},
    {/* 1 cable length = 0.1 NM, 1 NM = 1852 meters.
        So, 1 cable length = 185.2 meters EXACTLY */
     "185.2",           "Cable Lengths (International)", {"cable_length.international","cable_lengths.international",NULL}},
    {/* 1 cable length = 608 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters.
        So, 1 cable length = 608*12*0.0254 meters */
     "185.3184",        "Cable Lengths (Imperial)", {"cable_length.Imp","cable_lengths.Imp",NULL}},
    {/* 1 calibre = 1 inch (copy the value from there) */
     "0.0254",          "Calibre",              {"cal","calibre","calibres",NULL}},
    {"1e-2",            "Centimeters",          {"cm","centimeter","centimeters",NULL}},
    {/* 1 Chain = 66 feet, 1 foot = 12 inches, 1 inch = 1/39.37 meters EXACTLY.
        So, 1 chain = 66*12/39.37 meters */
     "2.0116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325621e1",
                        "Chains (Gunter's)",  {"ch","chain","chains","chain_gunter","chain_surveyor",NULL}},
    {/* 1 chain = 100 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 chain = 100*12*0.0254 meters */
     "30.48",           "Chains (Ramsden's)", {"chain_ramsden","chain_engineer",NULL}},
    {/* 1 cubit =  18 inches, 1 inch = 0.0254 meters EXACTLY.
        So 1 cubit = 18*0.0254 meters */
     "0.4572",          "Cubits",               {"cubit","cubits",NULL}},
    {"10",              "Decameters",            {"dam","decameter","decameters",NULL}},
    {"0.1",             "Decimeters",           {"dm","decimeter","decimeters",NULL}},
    {/* 1 ell = 45 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 ell = 45*0.0254 meters. */
     "1.143",           "ell",                  {"ell",NULL}},
    {"1e18",            "Exameters",            {"Em","exameter","exameters",NULL}},
    {/* 1 fathom = 1/1000 NM, 1 NM = 1852 meters. (or 6 ft) */
     "1.852",           "Fathoms",              {"fth","fath","fathom","fathoms",NULL}},
    {"1e-15",           "Fermis (femtometer)",  {"fm","fermi","fermis","femtometer","femtometers",NULL}},
    {/* 1 finger = 7/8 inches, 1 inch = 0.0254 meters EXACTLY.
        So 1 finger = 7/8*0.0254 meters */
     "2.2225e-2",       "Fingers",              {"finger","fingers",NULL}},
    {/* 1 finger = 4.5 inches, 1 inch = 0.0254 meters EXACTLY.
        So 1 finger = 4.5*0.0254 meters */
     "0.1143",         "Fingers (cloth)",      {"finger_cloth","fingers_cloth",NULL}},
    {/* 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 foot = 12*0.0254 = 0.3048 meters EXACTLY (international) */
     "0.3048",
                        "Feet",                 {"ft","feet","foot","'",NULL}},
    {/* 1 foot = 12 inches, 1 inch = 1/39.37 meters EXACTLY (US Survey).
        So, 1 foot = 12/39.37 meters. */
     "3.0480060960121920243840487680975361950723901447802895605791211582423164846329692659385318770637541275082550165100330200660401320802641605283210566421132842265684531369062738125476250952501905003810007620015240030480060960121920243840487680975361950723901447802895605791211582423164846329692659385318770637541275082550165100330200660401320802641605283210566421132842265684531369062738125476250952501905003810007620015240030480060960121920243840487680975361950723901447802895605791211582423164846329692659385318770637541275082550165100330200660401320802641605283210566421132842265684531369062738125476250952501905003810007620015240030480060960121920243840487680975361950723901447802895605791211582423164846329692659385318770637541275082550165100330200660401320802641605283210566421132842265684531369062738125476250952501905003810007620015240030480060960121920243840487680975361950723901447802895605791211582423164846329692659385318770637541275082550165100330200660401320802641605283210566421132842265684531369062738125476250952501905003810007620015240030480060960121920243840487680975361950723901447802895605791211582423164846329692659385318770637541275082550165100330200660401320802641605283210566421132842265684531369062738125476250942e-1",
                        "Feet (U.S. Survey)",   {"feet.US","foot.US",NULL}},
    {/* 1 furlong = 660 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 furlong = 660*12*0.0254 meters */
     "201.168",         "Furlongs",             {"fur","furlong","furlongs",NULL}},
    {/* 1 gm = 6082 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 gm = 6082*12*0.0254 meters */
     "1853.7936",       "Geographic Mile",      {"geographic_mile","geographic_miles",NULL}},
    {"1e9",             "Gigameters",           {"Gm","gigameter","gigameters",NULL}},
    {/* 1 hand = 4 inches, 1 inch = 0.0254 meters EXACTLY.
        So 1 hand = 4*0.0254 meters */
     "0.1016",          "Hands",                {"hand","hands",NULL}},
    {"1e-2",            "Hectometers",          {"hm","hectometer","hectometers",NULL}},
    {/* 1 inch = 0.0254 meters EXACTLY. (international) */
     "0.0254",          "Inches",               {"in","inch","inches","\"",NULL}},
    {/* 1 inch = 1/39.37 meters EXACTLY. (US Survey) */
     "39.37",           "Inches (U.S. Survey)", {"inch.US","inches.US","\"",NULL}},
    {"1e3",             "Kilometers",           {"km","kilometer","kilometers",NULL}},
    {/* 1 league = 3 miles, 1 mile = 5280 feet, aka 63360 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 league = 3*63360*0.0254 meters */
     "4828.032",        "Leagues",              {"lea","league","leagues",NULL}},
    {"299792458",       "Light-second",         {"lightsecond","light-second",NULL}},
    {"17987547480",     "Light-minute",         {"lightminute","light-minute",NULL}},
    {"1079252848800",   "Light-hour",           {"lighthour","light-hour",NULL}},
    {"25902068371200",  "Light-day",            {"lightday","light-day",NULL}},
    {/* this value is from wikipedia; 1 lightyear is exactly 360.25 light-days */
     "9460730472580800","Light-years",          {"ly","LY","lightyear","lightyears","light-year","light-years",NULL}},
    {/* 1 line = 1/12 inch, 1 inch = 0.0254 meters EXACTLY. */
     "2.1166666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666677e-3",
                        "Lines",                {"ln","line","lines",NULL}},
    {/* 1 link = 1 foot (international/engineer), so copy value from foot */
     "0.3048",          "Links",                {"lnk","link","links","link_ramsden","link_engineer",NULL}},
    {/* 1 link = 33/50 foot, 1 foot = 12 inches, 1 inch = 1/39.37 meters EXACTLY.
        So, 1 link = 33/50*12/39.37 meters */
     "2.0116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325628651257302514605029210058420116840233680467360934721869443738887477774955549911099822199644399288798577597155194310388620777241554483108966217932435864871729743459486918973837947675895351790703581407162814325638e-1",
                        "Links (Surveyor's)",   {"link_gunter","link_surveyor",NULL}},
    {"1e6",             "Megameters",           {"Mm","megameter","megameters",NULL}},
    {"1",               "Meters",               {"m","meter","meters",NULL}},
    {/* 1 mickey = 1/200 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 mickey = 1/(39.37*200) = 1/7874 meters */
     "1.2700025400050800101600203200406400812801625603251206502413004826009652019304038608077216154432308864617729235458470916941833883667767335534671069342138684277368554737109474218948437896875793751587503175006350012700025400050800101600203200406400812801625603251206502413004826009652019304038608077216154432308864617729235458470916941833883667767335534671069342138684277368554737109474218948437896875793751587503175006350012700025400050800101600203200406400812801625603251206502413004826009652019304038608077216154432308864617729235458470916941833883667767335534671069342138684277368554737109474218948437896875793751587503175006350012700025400050800101600203200406400812801625603251206502413004826009652019304038608077216154432308864617729235458470916941833883667767335534671069342138684277368554737109474218948437896875793751587503175006350012700025400050800101600203200406400812801625603251206502413004826009652019304038608077216154432308864617729235458470916941833883667767335534671069342138684277368554737109474218948437896875793751587503175006350012700025400050800101600203200406400812801625603251206502413004826009652019304038608077216154432308864617729235458470916941833883667767335534671069342138684277368554737109474218948437896e-4",
                        "Mickeys",              {"mickey",NULL}},
    {/* 1 microinch = 1e-6 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 microinch = 1e-6*0.0254 meters */
     "2.54e-8",         "Microinches",          {"mci","microinch","microinches",NULL}},
    {"1e-6",            "Micrometers",          {"µm","micrometer","micrometers",NULL}},
    {"1e-6",            "Microns",              {"µ","micron","microns",NULL}},
    {/* 1 mil = 1.000e-3 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 mil = 1e-3*0.0254 meters */
     "2.54e-5",         "Mils",                 {"mils","mil","thou","thous",NULL}},
    {/* 1 mile = 5280 feet, aka 63360 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 mile = 63360*0.0254 = approx 1.609344e3 meters  */
     "1609.344",        "Miles",                {"mi","mile","miles",NULL}},
    {/* 1 mile = 8000 links, 1 link = 33/50 feet, 1 foot = 12 inches, 1 inch = 1/39.37 meters EXACTLY.
        So, 1 mile = 8000*33/50*12/39.37 meters */
     "1.6093472186944373888747777495554991109982219964439928879857759715519431038862077724155448310896621793243586487172974345948691897383794767589535179070358140716281432562865125730251460502921005842011684023368046736093472186944373888747777495554991109982219964439928879857759715519431038862077724155448310896621793243586487172974345948691897383794767589535179070358140716281432562865125730251460502921005842011684023368046736093472186944373888747777495554991109982219964439928879857759715519431038862077724155448310896621793243586487172974345948691897383794767589535179070358140716281432562865125730251460502921005842011684023368046736093472186944373888747777495554991109982219964439928879857759715519431038862077724155448310896621793243586487172974345948691897383794767589535179070358140716281432562865125730251460502921005842011684023368046736093472186944373888747777495554991109982219964439928879857759715519431038862077724155448310896621793243586487172974345948691897383794767589535179070358140716281432562865125730251460502921005842011684023368046736093472186944373888747777495554991109982219964439928879857759715519431038862077724155448310896621793243586487172974345948691897383794767589535179070358140716281432562865125730251460502e3",
                        "Miles (U.S. Survey)",  {"mile.US",NULL}},
    {"1e-3",            "Millimeters",          {"mm","millimeter","millimeters",NULL}},
    {/* 1 NL = 3 NM, 1 NM = 1852 meters EXACTLY. */
     "5556",            "Nautical Leagues",     {"NL","nl","nautical_league","nautical_leagues",NULL}},
    {"1852",            "Nautical Miles",       {"nmi","n_mile","n_miles","naut_mi","NM","n.m.","nautical_mile","nautical_miles",NULL}},
    {/* 1 NM(adm) = 6080 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 NM(adm) = 6080*12*0.0254 meters */
     "1853.184",        "Nautical Miles (Admiralty)",{"nautical_mile_admiralty","nautical_miles_admiralty",NULL}},
    {"1e-9",            "Nanometers",           {"nm","nanometer","nanometers",NULL}},
    {/* 1 pace = 2.5 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 pace = 2.5*12*0.0254 meters */
     "0.762",           "Paces",                {"pace","paces",NULL}},
    {/* 1 palm = 3 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 palm = 3*0.0254 meters */
     "0.0762",          "Palms",                {"palm","palms",NULL}},
    {/* Accordingto wikipedia, a parsec is AU/(tan 1"), where 1" is 1
        arcsecond. Since an arcminute is one sixtieth of a degree, an arcsecond
        is one 3600th of a degree (1/(60*60)). 1/tan(1/3600) is 206264.80624548
        (or so). Since an AU is 149,597,870,691 meters (plus/minus 30),
        that makes a parsec equal to... */
     "3.0856775812815532551662293307624661169222692502222270426081550857417878384638999664699494407044042861602932526374318282499488847652952634794645249141496980063074636858211454807681864726160105407307641779418881311355100588628419722041218901230894205020947799677195720827607257117521428907774778761405872034835465103085631243748951756567312805730872050305047898113261823840201264119675037598195042586352981308836070658938595405030339374553212392440123798481742099013358058995563100552008571554273559596511759202860948271101418509640612111277024400471208264695880181534297105712763766329751662367524228052400468732985263042049136985383758443443380079254183864371360974040014983302470733443592542873383270001285046347408234028157433258037566336348468957134529293451389011373577259268212813972619554240216664384144944608982944614500982419232106751210208167706334127233172935836463139828831751573494313836207461334598375683284660892786805672511641013972755087507259833750048075706711919506474598156162153778257464270314209204366432992995467924164800500074633057880739313620879317654923853591805278058571465232061488212786096086032554448490686544572188808938011635920419618939912850107913457825562320650694313831446304849437714222898707245175e16",
                        "Parsec",               {"pc","psc","parsec","parsecs",NULL}},
    {"1e15",            "Petameters",           {"Pm","petameter","petameters",NULL}},
    {/* 1 pica = 1/6 inch, 1 inch = 0.0254 meters EXACTLY.
        So, 1 pica = 0.0254/6 meters */
     "4.2333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333353e-3",
                        "Pica",                 {"pi","pica","picas",NULL}},
    {"4.512e-3",        "Pica (French)",        {"pica_french",NULL}},
    {"4.2175e-3",       "Pica (American)",      {"pica_american",NULL}},
    {"1e-12",           "Picometer",            {"pm","picometer","picometers",NULL}},
    {/* 1 point = 1/72 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1point = 0.0254/72 meters */
     "3.5277777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777763e-4",
                        "Points (DTP)",         {"pt","point","points","point_dtp","points_dtp",NULL}},
    {"3.513666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666e-4",
                        "Points (Johnson)",     {"johnson_point","johnson_points","points_johnson","point_johnson",NULL}},
    {"1.879857552e-4",  "Points (Trebuchet)",   {"trebuchet_point","trebuchet_points","points_trebuchet","point_trebuchet",NULL}},
    {"3.45e-4",         "Points (Fournier)",    {"fournier_point","fournier_points","point_fournier","points_fournier",NULL}},
    {"3.759715104e-4",  "Points (Didot)",       {"didot_point","didot_points","point_didot","points_didot",NULL}},
    {/* 1 quarter = 1/4 yard, 1 yard = 3 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 quarter = 0.25*3*12*0.0254 meters */
     "0.1524",          "Quarters",             {"quarter","quarters",NULL}},
    {/* 1 rod = 25 links, 1 link = 33/50 foot, 1 foot = 12 inches, 1 inch = 1/39.37 meters EXACTLY.
        So, 1 rod = 25*33/50*12/39.37 meters*/
     "5.0292100584201168402336804673609347218694437388874777749555499110998221996443992887985775971551943103886207772415544831089662179324358648717297434594869189738379476758953517907035814071628143256286512573025146050292100584201168402336804673609347218694437388874777749555499110998221996443992887985775971551943103886207772415544831089662179324358648717297434594869189738379476758953517907035814071628143256286512573025146050292100584201168402336804673609347218694437388874777749555499110998221996443992887985775971551943103886207772415544831089662179324358648717297434594869189738379476758953517907035814071628143256286512573025146050292100584201168402336804673609347218694437388874777749555499110998221996443992887985775971551943103886207772415544831089662179324358648717297434594869189738379476758953517907035814071628143256286512573025146050292100584201168402336804673609347218694437388874777749555499110998221996443992887985775971551943103886207772415544831089662179324358648717297434594869189738379476758953517907035814071628143256286512573025146050292100584201168402336804673609347218694437388874777749555499110998221996443992887985775971551943103886207772415544831089662179324358648717297434594869189738379476758953517907035814052e0",
                        "Rods",                 {"rd","rod","rods","pole","poles","perch","perches",NULL}},
    {/* 1 rope = 20 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 rope = 20*12*0.0254 meters */
     "6.096",           "Ropes",                {"rope","ropes",NULL}},
    {/* 1 span = 6 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 span = 6*0.0254 meters */
     "0.1524",          "Spans",                {"span","spans",NULL}},
    {/* 1 span = 9 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 span = 9*0.0254 meters */
     "0.2286",          "Spans (cloth)",        {"span_cloth","spans_cloth",NULL}},
    {"1e12",            "Spat",                 {"S","spat","spats",NULL}},
    {/* 1 stick = 2 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 stick = 2*0.0254 meters */
     "0.0508",          "Sticks",               {"stick","sticks",NULL}},
    {"1e12",            "Stigma",               {"stigma","stigmas","pm",NULL}},
    {/* 1 telegraph mile = 6087 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 telegraph mile = 6087*12*0.0254 meters */
     "1.8553176e3",     "Telegraph Mile",       {"telegraph_mile","telegraph_miles",NULL}},
    {"1e12",            "Terameters",           {"Tm","terameter","terameters",NULL}},
    {/* 1 twip = 1/1440 inch, 1 inch = 0.0254 meters EXACTLY.
        So, 1 twip = 0.0254/1440 meters */
     "1.7638888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888893e-5",
                        "Twips",                {"twip","twips",NULL}},
    {"0.8359",          "Varas",                {"vara","varas",NULL}},
    {"1.0021e-13",      "X Unit (aka. Siegbahn)",{"xu","xunit","xunits","siegbahn","siegbahns",NULL}},
    {/* 1 yard = 3 feet, 1 foot = 12 inches, 1 inch = 0.0254 meters EXACTLY.
        So, 1 yard = 3*12*0.0254 meters */
     "0.9144",          "Yards",                {"yd","yard","yards",NULL}},
    {"1e-24",           "Yoctometers",          {"ym","yoctometer","yoctometers",NULL}},
    {"1e24",            "Yottameters",          {"Ym","yottameter","yottameters",NULL}},
    {"1e-21",           "Zeptometers",          {"zm","zeptometer","zeptometers",NULL}},
    {"1e21",            "Zettameters",          {"Zm","zettameter","zettameters",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on square meters */
const struct conversion areas[] = {
    {"4046.8564224",/* EXACT */
                            "Acre",                         {"ac","A","acre","acres",NULL}},
    {"3344.51",             "Acre (Commercial)",            {"commercial_acre","commercial_acres","acre_commercial","acres_commercial",NULL}},
    {"6555",                "Acre (Irish)",                 {"irish_acre","irish_acres","acre_irish","acres_irish",NULL}},
    {"4046.873",            "Acre (U.S. Survey)",           {"survey_acre","survey_acres","acre_survey","acres_survey",NULL}},
    {"100",                 "Are",                          {"a","are","ares",NULL}},
    {"1e-28",               "Barn",                         {"b","barn","barns",NULL}},
    {/* 1 barony = 4000 acres, 1 acre = 4046.8564224 sq.m. EXACTLY.
        So, 1 barony = 4000*4046.8564224 sq.m. */
     "1.61874256896e7",     "Barony",                       {"barony","baronies",NULL}},
    {/* estimate... */
     "2.023221770682737240319370214656998546438306899547039215588833471320921126417471625370063492421868196787881732426840489464859900344367864146821099843183000038506071225552856842016409065783018158657272198517585250776863236643965216228988156338408792150936839914223551771012803563652400545572335254633969696071915e1",
                            "Base boxe (tin plated steel)", {"base_boxe",NULL}},
    {/* 1 board = 1inch x 1ft */
     "7.74192e-3",          "Board",                        {"bd","board",NULL}},
    {/* (1 sq.ft.)(1 bhp)/(240 BTU(IT)/h) */
     "12.958174",           "Boiler Horsepower Equivalent Direct Radiation",
                                                            {"bhp_EDR",NULL}},
    {/* estimate... */
     "3.305999986112183332250900563641124112558303300929789325905802207264226740664221594002750604630606542223660019712530364281515847019980891998125961680246092111352514042012279280725466865507524133144949184887277902155429993354271259285945652071796744450693961944931539190805938593793207959026039247786833451175548",
                            "Bin (Taiwan)",                 {"bin","bins",NULL}},
    {"3.305999986112183332250900563641124112558303300929789325905802207264226740664221594002750604630606542223660019712530364281515847019980891998125961680246092111352514042012279280725466865507524133144949184887277902155429993354271259285945652071796744450693961944931539190805938593793207959026039247786833451175548e0",
                            "Bu (Japan)",                   {"bu.JP",NULL}},
    {"4.410000059935411421183026945326372834220286611137416097234852550714893241410574459123776977064897518619480425367063529901504324339934038355394105297799988993678993564571415494611534079547367772947225780508895232355968395512690233633682393537001126240867341684697236269205893939385683827627936222183839342584823e2",
                            "Cantero (Ecuador)",            {"cantero",NULL}},
    {"3.59999992883365115038623756679406629216561690371584415565930818480619801965401056179783277117848271209262157970079531953156885860207229291215190314829252358363842940138069833403135660496662831986569486645100890753605376190657495222069254109056024640636625690269104873222487000576803257426119544521134092701997e2",
                            "Cao (Vietnam)",                {"cao",NULL}},
    {"1",                   "Centaire",                     {"ca","centiare","cetaires",NULL}},
    {/* 1 circular inch = pi/4 sq.in. */
     "5.067075e-4",         "Circular Inch",                {"circ_in","circular_inch","circular_inches",NULL}},
    {/* 1 circular mil = pi/4 sq.mil. */
     "5.067075e-10",        "Circular Mil",                 {"circ_mil","circular_mil","circular_mils",NULL}},
    {"5.067075e-10",        "Circular Thou",                {"circ_thou","circular_thou","circular_thous",NULL}},
    {"6.555004219518634722507932857010961029789315628092761379337154926794186297011997686958690831577225289899557638948505674888113915664699717672899027971926028031857770636361722081931551470838347892427381355396853421535580827673563136471057017764092463340400959555792402922725306907214324010437841874146580084196008e3",
                            "Colpa (Ireland)",              {"colpa","colpas","colps","collops",NULL}},
    {/* 1 cord = 192 bd */
     "1.48644864",          "Cord",                         {"cord","cords",NULL}},
    {"1000",                "Dunam",                        {"dunam",NULL}},
    {"10000",               "Hectare",                      {"ha","hectare","hectares",NULL}},
    {/* 1 hide = 100 ac */
     "4.0468564224e5",      "Hide",                         {"hide","hides",NULL}},
    {"7.168000005668004433796468337287356843034890464043065497687985566426709337293379097745799211651433314333710137544755094384672582410127590322307529625065049207134742281152149431268892610091244243875809771852194031378781046051473273764968402646532970250415978387899093934066229923308357507962660603791257315711045e2",
                            "Labor (Canada)",               {"labors.CA",NULL}},
    {"7.168921877767936226749335695305580159433126660761736049601417183348095659875996457041629760850310008857395925597874224977856510186005314437555358724534986713906111603188662533215234720992028343666961913197519929140832595217006200177147918511957484499557130203720106288751107174490699734278122232063773250664306e5",
                            "Labor (US)",                   {"labors.US","labor","labors",NULL}},
    {"6.988954745473511146111813427835229588488453222252157033414157323551558106922540222024191932798651551889692523435503146609191829178151272114341768042056107046884822359922353705123688337167291843685021311282890873797138692585607803314474402282414219495851725972132993458092415670182855642827043568966350969542482e3",
                            "Manzana (Costa Rica)",         {"manzana","manzanas","manzanas.CR",NULL}},
    {"9.999991104214131450062517606242864838417933904310996673964505814384485749445248909030705287556278200876729118378199394098140286543147032514097349550021498149182329015582451579743307156659731248424704585777615237492772173981803175795555072327681214571297252684797596160974187394671424264738587448046139476038248e3",
                            "Manzana (Argentina)",          {"manzanas.AR",NULL}},
    {"1.600000158147460645992839873719252674174717360835176739671488187373111373372316682382570568362205129117517401162779204399662355171520805286869609395540636978434616897660801372719958407217850103883113211836546692049334100348517466398606720871708803080712533383940520740051042092923494189069242888800589890028205e3",
                            "Rai (Thailand)",               {"rai",NULL}},
    {/* 1 rood = 1/4 acre, 1 acre = 4046.8564224 sq.m. EXACTLY
        So, 1 rood = 4046.8564224/4 sq.m. */
     "1.0117141056e3",      "Rood",                         {"rood","roods",NULL}},
    {"3.59999992883365115038623756679406629216561690371584415565930818480619801965401056179783277117848271209262157970079531953156885860207229291215190314829252358363842940138069833403135660496662831986569486645100890753605376190657495222069254109056024640636625690269104873222487000576803257426119544521134092701997e2",
                            "Sao (Vietnam)",                {"sao",NULL}},
    {"8.739999699024072597190560766011724964742017758167155430033351139059983759394245689275735294816373843754411729159745185002836032927872224764814179414791183782725927225713959562415556771860748079295857315156852256912567858020497911975184241066306223516990790845003392717450313012372857755891348934195345436654664",
                            "Scruple (ancient Rome)",       {"scruples.Rome",NULL}},
    {"2.589988110336e6",    "Section (US Survey)",          {"sec","section","sections",NULL}},
    {"9.290304353801154091418566889239602606869718848771024165861935609221250172644061708533453769824659381700018944674839184623380412865536832977961034392239792131671373970973612140722292639402112370615553355591170239284285000921522078762441451988952221740340250382150498762303665022427996846032920281509378670060128",
                            "Square",                       {"sq","square","squares",NULL}},
    {"8.360999920994321177587907040397686158370354160310125387631941962891131827421185224020767584796398398862318224957265861381726680680533099908465805581899964265632298314865067225899053221068136943617382439384628403896864201799676638681907600379359485425105095734056968052946366383381171988090307127138649016114877e1",
                            "Square (Sri Lanka)",           {"sq.SL",NULL}},
    {"1e-36",               "Square Attometer",             {"sq.am.","am2","square_attometer",NULL}},
    {"6.451600000025805371469845509191320707435529442123245191140166144103673088731225441279514880831979513788959750813070571019621127355870604679843533668010227339185504432925734940505033877235572640058484439547273739780365859541347322329319040875363552976419152218301495776911823523529142004795473690535005333951172e-8",
                            "Square Caliber",               {"sq.cal.","cal2","square_caliber",NULL}},
    {"1e-4",                "Square Centimeter",            {"sq.cm.","cm2","square_centimeter",NULL}},
    {"404.68564224",        "Square Chain",                 {"sq.chain","chain2","square_chain",NULL}},
    {"3.629024956115121479274781435318412530929479925610046753027408418176592990668264764512352827840701682245111697958272531341770230097648193034695161500021667325726687609305971817060116874144668763388228163064633134230769827737668372006218311369490336855293849998545641518734165007769944620370522491497465088272311e-4",
                            "Square Digit",                 {"sq.digit.","square_digit",NULL}},
    {"1e2",                 "Square Decameter",             {"sq.dam.","dam2","square_decameter",NULL}},
    {"1e-2",                "Square Decimeter",             {"sq.dm.","dm2","square_decimeter",NULL}},
    {"1e36",                "Square Exameter",              {"sq.Em.","Em2","square_exameter",NULL}},
    {"1e-30",               "Square Femtometer",            {"sq.fm.","fm2","square_femtometer",NULL}},
    {"9.290303999976199816213594024505880622266235765981699256234837482841961280186241242536366837724159960778754603941227262392924018289593397791368431306698621324939571609514158812968601329784961443484290814502159924128111609616044488463423363629132934679087173040863377016714275964833344984500370552446953699258831e-2",
                            "Square Foot",                  {"sq.ft","sq.ft.IN","ft2",NULL}},
    {"9.290341199944321962239728391079895381007507748698769587339156653622117423009029788869627721462743024004179490771308932469255347683139050331331702733192442260292701268222885912435703348136322134671641093162915840707221524442743393444595634067094523114407626461543917038191866957480866104752883111755511144193929e-2",
                            "Square Foot (U.S. Survey)",    {"sq.ft.US",NULL}},
    {"1e18",                "Square Gigameter",             {"sq.Gm.","Gm2","square_gigameter",NULL}},
    {"1e4",                 "Square Hectometer",            {"sq.hm.","hm2","square_hectometer",NULL}},
    {"6.451599999999928578716062176854032557323871669709217231070812650300131239545160671903373783364761529184604934013347379882062581226598819665644930809227374390428023740508140794058426495017818320040469733924648818019814139126101116987142699372767211608266628889752537217536490567427263879494854143719603846218604e-4",
                            "Square Inch",                  {"sq.in","sq.in.IN","in2","square_inch",NULL}},
    {"6.45162583333375249212597612369649690590669930475143890753797286751901735797328249972043220454036172672105543331756798712291379892548324552912891079113363578200382356940299355312792546055537686633488182355297146467925717147321189284121397085636435723290004199447084773018499262702741902422391746036979637780008e-4",
                            "Square Inch (U.S. Survey)",    {"sq.in.US",NULL}},
    {"1e6",                 "Square Kilometer",             {"sq.km.","km2","square_kilometer",NULL}},
    {"4.0468564224e-2",     "Square Link",                  {"sq.lnk.","square_link",NULL}},
    {"1e12",                "Square Megameter",             {"sq.Mm.","Mm2","square_megameter",NULL}},
    {"1",                   "Square Meter",                 {"sq.m.","square_meter","square_meters","m2",NULL}},
    {"6.4516e-10",          "Square Mil",                   {"sq.mil.","mil2","square_mil",NULL}},
    {"2.589988110336e6",    "Square Mile",                  {"sq.mi.","square_mile","square_miles",NULL}},
    {"2.589998e6",          "Square Mile (U.S. Survey)",    {"sq.mi.US","square_mile.US","square_miles.US",NULL}},
    {"1e-6",                "Square Millimeter",            {"sq.mm.","mm2","square_millimeter",NULL}},
    {"1e-12",               "Square Micrometer",            {"sq.µm.","µm2","square_micrometer",NULL}},
    {"1e-18",               "Square Nanometer",             {"sq.nm.","nm2","square_nanometer",NULL}},
    {"1e30",                "Square Petameter",             {"sq.Pm.","Pm2","square_petameter",NULL}},
    {"1e-24",               "Square Picometer",             {"sq.pm.","pm2","square_picometer",NULL}},
    {"25.29285264",         "Square Rod",                   {"sq.rd.","rd2","square_rod",NULL}},
    {"25.29285264",         "Square Pole",                  {"sq.pole","pole2","square_pole",NULL}},
    {"25.29285264",         "Square Perch",                 {"sq.perch","perch2","square_perch",NULL}},
    {"1e24",                "Square Terameter",             {"sq.Tm.","Tm2","square_terameter",NULL}},
    {"6.4516e-10",          "Square Thou",                  {"sq.thou","mil2","square_thou",NULL}},
    {"1e-48",               "Square Yoctometer",            {"sq.ym.","ym2","square_yoctometer",NULL}},
    {"1e48",                "Square Yottameter",            {"sq.Ym.","Ym2","square_yottameter",NULL}},
    {"1e-42",               "Square Zeptometer",            {"sq.zm.","zm2","square_zeptometer",NULL}},
    {"1e42",                "Square Zettameter",            {"sq.Zm.","Zm2","square_zettameter",NULL}},
    {"0.83612736",          "Square Yard",                  {"sq.yd.","yd2","square_yard",NULL}},
    {"1000",                "Stremma",                      {"stremma","stremmas",NULL}},
    {/* 1 township = 36 sq.mi.(US) */
     "9.323994e7",          "Township",                     {"twp","township","townships",NULL}},
    {/* 1 yardland = 30 acres */
     "1.21405692672e5",     "Yardland",                     {"yardland","yardlands",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on cubic meters */
const struct conversion volumes[] = {
    {"9.46352946e-4",       "#2.5 can",                 {"2.5can",NULL}},
    {"3.785411784e-3",      "#10 can",                  {"10can",NULL}},
    {"102.79015312896",     "Acre-inch",                {"ac.in.","acre_inch","acre_inches",NULL}},
    {"1233.48183754752",    "Acre-foot",                {"ac.ft.","acre_foot","acre_feet",NULL}},
    {"1e-21",               "Attoliter",                {"al","aL","attoliter","attoliters",NULL}},
    {/* 3 bu (Imp) */
     "109.10616e-3",        "Bag",                      {"bag",NULL}},
    {/* 105 qt US */
     "115.628198985075e-3", "Barrel (U.S. dry)",        {"bbl","bl.US","barrel.US","barrels.US",NULL}},
    {/* 31.5 gal US */
     "119.240471196e-3",    "Barrel (U.S. fluid)",      {"fl.bl.US","fluid_barrel.US","fluid_barrels.US",NULL}},
    {/* 42 gal (US) */
     "158.987.294.928e-3",  "Barrel (petroleum)",       {"bl","bl.petrol","barrel.petrol","barrels.petrol",NULL}},
    {/* 36 gal Imp */
     "163.65924e-3",        "Barrel (Imperial)",        {"bl.Imp","barrel.Imp","barrels.Imp",NULL}},
    {/* 282 cu. in. */
     "4.621152048e-3",      "Beer Gallon",              {"beer_gal","beer_gallon",NULL}},
    {/* 10 fl.oz. */
     "284.130625e-6",       "Breakfast Cup",            {"breakfast_cup",NULL}},
    {/* 144 cu.in. */
     "2.359737216e-3",      "Board Foot",               {"bd_ft","fbm","BF","board_feet",NULL}},
    {/* 4 gal (Imp) */
     "18.18436e-3",         "Bucket",                   {"bkt","bucket","buckets",NULL}},
    {/* 2150.42 cu.in. */
     "35.23907016688e-3",   "Bushel (U.S. dry level)",  {"bu.USlvl","bushel.USlvl",NULL}},
    {/* 8 gal Imp. */
     "36.36872e-3",         "Bushel (Imperial)",        {"bu.Imp","bushel.Imp",NULL}},
    {/* 1.25 bu (US lvl) */
     "44.0488377086e-3",    "Bushel (U.S. dry heaped)", {"bu.US","bushel.US",NULL}},
    {/* 126 gal (US.fl.) */
     "0.476961884784",      "Butt",                     {"butt","butts",NULL}},
    {"1e-5",                "Centiliter",               {"cl","cL","centiliter","centiliters",NULL}},
    {"1.232593620345297455720952457494033544533373789699499794656285986149681595299792021394950518524218500664033568694139341334679893333411134037499605108242281057079990961138502682597151008112501276987632104114117951307487179636796225960779054312078955923059540650966697302213477139425532668523277142534521745167648e-6",
                            "Coffeespoon",              {"coffeespoon","coffeespoons",NULL}},
    {"1.000294685410741114246915054245905202113032996728842296055257383264619987013613069045568076424301273063504348088220559504362338689167130603830071293351980113434063071578892230315074771833285106977873840849283091091490651752710249437379790457846635698447513366385554038047948391165840769943644113517553355986187e-6",
                            "Cooking Milliliter",       {"cooking_milliliter","cooking_milliliters",NULL}},
    {/* 4 bu (Imp) */
     "145.47488e-3",        "Coomb",                    {"coomb","coombs",NULL}},
    {/* 8ftx4ftx4ft */
     "3.624556363776",      "Cord (firewood)",          {"cd","cord","cords",NULL}},
    {/* 16 cu.ft. */
     "0.453069545472",      "Cord Foot (timber)",       {"cd_ft","cord_foot","cord_feet",NULL}},
    {"1e-6",                "Cubic Centimeter",         {"cc","cm3","cu.cm.","cubic_centimeter","cubic_centimeters",NULL}},
    {"1e-3",                "Cubic Decimeter",          {"dm3","cu.dm.","cubic_decimeter","cubic_decimeters",NULL}},
    {"1e54",                "Cubic Exameter",           {"Em3","cu.Em.","cubic_exameter","cubic_exameters",NULL}},
    {"6.116438863872",      "Cubic Fathom",             {"fm3","cu.fm.","cubic_fathom","cubic_fathoms",NULL}},
    {"1e-45",               "Cubic Femtometer",         {"cubic_femtometer","cubic_femtometers",NULL}},
    {"28.316846592e-3",     "Cubic Foot",               {"ft3","cu.ft.","cubic_foot","cubic_feet",NULL}},
    {"1e27",                "Cubic Gigameter",          {"Gm3","cu.Gm.","cubic_gigameter","cubic_gigameters",NULL}},
    {"16.387064e-6",        "Cubic Inch",               {"i3","cu.in.","cubic_inch","cubic_inches",NULL}},
    {"1e9",                 "Cubic Kilometer",          {"km3","cu.km.","cubic_kilometer","cubic_kilometers",NULL}},
    {"1",                   "Cubic Meter",              {"m3","cu.m.","cubic_meter","cubic_meters",NULL}},
    {"1e18",                "Cubic Megameter",          {"Mm3","cu.Mm.","cubic_megameter","cubic_megameter",NULL}},
    {"1e-18",               "Cubic Micrometer",         {"µm3","cu.µm.","cubic_micrometer","cubic_micrometers",NULL}},
    {"4.168181825440579584e9","Cubic Mile",             {"mi3","cu.mi.","cubic_mile","cubic_miles",NULL}},
    {"1e-9",                "Cubic Millimeter",         {"mm3","cu.mm.","cubic_millimeter","cubic_millimeters",NULL}},
    {"1e-27",               "Cubic Nanometer",          {"nm3","cu.nm.","cubic_nanometer","cubic_nanometers",NULL}},
    {"1e45",                "Cubic Petameter",          {"Pm3","cu.Pm.","cubic_petameter","cubic_petameters",NULL}},
    {"1e-36",               "Cubic Picometer",          {"pm3","cu.pm.","cubic_picometer","cubic_picometers",NULL}},
    {"1e36",                "Cubic Terameter",          {"Tm3","cu.Tm.","cubic_terameter","cubic_terameters",NULL}},
    {/* 27 cu.ft. */
     "0.764554857984",      "Cubic Yard",               {"yd3","cu.yd.","cubic_yard","cubic_yards",NULL}},
    {"1e72",                "Cubic Yottameters",        {"Ym3","cu.Ym.","cubic_yottameter","cubic_yottameters",NULL}},
    {"1e63",                "Cubic Zettameters",        {"Zm3","cu.Zm.","cubic_zettameter","cubic_zettameters",NULL}},
    {/* 8 fl.oz. */
     "227.3045e-6",         "Cup (Canadian)",           {"cup.CA",NULL}},
    {/* 8 US fl.oz. */
     "236.5882365e-6",      "Cup (US)",                 {"c","cups","cup",NULL}},
    {"250e-6",              "Cup (metric)",             {"cup.metric",NULL}},
    {/* 1/96 US fl.oz. */
     "0.308057599609375e-6","Dash (U.S.)",              {"ds","dash","dashes","dash.US",NULL}},
    {/* 1/384 gi */
     "0.36996175130208e-6", "Dash (Imperial)",          {"dash.Imp",NULL}},
    {"1e-2",                "Decaliter",                {"dal","daL","decaliter","decaliters",NULL}},
    {"1e-4",                "Deciliter",                {"dl","dL","deciliter","deciliters",NULL}},
    {/* 1/12 gi */
     "11.838776041e-6",     "Dessertspoon",             {"dessertspoon",NULL}},
    {/* 1/20 mL */
     "5e-5",                "Drop",                     {"drops","drop",NULL}},
    {/* 1/360 US fl.oz. */
     "0.0821486932291e-6",  "Drop (U.S.)",              {"gtt","drop.US","drops.US",NULL}},
    {/* 1/12 mL */
     "0.08e-6",             "Drop (medical)",           {"drop_medical","drops_medical",NULL}},
    {/* 1/288 fl.oz. */
     "0.098656467013e-6",   "Drop (Imperial)",          {"drop.Imp","drops.Imp",NULL}},
    {"1e15",                "Exaliter",                 {"El","EL","exaliter","exaliters",NULL}},
    {"1e-18",               "Femtoliter",               {"fl","fL","femtoliter","femtoliters",NULL}},
    {/* 1/5 gal */
     "0.7570823568e-3",     "Fifth",                    {"fifths","fifth",NULL}},
    {/* 9 US gal */
     "34.068706056e-3",     "Firkin",                   {"firkin",NULL}},
    {/* 1/8 fl.oz. */
     "3.5516328125e-6",     "Fluid Drachm",             {"fl.dr.","fluid_drachm",NULL}},
    {/* 1/8 US fl.oz. */
     "3.6966911953125e-6",  "Fluid Dram",               {"fl.dram","fluid_dram",NULL}},
    {/* 1/128 gal */
     "29.5735295625e-6",    "Fluid Ounce (U.S.)",       {"fl.oz.","fl.oz.US",NULL}},
    {/* 1/160 gal */
     "28.4130625e-6",       "Fluid Ounce (Imperial)",   {"fl.oz.UK",NULL}},
    {/* 1/24 fl.oz. */
     "1.1838776041e-6",     "Fluid Scruple",            {"fl.s.","fluid_scruple","fluid_scruples",NULL}},
    {/* 231 cu. in. */
     "3.785411784e-3",      "Gallon (U.S. fluid)",      {"gal","gallon","gallons","gal.USfl","gallon.USfl",NULL}},
    {/* 1/8 bu */
     "4.40488377086e-3",    "Gallon (U.S. dry)",        {"gal.USdry","gallon.USdry",NULL}},
    {"4.54609e-3",          "Gallon (Imperial)",        {"gal.Imp","gallon.Imp",NULL}},
    {"1e6",                 "Gigaliter",                {"Gl","GL","gigaliter","gigaliters",NULL}},
    {/* 4 US fl.oz. */
     "118.29411825e-6",     "Gill (U.S.)",              {"gi","gill","gills","gill.US",NULL}},
    {/* 5 fl.oz. */
     "142.0653125e-6",      "Gill (Imperial)",          {"gill.Imp",NULL}},
    {"0.1",                 "Hectoliter",               {"hl","hL","hectoliter","hectoliters",NULL}},
    {/* 2 fl.bl. (US) */
     "238.480942392e-3",    "Hogshead (U.S.)",          {"hhd","hhd.US","hogshead","hogshead.US","hogsheads","hogsheads.US",NULL}},
    {/* 2 bl (Imp) */
     "327.31848e-3",        "Hogshead (Imperial)",      {"hhd.Imp","hogshead.Imp","hogsheads.Imp",NULL}},
    {"44.36029434375e-6",   "Jigger",                   {"jigger","jiggers",NULL}},
    {"81.82962e-3",         "Kilderkin",                {"kilderkin",NULL}},
    {"1",                   "Kiloliter",                {"kl","kL","kiloliter","kiloliters",NULL}},
    {"1e-9",                "Lambda",                   {"lambda","lambdas",NULL}},
    {/* 80 bu (Imp) */
     "2.9094976",           "Last",                     {"last","lasts",NULL}},
    {"1e-3",                "Liter",                    {"l","liter","liters",NULL}},
    {/* 50 cu.ft. */
     "1.4158423296",        "Load",                     {"load","loads",NULL}},
    {"44.36029434375e-6",   "Measure",                  {"measure","measures",NULL}},
    {"1e3",                 "Megaliter",                {"Ml","ML","megaliter","megaliters",NULL}},
    {"1e-9",                "Microliter",               {"µl","microliter","microliters",NULL}},
    {"1e-6",                "Milliliter",               {"ml","mL","milliliter","milliliters",NULL}},
    {/* 1 minim = 1/480 fluid ounce */
     "0.0591938802083333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333e-6",
                            "Minim (Imperial)",         {"minim","minims","minim.Imp",NULL}},
    {"0.061611519921875e-6","Minim (U.S.)",             {"minim.US",NULL}},
    {"1e-12",               "Nanoliter",                {"nl","nL","nanoliter","nanoliters",NULL}},
    {"7.103515625e-5",      "Nipperkin (US)",           {"nipperkin","nipperkins",NULL}},
    {/* 5 fl.oz. */
     "142.0653125e-6",      "Noggin",                   {"nog","noggin",NULL}},
    {/* 8 bu (Imp) */
     "290.94976e-3",        "Pail",                     {"pail","pails",NULL}},
    {"1e-15",               "Picoliter",                {"pl","pL","picoliter","picoliters",NULL}},
    {/* 1/8 gal */
     "473.176473e-6",       "Pint (U.S. fluid)",        {"pt","pint","pints","pint.USfl",NULL}},
    {/* 1/64 bu */
     "0.5506104713575e-3",  "Pint (U.S. dry)",          {"pint.USdry",NULL}},
    {/* 1/8 gal */
     "0.56826125e-3",       "Pint (Imperial)",          {"pint.Imp",NULL}},
    {/* 1/4 US lvl bu */
     "8.80976754172e-3",    "Peck (U.S.)",              {"pk","peck","pecks","pk.US","peck.US",NULL}},
    {/* 2 gal */
     "9.09218e-3",          "Peck (Imperial)",          {"pk.Imp","peck.Imp",NULL}},
    {/* 16.5ftx1.5ftx1ft */
     "0.700841953152",      "Perch",                    {"per","perch","perches",NULL}},
    {"1e12",                "Petaliter",                {"Pl","PL","petaliter","petaliters",NULL}},
    {"0.61611519921875e-6", "Pinch (U.S.)",             {"pinch","pinches","pinch.US",NULL}},
    {"0.7399235026041e-6",  "Pinch (Imperial)",         {"pinch.Imp",NULL}},
    {/* 126 gal (US.fl.) */
     "0.476961884784",      "Pipe",                     {"pipe","pipes",NULL}},
    {/* 3/4 US fl.oz. */
     "22.180147171875e-6",  "Pony",                     {"pony","ponies",NULL}},
    {/* 1/2 gal (Imp) */
     "2.273045e-3",         "Pottle",                   {"pottle",NULL}},
    {/* 1/4 gal */
     "0.946352946e-3",      "Quart (U.S. fluid)",       {"qt","quart","quarts","qt.USfl","quart.USfl",NULL}},
    {/* 1/32 bu */
     "1.101220942715e-3",   "Quart (U.S. dry)",         {"qt.USdry","quart.USdry",NULL}},
    {/* 1/4 gal */
     "1.1365225e-3",        "Quart (Imperial)",         {"qt.Imp","quart.Imp",NULL}},
    {/* 8 bu (Imp) */
     "290.94976e-3",        "Quarter",                  {"qtr","quarter","quarters",NULL}},
    {/* 1/2 gal (Imp) */
     "2.273045e-3",         "Quatern",                  {"pottle",NULL}},
    {/* 3 bu (US lvl) */
     "105.71721050064e-3",  "Sack (U.S.)",              {"sack","sack.US",NULL}},
    {/* 3 bu (Imp) */
     "109.10616e-3",        "Sack (Imperial)",          {"sack.Imp",NULL}},
    {/* 8 bu (US lvl) */
     "281.91256133504e-3",  "Seam",                     {"seam","seams",NULL}},
    {"1",                   "Stere",                    {"st","stere","steres",NULL}},
    {/* 2 bu (US lvl) */
     "70.47814033376e-3",   "Strike (U.S.)",            {"strike","strike.US",NULL}},
    {/* 2 bu (Imp) */
     "72.73744e-3",         "Strike (Imperial)",        {"strike.Imp",NULL}},
    {/* 1 US fl.oz. */
     "29.5735295625e-6",    "Shot",                     {"shot","shots",NULL}},
    {/* 1/2 fl.oz. */
     "14.20653125e-6",      "Tablespoon (Canadian)",    {"tbsp.CA",NULL}},
    {/* 1/2 US fl.oz. */
     "14.7867647825e-6",    "Tablespoon (US)",          {"tbsp","tbsp.US","tablespoon","tablespoons",NULL}},
    {"15e-6",               "Tablespoon (metric)",      {"tbsp.metric",NULL}},
    {/* 5/8 fl.oz. */
     "17.7581640625e-6",    "Tablespoon (Imperial)",    {"tbsp.Imp",NULL}},
    {/* 1/6 fl.oz. */
     "4.735510416667e-6",   "Teaspoon (Canadian)",      {"tsp.CA",NULL}},
    {"4.928921595e-6",      "Teaspoon (U.S.)",          {"tsp","tsp.US","teaspoon","teaspoons",NULL}},
    {"5e-6",                "Teaspoon (metric)",        {"tsp.metric",NULL}},
    {/* 1/24 gi */
     "5.9193880208e-6",     "Teaspoon (Imperial)",      {"tsp.Imp",NULL}},
    {"1e9",                 "Teraliter",                {"Tl","TL","teraliter","teraliters",NULL}},
    {/* 1 cu.ft. */
     "28.316846592e-3",     "Timber Foot",              {"timber_foot","timber_feet",NULL}},
    {/* 35 cu.ft. */
     "0.99108963072",       "Ton (Displacement)",       {"displacement_ton","displacement_tons","ton.displacement",NULL}},
    {/* 28 bu (Imp) */
     "1.01832416",          "Ton (Water)",              {"water_ton","water_tons","ton.water",NULL}},
    {/* 40 cu.ft. */
     "1.13267386368",       "Ton (Freight)",            {"freight_ton","freight_tons","ton.freight",NULL}},
    {/* 100 cu.ft. */
     "2.8316846592",        "Ton (Register)",           {"register_ton","register_tons","ton.register",NULL}},
    {/* 252 gal (US.fl.) */
     "0.953923769568",      "Tun",                      {"tun","tuns",NULL}},
    {/* 40 bu (US lvl) */
     "1.4095628066752",     "Wey",                      {"wey","weys",NULL}},
    {"1e-27",               "Yoctoliter",               {"yl","yL","yoctoliter","yoctoliters",NULL}},
    {"1e21",                "Yottaliter",               {"Yl","YL","yottaliter","yottaliters",NULL}},
    {"1e-24",               "Zeptoliter",               {"zl","zL","zeptoliter","zeptoliters",NULL}},
    {"1e18",                "Zettaliter",                {"Zl","ZL","zettaliter","zettaliters",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on the radian */
const struct conversion angles[] = {
    {/* 2pi/6400 */
     "9.8174770424681038701957605727484465131161543730472056905467018509619262696444031207126088291941158374442127703540072082721682644048455694741417481504015034920321378209435578784548738894467154841574436388400342989560420191514864007305836973914725059660801517716358135894076697707756668550226882785664803893343956447362130465047528780088414190366105136496643448750354079079650639578793254592248172237794783039698929987248534130816168185394118534534954627326949373835923979746429772652850873068219123410307302292576270520096438168592074850773037209443120654490711579350536759161773512021088077114668910375017753973289486275868305354461180925286676161835045888778163279794692079674542845337258740289341360688123628504031936277001075567494175930461603436266209600354687499949155564060970616616650503009956094514357979590884457578822132917021407351935349630344062598057461027705860412761919394286802160853237445421575898608397848736339496323555862304748680683055641630350837706656310246144099872163013131216815476787533296433227464560504793182124007196936001103182905281117925706231684101553805477608722859861714288826692209248463422909901928995496533596824429832420779353726720637528990677223094057775750401823800111365864378272193185717336e-4",
                        "Angular Mil",              {"angular_mil",NULL}},
    {/* 1°/60 */
     "2.9088820866572159615394846141476878557381198142362090934953190669516818576724157394704026160575158036871741541789650987473091153792135020664123698223411862198613741691684615936162589302064342175281314485451953478388272649337737483646173918196955573232830079323365373598244947468964938829696854158715497449879690799218409026680749268174344945293660781184190651481586393801377967282605408768073532514902157937688571848073639742464049832709368454677023593282059073729162660665608821526770629057250110640091052531133709783732277975879133289117936950946109823552803430918677558270155114672914985811753751222227482658752440378035053438358868422307163307210383967045381712531760616199864546766595182307952995759444038075268721859852170538516792868284919536671469511216203703688638685647694997516044593484431435411661623582484283727058409753191528104277140631213055584609618082283217900077605746455348788400959243087874340328414177403359850762535070312518127609794264186770618579750017850709362925085337224064241622751861717461697026536445864646555261391684741067609749712923829838883461956015942363735917884403470900393093987925470643825156127109776750695355386617013564252956065374082663904362398239340963082021866699663959815784353536508892e-4",
                        "Arcminute",                {"arcminute",NULL}},
    {/* 1°/3600 */
     "4.8481368110953599358991410235794797595635330237270151558255317782528030961206928991173376934291930061452902569649418312455151922986891701106872830372353103664356236152807693226937648836773903625468857475753255797313787748896229139410289863661592622054716798872275622663741579114941564716161423597859162416466151332030681711134582113623908242156101301973651085802643989668963278804342347946789220858170263229480953080122732904106749721182280757795039322136765122881937767776014702544617715095416851066818420885222849639553796626465222148529894918243516372588005718197795930450258524454858309686256252037045804431254067296725089063931447370511938845350639945075636187552934360333107577944325303846588326265740063458781203099753617564194654780474865894452449185360339506147731142746158329193407655807385725686102705970807139545097349588652546840461901052021759307682696803805363166796009577425581314001598738479790567214023629005599751270891783854196879349657106977951030966250029751182271541808895373440402704586436195769495044227409774410925435652807901779349582854873049731472436593359903939559863140672451500655156646542451073041926878516294584492258977695022607088260108956804439840603997065568271803369777832773266359640589227514762e-6",
                        "Arcsecond",                {"arcsecond",NULL}},
    {/* 1 gr/100 */
     "1.570796326794896619231321691639751442098584699687552910487472296153908203143104499314017412671058533991074043256641153323546922304775291115862679704064240558725142051350969260552779822311474477465190982214405487832966723064237824116893391582635600954572824283461730174305227163324106696803630124570636862293503303157794087440760460481414627045857682183946295180005665265274410233260692073475970755804716528635182879795976546093058690966305896552559274037231189981374783675942876362445613969091505974564916836681220328321543010697473197612368595351089930471851385269608588146588376192337409233834702566000284063572631780413892885671378894804586818589360734220450612476715073274792685525396139844629461771009978056064510980432017209079906814887385654980259353605674999999186489024975529865866408048159297512229727673454151321261154126672342517630965594085505001568919376443293766604190710308588834573651799126745214377734365579781431941176893796875978890928890266085613403306500963938305597954608210099469047628600532742931639432968076690913984115150976017650926484497886811299706945624860887641739565757787428621227075347975414766558430863927944537549190877318732469659627530200463850835569504924412006429180801781853830052355090971477e-4",
                        "Centesimal Minute of Arc", {"centesimal_minute",NULL}},
    {/* 1 gr/10000 */
     "1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432566411533235469223047752911158626797040642405587251420513509692605527798223114744774651909822144054878329667230642378241168933915826356009545728242834617301743052271633241066968036301245706368622935033031577940874407604604814146270458576821839462951800056652652744102332606920734759707558047165286351828797959765460930586909663058965525592740372311899813747836759428763624456139690915059745649168366812203283215430106974731976123685953510899304718513852696085881465883761923374092338347025660002840635726317804138928856713788948045868185893607342204506124767150732747926855253961398446294617710099780560645109804320172090799068148873856549802593536056749999991864890249755298658664080481592975122297276734541513212611541266723425176309655940855050015689193764432937666041907103085888345736517991267452143777343655797814319411768937968759788909288902660856134033065009639383055979546082100994690476286005327429316394329680766909139841151509760176509264844978868112997069456248608876417395657577874286212270753479754147665584308639279445375491908773187324696596275302004638508355695049244120064291808017818538300523550909714765e-6",
                        "Centesimal Second of Arc", {"centesimal_second",NULL}},
    {/* pi/180 */
     "1.7453292519943295769236907684886127134428718885417254560971914401710091146034494436822415696345094822123044925073790592483854692275281012398474218934047117319168245015010769561697553581238605305168788691271172087032963589602642490187704350918173343939698047594019224158946968481378963297818112495229298469927814479531045416008449560904606967176196468710514390888951836280826780369563245260844119508941294762613143108844183845478429899625621072806214155969235444237497596399365292916062377434350066384054631518680225870239366785527479973470762170567665894131682058551206534962093068803748991487052250733336489595251464226821032063015321053384297984326230380227229027519056369719918728059957109384771797455666422845161233115911302323110075720970951722002881706729722222213183211388616998509626756090658861246996974149490570236235045851914916862566284378727833350765770849369930740046563447873209273040575545852724604197048506442015910457521042187510876565876558512062371147850010710425617755051202334438544973651117030477018215921867518787933156835010844640565849827754297903330077173609565418241550730642082540235856392755282386295093676265866050417213231970208138551773639224449598342617438943604577849213120019798375889470612121905324e-2",
                        "Degree",                   {"degree","°",NULL}},
    {/* 2pi/400 */
     "1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432566411533235469223047752911158626797040642405587251420513509692605527798223114744774651909822144054878329667230642378241168933915826356009545728242834617301743052271633241066968036301245706368622935033031577940874407604604814146270458576821839462951800056652652744102332606920734759707558047165286351828797959765460930586909663058965525592740372311899813747836759428763624456139690915059745649168366812203283215430106974731976123685953510899304718513852696085881465883761923374092338347025660002840635726317804138928856713788948045868185893607342204506124767150732747926855253961398446294617710099780560645109804320172090799068148873856549802593536056749999991864890249755298658664080481592975122297276734541513212611541266723425176309655940855050015689193764432937666041907103085888345736517991267452143777343655797814319411768937968759788909288902660856134033065009639383055979546082100994690476286005327429316394329680766909139841151509760176509264844978868112997069456248608876417395657577874286212270753479754147665584308639279445375491908773187324696596275302004638508355695049244120064291808017818538300523550909714774e-2",
                        "Gradian",                  {"grad","gradian","gon","gradians",NULL}},
    {/* 45° */
     "7.853981633974483096156608458198757210492923498437764552437361480769541015715522496570087063355292669955370216283205766617734611523876455579313398520321202793625710256754846302763899111557372387325954911072027439164833615321189120584466957913178004772864121417308650871526135816620533484018150622853184311467516515788970437203802302407073135229288410919731475900028326326372051166303460367379853779023582643175914398979882730465293454831529482762796370186155949906873918379714381812228069845457529872824584183406101641607715053487365988061842976755449652359256926348042940732941880961687046169173512830001420317863158902069464428356894474022934092946803671102253062383575366373963427626980699223147308855049890280322554902160086045399534074436928274901296768028374999995932445124877649329332040240796487561148638367270756606305770633361712588154827970427525007844596882216468833020953551542944172868258995633726071888671827898907159705884468984379894454644451330428067016532504819691527989773041050497345238143002663714658197164840383454569920575754880088254632422489434056498534728124304438208697828788937143106135376739877073832792154319639722687745954386593662348298137651002319254177847524622060032145904008909269150261775454857391e-1",
                        "Octant",                   {"octant",NULL}},
    {/* 60° */
     "1.0471975511965977461542144610931676280657231331250352736583148641026054687620696662093449417807056893273826955044274355490312815365168607439084531360428270391500947009006461737018532148743163183101273214762703252219778153761585494112622610550904006363818828556411534495368181088827377978690867497137579081956688687718627249605069736542764180305717881226308634533371101768496068221737947156506471705364776857567885865306510307287057939775372643683728493581541266542498557839619175749637426460610039830432778911208135522143620071316487984082457302340599536479009235130723920977255841282249394892231350440001893757150878536092619237809192632030578790595738228136337416511433821831951236835974265630863078473399853707096739869546781393866045432582571033201729024037833333327909926833170199105776053654395316748198184489694342141741027511148950117539770627236700010459462509621958444027938068723925563824345327511634762518229103865209546274512625312506525939525935107237422688710006426255370653030721400663126984190670218286210929553120511272759894101006506784339509896652578741998046304165739250944930438385249524141513835653169431777056205759519630250327939182124883131064183534669759005570463366162746709527872011879025533682367273143194",
                        "Sextant",                  {"sextant",NULL}},
    {/* 30° */
     "5.2359877559829887307710723054658381403286156656251763682915743205130273438103483310467247089035284466369134775221371777451564076825843037195422656802141351957504735045032308685092660743715815915506366073813516261098890768807927470563113052754520031819094142782057672476840905444136889893454337485687895409783443438593136248025348682713820901528589406131543172666855508842480341108689735782532358526823884287839429326532551536435289698876863218418642467907706332712492789198095878748187132303050199152163894556040677610718100356582439920412286511702997682395046175653619604886279206411246974461156752200009468785754392680463096189045963160152893952978691140681687082557169109159756184179871328154315392366999268535483699347733906969330227162912855166008645120189166666639549634165850995528880268271976583740990922448471710708705137555744750587698853136183500052297312548109792220139690343619627819121726637558173812591145519326047731372563126562532629697629675536187113443550032131276853265153607003315634920953351091431054647765602556363799470505032533921697549483262893709990231520828696254724652191926247620707569178265847158885281028797598151251639695910624415655320917673348795027852316830813733547639360059395127668411836365715972e-1",
                        "Sign",                     {"sign",NULL}},
    {"1",               "Radian",                   {"rad","radian","radians",NULL}},
    {/* 90° */
     "1.5707963267948966192313216916397514420985846996875529104874722961539082031431044993140174126710585339910740432566411533235469223047752911158626797040642405587251420513509692605527798223114744774651909822144054878329667230642378241168933915826356009545728242834617301743052271633241066968036301245706368622935033031577940874407604604814146270458576821839462951800056652652744102332606920734759707558047165286351828797959765460930586909663058965525592740372311899813747836759428763624456139690915059745649168366812203283215430106974731976123685953510899304718513852696085881465883761923374092338347025660002840635726317804138928856713788948045868185893607342204506124767150732747926855253961398446294617710099780560645109804320172090799068148873856549802593536056749999991864890249755298658664080481592975122297276734541513212611541266723425176309655940855050015689193764432937666041907103085888345736517991267452143777343655797814319411768937968759788909288902660856134033065009639383055979546082100994690476286005327429316394329680766909139841151509760176509264844978868112997069456248608876417395657577874286212270753479754147665584308639279445375491908773187324696596275302004638508355695049244120064291808017818538300523550909714782",
                        "Quadrant",                 {"quadrant",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on the gram */
const struct conversion masses[] = {
    {"9.10938188e-28",  "Atomic Mass Unit",             {"atomic_mass_unit","atomic_mass_units",NULL}},
    {"1.66053873e-24",  "Atomic Mass Unit (Unified)",   {"atomic_mass_unit.unified","atomic_mass_units.unified",NULL}},
    {"1e-18",           "Attogram",                     {"ag","attogram","attograms",NULL}},
    {/* 60 kg */
     "60e3",            "Bag (Coffee)",                 {"bag.coffee","bags.coffee",NULL}},
    {/* 94 lb. av */
     "42.63768278e3",   "Bag (Portland Cement)",        {"bag.portland","bags.portland",NULL}},
    {/* 22.5 sh tn */
     "20411.65665e3",   "Barge",                        {"barge","barges",NULL}},
    {/* 3 1/6 gr */
     "205.196548333e-3","Carat",                        {"ct","carat","carats",NULL}},
    {"200e-3",          "Carat (metric)",               {"kt","carat.metric","carats.metric",NULL}},
    {"45.359237e3",     "Cental",                       {"cental","centals",NULL}},
    {"1e-2",            "Centigram",                    {"cg","centigram","centigrams",NULL}},
    {/* 8 lb. av */
     "3.62873896e3",    "Clove",                        {"clove","cloves",NULL}},
    {"89.9349e-3",      "Crith",                        {"crith","criths",NULL}},
    {"1.66053873e-24",  "Dalton",                       {"Da","dalton","daltons",NULL}},
    {"10",              "Decagram",                     {"dag","decagram","decagrams",NULL}},
    {"1e-1",            "Decigram",                     {"dg","decigram","decigrams",NULL}},
    {/* 12 11/32 gr */
     "1.7718451953125", "Dram (Avoirdupois)",           {"dr.Av","dr","dram","drams","dram.Av","drams.Av",NULL}},
    {/* 60 gr */
     "3.8879346",       "Dram (Apothecary)",            {"dr.Ap","dram.Ap","drams.Ap",NULL}},
    {"9.10938188e-31",  "Electron Rest Mass",           {"me","electron_rest_mass","electron_rest_masses",NULL}},
    {"1e18",            "Exagram",                      {"Eg","exagram","exagrams",NULL}},
    {"1e-15",           "Femtogram",                    {"fg","femtogram","femtograms",NULL}},
    {"1e-6",            "Gamma",                        {"gamma","gammas",NULL}},
    {"14.593903e3",     "Geepound",                     {"geepound","geepounds",NULL}},
    {"1e9",             "Gigagram",                     {"Gg","gigagram","gigagrams",NULL}},
    {"64.79891e-3",     "Grain",                        {"gr","grain","grains",NULL}},
    {"1",               "Gram",                         {"g","gram","grams",NULL}},
    {"1e3",             "Grave",                        {"grave","graves",NULL}},
    {"100",             "Hectogram",                    {"hg","hectogram","hectograms",NULL}},
    {"9.80665",         "hyl (CGS unit)",               {"hyl_CGS",NULL}},
    {"9.80665e3",       "hyl (MKS unit)",               {"hyl_MKS",NULL}},
    {/* 100 lb av */
     "45.359237e3",     "Hundredweight (Short)",        {"sh.Cwt","sh.cwt","hundredweight.short","hundredweights.short",NULL}},
    {/* 112 lb av */
     "50.80234544e3",   "Hundredweight (Long)",         {"l.Cwt","l.cwt","hundredweight.long","hundredweights.long",NULL}},
    {"1e3",             "Kilogram",                     {"kg","kilogram","kilograms",NULL}},
    {/* 1000 lb av */
     "453.59237e3",     "Kip",                          {"kip","kips",NULL}},
    {/* 8 oz t */
     "248.8278144",     "Mark",                         {"mark","marks",NULL}},
    {"1e6",             "Megagram",                     {"Mg","megagram","megagrams",NULL}},
    {"1e-6",            "Microgram",                    {"µg","mcg","microgram","micrograms",NULL}},
    {"1e-3",            "Milligram",                    {"mg","milligram","milligrams",NULL}},
    {/* 1/20 gr */
     "3.2399455e-3",    "Mite",                         {"mite","mites",NULL}},
    {/* 1/20 g */
     "50e-3",           "Mite (metric)",                {"mite.metric","mites.metric",NULL}},
    {"1e-9",            "Nanogram",                     {"ng","nanogram","nanograms",NULL}},
    {/* 1/12 lb */
     "28.349523125",    "Ounce (Avoirdupois)",          {"oz","ounce","ounces","oz.Av","oz.av",NULL}},
    {/* 1/12 lb t */
     "31.1034768",      "Ounce (Apothecary)",           {"ounce.Ap","ounces.Ap","oz.Ap","oz.Tr",NULL}},
    {/* 1/20 oz t */
     "1.55517384",      "Pennyweight",                  {"dwt","pwt","pennyweight","pennyweights",NULL}},
    {"1e15",            "Petagram",                     {"Pg","petagram","petagrams",NULL}},
    {"1e-12",           "Picogram",                     {"pg","picogram","picograms",NULL}},
    {/*5760 grains */
     "0.3732417216e3",  "Pound (Troy)",                 {"lb_t","lb.Tr","#",NULL}},
    {/* 7000 grains */
     "0.45359237e3",    "Pound (Avoirdupois)",          {"lb","lb_av","lb.Av","lbs","pounds","pound",NULL}},
    {"500",             "Pound (metric)",               {"lb.metric","pound.metric","pounds.metric",NULL}},
    {"100e3",           "Quintal (metric)",             {"q","quintal","quintals",NULL}},
    {/*1/4 long ton */
     "254.0017272e3",   "Quarter (Long)",               {"qtr.long","quarter.long","quarters.long",NULL}},
    {"12.70058636e3",   "Quarter (Imperial)",           {"qtr","Q","Qr","quarter","quarters",NULL}},
    {/* 20 gr */
     "1.2959782",       "Scruple (Avoirdupois)",        {"s.Av","scruple.Av","scruples.Av",NULL}},
    {/* 1/700 lb.av. */
     "647.9891e-3",     "Sheet",                        {"sheet","sheets",NULL}},
    {"14.593903e3",     "Slug",                         {"slug","slugs",NULL}},
    {/* 14 lb av */
     "6.35029318e3",    "Stone",                        {"stone","stones",NULL}},
    {"1e12",            "Teragram",                     {"Tg","teragram","teragrams",NULL}},
    {"29.16666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666",
                        "Ton (Short Assay)",            {"AT","AT.s","assay_ton.short",NULL}},
    {"32.66666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666",
                        "Ton (Long Assay)",             {"AT.l","assay_ton.long",NULL}},
    {/* 2000 lb */
     "907.18474e3",     "Ton (Short)",                  {"sh.tn","T","short_ton","short_tons",NULL}},
    {/* 2240 lb */
     "1016.0469088e3",  "Ton (Long)",                   {"long.tn.","L.T.","LT","ton","tons","long_ton","long_tons",NULL}},
    {"1000e3",          "Tonne / Metric Ton",           {"MT","t","metric_ton","metric_tons","tonne","tonnes",NULL}},
    {/* 252 lb */
     "114.30527724e3",  "Wey",                          {"wey","weys",NULL}},
    {"1e-24",           "Yoctogram",                    {"yg","yoctogram","yoctograms",NULL}},
    {"1e24",            "Yottagram",                    {"Yg","yottagram","yottagrams",NULL}},
    {"1e-21",           "Zeptogram",                    {"zg","zeptogram","zeptograms",NULL}},
    {"1e21",            "Zettagram",                    {"Zg","zettagram","zettagrams",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on meter per second */
const struct conversion speeds[] = {
    {"100",                 "Centimeters/second",           {"cm/s",NULL}},
    {"1",                   "Meters/second",                {"m/s",NULL}},
    {"3.048e-1",            "Feet/second",                  {"f/s","feet_per_second",NULL}},
    {"5.08e-3",             "Feet/minute",                  {"f/m","feet_per_minute",NULL}},
    {"8.4666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666666e-5",
                            "Feet/hour",                    {"fph","f/h","feet_per_hour",NULL}},
    {"1.663095e-4",         "Furlongs/fortnight",           {"furlong_per_fortnight",NULL}},
    {"3.352799999999999999664720000000000000033527999999999999996647200000000000000335279999999999999966472000000000000003352799999999999999664720000000000000033527999999999999996647200000000000000335279999999999999966472000000000000003352799999999999999664720000000000000033527999999999999996647200000000000000335283",
                            "Furlongs/minute",               {"fur/m",NULL}},
    {"4.2333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333e-4",
                            "inches/minute",                {"ipm","in/m","inches_per_minute",NULL}},
    {"2.54e-2",             "inches/second",                {"ips","in/s","inches_per_minute",NULL}},
    {"2.7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777e-1",
                            "Kilometers/hour",              {"km/h","kph","kilometer/hr","kilometers/hr","kilometer/hour","kilometers/hour",NULL}},
    {"0.5144444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444",
                            "Knots",                        {"k","knot","knots",NULL}},
    {"0.514773",            "Knots (Admiralty)",            {"knot.admiralty","knots.admiralty",NULL}},
    {"5.587898399999999857732106736000003622140562501439907780301278713339947913529443958364926121540356820028980945582515362062145125469158881897785105555214866882391212564229489174319728114717205621819722199299944868469872805823403648757038363736143102645803259277796606637849018787298395000363981675382863290733015e-2",
                            "Leagues/day",                   {"leagues/day","league/day",NULL}},
    {"331.5",               "Mach (dry air, 273 kelvin)",   {"M","Ma","mach","machs",NULL}},
    {"1.609344e3",          "Miles/second",                 {"mps","mile/s","miles/s","mile/second","miles/second",NULL}},
    {"26.8224",             "Miles/minute",                 {"mpm","mile/m","miles/m","mile/minute","miles/minute",NULL}},
    {"0.44704",             "Miles/hour",                   {"mph","mile/hr","miles/hr","mile/hour","miles/hour",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on Watts */
const struct conversion powers[] = {
    {"1.68875e-3",              "Atmosphere Cubic Centimeter/minute",{"atm.ccm.","atmosphere_cubic_centimeter/minute",NULL}},
    {"0.101325",                "Atmosphere Cubic Centimeter/second",{"atm.ccs.","atmosphere_cubic_centimeter/second",NULL}},
    {"47.82007468224",          "Atmosphere Cubic foot/minute",     {"atm.cfm.","atmosphere_cubic_foot/minute",NULL}},
    {"2.8692044809344e3",       "Atmosphere Cubic foot/second",     {"atm.cfs.","atmosphere_cubic_foot/second",NULL}},
    {"1e-18",                   "Attowatt",                         {"aW","attowatt","attowatts",NULL}},
    {"0.293071",                "BTU/hour",                         {"BTU/hr",NULL}},
    {"1.05505585262e3",         "BTU/s",                            {"BTU/s",NULL}},
    {"4.1868",                  "Calorie/second",                   {"cal/s","calorie_per_second",NULL}},
    {"1e-2",                    "Centiwatt",                        {"cW","centiwatt","centiwatts",NULL}},
    {"10",                      "Decawatt",                         {"daW","decawatt","decawatts",NULL}},
    {"0.1",                     "Deciwatt",                         {"dW","deciwatt","deciwatts",NULL}},
    {"1e-7",                    "Ergs/sec",                         {"ergs/s",NULL}},
    {"1e18",                    "Exawatt",                          {"EW","exawatt","exawatts",NULL}},
    {"1e-15",                   "Femtowatt",                        {"fW","femtowatt","femtowatts",NULL}},
    {"3.766161e-4",             "Foot-lbs/hour",                    {"ft-lbf/h","ft-lb/h","foot-pound_per_hour","foot-pounds_per_hour",NULL}},
    {"2.259696580552334e-2",    "Foot-lbs/minute",                  {"ft-lbf/m","ft-lb/m","foot-pound_per_minute","foot-pounds_per_minute",NULL}},
    {"1.3558179483314004",      "Foot-lbs/sec",                     {"ft-lbf/s","ft-lb/s","foot-pound_per_second","foot-pounds_per_second",NULL}},
    {"100",                     "Hectowatt",                        {"hW","hectowatt","hectowatts",NULL}},
    {"735.49875",               "Horsepower (metric)",              {"hp","horse","horsepower","horsepower.metric",NULL}},
    {"736",                     "Horsepower (European electrical)", {"horsepower.ee",NULL}},
    {"745.69987158227022",      "Horsepower (Imperial mechanical)", {"horsepower.ImpMech",NULL}},
    {"746",                     "Horsepower (Imperial electrical)", {"horsepower.ImpElec",NULL}},
    {"9.810657e3",              "Horsepower (Boiler)",              {"horsepower.boiler","boiler_horsepower",NULL}},
    {"1e3",                     "Kilowatt",                         {"kW","kilowatt","kilowatts",NULL}},
    {"1.68875",                 "Liter-atmosphere/minute",          {"L-atm/min","liter-atmosphere_per_minute",NULL}},
    {"101.325",                 "Liter-atmosphere/sec",             {"L-atm/s","liter-atmosphere_per_second",NULL}},
    {"1.3333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333e-4",
                                "Lusec",                            {"lusec","lusecs",NULL}},
    {"1e9",                     "Gigawatts",                        {"GW","gigawatt","gigawatts",NULL}},
    {"1e6",                     "Megawatt",                         {"MW","megawatt","megawatts",NULL}},
    {"1e-6",                    "Microwatt",                        {"µW","microwatt","microwatts",NULL}},
    {"1e-3",                    "Milliwatt",                        {"mW","milliwatt","milliwatts",NULL}},
    {"1e-9",                    "Nanowatt",                         {"nW","nanowatt","nanowatts",NULL}},
    {"1e15",                    "Petawatt",                         {"PW","petawatt","petawatts",NULL}},
    {"1e-12",                   "Picowatt",                         {"pW","picowatt","picowatts",NULL}},
    {"980.665",                 "Poncelet",                         {"p","poncelet","poncelets",NULL}},
    {"70.337057",               "Square Foot Equivalent Direct Radiation",{"sq.ft.EDR",NULL}},
    {"1e12",                    "Terawatt",                         {"TW","terawatt","terawatts",NULL}},
    {"844.2",                   "Ton of A.C.",                      {"ton_of_a.c.",NULL}},
    {"3.516853e3",              "Ton of Refrigeration",             {"ton_of_refrigeration",NULL}},
    {"3.938875e3",              "Ton of Refrigeration (Imperial)",  {"ton_of_refrigeration.Imp",NULL}},
    {"1",                       "Watts (joules/sec)",               {"W","watt","watts","J/s",NULL}},
    {"1e-24",                   "Yoctowatt",                        {"yW","yoctowatt","yoctowatts",NULL}},
    {"1e24",                    "Yottawatt",                        {"YW","yottawatt","yottawatts",NULL}},
    {"1e-21",                   "Zeptowatt",                        {"zW","zeptowatt","zeptowatts",NULL}},
    {"1e21",                    "Zettawatt",                        {"ZW","zettawatt","zettawatts",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on Newtons */
const struct conversion forces[] = {
    {"1e-5",                "Dyne",             {"dyn","dyne","dynes",NULL}},
    {"9.80665",             "Grave-force",      {"Gf","grave_force","graves_force",NULL}},
    {"9.80665e-3",          "Gravet",           {"gravet",NULL}},
    {"9.80665",             "Kilopond",         {"kp","kilopond","kiloponds",NULL}},
    {"9.80665",             "Kilogram-force",   {"kgf","kilogram_force","kilograms_force",NULL}},
    {"4.4482216152605e3",   "Kip (kilopound-force)",{"kip","kipf","klbf","kips","kilopound","kilopounds",NULL}},
    {"9.80665e-3",          "Milligrave-force", {"gf","milligrave_force","milligraves_force",NULL}},
    {"1",                   "Newton",           {"N","newton","newtons",NULL}},
    {"0.2780138509537812",  "Ounce-force",      {"ozf","ounce_force","ounces_force",NULL}},
    {"4.4482216152605",     "Pound",            {"lbf","pound_force","pounds_force",NULL}},
    {"0.138254954376",      "Poundal",          {"pdl","pl","poundal","poundals",NULL}},
    {"1e3",                 "Sthene",           {"sn","sthene","sthenes",NULL}},
    {"8.896443230521e3",    "Ton-force",        {"tnf","ton_force","tons_force",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on meters per square second */
const struct conversion accelerations[] = {
    {"3.28084",    "Celo",                      {"celo","celos",NULL}},
    {"10000",      "Centigal",                  {"centigal","centigals",NULL}},
    {"100",        "Centimeter/square second",  {"cm/s/s","cm/s^2",NULL}},
    {"1000",       "Decigal",                   {"Dg","decigal","decigals",NULL}},
    {"3.280839895","Foot/square second",        {"f/s/s","f/s^2",NULL}},
    {"11811.023622047","Foot/hour/second",      {"f/h/s",NULL}},
    {"196.8503937","Foot/minute/second",        {"f/m/s",NULL}},
    {"0.101972",   "G-unit (G)",                {"G","G-units",NULL}},
    {"100",        "Galileo",                   {"Gal","gal","galileo","galileos",NULL}},
    {"0.101971621","Grav",                      {"grav","gravs",NULL}},
    {"39.370079",  "Inch/square second",        {"in/s/s","in/s^2",NULL}},
    {"2362.204724409","Inch/minute/second",     {"in/m/s",NULL}},
    {"3.6",        "Kilometer/hour/second",     {"km/h/s",NULL}},
    {"0.001",      "Kilometer/square second",   {"km/s/s","km/s^2",NULL}},
    {"1.943844492","Knot/second",               {"k/s",NULL}},
    {"0.1",        "Leo",                       {"leo","leos",NULL}},
    {"1",          "Meter/square second",       {"m/s/s","m/s^2",NULL}},
    {"134.216178", "Mile/hour/minute",          {"mi/h/m",NULL}},
    {"2.236936292","Mile/hour/second",          {"mi/h/s",NULL}},
    {"0.037282271","Mile/minute/second",        {"mi/m/s",NULL}},
    {"0.000621371","Mile/square second",        {"mi/s/s","mi/s^2",NULL}},
    {"100000",     "Milligal",                  {"mGal","milligals",NULL}},
    {"1000",       "Millimeter/square second",  {"mm/s/s","mm/s^2",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on none */
const struct conversion metric[] = {
    {"10e-18",          "Atto",           {"a","atto",NULL}},
    {"1e-2",            "Centi",          {"c","centi",NULL}},
    {"10",              "Deca",            {"da","deca",NULL}},
    {"0.1",             "Deci",           {"d","deci",NULL}},
    {"1e18",            "Exa",            {"E","exa",NULL}},
    {"1e-15",           "Femto",          {"f","femto",NULL}},
    {"1e9",             "Giga",           {"G","giga",NULL}},
    {"1e-2",            "Hecto",          {"h","hecto",NULL}},
    {"1e3",             "Kilo",           {"k","kilo",NULL}},
    {"1e6",             "Mega",           {"M","mega",NULL}},
    {"1",               "none",           {"none",NULL}},
    {"1e-6",            "Micro",          {"µ","micro",NULL}},
    {"1e-3",            "Milli",          {"m","milli",NULL}},
    {"1e-9",            "Nano",           {"n","nano",NULL}},
    {"1e15",            "Peta",           {"P","peta",NULL}},
    {"1e-12",           "Pico",           {"p","pico",NULL}},
    {"1e12",            "Tera",           {"T","tera",NULL}},
    {"1e-24",           "Yocto",          {"y","yocto",NULL}},
    {"1e24",            "Yotta",          {"Y","yotta",NULL}},
    {"1e-21",           "Zepto",          {"z","zepto",NULL}},
    {"1e21",            "Zetta",          {"Z","zetta",NULL}},
    {NULL,NULL,{NULL}}
};

/* based on the Pascal */
const struct conversion pressures[] = {
    {"98.0665e3",   "Atmosphere (technical)",   {"atm.Tech","atmosphere.Tech",NULL}},
    {"101325",      "Atmosphere (standard)",    {"atm","atmosphere",NULL}},
    {"1e-18",       "Attopascal",               {"aPa","attopascal","attopascals",NULL}},
    {"1e5",         "Bar",                      {"bar",NULL}},
    {"0.1",         "Barye",                    {"barye","baryes",NULL}},
    {"98.0638",     "Centimeter of Water (3.98°C)",{"cmH2O","centimeter_of_water","centimeters_of_water",NULL}},
    {"1.33322387415e3","Centimeter of Mercury", {"cmHg","centimeter_of_mercury","centimeters_of_mercury",NULL}},
    {"1e-2",        "Centipascal",              {"cPa","centipascal","centipascals",NULL}},
    {"10",          "Decapascal",               {"daPa","decapascal","decapascals",NULL}},
    {"0.1",         "Decipascal",               {"dPa","decipascal","decipascals",NULL}},
    {"1e18",        "Exapascal",                {"EPa","exapascal","exapascals",NULL}},
    {"1e-15",       "Femtopascal",              {"fPa","femtopascal","femtopascals",NULL}},
    {"2.98898e3",   "Foot of Water (3.98°C)",   {"ftH2O","foot_of_water","feet_of_water",NULL}},
    {"40.6366636840919e3","Foot of Mercury",    {"ftHg","foot_of_mercury","feet_of_mercury",NULL}},
    {"1e9",         "Gigapascal",               {"GPa","gigapascal","gigapascals",NULL}},
    {"100",         "Hectopascal",              {"hPa","hectopascal","hectopascals",NULL}},
    {"249.082",     "Inch of Water (3.98°C)",   {"inH2O","inch_of_water","inches_of_water",NULL}},
    {"3.386388640341e3","Inch of Mercury",      {"inHg","inch_of_mercury","inches_of_mercury",NULL}},
    {"9.80665e6",   "Kilogram-force/sq.mm.",    {"kgf/mm2","kilogram-force_per_square_millimeter",NULL}},
    {"1e3",         "Kilopascal",               {"kPa","kilopascal","kilopascals",NULL}},
    {"6.894757e6",  "kip/sq.in.",               {"kip/sq.in.","kip_per_square_inch",NULL}},
    {"1e6",         "Megapascal",               {"MPa","megapascal","megapascals",NULL}},
    {"1e-6",        "Micropascal",              {"µPa","micropascal","micropascals",NULL}},
    {"133.322387415","Millimeter of Mercury",   {"mmHg","millimeter_of_mercury","millimeters_of_mercury",NULL}},
    {"9.80638",     "Millimeter of Water (3.98°C)",{"mmH2O","millimeter_of_water","millimeters_of_water",NULL}},
    {"1e-3",        "Millipascal",              {"mPa","millipascal","millipascals",NULL}},
    {"1e-9",        "Nanopascal",               {"nPa","nanopascal","nanopascals",NULL}},
    {"1",           "Pascal",                   {"Pa","pascal","pascals",NULL}},
    {"1e15",        "Petapascal",               {"PPa","petapascal","petapascals",NULL}},
    {"1e-12",       "Picopascal",               {"pPa","picopascal","picopascals",NULL}},
    {"1e3",         "Pièze",                    {"pz","pieze",NULL}},
    {"1.488164",    "Poundal/sq.ft.",           {"pdl/sq.ft.","poundal_per_sqare_foot",NULL}},
    {"47.880259",   "Pound/sq.ft.",             {"psf","lb/sq.ft.","pound_per_square_foot",NULL}},
    {"6.894757e3",  "Pound/sq.in.",             {"psi","lb/sq.ft.","pound_per_square_inch",NULL}},
    {"95.760518e3", "Short Ton/sq.ft.",         {"ST/sq.ft","short_ton_per_square_foot",NULL}},
    {"1e12",        "Terapascal",               {"TPa","terapascal","terapascals",NULL}},
    {"133.3223684", "Torr",                     {"torr",NULL}},
    {"1e-24",       "Yoctopascal",              {"yPa","yoctopascal","yoctopascals",NULL}},
    {"1e24",        "Yottapascal",              {"YPa","yottapascal","yottapascals",NULL}},
    {"1e-21",       "Zeptopascal",              {"zPa","zeptopascal","zeptopascals",NULL}},
    {"1e21",        "Zettapascal",              {"ZPa","zettapascal","zettapascals",NULL}},
    {NULL,NULL,{NULL}}
};

const struct conversion temperatures[] = {
    {"", "Kelvin", {"kelvin", "Kelvin", "K", NULL}},
    {"", "Celsius", {"celsius", "Celsius", "C", NULL}},
    {"", "Rankine", {"rankine", "Rankine", NULL}},
    {"", "Farenheit", {"farenheit", "Farenheit", "F", NULL}},
    {"", "Reaumur", {"reaumur", "Reaumur", "R", NULL}},
    {NULL,NULL,{NULL}}
};

const char * from_temperatures[] = {
    "[%1.15f]",                     // kelvin
    "[%1.15f + 273.15]",            // celsius
    "[%1.15f / 1.8]",               // rankine
    "[(%1.15f - 32)/1.8 + 273.15]", // farenheit
    "[((5/4) * %1.15f) + 273.15]"   // reaumur
};

const char * to_temperatures[] = {
    "%s",                 // kelvin
    "%s - 273.15",        // celsius
    "%s * 1.8",           // rankine
    "1.8*(%s-273.15)+32", // farenheit
    "(%s-273.15)*(4/5)"   // reaumur
};

/* time, energy */

const struct conversion *conversions[] = {
    lengths,
    areas,
    volumes,
    masses,
    speeds,
    powers,
    forces,
    accelerations,
    temperatures,
    angles,
    pressures,
    metric,
    NULL
};

/* returns the category number of the unit */
ssize_t identify_unit(const char *unit)
{
    ssize_t cat_num;

    for (cat_num = 0; conversions[cat_num] != NULL; cat_num++) {
        size_t                   unit_num;
        const struct conversion *category = conversions[cat_num];

        for (unit_num = 0; category[unit_num].name != NULL; unit_num++) {
            size_t abbrev_num;

            for (abbrev_num = 0; category[unit_num].aka[abbrev_num] != NULL; abbrev_num++) {
                if (!strcmp(category[unit_num].aka[abbrev_num], unit)) {
                    return cat_num;
                }
            }
        }
    }
    return -1;
}

ssize_t identify_units(const char *unit1,
                       const char *unit2)
{
    ssize_t cat_num;
    ssize_t u1 = -1, u2 = -1;

    for (cat_num = 0; conversions[cat_num] != NULL; cat_num++) {
        size_t                   unit_num;
        const struct conversion *category = conversions[cat_num];

        for (unit_num = 0; category[unit_num].name != NULL; unit_num++) {
            size_t abbrev_num;

            for (abbrev_num = 0;
                 category[unit_num].aka[abbrev_num] != NULL;
                 abbrev_num++) {
                if (!strcmp(category[unit_num].aka[abbrev_num], unit1)) {
                    if (u2 == -1) {
                        u1 = cat_num;
                    } else if (u2 == cat_num) {
                        return u2;
                    } else {
                        return -1;
                    }
                } else if (!strcmp(category[unit_num].aka[abbrev_num], unit2)) {
                    if (u1 == -1) {
                        u2 = cat_num;
                    } else if (u1 == cat_num) {
                        return u1;
                    } else {
                        return -1;
                    }
                }
            }
        }
    }
    if (-1 == u1) {
        return -3;
    } else if (-1 == u2) {
        return -4;
    }
    return -2;
}

ssize_t unit_id(const int   utype,
                const char *unit)
{
    size_t unit_num;

    for (unit_num = 0; (conversions[utype])[unit_num].name != NULL; unit_num++) {
        size_t abbrev_num;

        for (abbrev_num = 0;
             (conversions[utype])[unit_num].aka[abbrev_num] != NULL;
             abbrev_num++) {
            if (!strcmp((conversions[utype])[unit_num].aka[abbrev_num], unit)) {
                return (ssize_t)unit_num;
            }
        }
    }
    return -1;
}

void uber_conversion (Number    output,
                      const int utype,
                      const int fromunit,
                      const int tounit,
                      Number    value)
{
    if (utype != TEMPERATURE_C) {
        const struct conversion *ltable = conversions[utype];
        Number                   tofac, fromfac;

        /* Moving to a more precise model...
         *
         * The essential computation is:
         *
         *   value * (tofac/fromfac)
         *
         *   (it used to be :
         *      *((fromfac)/(tofac))
         */
        (void)num_init_set_str(tofac, ltable[tounit].factor, 10);
        (void)num_init_set_str(fromfac, ltable[fromunit].factor, 10);
        switch (utype) {
            case LENGTH_C:
            case METRIC_C:
            case AREA_C:
            case VOLUME_C:
            case ANGLE_C:
            case MASS_C:
            case SPEED_C:
            case FORCE_C:
            case POWER_C:
            case PRESSURE_C:
                num_div(fromfac, fromfac, tofac);
                num_mul(output, fromfac, value);
                break;
            case ACCELERATION_C:
            default:
                num_div(tofac, tofac, fromfac);
                num_mul(output, tofac, value);
        }
        num_free(tofac);
        num_free(fromfac);
    } else {
        Number temp;

        num_init(temp);
        /* convert to kelvin */
        switch (fromunit) {
            case KELVIN:
                break;
            case CELSIUS:
                num_set_d(temp, 273.15);
                num_add(value, value, temp);
                break;
            case RANKINE:
                num_set_d(temp, 1.8);
                num_div(value, value, temp);
                break;
            case FARENHEIT:
                num_sub_ui(value, value, 32);
                num_set_d(temp, 1.8);
                num_div(value, value, temp);
                num_set_d(temp, 273.15);
                num_add(value, value, temp);
                break;
            case REAUMUR:
                num_set_d(temp, (5.0 / 4.0));
                num_mul(value, value, temp);
                num_set_d(temp, 273.15);
                num_add(value, value, temp);
                break;
        }
        /* convert from kelvin */
        switch (tounit) {
            case KELVIN:
                break;
            case CELSIUS:
                num_set_d(temp, 273.15);
                num_sub(output, value, temp);
                break;
            case RANKINE:
                num_set_d(temp, 1.8);
                num_mul(output, value, temp);
                break;
            case FARENHEIT:
                num_set_d(temp, 273.15);
                num_sub(value, value, temp);
                num_set_d(temp, 1.8);
                num_mul(value, value, temp);
                num_add_ui(output, value, 32);
                break;
            case REAUMUR:
                num_set_d(temp, 273.15);
                num_sub(value, value, temp);
                num_set_d(temp, (4.0 / 5.0));
                num_mul(output, value, temp);
                break;
        }
        num_free(temp);
/* The old way *//*
 *      //char stage1[100];
 *      //char composite[100];
 *      //sprintf(stage1,from_temperatures[fromunit],value);
 *      //sprintf(composite,to_temperatures[tounit],stage1);
 *      //return parseme(composite);
 */
    }
}

/* vim:set expandtab: */
