/* msm-driver.c
 *
 * Copyright (c) 2009-2010, Code Aurora Forum. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Code Aurora nor
 *       the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NON-INFRINGEMENT ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <sys/types.h>
#include <grp.h>

#include "xf86.h"
#include "damage.h"
#include "xf86_OSlib.h"
#include "xf86Crtc.h"

#include "mipointer.h"
#include "mibstore.h"
#include "micmap.h"
#include "fb.h"
#include "dixstruct.h"

#include "msm.h"
#include "msm-drm.h"

#if USEDRI
#define _XF86DRI_SERVER_
#endif

#if USEDRI2
#include "xf86drm.h"
#endif

#define MSM_NAME        "msm"
#define MSM_DRIVER_NAME "msm"

#define MSM_VERSION_MAJOR PACKAGE_VERSION_MAJOR
#define MSM_VERSION_MINOR PACKAGE_VERSION_MINOR
#define MSM_VERSION_PATCH PACKAGE_VERSION_PATCHLEVEL

#define MSM_VERSION_CURRENT \
	((MSM_VERSION_MAJOR << 20) |\
	(MSM_VERSION_MINOR << 10) | \
	 (MSM_VERSION_PATCH))

/* List of available strings for fbCache support */


static const char *fbCacheStrings[] = {
#ifdef MSMFB_GET_PAGE_PROTECTION
   [MDP_FB_PAGE_PROTECTION_NONCACHED] = "Noncached",
   [MDP_FB_PAGE_PROTECTION_WRITECOMBINE] = "WriteCombine",
   [MDP_FB_PAGE_PROTECTION_WRITETHROUGHCACHE] = "WriteThroughCache",
   [MDP_FB_PAGE_PROTECTION_WRITEBACKCACHE] = "WriteBackCache",
   [MDP_FB_PAGE_PROTECTION_WRITEBACKWACACHE] = "WriteBackWACache",
#endif
   NULL
};

/* This enumerates all of the available options */

typedef enum
{
   OPTION_FB,
   OPTION_NOACCEL,
   OPTION_SWBLIT,
   OPTION_DRI,
   OPTION_DRI2,
   OPTION_SWCURSOR,
   OPTION_VSYNC,
   OPTION_SOCKGID,
   OPTION_NOSIGBLOCK,
   OPTION_FASTFILL,
   OPTION_FASTCOMPOSITE,
   OPTION_FASTCOMPOSITEREPEAT,
   OPTION_FASTVIDEOMEMCOPY,
   OPTION_FASTAPPFBMEMCOPY,
   OPTION_FBCACHE,
   OPTION_PIXMAP_MEMTYPE,
} MSMOpts;

/* An aray containing the options that the user can
   configure in xorg.conf
 */

static const OptionInfoRec MSMOptions[] = {
   {OPTION_FB, "fb", OPTV_STRING, {0}, FALSE},
   {OPTION_NOACCEL, "NoAccel", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_SWBLIT, "SWBlit", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_DRI, "DRI", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_DRI2, "DRI2", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_SWCURSOR, "SWCursor", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_SOCKGID, "SocketGID", OPTV_STRING, {0}, FALSE},
   {OPTION_VSYNC, "DefaultVsync", OPTV_INTEGER, {0}, FALSE},
   {OPTION_NOSIGBLOCK, "NoSigBlock", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_FASTFILL, "FastFill", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_FASTCOMPOSITE, "FastComposite", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_FASTCOMPOSITEREPEAT, "FastCompositeRepeat", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_FASTVIDEOMEMCOPY, "FastVideoMemCopy", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_FASTAPPFBMEMCOPY, "FastAppFBMemCopy", OPTV_BOOLEAN, {0}, FALSE},
   {OPTION_FBCACHE, "FBCache", OPTV_STRING, {0}, FALSE},
   {OPTION_PIXMAP_MEMTYPE, "PixmapMemtype", OPTV_STRING, {0}, FALSE},
   {-1, NULL, OPTV_NONE, {0}, FALSE}
};

/** Return a string for the given chipset type */

static char *
msmGetChipset(int chipID)
{
   switch (chipID) {
   case MSM_TYPE_7201:
      return "7201";
      break;
   case MSM_TYPE_8X50:
      return "8x50";
   }

   return "";
}

#if USEDRI2

static Bool
MSMInitDRM(ScrnInfoPtr pScrn)
{
   MSMPtr pMsm = MSMPTR(pScrn);   int i, fd;
   drmVersionPtr version;
   drmSetVersion sv;
   int ret;

   /* Ugly, huh? */

     pMsm->drmFD = 0;
     pMsm->drmDevName[0] = '\0';

   for(i = 0; i < DRM_MAX_MINOR; i++) {
     int ret = -1;

     snprintf(pMsm->drmDevName, sizeof(pMsm->drmDevName),
	      DRM_DEV_NAME, DRM_DIR_NAME, i);
     fd = open(pMsm->drmDevName, O_RDWR);
     if (fd < 0)
       continue;

     version = drmGetVersion(fd);
     if (version)
       ret = strcmp(version->name, "kgsl");

     drmFreeVersion(version);

     if (!ret)
       break;

     close(fd);
   }

   if (i ==DRM_MAX_MINOR) {
     xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
		"Unable to open a DRM device\n");
     close(fd);
     return FALSE;
   }


   sv.drm_di_major = 1;
   sv.drm_di_minor = 1;
   sv.drm_dd_major = -1;
   sv.drm_dd_minor = -1;

   ret = drmSetInterfaceVersion(fd, &sv);
   if (ret != 0) {
	xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                "Unable to set the DRM version: %d\n", ret);
	close(fd);
	return FALSE;
   }

   pMsm->drmFD = fd;
   return TRUE;
}
#endif

/* Get the current mode from the framebuffer mode and
 * convert it into xfree86 timings
 */

static void
MSMGetDefaultMode(MSMPtr pMsm)
{
   char name[32];
   sprintf(name, "%dx%d", pMsm->mode_info.xres, pMsm->mode_info.yres);

   pMsm->default_mode.name = strdup(name);

   if (pMsm->default_mode.name == NULL)
	pMsm->default_mode.name = "";

   pMsm->default_mode.next = &pMsm->default_mode;
   pMsm->default_mode.prev = &pMsm->default_mode;
   pMsm->default_mode.type |= M_T_BUILTIN | M_T_PREFERRED;

   pMsm->default_mode.HDisplay = pMsm->mode_info.xres;
   pMsm->default_mode.HSyncStart =
	 pMsm->default_mode.HDisplay + pMsm->mode_info.right_margin;
   pMsm->default_mode.HSyncEnd =
	 pMsm->default_mode.HSyncStart + pMsm->mode_info.hsync_len;
   pMsm->default_mode.HTotal =
	 pMsm->default_mode.HSyncEnd + pMsm->mode_info.left_margin;

   pMsm->default_mode.VDisplay = pMsm->mode_info.yres;
   pMsm->default_mode.VSyncStart =
	 pMsm->default_mode.VDisplay + pMsm->mode_info.lower_margin;
   pMsm->default_mode.VSyncEnd =
	 pMsm->default_mode.VSyncStart + pMsm->mode_info.vsync_len;
   pMsm->default_mode.VTotal =
	 pMsm->default_mode.VSyncEnd + pMsm->mode_info.upper_margin;

   /* The clock number we get is not the actual pixclock for the display,
    * which automagically updates at a fixed rate.  There is no good way
    * to automatically figure out the fixed rate, so we use a config
    * value */

   pMsm->default_mode.Clock = (pMsm->defaultVsync *
			       pMsm->default_mode.HTotal *
			       pMsm->default_mode.VTotal) / 1000;

   pMsm->default_mode.CrtcHDisplay = pMsm->default_mode.HDisplay;
   pMsm->default_mode.CrtcHSyncStart = pMsm->default_mode.HSyncStart;
   pMsm->default_mode.CrtcHSyncEnd = pMsm->default_mode.HSyncEnd;
   pMsm->default_mode.CrtcHTotal = pMsm->default_mode.HTotal;

   pMsm->default_mode.CrtcVDisplay = pMsm->default_mode.VDisplay;
   pMsm->default_mode.CrtcVSyncStart = pMsm->default_mode.VSyncStart;
   pMsm->default_mode.CrtcVSyncEnd = pMsm->default_mode.VSyncEnd;
   pMsm->default_mode.CrtcVTotal = pMsm->default_mode.VTotal;

   pMsm->default_mode.CrtcHAdjusted = FALSE;
   pMsm->default_mode.CrtcVAdjusted = FALSE;
}

static Bool
MSMCrtcResize(ScrnInfoPtr pScrn, int width, int height)
{
   return TRUE;
}

static const xf86CrtcConfigFuncsRec MSMCrtcConfigFuncs = {
   MSMCrtcResize,
};

static int
_getgid(const char *gid, gid_t *ret)
{
    struct group *grp;
    gid_t g;

    grp = getgrnam(gid);

    if (grp != NULL) {
	*ret = grp->gr_gid;
	return 0;
    }

    g = strtoul(gid, NULL, 0);

    if (g != 0) {
	grp = getgrgid(g);
	if (grp != NULL) {
	    *ret = grp->gr_gid;
	    return 0;
	}
    }

    return -1;
}

/* A simple case-insenstive string comparison function. */

static int stricmp(const char *left, const char *right)
{
   const int MAXSTRINGLEN = 100;
   char leftCopy[MAXSTRINGLEN],
        rightCopy[MAXSTRINGLEN];
   int i;

   // Make temporary copies of comparison strings.
   strncpy(leftCopy,left, MAXSTRINGLEN);
   strncpy(rightCopy,right, MAXSTRINGLEN);

   // Convert English upper-case characters to lower-case.
   i = 0;
   while (leftCopy[i] != '\0')
   {
      if (leftCopy[i] >= 'A' && leftCopy[i] <= 'Z')
         leftCopy[i] += 'a' - 'A';
      i++;
   }
   i = 0;
   while (rightCopy[i] != '\0')
   {
      if (rightCopy[i] >= 'A' && rightCopy[i] <= 'Z')
         rightCopy[i] += 'a' - 'A';
      i++;
   }

   return strcmp(leftCopy, rightCopy);
}

/* This is the main initialization function for the screen */

static Bool
MSMPreInit(ScrnInfoPtr pScrn, int flags)
{
   MSMPtr pMsm;
   EntityInfoPtr pEnt;
   char *dev, *gid, *str;
   int mdpver, panelid;
   int depth, fbbpp;
   OptionInfoPtr options;
   rgb defaultWeight = { 0, 0, 0 };
   int vsync;

   /* Omit ourselves from auto-probing (which is bound to
    * fail on our hardware anyway)
    */

   if (flags & PROBE_DETECT)
      return FALSE;

   if (pScrn->numEntities != 1) {
      return FALSE;
   }

   /* Just use the current monitor specified in the
    * xorg.conf.  This really means little to us since
    * we have no choice over which monitor is used,
    * but X needs this to be set
    */

   pScrn->monitor = pScrn->confScreen->monitor;

   /* Allocate room for our private data */
   if (pScrn->driverPrivate == NULL)
      pScrn->driverPrivate = xnfcalloc(sizeof(MSMRec), 1);

   pMsm = MSMPTR(pScrn);

   if (pMsm == NULL) {
      ErrorF("Unable to allocate memory\n");
      return FALSE;
   }

   pEnt = xf86GetEntityInfo(pScrn->entityList[0]);

   /* Open the FB device specified by the user */
   dev = xf86FindOptionValue(pEnt->device->options, "fb");

   pMsm->fd = open(dev, O_RDWR, 0);

   if (pMsm->fd < 0) {
      xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		 "Opening '%s' failed: %s\n", dev, strerror(errno));
      xfree(pMsm);
      return FALSE;
   }

   /* Unblank the screen if it was previously blanked */
   ioctl(pMsm->fd, FBIOBLANK, FB_BLANK_UNBLANK);

   /* Make sure the software refresher is on */
   ioctl(pMsm->fd, MSMFB_RESUME_SW_REFRESHER, 0);

   /* Get the fixed info (par) structure */

   if (ioctl(pMsm->fd, FBIOGET_FSCREENINFO, &pMsm->fixed_info)) {
      xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		 "Unable to read hardware info from %s: %s\n",
		 dev, strerror(errno));
      xfree(pMsm);
      return FALSE;
   }

   /* Parse the ID and figure out what version of the MDP and what
    * panel ID we have */

   if (sscanf(pMsm->fixed_info.id, "msmfb%d_%x", &mdpver, &panelid) < 2) {

      xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		 "Unable to determine the MDP and panel type\n");
      xfree(pMsm);
      return FALSE;
   }

   switch (mdpver) {
   case 22:
      pMsm->chipID = MSM_TYPE_7201;
      break;
   case 31:
      pMsm->chipID = MSM_TYPE_8X50;
      break;
   }

   /* FIXME:  If we want to parse the panel type, it happens here */

   /* Setup memory */

   /* FIXME:  This is where we will be in close communication with
    * the fbdev driver to allocate memory.   In the mean time, we
    * just reuse the framebuffer memory */

   pScrn->videoRam = pMsm->fixed_info.smem_len;

   /* Get the current screen setting */
   if (ioctl(pMsm->fd, FBIOGET_VSCREENINFO, &pMsm->mode_info)) {
      xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		 "Unable to read the current mode from %s: %s\n",
		 dev, strerror(errno));

      xfree(pMsm);
      return FALSE;
   }

   /* Fixme: 16bpp and 24bpp for now; other depths later    */
   /*        -- but only when they can be tested.           */
   /* (Tested on 16bpp Alaska and SURF and 24bpp ASUS CAT.) */
   if (pMsm->mode_info.bits_per_pixel != 16
       && pMsm->mode_info.bits_per_pixel != 24) {
      xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		 "The driver can only support 16bpp and 24bpp output\n");
      xfree(pMsm);
      return FALSE;
   }

   switch(pMsm->mode_info.bits_per_pixel) {
	case 16:
		depth = 16;
		fbbpp = 16;
		break;

	case 24:
		depth = 24;

		if (pMsm->mode_info.transp.offset == 24 &&
		    pMsm->mode_info.transp.length == 8)
			fbbpp = 32;
		else
			fbbpp = 24;
		break;
	default:
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
			"The driver can only support 16bpp and 24bpp output\n");
		xfree(pMsm);
		return FALSE;
   }

   if (!xf86SetDepthBpp(pScrn, depth, 16, fbbpp,
			Support24bppFb | Support32bppFb |
			SupportConvert32to24 | SupportConvert24to32)) {
      ErrorF("Unable to set bitdepth\n");
      xfree(pMsm);
      return FALSE;
   }

   xf86PrintDepthBpp(pScrn);
   pScrn->rgbBits = 8;

   if (!xf86SetWeight(pScrn, defaultWeight, defaultWeight)) {
      xfree(pMsm);
      return FALSE;
   }

   /* Initialize default visual */
   if (!xf86SetDefaultVisual(pScrn, -1)) {
      xfree(pMsm);
      return FALSE;
   }

   {
      Gamma zeros = { 0.0, 0.0, 0.0 };

      if (!xf86SetGamma(pScrn, zeros)) {
         xfree(pMsm);
	 return FALSE;
      }
   }

   pScrn->progClock = TRUE;
   pScrn->chipset = MSM_DRIVER_NAME;

   xf86DrvMsg(pScrn->scrnIndex, X_INFO, "MSM %s variant (video memory:"
	      " %dkB)\n", msmGetChipset(pMsm->chipID),
	      pScrn->videoRam / 1024);

   /* Default options with no conf settings */
   pMsm->xvports = 3;

   xf86CollectOptions(pScrn, NULL);

   /* We need to allocate this memory here because we have multiple
    * screens, and we can't go writing on the MSMOptions structure */

   options = xalloc(sizeof(MSMOptions));

   if (options == NULL) {
      xfree(pMsm);
      return FALSE;
   }

   memcpy(options, MSMOptions, sizeof(MSMOptions));

   xf86ProcessOptions(pScrn->scrnIndex, pScrn->options, options);

   /* NoAccel - default TRUE */
   pMsm->accel = xf86ReturnOptValBool(options, OPTION_NOACCEL, TRUE);

   /* SWBlit - default FALSE */
   pMsm->useSWBlit = xf86ReturnOptValBool(options, OPTION_SWBLIT, FALSE);

#if USEDRI2
   /* DRI2 - default TRUE */
   pMsm->useDRI2 = xf86ReturnOptValBool(options, OPTION_DRI2, TRUE);
#else
   pMsm->useDRI2 = FALSE;
#endif

   /* DRI - default FALSE */

   if (pMsm->useDRI2 == FALSE)
       pMsm->useDRI = xf86ReturnOptValBool(options, OPTION_DRI, FALSE);
   else
       pMsm->useDRI = FALSE;

   /* SWCursor - default FALSE */
   pMsm->HWCursor = !xf86ReturnOptValBool(options, OPTION_SWCURSOR, FALSE);

   /* DefaultVsync - default 60 */
   pMsm->defaultVsync = 60;

   if (xf86GetOptValInteger(options, OPTION_VSYNC, &vsync)) {
       if (vsync > 0 && vsync < 120)
	   pMsm->defaultVsync = vsync;
   }

   /* NoSigBlock - default TRUE */
   pMsm->NoSigBlock = xf86ReturnOptValBool(options, OPTION_NOSIGBLOCK, TRUE);

   /* SocketGID - default effective GID */
   pMsm->socketGID = getegid();

   gid = xf86GetOptValString(options, OPTION_SOCKGID);

   if (gid && _getgid(gid, &pMsm->socketGID))
       xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Socket GID %s is not valid\n",
		  gid);

   /* FastFill - default TRUE */
   pMsm->FastFill = xf86ReturnOptValBool(options, OPTION_FASTFILL, TRUE);

   /* FastComposite - default FALSE */
   pMsm->FastComposite = xf86ReturnOptValBool(options, OPTION_FASTCOMPOSITE, FALSE);

   /* FastCompositeRepeat - default FALSE */
   pMsm->FastCompositeRepeat = xf86ReturnOptValBool(options, OPTION_FASTCOMPOSITEREPEAT, FALSE);

   /* FastVideoMemCopy - default FALSE */
   pMsm->FastVideoMemCopy = xf86ReturnOptValBool(options, OPTION_FASTVIDEOMEMCOPY, FALSE);

   /* FastAppFBMemCopy - default FALSE */
   pMsm->FastAppFBMemCopy = xf86ReturnOptValBool(options, OPTION_FASTAPPFBMEMCOPY, FALSE);

   /* FBCache - default WriteThroughCache */
   pMsm->FBCache = MDP_FB_PAGE_PROTECTION_WRITETHROUGHCACHE;

   str = xf86GetOptValString(options, OPTION_FBCACHE);

   if (str) {
       int i;

       for(i = 0; fbCacheStrings[i] != NULL; i++) {
	   if (!stricmp(str, fbCacheStrings[i])) {
	       pMsm->FBCache = i;
	       break;
	   }
       }

       if (fbCacheStrings[i] == NULL)
	   xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Invalid FBCache '%s'\n",
		str);
   }

   /* PixmapMemtype - default KMEM */
   pMsm->pixmapMemtype = MSM_DRM_MEMTYPE_KMEM;

   str = xf86GetOptValString(options, OPTION_PIXMAP_MEMTYPE);

   if (str) {
       /* No for loop here because the memory types are masks, not indexes */

       if (!stricmp(str, "KMEM"))
	   pMsm->pixmapMemtype = MSM_DRM_MEMTYPE_KMEM;
       else if (!stricmp(str, "UncachedKMEM"))
	   pMsm->pixmapMemtype = MSM_DRM_MEMTYPE_KMEM_NOCACHE;
       else if (!stricmp(str, "EBI"))
	   pMsm->pixmapMemtype = MSM_DRM_MEMTYPE_EBI;
       else if (!stricmp(str, "SMI"))
	   pMsm->pixmapMemtype = MSM_DRM_MEMTYPE_SMI;
       else
	   xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		      "Invalid pixmap memory type %s\n", str);
   }

   /* Other drivers will keep this copy of the options in the private rec.
    * I don't see any reason to do that unless we have other functions
    * that need it */

   xfree(options);

#if USEDRI2
   if (pMsm->useDRI2 && !MSMInitDRM(pScrn)) {
     xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "DRI2: Unable to open DRM\n");
     pMsm->useDRI2 = FALSE;
   }
#endif

   /* Set up the binder socket */
   MSMBinderInit(pMsm);

   /* Set up the virtual size */

   pScrn->virtualX = pScrn->display->virtualX > pMsm->mode_info.xres ?
	 pScrn->display->virtualX : pMsm->mode_info.xres;

   pScrn->virtualY = pScrn->display->virtualY > pMsm->mode_info.yres ?
	 pScrn->display->virtualY : pMsm->mode_info.yres;

   if (pScrn->virtualX > pMsm->mode_info.xres_virtual)
      pScrn->virtualX = pMsm->mode_info.xres_virtual;

   if (pScrn->virtualY > pMsm->mode_info.yres_virtual)
      pScrn->virtualY = pMsm->mode_info.yres_virtual;

   /* displayWidth is the width of the line in pixels */

   /* The framebuffer driver should always report the line length,
    * but in case it doesn't, we can calculate it ourselves */

   if (pMsm->fixed_info.line_length) {
      pScrn->displayWidth = pMsm->fixed_info.line_length;
   } else {
      pScrn->displayWidth = pMsm->mode_info.xres_virtual *
	    pMsm->mode_info.bits_per_pixel / 8;
   }

   pScrn->displayWidth /= (pScrn->bitsPerPixel / 8);

   /* Set up the view port */
   pScrn->frameX0 = 0;
   pScrn->frameY0 = 0;
   pScrn->frameX1 = pMsm->mode_info.xres;
   pScrn->frameY1 = pMsm->mode_info.yres;

   MSMGetDefaultMode(pMsm);

   /* Make a copy of the mode - this is important, because some
    * where in the RandR setup, these modes get deleted */

   pScrn->modes = xf86DuplicateMode(&pMsm->default_mode);
   pScrn->currentMode = pScrn->modes;

   /* Set up the colors - this is from fbdevhw, which implies
    * that it is important for TrueColor and DirectColor modes
    */

   pScrn->offset.red = pMsm->mode_info.red.offset;
   pScrn->offset.green = pMsm->mode_info.green.offset;
   pScrn->offset.blue = pMsm->mode_info.blue.offset;

   pScrn->mask.red = ((1 << pMsm->mode_info.red.length) - 1)
	 << pMsm->mode_info.red.offset;

   pScrn->mask.green = ((1 << pMsm->mode_info.green.length) - 1)
	 << pMsm->mode_info.green.offset;

   pScrn->mask.blue = ((1 << pMsm->mode_info.blue.length) - 1)
	 << pMsm->mode_info.blue.offset;

   xf86CrtcConfigInit(pScrn, &MSMCrtcConfigFuncs);
   MSMCrtcSetup(pScrn);

   xf86CrtcSetSizeRange(pScrn,
			pMsm->mode_info.xres, pMsm->mode_info.yres,
			pMsm->mode_info.xres, pMsm->mode_info.yres);

   /* Setup the output */
   MSMOutputSetup(pScrn);

   if (!xf86InitialConfiguration(pScrn, FALSE)) {
      xfree(pMsm);
      return FALSE;
   }

   xf86PrintModes(pScrn);

   /* FIXME:  We will probably need to be more exact when setting
    * the DPI.  For now, we just use the default (96,96 I think) */

   xf86SetDpi(pScrn, 0, 0);

   xf86DrvMsg(pScrn->scrnIndex, X_INFO, "MSM Options:\n");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " HW Accel: %s\n",
	      pMsm->accel ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " SW Blit: %s\n",
	      pMsm->useSWBlit ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " DRI: %s\n",
	      pMsm->useDRI ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " DRI2: %s\n",
	      pMsm->useDRI2 ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " HW Cursor: %s\n",
	      pMsm->HWCursor ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " Default Vsync: %d\n",
	      pMsm->defaultVsync);
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " NoSigBlock: %s\n",
	      pMsm->NoSigBlock ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " FastFill: %s\n",
	      pMsm->FastFill ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " FastComposite: %s\n",
	      pMsm->FastComposite ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " FastCompositeRepeat: %s\n",
	      pMsm->FastCompositeRepeat ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " FastVideoMemCopy: %s\n",
	      pMsm->FastVideoMemCopy ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " FastAppFBMemCopy: %s\n",
	      pMsm->FastAppFBMemCopy ? "Enabled" : "Disabled");
   xf86DrvMsg(pScrn->scrnIndex, X_INFO, " FBCache: %s\n",
	      fbCacheStrings[pMsm->FBCache]);

   switch(pMsm->pixmapMemtype) {
   case MSM_DRM_MEMTYPE_KMEM:
       xf86DrvMsg(pScrn->scrnIndex, X_INFO,
		  " Pixmap: KMEM\n");
       break;
   case MSM_DRM_MEMTYPE_KMEM_NOCACHE:
       xf86DrvMsg(pScrn->scrnIndex, X_INFO,
		  " Pixmap: Uncached KMEM\n");
       break;
   case MSM_DRM_MEMTYPE_EBI:
       xf86DrvMsg(pScrn->scrnIndex, X_INFO,
		  " Pixmap: EBI\n");
       break;
   case MSM_DRM_MEMTYPE_SMI:
       xf86DrvMsg(pScrn->scrnIndex, X_INFO,
		  " Pixmap: SMI\n");
       break;
   }

   return TRUE;
}

static Bool
MSMSaveScreen(ScreenPtr pScreen, int mode)
{
   /* Nothing to do here, yet */
   return TRUE;
}

static Bool
MSMCloseScreen(int scrnIndex, ScreenPtr pScreen)
{
   ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];

   MSMPtr pMsm = MSMPTR(pScrn);

   /* Close EXA */
   if (pMsm->accel && pMsm->pExa) {
      exaDriverFini(pScreen);
      xfree(pMsm->pExa);
      pMsm->pExa = NULL;
   }

   /* Close DRI2 */
#if USEDRI2
   if (pMsm->useDRI2)
	MSMDRI2ScreenClose(pScreen);
#endif

   /* Unmap the framebuffer memory */
   munmap(pMsm->fbmem, pMsm->fixed_info.smem_len);

   pScreen->CloseScreen = pMsm->CloseScreen;

   return (*pScreen->CloseScreen) (scrnIndex, pScreen);
}

static Bool
MSMScreenInit(int scrnIndex, ScreenPtr pScreen, int argc, char **argv)
{
   ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];

   MSMPtr pMsm = MSMPTR(pScrn);

#if defined (MSMFB_GET_PAGE_PROTECTION) && defined (MSMFB_SET_PAGE_PROTECTION)
   /* If the frame buffer can be cached, do so.                                      */
   /* CAUTION: This needs to be done *BEFORE* the mmap() call, or it has no effect.  */
   /* FIXME:  The current page protection should ideally be saved here and restored  */
   /*         when exiting the driver, but there may be little point in doing this   */
   /*         since the XServer typically won't exit for most applications.          */
   {
      const int desired_fb_page_protection = pMsm->FBCache;
      struct mdp_page_protection fb_page_protection;

      // If the kernel supports the FB Caching settings infrastructure,
      // then set the frame buffer cache settings.
      // Otherwise, issue a warning and continue.
      if (ioctl(pMsm->fd, MSMFB_GET_PAGE_PROTECTION, &fb_page_protection)) {
               xf86DrvMsg(scrnIndex, X_WARNING,
                          "MSMFB_GET_PAGE_PROTECTION IOCTL: Unable to get current FB cache settings.\n");
      }
      else {
         if (fb_page_protection.page_protection != desired_fb_page_protection) {
            fb_page_protection.page_protection = desired_fb_page_protection;
            if (ioctl(pMsm->fd, MSMFB_SET_PAGE_PROTECTION, &fb_page_protection)) {
               xf86DrvMsg(scrnIndex, X_ERROR,
                          "MSMFB_SET_PAGE_PROTECTION IOCTL: Unable to set requested FB cache settings: %s.\n",
                          fbCacheStrings[desired_fb_page_protection]);
              return FALSE;
            }
         }
      }
   }
#endif // defined (MSMFB_GET_PAGE_PROTECTION) && defined (MSMFB_SET_PAGE_PROTECTION)

   /* Map the framebuffer memory */
   pMsm->fbmem = mmap(NULL, pMsm->fixed_info.smem_len,
		      PROT_READ | PROT_WRITE, MAP_SHARED, pMsm->fd, 0);

   /* If we can't map the memory, then this is a short trip */

   if (pMsm->fbmem == MAP_FAILED) {
      xf86DrvMsg(scrnIndex, X_ERROR, "Unable to map the "
		 "framebuffer memory: %s\n", strerror(errno));
      return FALSE;
   }

   /* Set up the mode - this doesn't actually touch the hardware,
    * but it makes RandR all happy */

   if (!xf86SetDesiredModes(pScrn)) {
      xf86DrvMsg(scrnIndex, X_ERROR, "Unable to set the mode");
      return FALSE;
   }

   /* Set up the X visuals */
   miClearVisualTypes();

   /* We only support TrueColor at the moment, and I suspect that is all
    * we will ever support */

   if (!miSetVisualTypes(pScrn->depth, TrueColorMask,
			 pScrn->rgbBits, TrueColor)) {
      xf86DrvMsg(scrnIndex, X_ERROR, "Unable to set up the visual"
		 " for %d BPP\n", pScrn->bitsPerPixel);
      return FALSE;
   }

   if (!miSetPixmapDepths()) {
      xf86DrvMsg(scrnIndex, X_ERROR, "Unable to set the pixmap depth\n");
      return FALSE;
   }

#if USEDRI2
   if (pMsm->useDRI2)
     pMsm->useDRI2 = MSMDRI2ScreenInit(pScreen);
#endif

#if USEDRI
   pMsm->DRIEnabled = FALSE;
   if (!pMsm->useDRI2 && pMsm->useDRI) {
     pMsm->dri = xcalloc(1, sizeof(struct msm_dri));
     pMsm->dri->depthBits = pScrn->depth;
     pMsm->DRIEnabled = MSMDRIScreenInit(pScreen);
   }
#endif

   /* Set up the X drawing area */

   xf86LoadSubModule(pScrn, "fb");

   if (!fbScreenInit(pScreen, pMsm->fbmem,
		     pScrn->virtualX, pScrn->virtualY,
		     pScrn->xDpi, pScrn->yDpi,
		     pScrn->displayWidth, pScrn->bitsPerPixel)) {

      xf86DrvMsg(scrnIndex, X_ERROR, "fbScreenInit failed\n");
      return FALSE;
   }

   /* Set up the color information for the visual(s) */

   if (pScrn->bitsPerPixel > 8) {
      VisualPtr visual = pScreen->visuals + pScreen->numVisuals;

      while (--visual >= pScreen->visuals) {
	 if ((visual->class | DynamicClass) == DirectColor) {
	    visual->offsetRed = pScrn->offset.red;
	    visual->offsetGreen = pScrn->offset.green;
	    visual->offsetBlue = pScrn->offset.blue;
	    visual->redMask = pScrn->mask.red;
	    visual->greenMask = pScrn->mask.green;
	    visual->blueMask = pScrn->mask.blue;
	 }
      }
   }

   /* Set up the Render fallbacks */
   if (!fbPictureInit(pScreen, NULL, 0)) {
      xf86DrvMsg(scrnIndex, X_ERROR, "fbPictureInit failed\n");
      return FALSE;
   }

   /* Set default colors */
   xf86SetBlackWhitePixels(pScreen);

   /* Set up the backing store */
   miInitializeBackingStore(pScreen);
   xf86SetBackingStore(pScreen);

   if (pMsm->accel) {
      /* Set up EXA */
      xf86LoadSubModule(pScrn, "exa");

      if (!MSMSetupExa(pScreen))
		ErrorF("Unable to setup EXA\n");
   }

   /* Set up the software cursor */
   miDCInitialize(pScreen, xf86GetPointerScreenFuncs());

   /* Try to set up the HW cursor */

   if (pMsm->HWCursor == TRUE)
      pMsm->HWCursor = MSMCursorInit(pScreen);

   /* Set up the default colormap */

   if (!miCreateDefColormap(pScreen)) {
      xf86DrvMsg(scrnIndex, X_ERROR, "miCreateDefColormap failed\n");
      return FALSE;
   }

#if USEDRI
   if (pMsm->DRIEnabled)
     MSMDRIFinishScreenInit(pScreen);
#endif

   /* Set up Xv */
   MSMInitVideo(pScreen);

   /* FIXME: Set up DPMS here */

   pScreen->SaveScreen = MSMSaveScreen;

   /* Set up our own CloseScreen function */

   pMsm->CloseScreen = pScreen->CloseScreen;
   pScreen->CloseScreen = MSMCloseScreen;

   if (!xf86CrtcScreenInit(pScreen)) {
        xf86DrvMsg(scrnIndex, X_ERROR, "CRTCScreenInit failed\n");
        return FALSE;
    }

   return TRUE;
}

static Bool
MSMSwitchMode(int scrnIndex, DisplayModePtr mode, int flags)
{
   /* FIXME:  We should only have the one mode, so we shouldn't ever call
    * this function - regardless, it needs to be stubbed - so what
    * do we return, TRUE or FALSE? */

   return TRUE;
}

static Bool
MSMEnterVT(int ScrnIndex, int flags)
{
   /* Nothing to do here yet - there might be some triggers that we need
    * to throw at the framebuffer */

   return TRUE;
}

static void
MSMLeaveVT(int ScrnIndex, int flags)
{
   /* Restore any framebufferish things here */
}

/* ------------------------------------------------------------ */
/* Following is the standard driver setup that probes for the   */
/* hardware and sets up the structures.                         */

static SymTabRec MSMChipsets[] = {
   {0, "MSM7201"},
   {1, "QSD8X50"},
   {-1, NULL}
};

static const OptionInfoRec *
MSMAvailableOptions(int chipid, int busid)
{
   return MSMOptions;
}

static void
MSMIdentify(int flags)
{
   xf86PrintChipsets(MSM_NAME, "Driver for Qualcomm MSM processors",
		     MSMChipsets);
}

static Bool
MSMProbe(DriverPtr drv, int flags)
{
   GDevPtr *sections;

   int nsects;

   char *dev;

   Bool foundScreen = FALSE;

   ScrnInfoPtr pScrn = NULL;

   int fd, i;

   /* For now, just return false during a probe */

   if (flags & PROBE_DETECT)
      return FALSE;

   /* Find all of the device sections in the config */

   nsects = xf86MatchDevice(MSM_NAME, &sections);
   if (nsects <= 0)
      return FALSE;

   /* We know that we will only have at most 4 possible outputs */

   for (i = 0; i < (nsects > 4 ? 4 : nsects); i++) {

      dev = xf86FindOptionValue(sections[i]->options, "fb");

      xf86Msg(X_WARNING, "Section %d - looking for %s\n", i, dev);

      /* FIXME:  There should be some discussion about how we
       * refer to devices - blindly matching to /dev/fbX files
       * seems like it could backfire on us.   For now, force
       * the user to set the backing FB in the xorg.conf */

      if (dev == NULL) {
	 xf86Msg(X_WARNING, "no device specified in section %d\n", i);
	 continue;
      }

      fd = open(dev, O_RDWR, 0);

      if (fd <= 0) {
	 xf86Msg(X_WARNING, "Could not open '%s': %s\n",
		 dev, strerror(errno));
	 continue;
      } else {
	 struct fb_fix_screeninfo info;

	 int entity;

	 if (ioctl(fd, FBIOGET_FSCREENINFO, &info)) {
	    xf86Msg(X_WARNING,
		    "Unable to read hardware info "
		    "from %s: %s\n", dev, strerror(errno));
	    close(fd);
	    continue;
	 }

	 close(fd);

	 /* Make sure that this is a MSM driver */
	 if (strncmp(info.id, "msmfb", 5)) {
	    xf86Msg(X_WARNING, "%s is not a MSM device: %s\n", dev, info.id);
	    continue;
	 }

	 foundScreen = TRUE;

	 entity = xf86ClaimFbSlot(drv, 0, sections[i], TRUE);
	 pScrn = xf86ConfigFbEntity(NULL, 0, entity, NULL, NULL, NULL, NULL);

	 xf86Msg(X_WARNING, "Add screen %p\n", pScrn);

	 /* Set up the hooks for the screen */

	 pScrn->driverVersion = MSM_VERSION_CURRENT;
	 pScrn->driverName = MSM_NAME;
	 pScrn->name = MSM_NAME;
	 pScrn->Probe = MSMProbe;
	 pScrn->PreInit = MSMPreInit;
	 pScrn->ScreenInit = MSMScreenInit;
	 pScrn->SwitchMode = MSMSwitchMode;
	 pScrn->EnterVT = MSMEnterVT;
	 pScrn->LeaveVT = MSMLeaveVT;
      }
   }

   xfree(sections);
   return foundScreen;
}

_X_EXPORT DriverRec msmDriver = {
   MSM_VERSION_CURRENT,
   MSM_DRIVER_NAME,
   MSMIdentify,
   MSMProbe,
   MSMAvailableOptions,
   NULL,
   0,
   NULL
};

MODULESETUPPROTO(msmSetup);

/* Versioning information for the module - most of these variables will
   come from config.h generated by ./configure
*/

static XF86ModuleVersionInfo msmVersRec = {
   MSM_DRIVER_NAME,
   MODULEVENDORSTRING,
   MODINFOSTRING1,
   MODINFOSTRING2,
   XORG_VERSION_CURRENT,
   MSM_VERSION_MAJOR, MSM_VERSION_MINOR, MSM_VERSION_PATCH,
   ABI_CLASS_VIDEODRV,
   ABI_VIDEODRV_VERSION,
   NULL,
   {0, 0, 0, 0},
};

_X_EXPORT XF86ModuleData msmModuleData = { &msmVersRec, msmSetup, NULL };

pointer
msmSetup(pointer module, pointer ops, int *errmaj, int *errmin)
{
   static Bool initDone = FALSE;

   if (initDone == FALSE) {
      initDone = TRUE;
      xf86AddDriver(&msmDriver, module, HaveDriverFuncs);

      /* FIXME: Load symbol references here */
      return (pointer) 1;
   } else {
      if (errmaj)
	 *errmaj = LDR_ONCEONLY;
      return NULL;
   }
}
