/*	$Id: xkbind.c,v 1.17 2005/05/25 07:41:24 chg Exp $

	XkbInd - X keyboard extension indicator.
	Copyright (C) 2004 CHG

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#if __minix
#define _POSIX_SOURCE
#endif

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <signal.h>
#include <time.h>

#if STDC_HEADERS || HAVE_STDLIB_H
# include <stdlib.h>
#endif

#if STDC_HEADERS
# include <string.h>
#endif

#if HAVE_STRINGS_H
# include <strings.h>
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xlibint.h>
#include <X11/Xutil.h>
#include <X11/XKBlib.h>
#include <X11/Xatom.h>
#include <X11/Xresource.h>
#include "common.h"

/* return type for ResValue function */
typedef enum {undef=-1, off, on} resvals;

typedef struct {
	char **items;
	int count;
	char *res_string;
} List;

const char *app_name=PACKAGE, *app_def_dir=APP_DEF_DIR, *user_rc_file=USER_RC_FILE;

Display *dpy;
Window root;
int def_group, xkb_event;
char *command, *def_label, *noname="";
char *group_labels[XKB_MAX_GROUP+1];
XkbEvent ev;
volatile sig_atomic_t term;

int(*DefErrHandler)(Display *, XErrorEvent *);

struct timespec delay = {0, 20000000};

int flags;

#define f_force (1L<<0)
#define f_reverse (1L<<1)
#define f_usewild (1L<<2)
#define f_nodetach (1L<<3)

List name_list, class_list;

char *atom_names[] = {"XKBIND_NAME", "XKBIND_GROUP", "COMPOUND_TEXT", "WM_TAKE_FOCUS"};
Atom atoms[sizeof(atom_names)/sizeof(char*)]; 

#define A_XKBIND_NAME atoms[0] 
#define A_XKBIND_GROUP atoms[1]
#define A_COMPOUND_TEXT atoms[2]
#define A_WM_TAKE_FOCUS atoms[3]

void QueryTree(Window, void(*)(Window));
void AddWindow(Window);
void DelWindow(Window);
void SetWM_NAME(Window, XTextProperty*, int group);
void SetWindow(Window, int group);
void ChangeWindow(Window, int group);
int GetWindowGroup(Window);
void CheckInstance(void);
Display *Initialize(int *p_argc, char **argv);
char *GetRes(XrmDatabase, const char *name1, const char *name2, const char *def_value);
void CleanUp(Bool full);
void Usage(void);
Bool IsInputWindow(Window);
Bool IgnoreWindow(Window);
void ParseList(List*);
void FreeList(List*);
int PatternMatch(const char *pattern, const char *string);
resvals ResValue(const char*);
void SigHandler(int); 
Bool Predicate(Display*, XEvent*, XPointer arg);
int ErrHandler(Display*, XErrorEvent*);
void SetSignals(void);
void Deamonize(void);

int main(int argc, char **argv)
{
	int revert_to, group;
	Window window;

	command = argv[0];
	SetSignals();
	DefErrHandler = XSetErrorHandler(ErrHandler);
	dpy = Initialize(&argc, argv);
	root = DefaultRootWindow(dpy);

	if(!XInternAtoms(dpy, atom_names, sizeof(atoms)/sizeof(Atom), False,
	 atoms)) {
		fprintf(stderr, "%s: XInternAtoms() error\n", command);
		return 1;
	}

	CheckInstance();
	if(!(flags&f_nodetach)) Deamonize();

	QueryTree(root, AddWindow);
	XGetInputFocus(dpy, &window, &revert_to);
	if(window != PointerRoot && window != None && (group=GetWindowGroup(window))!=-1)
		XkbLockGroup(dpy, XkbUseCoreKbd, group);

	/* for XkbStateNotify events */
	XkbSelectEventDetails(dpy, XkbUseCoreKbd, XkbStateNotify,
		XkbGroupLockMask, XkbGroupLockMask);
	/* for ReparentNotify events */
	XSelectInput(dpy, root, SubstructureNotifyMask);

	while(1) {
		if(term) {
			CleanUp(True);
#ifdef psignal
			psignal(term, command);
#endif
			exit(EXIT_SUCCESS);
		}

		if(!(XCheckMaskEvent(dpy, -1, (XEvent*)&ev) || 
		     XCheckTypedEvent(dpy, xkb_event, (XEvent*)&ev))) {
			nanosleep(&delay, NULL);
			continue;
		}

		switch(ev.type) {
			case ReparentNotify:
#ifdef DEBUG
				fprintf(stderr, "ReparentNotify\nwindow: %d\n\n",
					(int)ev.core.xreparent.window);
#endif
				AddWindow(ev.core.xreparent.window);
				break;
			/* case DestroyNotify:
				break; */
			case PropertyNotify:
				if(ev.core.xproperty.atom == XA_WM_NAME) {
					window = ev.core.xproperty.window;
#ifdef DEBUG
					fprintf(stderr, "PropertyNotify - WM_NAME\nwindow: %d\n\n",
					                                         (int)window);
#endif
					group = GetWindowGroup(window);
					if(group != -1) {

						/* when the type of WM_NAME property is changed
						   from STRING to COMPAUND_TEXT and vice versa
						   we receive more than one notification event
						   with state "PropertyNewValue" from the same
						   window in raw (for example, from Netscape
						   Navigator). remove them from the message
						   query before farther processing. */

						while(XCheckIfEvent(dpy, (XEvent*)&ev, Predicate,
						                                (XPointer)window));
						SetWindow(window, group);
					}
				}
				break;
			case FocusIn:
#ifdef DEBUG
				fprintf(stderr, "FocusIn\nmode: ");
				switch(ev.core.xfocus.mode) {
					case NotifyNormal:
						fprintf(stderr, "NotifyNormal\n");
						break;
					case NotifyGrab:
						fprintf(stderr, "NotifyGrab\n");
						break;
					case NotifyUngrab:
						fprintf(stderr, "NotifyUngrab\n");
						break;
					case NotifyWhileGrabbed:
						fprintf(stderr, "NotifyWhileGrabbed\n");
						break;
				}
				fprintf(stderr, "detail: ");
				switch(ev.core.xfocus.detail) {
					case NotifyAncestor:
						fprintf(stderr, "NotifyAncestor\n");
						break;
					case NotifyVirtual:
						fprintf(stderr, "NotifyVirtual\n");
						break;
					case NotifyInferior:
						fprintf(stderr, "NotifyInferior\n");
						break;
					case NotifyNonlinear:
						fprintf(stderr, "NotifyNonlinear\n");
						break;
					case NotifyNonlinearVirtual:
						fprintf(stderr, "NotifyNonlinearVirtual\n");
						break;
					case NotifyPointer:
						fprintf(stderr, "NotifyPointer\n");
						break;
				}
				fprintf(stderr, "window: %d\n\n", ev.core.xfocus.window);
#endif
				if(ev.core.xfocus.detail == NotifyNonlinear &&
				  (ev.core.xfocus.mode==NotifyNormal ||
				  ev.core.xfocus.mode==NotifyWhileGrabbed) &&
				  (group=GetWindowGroup(ev.core.xfocus.window))!=-1)
					XkbLockGroup(dpy, XkbUseCoreKbd, group);
				break;
			/* case FocusOut:
				break; */
			default: 
				if(ev.type == xkb_event && ev.any.xkb_type == XkbStateNotify) {
#ifdef DEBUG
					printf("XkbStateNotify\n"
						"event_type: %s\n"
						"group: %d\n"
						"locked_group: %d\n\n",
					 ev.state.event_type==KeyPress?"KeyPress":"KeyRelease",
					 ev.state.group, ev.state.locked_group);
#endif
					XGetInputFocus(dpy, &window, &revert_to);
					if(window != PointerRoot && window != None &&
					   (group=GetWindowGroup(window))!=-1 &&
					   group!=ev.state.locked_group)
						ChangeWindow(window, ev.state.locked_group);
				}
		}
	}

	/* XCloseDisplay(dpy); */
	return 0;
}

Bool Predicate(Display *dpy, XEvent *ev, XPointer arg)
{
	return (ev->type==PropertyNotify &&
			ev->xproperty.window==(Window)arg &&
			ev->xproperty.atom==XA_WM_NAME);
}

void QueryTree(Window win, void(*Action)(Window))
{
	Window rwin, parent, *childrens;
	unsigned int num, i=0;

	XQueryTree(dpy, win, &rwin, &parent, &childrens, &num);
	if(!childrens) return;
	while(num) {
		Action(childrens[i]);
		QueryTree(childrens[i], Action);
		num--; i++;
	}
	XFree(childrens);
}

void AddWindow(Window window)
{
	int group;

	if(!IsInputWindow(window) || IgnoreWindow(window)) return;

	if((group=GetWindowGroup(window)) == -1)
		SetWindow(window, def_group);
	else
		ChangeWindow(window, group);

	XSelectInput(dpy, window, FocusChangeMask|PropertyChangeMask);
}

void DelWindow(Window window)
{
	XTextProperty tp;

	if(XGetTextProperty(dpy, window, &tp, A_XKBIND_NAME)) {
		SetWM_NAME(window, &tp, -1);
		XDeleteProperty(dpy, window, A_XKBIND_NAME);
		XDeleteProperty(dpy, window, A_XKBIND_GROUP);
	}
}

void SetWM_NAME(Window window, XTextProperty *p_tp, int group)
{
	char *prefix, *s;
	XWindowAttributes wa;

	if(group >= 0) 
		prefix = (group<=XKB_MAX_GROUP) ? group_labels[group] : def_label;
	else
		prefix = "\0";

	p_tp->nitems += strlen(prefix);

	s = malloc(p_tp->nitems + 1);
	if(!s) {
		perror("malloc");
		return;
	}

	strcpy(s, prefix);
	strcat(s, (const char *)p_tp->value);
	p_tp->value = (unsigned char *)s;

	if(XGetWindowAttributes(dpy, window, &wa)) {
		XSelectInput(dpy, window, wa.your_event_mask&~PropertyChangeMask);
		XSync(dpy, False);
		XSetWMName(dpy, window, p_tp);
		XSelectInput(dpy, window, wa.your_event_mask);
	}

	free(s);
}

void ChangeWindow(Window window, int group)
{
	XTextProperty tp;

	XChangeProperty(dpy, window, A_XKBIND_GROUP, XA_INTEGER, 32,
		PropModeReplace, (unsigned char*)&group, 1);

	if(!XGetTextProperty(dpy, window, &tp, A_XKBIND_NAME)) {
		if(!XStringListToTextProperty(&noname, 1, &tp)) {
			fprintf(stderr, "%s: can't change window %d\n", command, (int)window);
			return;
		}
	}

	SetWM_NAME(window, &tp, group);
}

void SetWindow(Window window, int group)
{
	XTextProperty tp;

	XChangeProperty(dpy, window, A_XKBIND_GROUP, XA_INTEGER, 32,
	        PropModeReplace, (unsigned char *)&group, 1);

	/* using XGetWMName instead XFetchName allows properly process
	   properties with both COMPAUND_TEXT and STRING types */

	if(!XGetWMName(dpy, window, &tp)) {
		if(!XStringListToTextProperty(&noname, 1, &tp)) {
			fprintf(stderr, "%s: can't set window %d\n", command, (int)window);
			return;
		}
	}

	XSetTextProperty(dpy, window, &tp, A_XKBIND_NAME);
	SetWM_NAME(window, &tp, group);
}

/*
* returns xkb group bound to the window or -1 if property not found
*/
int GetWindowGroup(Window window)
{
	Atom actual_type;
	int actual_format, *data, group;
	unsigned long nitems, bytes_after;

	if(XGetWindowProperty(dpy, window, A_XKBIND_GROUP, 0, 1, False,
	 AnyPropertyType, &actual_type, &actual_format, &nitems,
	 &bytes_after, (unsigned char**)&data) == Success && actual_format) {
		group = *data;
		XFree(data);
		return group;
	} else 
		return -1; 
}

void CheckInstance()
{
	Atom actual_type;
	int actual_format;
	unsigned long nitems, bytes_after;
	unsigned char *data;

	XGrabServer(dpy);	/* check property atomically */
	XSync(dpy, False);
	if(!(flags&f_force) && XGetWindowProperty(dpy, root, A_XKBIND_GROUP, 0, 1, False,
	 AnyPropertyType, &actual_type, &actual_format, &nitems,
	 			&bytes_after, &data) == Success && actual_format) {
		XFree(data);
		fprintf(stderr, "%s is already running on display \"%s\"\n"
			"If you feel sure, try with -f option.\n",
			app_name, XDisplayString(dpy));
		CleanUp(False);
		exit(EXIT_FAILURE);
	}
	XChangeProperty(dpy, root, A_XKBIND_GROUP, XA_INTEGER, 32, PropModeReplace,
		(unsigned char*)&def_group, 1);
	XUngrabServer(dpy);
}

Display *Initialize(int *p_argc, char **argv)
{
	static XrmOptionDescRec opt_table[] = {
		{"-display",	".display",			XrmoptionSepArg,	NULL},
		{"-help",		".help",			XrmoptionIsArg,		NULL},
		{"-version",	".version",			XrmoptionIsArg,		NULL},
		{"-force",		".force",			XrmoptionIsArg,		NULL},
		{"-nodetach",	".nodetach",		XrmoptionIsArg,		NULL},
		{"-defgrp",		".defgroup",		XrmoptionSepArg,	NULL},
		{"-label0",		".label.group0",	XrmoptionSepArg,	NULL},
		{"-label1",		".label.group1",	XrmoptionSepArg,	NULL},
		{"-label2",		".label.group2",	XrmoptionSepArg,	NULL},
		{"-label3",		".label.group2",	XrmoptionSepArg,	NULL},
		{"-deflbl",		".label.default",	XrmoptionSepArg,	NULL},
		{"-iname",		".ignore.name",		XrmoptionSepArg,	NULL},
		{"-iclass",		".ignore.class",	XrmoptionSepArg,	NULL},
		{"-ireverse",	".ignore.reverse",	XrmoptionNoArg,		"on"},
		{"-inowild",	".ignore.usewild",	XrmoptionNoArg,		"off"}
	};
	XrmDatabase db=0;
	char *p_str, *p, *buf;
	Display *dpy;
	int xkb_error, reason, i;
	unsigned int l;

	XrmInitialize();
	XrmParseCommand(&db, opt_table, sizeof(opt_table)/sizeof(XrmOptionDescRec), app_name, p_argc, argv);

	/* following options can be specified only as command line arguments */

	p_str = GetRes(db, "help", NULL, "");
	i = (int)*p_str;
	free(p_str);
	if(i || *p_argc>1) {
		XrmDestroyDatabase(db);
		Usage();
	}

	p_str = GetRes(db, "version", NULL, "");
	i = (int)*p_str;
	free(p_str);
	if(i) {
		static char msg[] =
			"%s - X Keyboard Extention Indicator - Version %s\n"
			"Copyright (C) 2004 CHG <%s>\n\n"
			"This program is free software; you can redistribute it and/or modify\n"
			"it under the terms of the GNU General Public License as published by\n"
			"the Free Software Foundation; either version 2 of the License, or\n"
			"(at your option) any later version.\n";

		XrmDestroyDatabase(db);
		fprintf(stderr, msg, app_name, VERSION, EMAIL);
		exit(EXIT_FAILURE);
	}

	p_str = GetRes(db, "force", NULL, "");
	if(*p_str) flags |= f_force;
	free(p_str);

	p_str = GetRes(db, "nodetach", NULL, "");
	if(*p_str) flags |= f_nodetach;
	free(p_str);

	p_str = getenv("HOME");
	if(p_str) {
		l = strlen(p_str);

		/* two extra bytes for slash and zero */
		buf = malloc(l+strlen(user_rc_file)+2);
		if(buf) {
			strcpy(buf, p_str);
			if(!*(p=buf+l)) *p='/',*(p+1)='\0';
			strcat(buf, user_rc_file);
			XrmCombineFileDatabase(buf, &db, False);
			free(buf);
		} else {
			perror("malloc");
		}
	}

	/* following option can be specified either as command line arguments
	   or as user's private X resources */

	p_str = GetRes(db, "display", NULL, "");
	dpy = XkbOpenDisplay(p_str, &xkb_event, &xkb_error, NULL, NULL, &reason);
	if(!dpy) {
		fprintf(stderr, "%s: can't open display \"%s\"\n", command, XDisplayName(p_str));
		free(p_str);
		exit(EXIT_FAILURE);
	}
	free(p_str);

	p_str=XResourceManagerString(dpy);
	if(p_str) {
		XrmDatabase manager_db = XrmGetStringDatabase(p_str);
		XrmCombineDatabase(manager_db, &db, False);
	}

	l = strlen(app_def_dir);
	buf = malloc(l+strlen(app_name)+2);
	if(buf) {
		strcpy(buf, app_def_dir);
		if(!*(p=buf+l)) *p='/',*(p+1)='\0';
		l = strlen(buf);
		strcat(buf, app_name);
		buf[l] = toupper(buf[l]);
		XrmCombineFileDatabase(buf, &db, False);
		free(buf);
	} else {
		perror("malloc");
	}

	/* following option can be specified either as command line arguments,
	   user's private or global X resources */

	p_str = GetRes(db, "ignore", "reverse", "off");
	if(ResValue(p_str)==on) flags |= f_reverse;
	free(p_str);

	flags |= f_usewild;
	p_str = GetRes(db, "ignore", "usewild", "on");
	if(ResValue(p_str)==off) flags &= ~f_usewild;
	free(p_str);

	def_label = GetRes(db, "label", "default", "");
	for(i=0;i<=XKB_MAX_GROUP;i++) {
		char buf[32];

		snprintf(buf, sizeof(buf), "group%d", i);
		group_labels[i] = GetRes(db, "label", buf, def_label);
	}

	{
		XkbStateRec xkb_state;
		int grp;
		char buf[16];

		XkbGroupLock(&xkb_state) = 0;
		XkbGetState(dpy, XkbUseCoreKbd, &xkb_state);
		snprintf(buf, sizeof(buf), "%d", XkbGroupLock(&xkb_state));

		p_str = GetRes(db, "defgroup", NULL, buf);
		def_group = strtol(p_str, &p, 10);
		if(*p || def_group>XKB_MAX_GROUP)
			def_group=grp;
		free(p_str);
	}

	name_list.res_string = GetRes(db, "ignore", "name", "");
	ParseList(&name_list);
	class_list.res_string = GetRes(db, "ignore", "class", "");
	ParseList(&class_list);

	XrmDestroyDatabase(db);
	
		return dpy;
}

/*
* retrieve resource 'app_name.name1.name2' or 'app_name.name1' (if
* name2 = NULL) from the specified database. to free storage for
* returned value, use free().
*/
char *GetRes(XrmDatabase db, const char *name1, const char *name2, const char *def_value)
{
	char *str_type, *ptr, *buf, *class;
	XrmValue xv;
	int len;

	len = strlen(app_name) + strlen(name1) + 2;
	if(name2) len += strlen(name2) + 1;

	buf = malloc(len);
	class = malloc(len);
	if(buf && class) {
		if(!name2) sprintf(buf, "%s.%s", app_name, name1);
		else sprintf(buf, "%s.%s.%s", app_name, name1, name2);
		strcpy(class, buf);
		*class = toupper(*class);

		if(XrmGetResource(db, buf, class, &str_type, &xv) && (ptr = malloc(xv.size+1))) {
			memcpy(ptr, xv.addr, xv.size);
			*(ptr+xv.size) = '\0';
		} else if (ptr=malloc(strlen(def_value)+1))
			strcpy(ptr, def_value);

		free(buf);
		free(class);
	}

	if(!(ptr && buf && class)) {
		fprintf(stderr, "%s: memory allocation error.\n", command);
		exit(EXIT_FAILURE);
	}

	return ptr;
}

void CleanUp(Bool full)
{
	int i;

	if(root) {
		if(full) {
			QueryTree(root, DelWindow);
			XDeleteProperty(dpy, root, A_XKBIND_GROUP);
			XkbLockGroup(dpy, XkbUseCoreKbd, def_group);
		}
		XSync(dpy, False);
		XCloseDisplay(dpy);
	}

	for(i=0;i<=XKB_MAX_GROUP;i++)
		if(group_labels[i] != def_label) free(group_labels[i]);
	free(def_label);

	FreeList(&name_list);
	FreeList(&class_list);
}

void FreeList(List *list)
{
	free(list->items);
	free(list->res_string);
}

void Usage()
{
	static char msg[] = 
		"Usage: %s [-display <displayname>] [-force] [-nodetach]\n"
		"       [-defgrp <group>] [-deflbl <label>]\n"
		"       [-label0 <label>] [-label1 <label>]\n"
		"       [-label2 <label>] [-label3 <label>]\n"
		"       [-iname <name list>] [-iclass <class list>]\n"
		"       [-ireverse] [-inowild] | [-help] | [-version]\n"
		"\n"
		"Try \"man xkbind\" for more information.\n";

	fprintf(stderr, msg, command);
	exit(EXIT_FAILURE);
}

Bool IsInputWindow(Window window)
{
	XWMHints *hints;
	Atom *protocols;
	int count;
	Bool ret=False;

	hints = XGetWMHints(dpy, window);
	if(hints) {
		if(hints->flags&InputHint && hints->input) ret=True;
		XFree(hints);
	}

	if(!ret && XGetWMProtocols(dpy, window, &protocols, &count)) {
		int i;
		for(i=0;i<count;i++)
			if(protocols[i]==A_WM_TAKE_FOCUS) {
				ret=True;
				break;
			}
		XFree(protocols);
	}

	return ret;
}

Bool IgnoreWindow(Window window)
{
	XClassHint hints;

	if(XGetClassHint(dpy, window, &hints)) {
		Bool ret=False;
		int i;
		int (*Compare)(const char *, const char *) = (flags&f_usewild) ? PatternMatch:strcmp;

		if(name_list.items && name_list.count)
			for(i=0; i<name_list.count; i++)
				if(!Compare(name_list.items[i], hints.res_name)) {
					ret = True;
					break;
				}
		if(!ret && class_list.items && class_list.count)
			for(i=0; i<class_list.count; i++)
				if(!Compare(class_list.items[i], hints.res_class)) {
					ret = True;
					break;
				}
		XFree(hints.res_name);
		XFree(hints.res_class);

		return ((flags&f_reverse) ? !ret : ret);
	} else 
		return True;
}

void ParseList(List* list)
{
	Bool in_word, quoted;
	char *p, q, **items;
	int index;

	in_word = quoted = False;
	p = list->res_string;
	list->items = items = NULL;
	list->count = index = 0;

	while(*p) {
		if(*p=='"'||*p=='\'') {
			if(!quoted) {
				quoted = in_word = True;
				q = *p++;
				items = realloc(items, (index+1)*4);
				items[index++] = p;
			} else if(*p==q) {
				quoted = in_word = False;
				*p = '\0';
			}
		} else if(*p==' '||*p=='\t') {
			if(in_word) {
				in_word = False;
				*p = '\0';
			}
		} else if(!in_word) {
			in_word = True;
			items = realloc(items, (index+1)*4);
			items[index++] = p;
		}
		p++;
	}

	if(!quoted) {
		list->count = index;
		list->items = items;
	}
	else
		free(items); /* non-balanced quotes */
}

/* 
* test whether the string matches the pattern p.
* return zero if they do match, otherwise - nonzero.
*/
int PatternMatch(const char *p, const char *s)
{
	char *subp=NULL, *subs;
	int esc;

	while(*p&&*s) {
		if(esc=*p=='\\') p++;
		if(*p=='*'&&!esc) {
			if(!*++p) return 0; 
			subp = (char*)p;
			subs = (char*)s;
		} else if(*p=='?'&&!esc) {
			p++; s++;
			if(subp && !*p && *s) {
				p = subp;
				s = ++subs;
			}
		} else if(*p==*s) {
			p++; s++;
		} else if(subp) {
			p = subp;
			s = ++subs;
		} else return 1;
	}
	while(*p=='*') p++;
	return *p!=*s;
}

resvals ResValue(const char *s)
{
	int ret = undef;

	if(!strcasecmp(s, "on") || !strcasecmp(s, "yes") || !strcasecmp(s, "true"))
		ret = on;
	else if(!strcasecmp(s, "off") || !strcasecmp(s, "no") || !strcasecmp(s, "false"))
		ret = off;

	return ret;
}

void SigHandler(int sig_num)
{
	/* CleanUp(True);
	psignal(sig_num, command);
	exit(0); */
	term = sig_num;
}

int ErrHandler(Display* dpy, XErrorEvent* err_event)
{
	if(err_event->error_code == BadWindow) {
#ifdef DEBUG
		fprintf(stderr, "X Error: BadWindow\n\n");
#endif
		return 0;
	}

	/* we can't cleanup here. */
	/* SetErrorHandler(DefErrHandler);
	CleanUp(True); */

	DefErrHandler(dpy, err_event);
	return 0;
}

void SetSignals()
{
	struct sigaction sa;
	sigset_t sig_mask;

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = SIG_IGN;

	/* signals to ignore */
	sigaction(SIGALRM, &sa, NULL);
	sigaction(SIGVTALRM, &sa, NULL);
	sigaction(SIGPROF, &sa, NULL);
	sigaction(SIGHUP, &sa, NULL);

	sa.sa_handler = SigHandler;
	sigfillset(&sig_mask);	/* block all signals during handler execution */
	sa.sa_mask = sig_mask;

	/* signals to raise */
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);
	sigaction(SIGQUIT, &sa, NULL);
	sigaction(SIGABRT, &sa, NULL);
}

/* fork itself in background */
void Deamonize()
{
	int i;

	switch(fork()) {
		case -1: /* if can't fork, warn and continue in foreground */
			perror(command);
			return;
		case 0: /* deamonize child process */
			setsid();
			umask(0);
			chdir("/"); /* doesn't prevent unmounting of parent wd */
			for(i=2; i>=0; --i) close(i);
			if((i = open("/dev/null", O_RDWR)) != -1) { /* stdin */
				dup(i);                               /* stdout */
				dup(i);                               /* stderr */
			}
			return;
		default: /* exit parent process */
			exit(EXIT_SUCCESS);
	}
}
